#include  <stdio.h>
#include  <string.h>
#include  <time.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "xnor2.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "xnor2.c"                                           */


/* Module internal constants. */
#define        E_XNOR2_WIDTH         2
#define        E_XNOR2_SLICE         3
#define        E_XNOR2_DRIVE         4
#define        E_XNOR2_NOVIEW        5
#define        E_XNOR2_OPENVBE       6
#define        E_XNOR2_MODELNAME     7


/* Names of xnor2 leaf cells. */
static char *TablLeafCellXnor2[1] = {  "nxr2_fp" };


/* Interface function of module.
   extern       char *Dgn_DispachXnor2();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorXnor2();
static       void  Dgn_VhdlXnor2();
static       void  Dgn_NetlistXnor2();


/*  Function  :  'Dgn_DispatchXnor2'                               */
/* -------------------------------------------------------------- */
/*  Interface of xnor2 generator.                                  */

extern char *Dgn_DispatchXnor2(xnor2_modelname,
							  xnor2_width    ,
							  xnor2_slice    ,
							  xnor2_msb0     ,
							  xnor2_vhdl     ,
							  xnor2_netlist  ,
							  xnor2_icon     )
	char *xnor2_modelname;  /* Model name.                    */
	long  xnor2_width;      /* Bus wide.                      */
	long  xnor2_slice;      /* LSB starting slice.            */
	long  xnor2_msb0;       /* Zero as MSB index.             */
	long  xnor2_vhdl;       /* Generation of behavioral view. */
	long  xnor2_netlist;    /* Generation of netlist view.    */
	long  xnor2_icon;       /* Generation of icon.            */
{
	static short  DispachXnor2_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachXnor2_FirstCall)
	{
		Dgn_ErrorXnor2();
		DispachXnor2_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (xnor2_netlist == FALSE)
		 &&(xnor2_vhdl    == FALSE)
		 &&(xnor2_icon    == FALSE) )
		Dut_PrintError( E_XNOR2_ERROR, E_XNOR2_NOVIEW);

	/* Check the bus wide. */
	if ((xnor2_width < 1) || (xnor2_width > DP_WIDTH))
		Dut_PrintError( E_XNOR2_ERROR, E_XNOR2_WIDTH, xnor2_width);

	/* Check the LSB starting slice. */
	if ((xnor2_slice < 0) || ( xnor2_slice
							 +xnor2_width > DP_WIDTH))
		Dut_PrintError( E_XNOR2_ERROR, E_XNOR2_SLICE, xnor2_slice);

	/* Check the model name. */
	if (xnor2_modelname == (char*)NULL)
#ifndef  CRUNCH_FILENAME
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "xnor2_%ldx%ldx%ldx%ld%c_cl",
					                   DP_WIDTH              ,
					                 xnor2_width              ,
					                 xnor2_slice              ,
					               ((xnor2_msb0 ) ? 'm' : 'l'));
#else
		/* Build the crunched model name. */
		strcpy( ModelName, Dut_MakeCrunchModelName(
									    "od"       ,
								   8L,   DP_WIDTH-1,
								   8L, xnor2_width-1,
								   8L, xnor2_slice  ,
								   0L, xnor2_msb0   ));
#endif
	else
		strcpy( ModelName, xnor2_modelname);


	/* Generation of behaviorial view. */
	if (xnor2_vhdl)
		Dgn_VhdlXnor2( ModelName,
			   		  xnor2_width,
					  xnor2_slice,
			   		  xnor2_msb0 );

	/* Generation of netlist view. */
	if (xnor2_netlist)
		Dgn_NetlistXnor2( ModelName,
						 xnor2_width,
						 xnor2_slice,
						 xnor2_msb0 );

	/* Generation of icone. */
	/* if (xnor2_icon) */
	
	/* Return name of the generated model. */
	return((xnor2_modelname == (char*)NULL) ?      ModelName 
		                                   : xnor2_modelname);
}


/*  Function  :  'Dgn_ErrorXnor2'                                  */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorXnor2()
{
	Dut_AddError( E_XNOR2_ERROR, E_XNOR2_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_XNOR2_ERROR, E_XNOR2_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_XNOR2_ERROR, E_XNOR2_DRIVE,
				  "invalid output drive : %ld.");
	Dut_AddError( E_XNOR2_ERROR, E_XNOR2_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_XNOR2_ERROR, E_XNOR2_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_XNOR2_ERROR, E_XNOR2_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_NetListXnor2'                                */
/* -------------------------------------------------------------- */
/*  Generation of xnor2 behavioral view.                           */

static void  Dgn_VhdlXnor2(xnor2_modelname,
					      xnor2_width    ,
					      xnor2_slice    ,
						  xnor2_msb0     )
	char *xnor2_modelname;  /* Nom du modele.           */
	long  xnor2_width;      /* Largeur du bus.          */
	long  xnor2_slice;      /* LSB starting slice.      */
	long  xnor2_msb0;       /* Zero comme index du MSB. */
{
	FILE *Xnor2File;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((Xnor2File=mbkfopen(xnor2_modelname,
						            "vbe",
						       WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_XNOR2_ERROR, E_XNOR2_OPENVBE,
					                  xnor2_modelname);

	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( Xnor2File, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( Xnor2File, 
"-- #                                                                 #\n"
			);
	fprintf( Xnor2File,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( Xnor2File,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( Xnor2File,
"-- #                                                                 #\n"
			);
    fprintf( Xnor2File,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( Xnor2File,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( Xnor2File, 
"-- # *************************************************************** #\n"
			);
	fprintf( Xnor2File,
"-- #                                                                 #\n"
			);
    fprintf( Xnor2File,
"-- # Generator : \"xnor2g\" (xnor2)                                      #\n"
			);
	fprintf( Xnor2File, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	sprintf( TmpString, "\"%s.vbe\"."           , xnor2_modelname);
    fprintf( Xnor2File , "-- # File : %-55s  #\n",     TmpString);
	fprintf( Xnor2File ,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( Xnor2File , "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"xnor2g %ld %ld %ld%s -vhdl -o %s\"" ,
			                         DP_WIDTH                      ,
			                       xnor2_width                      ,
			                       xnor2_slice                      ,
			                      (xnor2_msb0     ) ? " -msb0"  : "",
			                       xnor2_modelname                  );
    fprintf( Xnor2File , "-- # Generator Call : %-45s  #\n", TmpString);

	fprintf( Xnor2File, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "xnor2_msb0")            */
	if (!xnor2_msb0)
		sprintf( BusWide, "(%ld downto %ld)", xnor2_width-1
				                             +xnor2_slice  ,
				                              xnor2_slice  );
	else
		sprintf( BusWide,     "(%ld to %ld)", DP_WIDTH-xnor2_slice
				                                      -xnor2_width  ,
				                              DP_WIDTH-xnor2_slice-1);


	/* *** Interface description. *** */
	fprintf( Xnor2File, "\nENTITY %s IS\n\tPORT(\n", xnor2_modelname);

	/* Terminals of data buses. */
	fprintf( Xnor2File, "%14s :  in BIT_VECTOR%s;\n", "i0", BusWide);
	fprintf( Xnor2File, "%14s :  in BIT_VECTOR%s;\n", "i1", BusWide);
	fprintf( Xnor2File, "%14s : out BIT_VECTOR%s;\n",  "o", BusWide);

	/* Supplies terminals. */
	fprintf( Xnor2File, "%14s :  in BIT;\n", "vdd");
	fprintf( Xnor2File, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( Xnor2File, "\t);\nEND %s;\n\n\n", xnor2_modelname);


	/* *** Behavioral view *** */
	fprintf( Xnor2File, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            xnor2_modelname);
	fprintf( Xnor2File, "BEGIN\n\n");
	fprintf( Xnor2File, "\to%s <= not (i0%s xor i1%s);\n\n" , BusWide,
			                                                  BusWide,
			                                                  BusWide);

	/* Check supplies connections. */
	fprintf( Xnor2File, "\t-- ****************>");
	fprintf( Xnor2File, " Supplies Check "      );
	fprintf( Xnor2File, "<****************\n\n" );

	fprintf( Xnor2File, "\tASSERT((vdd = '1') and (vss = '0'))\n");
	fprintf( Xnor2File, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              xnor2_modelname);
	fprintf( Xnor2File, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( Xnor2File, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( Xnor2File);
}


/*  Function  :  'Dgn_NetListXnor2'                                */
/* -------------------------------------------------------------- */
/*  Generation of xnor2erter/buffer netlist view.                  */

static void  Dgn_NetlistXnor2(xnor2_modelname,
						     xnor2_width    ,
						     xnor2_slice    ,
						     xnor2_msb0     )
	char *xnor2_modelname;  /* Model name.            */
	long  xnor2_width;      /* Bus wide.              */
	long  xnor2_slice;      /* LSB starting slice.    */
	long  xnor2_msb0;       /* Zero as MSB index.     */
{
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameI0[C_SUPP_MBKSZNAME];
	char  BusNameI1[C_SUPP_MBKSZNAME];
	char  BusNameO[ C_SUPP_MBKSZNAME];
	char  PinNameI0[C_SUPP_MBKSZNAME];
	char  PinNameI1[C_SUPP_MBKSZNAME];
	char  PinNameO[ C_SUPP_MBKSZNAME];


	/* Select name of the leaf cell model. */
	LeafCellSliceName = TablLeafCellXnor2[0];

	/* Creation of the new logical figure. */
	DEF_LOFIG( xnor2_modelname);

	/* Name of data bus terminals. */
	if (!xnor2_msb0)
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", xnor2_width-1+xnor2_slice, xnor2_slice);
		sprintf( BusNameI1,"i1[%ld:%ld]", xnor2_width-1+xnor2_slice, xnor2_slice);
		sprintf( BusNameO , "o[%ld:%ld]", xnor2_width-1+xnor2_slice, xnor2_slice);
	}
	else
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", DP_WIDTH-xnor2_slice
				                                  -xnor2_width  ,
				                          DP_WIDTH-xnor2_slice-1);
		sprintf( BusNameI1,"i1[%ld:%ld]", DP_WIDTH-xnor2_slice
				                                  -xnor2_width  ,
				                          DP_WIDTH-xnor2_slice-1);
		sprintf( BusNameO , "o[%ld:%ld]", DP_WIDTH-xnor2_slice
				                                  -xnor2_width  ,
				                          DP_WIDTH-xnor2_slice-1);
	}

 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON( BusNameI0, IN   , BusNameI0);
	LOCON( BusNameI1, IN   , BusNameI1);
	LOCON( BusNameO ,   OUT, BusNameO );

	/* **** Data bus terminals **** */
	LOCON(     "vdd", INOUT,     "vdd"); 
	LOCON(     "vss", INOUT,     "vss"); 
	
	
	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=xnor2_slice;
		 LeafCellIndex<xnor2_slice+xnor2_width; LeafCellIndex++)
	{
		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "xnor2_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!xnor2_msb0) BusIndex =              LeafCellIndex;
		else            BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameI0, "i0[%ld]", BusIndex);
		sprintf( PinNameI1, "i1[%ld]", BusIndex);
		sprintf( PinNameO ,  "o[%ld]", BusIndex);

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
			          PinNameI0 ,
			          PinNameI1 ,
			          PinNameO  , "vdd", "vss", EOL);
	}

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
