#include  <time.h>
#include  <stdio.h>
#include  <string.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "pl1.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "pl1.c"                                           */


/* Module internal constants. */
#define        E_PL1_WIDTH         2
#define        E_PL1_SLICE         3
#define        E_PL1_NOVIEW        4
#define        E_PL1_OPENVBE       5
#define        E_PL1_MODELNAME     6


/* Names of tristate leaf cells. */
static char *TablLeafCellPl1[2] = { "pl1b0_fp",
									"pl1a0_fp"};


/* Interface function of module.
   extern       char *Dgn_DispachPl1();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorPl1();
static       void  Dgn_VhdlPl1();
static       void  Dgn_NetlistPl1();


/*  Function  :  'Dgn_DispatchPl1'                                */
/* -------------------------------------------------------------- */
/*  Interface one input latch.                                    */

extern char *Dgn_DispatchPl1(pl1_modelname,
							  pl1_width    ,
							  pl1_slice    ,
							  pl1_msb0     ,
							  pl1_vhdl     ,
							  pl1_netlist  ,
							  pl1_icon     )
	char *pl1_modelname;  /* Model name.                    */
	long  pl1_width;      /* Bus wide.                      */
	long  pl1_slice;      /* LSB starting slice.            */
	long  pl1_msb0;       /* Zero as MSB index.             */
	long  pl1_vhdl;       /* Generation of behavioral view. */
	long  pl1_netlist;    /* Generation of netlist view.    */
	long  pl1_icon;       /* Generation of icon.            */
{
	static short  DispachPl1_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachPl1_FirstCall)
	{
		Dgn_ErrorPl1();
		DispachPl1_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (pl1_netlist == FALSE)
		 &&(pl1_vhdl    == FALSE)
		 &&(pl1_icon    == FALSE) )
		Dut_PrintError( E_PL1_ERROR, E_PL1_NOVIEW);

	/* Check the bus wide. */
	if ((pl1_width < 2) || (pl1_width > DP_WIDTH))
		Dut_PrintError( E_PL1_ERROR, E_PL1_WIDTH, pl1_width);

	/* Check the LSB starting slice. */
	if ((pl1_slice < 0) || ( pl1_slice
							 +pl1_width > DP_WIDTH))
		Dut_PrintError( E_PL1_ERROR, E_PL1_SLICE, pl1_slice);

	/* Check the model name. */
	if (pl1_modelname == (char*)NULL)
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "pl1_%ldx%ldx%ld%c_cl"  ,
					                   DP_WIDTH              ,
					                pl1_width              ,
					                pl1_slice              ,
					              ((pl1_msb0 ) ? 'm' : 'l'));
	else
		strcpy( ModelName, pl1_modelname);

	/* Generation of behaviorial view. */
	if (pl1_vhdl)
		Dgn_VhdlPl1(ModelName  ,
					 pl1_width,
					 pl1_slice,
					 pl1_msb0 );

	/* Generation of netlist view. */
	if (pl1_netlist)
		Dgn_NetlistPl1(ModelName  ,
						pl1_width,
						pl1_slice,
						pl1_msb0 );

	/* Generation of icon. */
	/* if (pl1_icon) */
	
	/* Return name of the generated model. */
	return((pl1_modelname == (char*)NULL) ?      ModelName 
		                                   : pl1_modelname);
}


/*  Function  :  'Dgn_ErrorPl1'                                  */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorPl1()
{
	Dut_AddError( E_PL1_ERROR, E_PL1_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_PL1_ERROR, E_PL1_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_PL1_ERROR, E_PL1_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_PL1_ERROR, E_PL1_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_PL1_ERROR, E_PL1_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_VhdlPl1'                                   */
/* -------------------------------------------------------------- */
/*  Generation scannable D flip-flops behavioral view.            */

static void  Dgn_VhdlPl1(pl1_modelname,
						  pl1_width    ,
						  pl1_slice    ,
						  pl1_msb0     )
	char *pl1_modelname;  /* Model name.         */
	long  pl1_width;      /* Bus width.          */
	long  pl1_slice;      /* LSB starting slice. */
	long  pl1_msb0;       /* Zero as MSB index.  */
{
	FILE *Pl1File;
	long  ConIndex;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[  256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((Pl1File=mbkfopen(pl1_modelname,
						              "vbe",
						         WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_PL1_ERROR, E_PL1_OPENVBE,
					                   pl1_modelname);


	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( Pl1File, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( Pl1File, 
"-- #                                                                 #\n"
			);
	fprintf( Pl1File,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( Pl1File,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( Pl1File,
"-- #                                                                 #\n"
			);
    fprintf( Pl1File,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( Pl1File,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( Pl1File, 
"-- # *************************************************************** #\n"
			);
	fprintf( Pl1File,
"-- #                                                                 #\n"
			);
    fprintf( Pl1File,
"-- # Generator : \"pl1g\" (Scannable D Flip-Flop)                      #\n"
			);
	fprintf( Pl1File, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	sprintf(  TmpString, "\"%s.vbe\"."            ,pl1_modelname);
    fprintf( Pl1File  , "-- # File : %-55s  #\n",      TmpString);
	fprintf( Pl1File,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( Pl1File  , "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"pl1g %ld %ld %ld%s -vhdl -o %s\"" ,
			                      DP_WIDTH                      ,
			                   pl1_width                      ,
			                   pl1_slice                      ,
			                  (pl1_msb0     ) ? " -msb0"  : "",
			                   pl1_modelname                  );
    fprintf( Pl1File  , "-- # Generator Call : %-45s  #\n", TmpString);

	fprintf( Pl1File, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "pl1_msb0")            */
	if (!pl1_msb0)
	{
		sprintf( BusWide ,  "(%ld downto %ld)", pl1_width-1
				                               +pl1_slice  ,
				                                pl1_slice  );
	}
	else
	{
		sprintf( BusWide  ,   "(%ld to %ld)", DP_WIDTH-pl1_slice
				                                      -pl1_width  ,
				                              DP_WIDTH-pl1_slice-1);
	}


	/* *** Interface description. *** */
	fprintf( Pl1File, "\nENTITY %s IS\n\tPORT(\n", pl1_modelname);

	/* Controls terminals. */
	fprintf( Pl1File, "%14s : in    BIT;\n",    "ck");

	/* Terminals of data pl1s. */
	fprintf( Pl1File, "%14s : in    BIT_VECTOR%s;\n",  "d", BusWide  );
	fprintf( Pl1File, "%14s :   out BIT_VECTOR%s;\n",  "q", BusWide );

	/* Supplies terminals. */
	fprintf( Pl1File, "%14s :  in BIT;\n", "vdd");
	fprintf( Pl1File, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( Pl1File, "\t);\nEND %s;\n\n\n", pl1_modelname);


	/* *** Behavioral view *** */
	fprintf( Pl1File, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            pl1_modelname);
	
	/* Internal register declaration. */
	for( BitIndex=pl1_slice; 
		 BitIndex<pl1_slice+pl1_width; BitIndex++)
	{
		fprintf( Pl1File, "\tSIGNAL inst_pl1_%ld_mem_s", BitIndex);
	    fprintf( Pl1File, ": REG_BIT REGISTER;\n"      );
	}

	fprintf( Pl1File, "\nBEGIN\n\n");

	/* *** Write Latch *** */
	fprintf( Pl1File, "\twl:BLOCK(ck) BEGIN\n");
	/* Loop on bus wide. */
	for( BitIndex=pl1_slice; 
		 BitIndex<pl1_slice+pl1_width; BitIndex++)
	{
		if (!pl1_msb0) ConIndex =              BitIndex;
		else           ConIndex = DP_WIDTH -1 -BitIndex;

		fprintf( Pl1File, "\t\tinst_pl1_%ld_mem_s <= GUARDED ", ConIndex);
		fprintf( Pl1File,            "d(%ld);\n"              , ConIndex);
	}
	fprintf( Pl1File, "\tEND BLOCK wl;\n\n");

	/* *** Affect the outputs. *** */

	/* Loop on bus wide. */
	for( BitIndex=pl1_slice; 
		 BitIndex<pl1_slice+pl1_width; BitIndex++)
	{
		if (!pl1_msb0) ConIndex =              BitIndex;
		else           ConIndex = DP_WIDTH -1 -BitIndex;

		fprintf( Pl1File, "\t q(%ld) <= inst_pl1_%ld_mem_s;\n",
				                                      ConIndex,
				                                      ConIndex);
	}


	/* Check supplies connections. */
	fprintf( Pl1File, "\n\t-- ****************>");
	fprintf( Pl1File,   " Supplies Check "      );
	fprintf( Pl1File,   "<****************\n\n" );

	fprintf( Pl1File, "\tASSERT ((vdd = '1') and (vss = '0'))\n");
	fprintf( Pl1File, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              pl1_modelname);
	fprintf( Pl1File, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( Pl1File, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( Pl1File);
}


/*  Fonction  :  'Dgn_NetListPl1'                                */
/* -------------------------------------------------------------- */
/*  Generation of Scannable D Flip-Flop Netlist.                  */

static void  Dgn_NetlistPl1(pl1_modelname,
							 pl1_width    ,
							 pl1_slice    ,
							 pl1_msb0     )
	char *pl1_modelname;  /* Model name.         */
	long  pl1_width;      /* Bus wide.           */
	long  pl1_slice;      /* LSB starting slice. */
	long  pl1_msb0;       /* Zero as MSB index.  */
{
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char *LeafCellBuffrName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameD[  C_SUPP_MBKSZNAME];
	char  BusNameQ[  C_SUPP_MBKSZNAME];
	char  PinNameD[  C_SUPP_MBKSZNAME];
	char  PinNameQ[  C_SUPP_MBKSZNAME];


	/* Model names (slices and buffer). */

	/* Creation of the new logical figure. */
	DEF_LOFIG( pl1_modelname);

	/* Name of data bus terminals. */
	if (!pl1_msb0)
	{
		sprintf( BusNameD , "d[%ld:%ld]", pl1_width-1
				                         +pl1_slice  , 
				                          pl1_slice  );
		sprintf( BusNameQ , "q[%ld:%ld]", pl1_width-1
				                         +pl1_slice  ,
				                          pl1_slice  );
	}
	else
	{
		sprintf( BusNameD , "d[%ld:%ld]", DP_WIDTH-pl1_slice
				                                  -pl1_width  ,
				                          DP_WIDTH-pl1_slice-1);
		sprintf( BusNameQ , "q[%ld:%ld]", DP_WIDTH-pl1_slice
				                                  -pl1_width  ,
				                          DP_WIDTH-pl1_slice-1);
	}
 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON(      "ck", IN   ,       "ck");

	/* **** Data bus terminals **** */
	LOCON( BusNameD , IN   , BusNameD );
	LOCON( BusNameQ ,   OUT, BusNameQ );

	/* **** Supply terminals **** */
	LOCON(     "vdd", INOUT,      "vdd"); 
	LOCON(     "vss", INOUT,      "vss"); 


	/* Instance of bottom column. */	
	LeafCellIndex=pl1_slice;

	/* Select leaf cell model name. */
	LeafCellSliceName = TablLeafCellPl1[1];

	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=pl1_slice; 
		 LeafCellIndex<pl1_slice+pl1_width; LeafCellIndex++)
	{
		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "pl1_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!pl1_msb0) BusIndex =              LeafCellIndex;
		else           BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameD  ,   "d[%ld]", BusIndex  );
		sprintf( PinNameQ  ,   "q[%ld]", BusIndex  );

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
			          PinNameD  ,
			               "ckl",
			          PinNameQ  , "vdd", "vss", EOL);
	}


	/* Creation of buffer instance. */

	/* Select buffer model name. */
	LeafCellBuffrName = TablLeafCellPl1[0];
	/* Build buffer instance name. */
	sprintf( LeafCellInstaName, "bpl1_%ld", DP_WIDTH);

	/* Buffer instanciation. */
	LOINS( LeafCellBuffrName,
		   LeafCellInstaName,
		               "ck" ,
		               "ckl", "vdd", "vss", "vss", EOL);

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
