#include  <stdio.h>
#include  <string.h>
#include  <time.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "or2.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "or2.c"                                            */


/* Module internal constants. */
#define        E_OR2_WIDTH         2
#define        E_OR2_SLICE         3
#define        E_OR2_DRIVE         4
#define        E_OR2_NOVIEW        5
#define        E_OR2_OPENVBE       6
#define        E_OR2_MODELNAME     7


/* Names of or2 leaf cells. */
static char *TablLeafCellOr2[4] = {  "o2_fp",
									"o2p_fp" };


/* Interface function of module.
   extern       char *Dgn_DispachOr2();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorOr2();
static       void  Dgn_VhdlOr2();
static       void  Dgn_NetlistOr2();


/*  Function  :  'Dgn_DispatchOr2'                               */
/* -------------------------------------------------------------- */
/*  Interface of or2 generator.                                  */

extern char *Dgn_DispatchOr2(or2_modelname,
							  or2_width    ,
							  or2_slice    ,
							  or2_drive    ,
							  or2_msb0     ,
							  or2_vhdl     ,
							  or2_netlist  ,
							  or2_icon     )
	char *or2_modelname;  /* Model name.                    */
	long  or2_width;      /* Bus wide.                      */
	long  or2_slice;      /* LSB starting slice.            */
	long  or2_drive;      /* Output drive capacity.         */
	long  or2_msb0;       /* Zero as MSB index.             */
	long  or2_vhdl;       /* Generation of behavioral view. */
	long  or2_netlist;    /* Generation of netlist view.    */
	long  or2_icon;       /* Generation of icon.            */
{
	static short  DispachOr2_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachOr2_FirstCall)
	{
		Dgn_ErrorOr2();
		DispachOr2_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (or2_netlist == FALSE)
		 &&(or2_vhdl    == FALSE)
		 &&(or2_icon    == FALSE) )
		Dut_PrintError( E_OR2_ERROR, E_OR2_NOVIEW);

	/* Check the bus wide. */
	if ((or2_width < 1) || (or2_width > DP_WIDTH))
		Dut_PrintError( E_OR2_ERROR, E_OR2_WIDTH, or2_width);

	/* Check the output drive capacity. */
	if ((or2_drive < 1) || (or2_drive > C_OR2_NBDRIVE))
		Dut_PrintError( E_OR2_ERROR, E_OR2_DRIVE, or2_drive);

	/* Check the LSB starting slice. */
	if ((or2_slice < 0) || ( or2_slice
							 +or2_width > DP_WIDTH))
		Dut_PrintError( E_OR2_ERROR, E_OR2_SLICE, or2_slice);

	/* Check the model name. */
	if (or2_modelname == (char*)NULL)
#ifndef  CRUNCH_FILENAME
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "or2x%ldx%ldx%ldx%ld%c_cl",
					                   DP_WIDTH              ,
					                 or2_width              ,
					                 or2_slice              ,
					                 or2_drive              ,
					               ((or2_msb0 ) ? 'm' : 'l'));
#else
		/* Build the crunched model name. */
		strcpy( ModelName, Dut_MakeCrunchModelName(
									    "od"       ,
								   8L,   DP_WIDTH-1,
								   8L, or2_width-1,
								   8L, or2_slice  ,
								   8L, or2_drive  ,
								   0L, or2_msb0   ));
#endif
	else
		strcpy( ModelName, or2_modelname);


	/* Generation of behaviorial view. */
	if (or2_vhdl)
		Dgn_VhdlOr2( ModelName,
			   		  or2_width,
					  or2_slice,
			   		  or2_drive,
			   		  or2_msb0 );

	/* Generation of netlist view. */
	if (or2_netlist)
		Dgn_NetlistOr2( ModelName,
						 or2_width,
						 or2_slice,
					     or2_drive,
						 or2_msb0 );

	/* Generation of icone. */
	/* if (or2_icon) */
	
	/* Return name of the generated model. */
	return((or2_modelname == (char*)NULL) ?      ModelName 
		                                   : or2_modelname);
}


/*  Function  :  'Dgn_ErrorOr2'                                  */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorOr2()
{
	Dut_AddError( E_OR2_ERROR, E_OR2_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_OR2_ERROR, E_OR2_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_OR2_ERROR, E_OR2_DRIVE,
				  "invalid output drive : %ld.");
	Dut_AddError( E_OR2_ERROR, E_OR2_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_OR2_ERROR, E_OR2_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_OR2_ERROR, E_OR2_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_NetListOr2'                                */
/* -------------------------------------------------------------- */
/*  Generation of or2 behavioral view.                           */

static void  Dgn_VhdlOr2(or2_modelname,
					      or2_width    ,
					      or2_slice    ,
					      or2_drive    ,
						  or2_msb0     )
	char *or2_modelname;  /* Nom du modele.           */
	long  or2_width;      /* Largeur du bus.          */
	long  or2_slice;      /* LSB starting slice.      */
	long  or2_drive;      /* Output drive power.      */
	long  or2_msb0;       /* Zero comme index du MSB. */
{
	FILE *Or2File;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((Or2File=mbkfopen(or2_modelname,
						            "vbe",
						       WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_OR2_ERROR, E_OR2_OPENVBE,
					                  or2_modelname);

	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( Or2File, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( Or2File, 
"-- #                                                                 #\n"
			);
	fprintf( Or2File,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( Or2File,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( Or2File,
"-- #                                                                 #\n"
			);
    fprintf( Or2File,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( Or2File,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( Or2File, 
"-- # *************************************************************** #\n"
			);
	fprintf( Or2File,
"-- #                                                                 #\n"
			);
    fprintf( Or2File,
"-- # Generator : \"or2g\" (or2)                                      #\n"
			);
	fprintf( Or2File, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	sprintf( TmpString, "\"%s.vbe\"."           , or2_modelname);
    fprintf( Or2File , "-- # File : %-55s  #\n",     TmpString);
	fprintf( Or2File ,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( Or2File , "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"or2g %ld %ld %ld %ld%s -vhdl -o %s\"" ,
			                        DP_WIDTH                      ,
			                       or2_width                      ,
			                       or2_slice                      ,
			                       or2_drive                      ,
			                      (or2_msb0     ) ? " -msb0"  : "",
			                       or2_modelname                  );
    fprintf( Or2File , "-- # Generator Call : %-45s  #\n", TmpString);

	fprintf( Or2File, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "or2_msb0")            */
	if (!or2_msb0)
		sprintf( BusWide, "(%ld downto %ld)", or2_width-1
				                             +or2_slice  ,
				                              or2_slice  );
	else
		sprintf( BusWide,     "(%ld to %ld)", DP_WIDTH-or2_slice
				                                      -or2_width  ,
				                              DP_WIDTH-or2_slice-1);


	/* *** Interface description. *** */
	fprintf( Or2File, "\nENTITY %s IS\n\tPORT(\n", or2_modelname);

	/* Terminals of data buses. */
	fprintf( Or2File, "%14s :in    BIT_VECTOR%s;\n", "i0", BusWide);
	fprintf( Or2File, "%14s :in    BIT_VECTOR%s;\n", "i1", BusWide);
	fprintf( Or2File, "%14s :inout BIT_VECTOR%s;\n", "no", BusWide);
	fprintf( Or2File, "%14s :  out BIT_VECTOR%s;\n",  "o", BusWide);

	/* Supplies terminals. */
	fprintf( Or2File, "%14s :  in BIT;\n", "vdd");
	fprintf( Or2File, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( Or2File, "\t);\nEND %s;\n\n\n", or2_modelname);


	/* *** Behavioral view *** */
	fprintf( Or2File, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            or2_modelname);
	fprintf( Or2File, "BEGIN\n\n");
	fprintf( Or2File, "\to%s <= not no%s;\n" , BusWide,
															BusWide);
	fprintf( Or2File, "\tno%s <= not (i0%s or i1%s);\n\n" , BusWide,
			                                                BusWide,
			                                                BusWide);

	/* Check supplies connections. */
	fprintf( Or2File, "\t-- ****************>");
	fprintf( Or2File, " Supplies Check "      );
	fprintf( Or2File, "<****************\n\n" );

	fprintf( Or2File, "\tASSERT((vdd = '1') and (vss = '0'))\n");
	fprintf( Or2File, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              or2_modelname);
	fprintf( Or2File, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( Or2File, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( Or2File);
}


/*  Function  :  'Dgn_NetListOr2'                                */
/* -------------------------------------------------------------- */
/*  Generation of or2erter/buffer netlist view.                  */

static void  Dgn_NetlistOr2(or2_modelname,
						     or2_width    ,
						     or2_slice    ,
							 or2_drive    ,
						     or2_msb0     )
	char *or2_modelname;  /* Model name.            */
	long  or2_width;      /* Bus wide.              */
	long  or2_slice;      /* LSB starting slice.    */
	long  or2_drive;      /* Output drive capacity. */
	long  or2_msb0;       /* Zero as MSB index.     */
{
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameI0[C_SUPP_MBKSZNAME];
	char  BusNameI1[C_SUPP_MBKSZNAME];
	char  BusNameNO[C_SUPP_MBKSZNAME];
	char  BusNameO[ C_SUPP_MBKSZNAME];
	char  PinNameI0[C_SUPP_MBKSZNAME];
	char  PinNameI1[C_SUPP_MBKSZNAME];
	char  PinNameNO[C_SUPP_MBKSZNAME];
	char  PinNameO[ C_SUPP_MBKSZNAME];


	/* Select name of the leaf cell model. */
	/* (according to output drive power)   */
	LeafCellSliceName = TablLeafCellOr2[or2_drive-1];

	/* Creation of the new logical figure. */
	DEF_LOFIG( or2_modelname);

	/* Name of data bus terminals. */
	if (!or2_msb0)
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", or2_width-1+or2_slice, or2_slice);
		sprintf( BusNameI1,"i1[%ld:%ld]", or2_width-1+or2_slice, or2_slice);
		sprintf( BusNameNO,"no[%ld:%ld]", or2_width-1+or2_slice, or2_slice);
		sprintf( BusNameO , "o[%ld:%ld]", or2_width-1+or2_slice, or2_slice);
	}
	else
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", DP_WIDTH-or2_slice
				                                  -or2_width  ,
				                          DP_WIDTH-or2_slice-1);
		sprintf( BusNameI1,"i1[%ld:%ld]", DP_WIDTH-or2_slice
				                                  -or2_width  ,
				                          DP_WIDTH-or2_slice-1);
		sprintf( BusNameNO,"no[%ld:%ld]", DP_WIDTH-or2_slice
				                                  -or2_width  ,
				                          DP_WIDTH-or2_slice-1);
		sprintf( BusNameO , "o[%ld:%ld]", DP_WIDTH-or2_slice
				                                  -or2_width  ,
				                          DP_WIDTH-or2_slice-1);
	}

 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON( BusNameI0, IN   , BusNameI0);
	LOCON( BusNameI1, IN   , BusNameI1);
	LOCON( BusNameNO, INOUT, BusNameNO);
	LOCON( BusNameO ,   OUT, BusNameO );

	/* **** Data bus terminals **** */
	LOCON(     "vdd", INOUT,     "vdd"); 
	LOCON(     "vss", INOUT,     "vss"); 
	
	
	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=or2_slice;
		 LeafCellIndex<or2_slice+or2_width; LeafCellIndex++)
	{
		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "or2_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!or2_msb0) BusIndex =              LeafCellIndex;
		else            BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameI0, "i0[%ld]", BusIndex);
		sprintf( PinNameI1, "i1[%ld]", BusIndex);
		sprintf( PinNameNO, "no[%ld]", BusIndex);
		sprintf( PinNameO ,  "o[%ld]", BusIndex);

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
			          PinNameI0 ,
			          PinNameI1 ,
			          PinNameNO ,
			          PinNameO  , "vdd", "vss", EOL);
	}

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
