#include  <stdio.h>
#include  <string.h>
#include  <time.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "nor2.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "nor2.c"                                           */


/* Module internal constants. */
#define        E_NOR2_WIDTH         2
#define        E_NOR2_SLICE         3
#define        E_NOR2_DRIVE         4
#define        E_NOR2_NOVIEW        5
#define        E_NOR2_OPENVBE       6
#define        E_NOR2_MODELNAME     7


/* Names of nor2 leaf cells. */
static char *TablLeafCellNor2[2] = {  "no2_fp",
									 "nop2_fp" };


/* Interface function of module.
   extern       char *Dgn_DispachNor2();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorNor2();
static       void  Dgn_VhdlNor2();
static       void  Dgn_NetlistNor2();


/*  Function  :  'Dgn_DispatchNor2'                               */
/* -------------------------------------------------------------- */
/*  Interface of nor2 generator.                                  */

extern char *Dgn_DispatchNor2(nor2_modelname,
							  nor2_width    ,
							  nor2_slice    ,
							  nor2_drive    ,
							  nor2_msb0     ,
							  nor2_vhdl     ,
							  nor2_netlist  ,
							  nor2_icon     )
	char *nor2_modelname;  /* Model name.                    */
	long  nor2_width;      /* Bus wide.                      */
	long  nor2_slice;      /* LSB starting slice.            */
	long  nor2_drive;      /* Output drive capacity.         */
	long  nor2_msb0;       /* Zero as MSB index.             */
	long  nor2_vhdl;       /* Generation of behavioral view. */
	long  nor2_netlist;    /* Generation of netlist view.    */
	long  nor2_icon;       /* Generation of icon.            */
{
	static short  DispachNor2_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachNor2_FirstCall)
	{
		Dgn_ErrorNor2();
		DispachNor2_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (nor2_netlist == FALSE)
		 &&(nor2_vhdl    == FALSE)
		 &&(nor2_icon    == FALSE) )
		Dut_PrintError( E_NOR2_ERROR, E_NOR2_NOVIEW);

	/* Check the bus wide. */
	if ((nor2_width < 1) || (nor2_width > DP_WIDTH))
		Dut_PrintError( E_NOR2_ERROR, E_NOR2_WIDTH, nor2_width);

	/* Check the output drive capacity. */
	if ((nor2_drive < 1) || (nor2_drive > C_NOR2_NBDRIVE))
		Dut_PrintError( E_NOR2_ERROR, E_NOR2_DRIVE, nor2_drive);

	/* Check the LSB starting slice. */
	if ((nor2_slice < 0) || ( nor2_slice
							 +nor2_width > DP_WIDTH))
		Dut_PrintError( E_NOR2_ERROR, E_NOR2_SLICE, nor2_slice);

	/* Check the model name. */
	if (nor2_modelname == (char*)NULL)
#ifndef  CRUNCH_FILENAME
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "nor2_%ldx%ldx%ldx%ld%c_cl",
					                    DP_WIDTH              ,
					                  nor2_width              ,
					                  nor2_slice              ,
					                  nor2_drive              ,
					                ((nor2_msb0 ) ? 'm' : 'l'));
#else
		/* Build the crunched model name. */
		strcpy( ModelName, Dut_MakeCrunchModelName(
									    "od"       ,
								   8L,   DP_WIDTH-1,
								   8L, nor2_width-1,
								   8L, nor2_slice  ,
								   8L, nor2_drive  ,
								   0L, nor2_msb0   ));
#endif
	else
		strcpy( ModelName, nor2_modelname);


	/* Generation of behaviorial view. */
	if (nor2_vhdl)
		Dgn_VhdlNor2( ModelName,
			   		  nor2_width,
					  nor2_slice,
			   		  nor2_drive,
			   		  nor2_msb0 );

	/* Generation of netlist view. */
	if (nor2_netlist)
		Dgn_NetlistNor2( ModelName,
						 nor2_width,
						 nor2_slice,
					     nor2_drive,
						 nor2_msb0 );

	/* Generation of icone. */
	/* if (nor2_icon) */
	
	/* Return name of the generated model. */
	return((nor2_modelname == (char*)NULL) ?      ModelName 
		                                   : nor2_modelname);
}


/*  Function  :  'Dgn_ErrorNor2'                                  */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorNor2()
{
	Dut_AddError( E_NOR2_ERROR, E_NOR2_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_NOR2_ERROR, E_NOR2_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_NOR2_ERROR, E_NOR2_DRIVE,
				  "invalid output drive : %ld.");
	Dut_AddError( E_NOR2_ERROR, E_NOR2_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_NOR2_ERROR, E_NOR2_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_NOR2_ERROR, E_NOR2_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_NetListNor2'                                */
/* -------------------------------------------------------------- */
/*  Generation of nor2 behavioral view.                           */

static void  Dgn_VhdlNor2(nor2_modelname,
					      nor2_width    ,
					      nor2_slice    ,
					      nor2_drive    ,
						  nor2_msb0     )
	char *nor2_modelname;  /* Nom du modele.           */
	long  nor2_width;      /* Largeur du bus.          */
	long  nor2_slice;      /* LSB starting slice.      */
	long  nor2_drive;      /* Output drive power.      */
	long  nor2_msb0;       /* Zero comme index du MSB. */
{
	FILE *Nor2File;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((Nor2File=mbkfopen(nor2_modelname,
						            "vbe",
						       WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_NOR2_ERROR, E_NOR2_OPENVBE,
					                  nor2_modelname);

	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( Nor2File, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( Nor2File, 
"-- #                                                                 #\n"
			);
	fprintf( Nor2File,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( Nor2File,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( Nor2File,
"-- #                                                                 #\n"
			);
    fprintf( Nor2File,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( Nor2File,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( Nor2File, 
"-- # *************************************************************** #\n"
			);
	fprintf( Nor2File,
"-- #                                                                 #\n"
			);
    fprintf( Nor2File,
"-- # Generator : \"nor2g\" (nor2)                                      #\n"
			);
	fprintf( Nor2File, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	sprintf( TmpString, "\"%s.vbe\"."           , nor2_modelname);
    fprintf( Nor2File , "-- # File : %-55s  #\n",     TmpString);
	fprintf( Nor2File ,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( Nor2File , "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"nor2g %ld %ld %ld %ld%s -vhdl -o %s\"" ,
			                         DP_WIDTH                      ,
			                       nor2_width                      ,
			                       nor2_slice                      ,
			                       nor2_drive                      ,
			                      (nor2_msb0     ) ? " -msb0"  : "",
			                       nor2_modelname                  );
    fprintf( Nor2File , "-- # Generator Call : %-45s  #\n", TmpString);

	fprintf( Nor2File, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "nor2_msb0")            */
	if (!nor2_msb0)
		sprintf( BusWide, "(%ld downto %ld)", nor2_width-1
				                             +nor2_slice  ,
				                              nor2_slice  );
	else
		sprintf( BusWide,     "(%ld to %ld)", DP_WIDTH-nor2_slice
				                                      -nor2_width  ,
				                              DP_WIDTH-nor2_slice-1);


	/* *** Interface description. *** */
	fprintf( Nor2File, "\nENTITY %s IS\n\tPORT(\n", nor2_modelname);

	/* Terminals of data buses. */
	fprintf( Nor2File, "%14s :  in BIT_VECTOR%s;\n", "i0", BusWide);
	fprintf( Nor2File, "%14s :  in BIT_VECTOR%s;\n", "i1", BusWide);
	fprintf( Nor2File, "%14s : out BIT_VECTOR%s;\n",  "o", BusWide);

	/* Supplies terminals. */
	fprintf( Nor2File, "%14s :  in BIT;\n", "vdd");
	fprintf( Nor2File, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( Nor2File, "\t);\nEND %s;\n\n\n", nor2_modelname);


	/* *** Behavioral view *** */
	fprintf( Nor2File, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            nor2_modelname);
	fprintf( Nor2File, "BEGIN\n\n");
	fprintf( Nor2File, "\to%s <= not (i0%s or i1%s);\n\n" , BusWide,
			                                                BusWide,
			                                                BusWide);

	/* Check supplies connections. */
	fprintf( Nor2File, "\t-- ****************>");
	fprintf( Nor2File, " Supplies Check "      );
	fprintf( Nor2File, "<****************\n\n" );

	fprintf( Nor2File, "\tASSERT((vdd = '1') and (vss = '0'))\n");
	fprintf( Nor2File, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              nor2_modelname);
	fprintf( Nor2File, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( Nor2File, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( Nor2File);
}


/*  Function  :  'Dgn_NetListNor2'                                */
/* -------------------------------------------------------------- */
/*  Generation of nor2erter/buffer netlist view.                  */

static void  Dgn_NetlistNor2(nor2_modelname,
						     nor2_width    ,
						     nor2_slice    ,
							 nor2_drive    ,
						     nor2_msb0     )
	char *nor2_modelname;  /* Model name.            */
	long  nor2_width;      /* Bus wide.              */
	long  nor2_slice;      /* LSB starting slice.    */
	long  nor2_drive;      /* Output drive capacity. */
	long  nor2_msb0;       /* Zero as MSB index.     */
{
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameI0[C_SUPP_MBKSZNAME];
	char  BusNameI1[C_SUPP_MBKSZNAME];
	char  BusNameO[ C_SUPP_MBKSZNAME];
	char  PinNameI0[C_SUPP_MBKSZNAME];
	char  PinNameI1[C_SUPP_MBKSZNAME];
	char  PinNameO[ C_SUPP_MBKSZNAME];


	/* Select name of the leaf cell model. */
	/* (according to output drive power)   */
	LeafCellSliceName = TablLeafCellNor2[nor2_drive-1];

	/* Creation of the new logical figure. */
	DEF_LOFIG( nor2_modelname);

	/* Name of data bus terminals. */
	if (!nor2_msb0)
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", nor2_width-1+nor2_slice, nor2_slice);
		sprintf( BusNameI1,"i1[%ld:%ld]", nor2_width-1+nor2_slice, nor2_slice);
		sprintf( BusNameO , "o[%ld:%ld]", nor2_width-1+nor2_slice, nor2_slice);
	}
	else
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", DP_WIDTH-nor2_slice
				                                  -nor2_width  ,
				                          DP_WIDTH-nor2_slice-1);
		sprintf( BusNameI1,"i1[%ld:%ld]", DP_WIDTH-nor2_slice
				                                  -nor2_width  ,
				                          DP_WIDTH-nor2_slice-1);
		sprintf( BusNameO , "o[%ld:%ld]", DP_WIDTH-nor2_slice
				                                  -nor2_width  ,
				                          DP_WIDTH-nor2_slice-1);
	}

 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON( BusNameI0, IN   , BusNameI0);
	LOCON( BusNameI1, IN   , BusNameI1);
	LOCON( BusNameO ,   OUT, BusNameO );

	/* **** Data bus terminals **** */
	LOCON(     "vdd", INOUT,     "vdd"); 
	LOCON(     "vss", INOUT,     "vss"); 
	
	
	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=nor2_slice;
		 LeafCellIndex<nor2_slice+nor2_width; LeafCellIndex++)
	{
		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "nor2_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!nor2_msb0) BusIndex =              LeafCellIndex;
		else            BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameI0, "i0[%ld]", BusIndex);
		sprintf( PinNameI1, "i1[%ld]", BusIndex);
		sprintf( PinNameO ,  "o[%ld]", BusIndex);

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
			          PinNameI0 ,
			          PinNameI1 ,
			          PinNameO  , "vdd", "vss", EOL);
	}

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
