#include  <time.h>
#include  <stdio.h>
#include  <string.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "nmux2cs.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "nmux2cs.c"                                        */


/* Module internal constants. */
#define        E_NMUX2CS_WIDTH         2
#define        E_NMUX2CS_SLICE         3
#define        E_NMUX2CS_NOVIEW        4
#define        E_NMUX2CS_OPENVBE       5
#define        E_NMUX2CS_MODELNAME     6


/* Names of tristate leaf cells. */
static char *TablLeafCellNmux2cs[2] = { "nmx2sb0_fp",
									     "nmx2sc_fp"};


/* Interface function of module.
   extern       char *Dgn_DispachNmux2cs();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorNmux2cs();
static       void  Dgn_VhdlNmux2cs();
static       void  Dgn_NetlistNmux2cs();


/*  Function  :  'Dgn_DispatchNmux2cs'                            */
/* -------------------------------------------------------------- */
/*  Interface of a two input multiplexer generator.               */

extern char *Dgn_DispatchNmux2cs(nmux2cs_modelname,
								 nmux2cs_width    ,
								 nmux2cs_slice    ,
								 nmux2cs_msb0     ,
								 nmux2cs_vhdl     ,
								 nmux2cs_netlist  ,
								 nmux2cs_icon     )
	char *nmux2cs_modelname;  /* Model name.                    */
	long  nmux2cs_width;      /* Bus wide.                      */
	long  nmux2cs_slice;      /* LSB starting slice.            */
	long  nmux2cs_msb0;       /* Zero as MSB index.             */
	long  nmux2cs_vhdl;       /* Generation of behavioral view. */
	long  nmux2cs_netlist;    /* Generation of netlist view.    */
	long  nmux2cs_icon;       /* Generation of icon.            */
{
	static short  DispachNmux2cs_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachNmux2cs_FirstCall)
	{
		Dgn_ErrorNmux2cs();
		DispachNmux2cs_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (nmux2cs_netlist == FALSE)
		 &&(nmux2cs_vhdl    == FALSE)
		 &&(nmux2cs_icon    == FALSE) )
		Dut_PrintError( E_NMUX2CS_ERROR, E_NMUX2CS_NOVIEW);

	/* Check the bus wide. */
	if ((nmux2cs_width < 1) || (nmux2cs_width > DP_WIDTH))
		Dut_PrintError( E_NMUX2CS_ERROR, E_NMUX2CS_WIDTH, nmux2cs_width);

	/* Check the LSB starting slice. */
	if ((nmux2cs_slice < 0) || ( nmux2cs_slice
							 +nmux2cs_width > DP_WIDTH))
		Dut_PrintError( E_NMUX2CS_ERROR, E_NMUX2CS_SLICE, nmux2cs_slice);

	/* Check the model name. */
	if (nmux2cs_modelname == (char*)NULL)
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "nmux2cs_%ldx%ldx%ld%c_cl"  ,
					                     DP_WIDTH              ,
					                nmux2cs_width              ,
					                nmux2cs_slice              ,
					              ((nmux2cs_msb0 ) ? 'm' : 'l'));
	else
		strcpy( ModelName, nmux2cs_modelname);

	/* Generation of behaviorial view. */
	if (nmux2cs_vhdl)
		Dgn_VhdlNmux2cs(ModelName    ,
						nmux2cs_width,
						nmux2cs_slice,
						nmux2cs_msb0 );

	/* Generation of netlist view. */
	if (nmux2cs_netlist)
		Dgn_NetlistNmux2cs(ModelName    ,
						   nmux2cs_width,
						   nmux2cs_slice,
						   nmux2cs_msb0 );

	/* Generation of icon. */
	/* if (nmux2cs_icon) */
	
	/* Return name of the generated model. */
	return((nmux2cs_modelname == (char*)NULL) ?         ModelName 
		                                      : nmux2cs_modelname);
}


/*  Function  :  'Dgn_ErrorNmux2cs'                               */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorNmux2cs()
{
	Dut_AddError( E_NMUX2CS_ERROR, E_NMUX2CS_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_NMUX2CS_ERROR, E_NMUX2CS_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_NMUX2CS_ERROR, E_NMUX2CS_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_NMUX2CS_ERROR, E_NMUX2CS_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_NMUX2CS_ERROR, E_NMUX2CS_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_NetListNmux2cs'                             */
/* -------------------------------------------------------------- */
/*  Generation of multiplexer behavioral view.                    */

static void  Dgn_VhdlNmux2cs(nmux2cs_modelname,
							 nmux2cs_width    ,
							 nmux2cs_slice    ,
							 nmux2cs_msb0     )
	char *nmux2cs_modelname;  /* Model name.         */
	long  nmux2cs_width;      /* Bus width.          */
	long  nmux2cs_slice;      /* LSB starting slice. */
	long  nmux2cs_msb0;       /* Zero as MSB index.  */
{
	FILE *Nmux2csFile;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((Nmux2csFile=mbkfopen(nmux2cs_modelname,
							              "vbe",
						             WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_NMUX2CS_ERROR, E_NMUX2CS_OPENVBE,
					                     nmux2cs_modelname);


	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( Nmux2csFile, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( Nmux2csFile, 
"-- #                                                                 #\n"
			);
	fprintf( Nmux2csFile,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( Nmux2csFile,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( Nmux2csFile,
"-- #                                                                 #\n"
			);
    fprintf( Nmux2csFile,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( Nmux2csFile,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( Nmux2csFile, 
"-- # *************************************************************** #\n"
			);
	fprintf( Nmux2csFile,
"-- #                                                                 #\n"
			);
    fprintf( Nmux2csFile,
"-- # Generator : \"nmux2csg\" (non-inverter tristate)                  #\n"
			);
	fprintf( Nmux2csFile, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	sprintf(  TmpString, "\"%s.vbe\"."           , nmux2cs_modelname);
    fprintf( Nmux2csFile  , "-- # File : %-55s  #\n",      TmpString);
	fprintf( Nmux2csFile,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( Nmux2csFile  , "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"nmux2csg %ld %ld %ld%s -vhdl -o %s\"" ,
			                      DP_WIDTH                        ,
			                 nmux2cs_width                        ,
			                 nmux2cs_slice                        ,
			                (nmux2cs_msb0     ) ? " -msb0"  : ""  ,
			                 nmux2cs_modelname                    );
    fprintf( Nmux2csFile  , "-- # Generator Call : %-45s  #\n", TmpString);

	fprintf( Nmux2csFile, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "nmux2cs_msb0")            */
	if (!nmux2cs_msb0)
		sprintf( BusWide, "(%ld downto %ld)", nmux2cs_width-1
				                             +nmux2cs_slice  ,
				                              nmux2cs_slice  );
	else
		sprintf( BusWide,     "(%ld to %ld)", DP_WIDTH-nmux2cs_slice
				                                      -nmux2cs_width  ,
				                              DP_WIDTH-nmux2cs_slice-1);


	/* *** Interface description. *** */
	fprintf( Nmux2csFile, "\nENTITY %s IS\n\tPORT(\n", nmux2cs_modelname);

	/* Controls terminals. */
	fprintf( Nmux2csFile, "%14s :  in BIT;\n",  "sel");

	/* Terminals of data nmux2css. */
	fprintf( Nmux2csFile, "%14s :  in BIT_VECTOR%s;\n", "i0", BusWide);
	fprintf( Nmux2csFile, "%14s :  in BIT_VECTOR%s;\n", "i1", BusWide);
	fprintf( Nmux2csFile, "%14s : out BIT_VECTOR%s;\n",  "o", BusWide);

	/* Supplies terminals. */
	fprintf( Nmux2csFile, "%14s :  in BIT;\n", "vdd");
	fprintf( Nmux2csFile, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( Nmux2csFile, "\t);\nEND %s;\n\n\n", nmux2cs_modelname);


	/* *** Behavioral view *** */
	fprintf( Nmux2csFile, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            nmux2cs_modelname);
	fprintf( Nmux2csFile, "BEGIN\n\n");

	fprintf( Nmux2csFile,     "\tWITH  sel  SELECT\n");
	fprintf( Nmux2csFile,   "\t\to%s <= \n"               , BusWide);
	fprintf( Nmux2csFile, "\t\t\tnot i0%s  WHEN \'0\',\n" , BusWide);
	fprintf( Nmux2csFile, "\t\t\tnot i1%s  WHEN \'1\';\n" , BusWide);

	/* Check supplies connections. */
	fprintf( Nmux2csFile, "\t-- ****************>");
	fprintf( Nmux2csFile, " Supplies Check "      );
	fprintf( Nmux2csFile, "<****************\n\n" );

	fprintf( Nmux2csFile, "\tASSERT ((vdd = '1') and (vss = '0'))\n");
	fprintf( Nmux2csFile, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              nmux2cs_modelname);
	fprintf( Nmux2csFile, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( Nmux2csFile, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( Nmux2csFile);
}


/*  Fonction  :  'Dgn_NetListNmux2cs'                                */
/* -------------------------------------------------------------- */
/*  Generation of Tristate Netlist.                               */

static void  Dgn_NetlistNmux2cs(nmux2cs_modelname,
						     nmux2cs_width    ,
							 nmux2cs_slice    ,
						     nmux2cs_msb0     )
	char *nmux2cs_modelname;  /* Model name.         */
	long  nmux2cs_width;      /* Bus wide.           */
	long  nmux2cs_slice;      /* LSB starting slice. */
	long  nmux2cs_msb0;       /* Zero as MSB index.  */
{
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char *LeafCellBuffrName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameI0[C_SUPP_MBKSZNAME];
	char  BusNameI1[C_SUPP_MBKSZNAME];
	char  BusNameO[ C_SUPP_MBKSZNAME];
	char  PinNameI0[C_SUPP_MBKSZNAME];
	char  PinNameI1[C_SUPP_MBKSZNAME];
	char  PinNameO[ C_SUPP_MBKSZNAME];


	/* Model names (slices and buffer). */
	LeafCellSliceName = TablLeafCellNmux2cs[1];
	LeafCellBuffrName = TablLeafCellNmux2cs[0];

	/* Creation of the new logical figure. */
	DEF_LOFIG( nmux2cs_modelname);

	/* Name of data bus terminals. */
	if (!nmux2cs_msb0)
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", nmux2cs_width-1
				                         +nmux2cs_slice  ,
				                          nmux2cs_slice  );
		sprintf( BusNameI1,"i1[%ld:%ld]", nmux2cs_width-1
				                         +nmux2cs_slice  ,
				                          nmux2cs_slice  );
		sprintf( BusNameO , "o[%ld:%ld]", nmux2cs_width-1
				                         +nmux2cs_slice  ,
				                          nmux2cs_slice  );
	}
	else
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", DP_WIDTH-nmux2cs_slice
				                                  -nmux2cs_width  ,
				                          DP_WIDTH-nmux2cs_slice-1);
		sprintf( BusNameI1,"i1[%ld:%ld]", DP_WIDTH-nmux2cs_slice
				                                  -nmux2cs_width  ,
				                          DP_WIDTH-nmux2cs_slice-1);
		sprintf( BusNameO , "o[%ld:%ld]", DP_WIDTH-nmux2cs_slice
				                                  -nmux2cs_width  ,
				                          DP_WIDTH-nmux2cs_slice-1);
	}
 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON(     "sel", IN   ,      "sel");

	/* **** Data bus terminals **** */
	LOCON( BusNameI0, IN   , BusNameI0);
	LOCON( BusNameI1, IN   , BusNameI1);
	LOCON( BusNameO ,   OUT, BusNameO );

	/* **** Supply terminals **** */
	LOCON(     "vdd", INOUT,      "vdd"); 
	LOCON(     "vss", INOUT,      "vss"); 
	
	
	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=nmux2cs_slice; 
		 LeafCellIndex<nmux2cs_slice+nmux2cs_width; LeafCellIndex++)
	{
		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "nmux2cs_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!nmux2cs_msb0) BusIndex =              LeafCellIndex;
		else               BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameI0, "i0[%ld]", BusIndex);
		sprintf( PinNameI1, "i1[%ld]", BusIndex);
		sprintf( PinNameO ,  "o[%ld]", BusIndex);

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
			                "s0",
			                "s1",
			          PinNameI0 ,
			          PinNameI1 ,
			          PinNameO  , "vdd", "vss", EOL);
	}


	/* Creation of buffer instance. */

	/* Build buffer instance name. */
	sprintf( LeafCellInstaName, "bnmux2cs_%ld", DP_WIDTH);

	/* Buffer instanciation. */
	LOINS( LeafCellBuffrName,
		   LeafCellInstaName,
		               "sel",
		                "s0",
		                "s1", "vdd", "vss", VSS, EOL);

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
