#include  <stdio.h>
#include  <string.h>
#include  <time.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "nand2.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "nand2.c"                                          */


/* Constants of module "nand2.c". */
#define        E_NAND2_WIDTH         2
#define        E_NAND2_SLICE         3
#define        E_NAND2_DRIVE         4
#define        E_NAND2_NOVIEW        5
#define        E_NAND2_OPENVBE       6
#define        E_NAND2_MODELNAME     7


/* Names of nand2 leaf cells. */
static char *TablLeafCellNand2[2] = {  "na2_fp",
									  "na2p_fp" };


/* Interface function of module.
   extern       char *Dgn_DispachNand2();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorNand2();
static       void  Dgn_VhdlNand2();
static       void  Dgn_NetlistNand2();


/*  Function  :  'Dgn_DispatchNand2'                               */
/* -------------------------------------------------------------- */
/*  Interface of nand2 generator.                                  */

extern char *Dgn_DispatchNand2(nand2_modelname,
							  nand2_width    ,
							  nand2_slice    ,
							  nand2_drive    ,
							  nand2_msb0     ,
							  nand2_vhdl     ,
							  nand2_netlist  ,
							  nand2_icon     )
	char *nand2_modelname;  /* Model name.                    */
	long  nand2_width;      /* Bus wide.                      */
	long  nand2_slice;      /* LSB starting slice.            */
	long  nand2_drive;      /* Output drive capacity.         */
	long  nand2_msb0;       /* Zero as MSB index.             */
	long  nand2_vhdl;       /* Generation of behavioral view. */
	long  nand2_netlist;    /* Generation of netlist view.    */
	long  nand2_icon;       /* Generation of icon.            */
{
	static short  DispachNand2_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachNand2_FirstCall)
	{
		Dgn_ErrorNand2();
		DispachNand2_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (nand2_netlist == FALSE)
		 &&(nand2_vhdl    == FALSE)
		 &&(nand2_icon    == FALSE) )
		Dut_PrintError( E_NAND2_ERROR, E_NAND2_NOVIEW);

	/* Check the bus wide. */
	if ((nand2_width < 1) || (nand2_width > DP_WIDTH))
		Dut_PrintError( E_NAND2_ERROR, E_NAND2_WIDTH, nand2_width);

	/* Check the output drive capacity. */
	if ((nand2_drive < 1) || (nand2_drive > C_NAND2_NBDRIVE))
		Dut_PrintError( E_NAND2_ERROR, E_NAND2_DRIVE, nand2_drive);

	/* Check the LSB starting slice. */
	if ((nand2_slice < 0) || ( nand2_slice
							 +nand2_width > DP_WIDTH))
		Dut_PrintError( E_NAND2_ERROR, E_NAND2_SLICE, nand2_slice);

	/* Check the model name. */
	if (nand2_modelname == (char*)NULL)
#ifndef  CRUNCH_FILENAME
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "nand2_%ldx%ldx%ldx%ld%c_cl",
					                   DP_WIDTH              ,
					                 nand2_width              ,
					                 nand2_slice              ,
					                 nand2_drive              ,
					               ((nand2_msb0 ) ? 'm' : 'l'));
#else
		/* Build the crunched model name. */
		strcpy( ModelName, Dut_MakeCrunchModelName(
									    "od"       ,
								   8L,   DP_WIDTH-1,
								   8L, nand2_width-1,
								   8L, nand2_slice  ,
								   8L, nand2_drive  ,
								   0L, nand2_msb0   ));
#endif
	else
		strcpy( ModelName, nand2_modelname);


	/* Generation of behaviorial view. */
	if (nand2_vhdl)
		Dgn_VhdlNand2( ModelName,
			   		  nand2_width,
					  nand2_slice,
			   		  nand2_drive,
			   		  nand2_msb0 );

	/* Generation of netlist view. */
	if (nand2_netlist)
		Dgn_NetlistNand2( ModelName,
						 nand2_width,
						 nand2_slice,
					     nand2_drive,
						 nand2_msb0 );

	/* Generation of icone. */
	/* if (nand2_icon) */
	
	/* Return name of the generated model. */
	return((nand2_modelname == (char*)NULL) ?      ModelName 
		                                   : nand2_modelname);
}


/*  Function  :  'Dgn_ErrorNand2'                                  */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorNand2()
{
	Dut_AddError( E_NAND2_ERROR, E_NAND2_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_NAND2_ERROR, E_NAND2_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_NAND2_ERROR, E_NAND2_DRIVE,
				  "invalid output drive : %ld.");
	Dut_AddError( E_NAND2_ERROR, E_NAND2_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_NAND2_ERROR, E_NAND2_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_NAND2_ERROR, E_NAND2_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_NetListNand2'                                */
/* -------------------------------------------------------------- */
/*  Generation of nand2 behavioral view.                           */

static void  Dgn_VhdlNand2(nand2_modelname,
					      nand2_width    ,
					      nand2_slice    ,
					      nand2_drive    ,
						  nand2_msb0     )
	char *nand2_modelname;  /* Nom du modele.           */
	long  nand2_width;      /* Largeur du bus.          */
	long  nand2_slice;      /* LSB starting slice.      */
	long  nand2_drive;      /* Output drive power.      */
	long  nand2_msb0;       /* Zero comme index du MSB. */
{
	FILE *Nand2File;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((Nand2File=mbkfopen(nand2_modelname,
						            "vbe",
						       WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_NAND2_ERROR, E_NAND2_OPENVBE,
					                  nand2_modelname);

	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( Nand2File, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( Nand2File, 
"-- #                                                                 #\n"
			);
	fprintf( Nand2File,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( Nand2File,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( Nand2File,
"-- #                                                                 #\n"
			);
    fprintf( Nand2File,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( Nand2File,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( Nand2File, 
"-- # *************************************************************** #\n"
			);
	fprintf( Nand2File,
"-- #                                                                 #\n"
			);
    fprintf( Nand2File,
"-- # Generator : \"nand2g\" (nand2)                                      #\n"
			);
	fprintf( Nand2File, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	sprintf( TmpString, "\"%s.vbe\"."           , nand2_modelname);
    fprintf( Nand2File , "-- # File : %-55s  #\n",     TmpString);
	fprintf( Nand2File ,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( Nand2File , "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"nand2g %ld %ld %ld %ld%s -vhdl -o %s\"" ,
			                         DP_WIDTH                      ,
			                       nand2_width                      ,
			                       nand2_slice                      ,
			                       nand2_drive                      ,
			                      (nand2_msb0     ) ? " -msb0"  : "",
			                       nand2_modelname                  );
    fprintf( Nand2File , "-- # Generator Call : %-45s  #\n", TmpString);

	fprintf( Nand2File, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "nand2_msb0")            */
	if (!nand2_msb0)
		sprintf( BusWide, "(%ld downto %ld)", nand2_width-1
				                             +nand2_slice  ,
				                              nand2_slice  );
	else
		sprintf( BusWide,     "(%ld to %ld)", DP_WIDTH-nand2_slice
				                                      -nand2_width  ,
				                              DP_WIDTH-nand2_slice-1);


	/* *** Interface description. *** */
	fprintf( Nand2File, "\nENTITY %s IS\n\tPORT(\n", nand2_modelname);

	/* Terminals of data buses. */
	fprintf( Nand2File, "%14s :  in BIT_VECTOR%s;\n", "i0", BusWide);
	fprintf( Nand2File, "%14s :  in BIT_VECTOR%s;\n", "i1", BusWide);
	fprintf( Nand2File, "%14s : out BIT_VECTOR%s;\n",  "o", BusWide);

	/* Supplies terminals. */
	fprintf( Nand2File, "%14s :  in BIT;\n", "vdd");
	fprintf( Nand2File, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( Nand2File, "\t);\nEND %s;\n\n\n", nand2_modelname);


	/* *** Behavioral view *** */
	fprintf( Nand2File, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            nand2_modelname);
	fprintf( Nand2File, "BEGIN\n\n");
	fprintf( Nand2File, "\to%s <= not (i0%s and i1%s);\n\n" , BusWide,
			                                                BusWide,
			                                                BusWide);

	/* Check supplies connections. */
	fprintf( Nand2File, "\t-- ****************>");
	fprintf( Nand2File, " Supplies Check "      );
	fprintf( Nand2File, "<****************\n\n" );

	fprintf( Nand2File, "\tASSERT((vdd = '1') and (vss = '0'))\n");
	fprintf( Nand2File, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              nand2_modelname);
	fprintf( Nand2File, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( Nand2File, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( Nand2File);
}


/*  Function  :  'Dgn_NetListNand2'                                */
/* -------------------------------------------------------------- */
/*  Generation of nand2erter/buffer netlist view.                  */

static void  Dgn_NetlistNand2(nand2_modelname,
						     nand2_width    ,
						     nand2_slice    ,
							 nand2_drive    ,
						     nand2_msb0     )
	char *nand2_modelname;  /* Model name.            */
	long  nand2_width;      /* Bus wide.              */
	long  nand2_slice;      /* LSB starting slice.    */
	long  nand2_drive;      /* Output drive capacity. */
	long  nand2_msb0;       /* Zero as MSB index.     */
{
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameI0[C_SUPP_MBKSZNAME];
	char  BusNameI1[C_SUPP_MBKSZNAME];
	char  BusNameO[ C_SUPP_MBKSZNAME];
	char  PinNameI0[C_SUPP_MBKSZNAME];
	char  PinNameI1[C_SUPP_MBKSZNAME];
	char  PinNameO[ C_SUPP_MBKSZNAME];


	/* Select name of the leaf cell model. */
	/* (according to output drive power)   */
	LeafCellSliceName = TablLeafCellNand2[nand2_drive-1];

	/* Creation of the new logical figure. */
	DEF_LOFIG( nand2_modelname);

	/* Name of data bus terminals. */
	if (!nand2_msb0)
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", nand2_width-1+nand2_slice, nand2_slice);
		sprintf( BusNameI1,"i1[%ld:%ld]", nand2_width-1+nand2_slice, nand2_slice);
		sprintf( BusNameO , "o[%ld:%ld]", nand2_width-1+nand2_slice, nand2_slice);
	}
	else
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", DP_WIDTH-nand2_slice
				                                  -nand2_width  ,
				                          DP_WIDTH-nand2_slice-1);
		sprintf( BusNameI1,"i1[%ld:%ld]", DP_WIDTH-nand2_slice
				                                  -nand2_width  ,
				                          DP_WIDTH-nand2_slice-1);
		sprintf( BusNameO , "o[%ld:%ld]", DP_WIDTH-nand2_slice
				                                  -nand2_width  ,
				                          DP_WIDTH-nand2_slice-1);
	}

 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON( BusNameI0, IN   , BusNameI0);
	LOCON( BusNameI1, IN   , BusNameI1);
	LOCON( BusNameO ,   OUT, BusNameO );

	/* **** Data bus terminals **** */
	LOCON(     "vdd", INOUT,     "vdd"); 
	LOCON(     "vss", INOUT,     "vss"); 
	
	
	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=nand2_slice;
		 LeafCellIndex<nand2_slice+nand2_width; LeafCellIndex++)
	{
		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "nand2_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!nand2_msb0) BusIndex =              LeafCellIndex;
		else            BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameI0, "i0[%ld]", BusIndex);
		sprintf( PinNameI1, "i1[%ld]", BusIndex);
		sprintf( PinNameO ,  "o[%ld]", BusIndex);

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
			          PinNameI0 ,
			          PinNameI1 ,
			          PinNameO  , "vdd", "vss", EOL);
	}

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
