/****************************************************************************/ 
/*                                                                          */
/*                      ALLIANCE CAO & VLSI CAD                             */
/*                                                                          */
/*    Product : (S)ymbolic (2)to (R)eal translater                          */ 
/*    File    : rdsacces+.c                                                 */ 
/*                                                                          */ 
/*    (c) copyright 1992 MASI laboratory CAO & VLSI team                    */ 
/*    All Right Reserved                                                    */ 
/*    Support : e-mail cao-vlsi@masi.ibp.fr                                 */ 
/*                                                                          */ 
/****************************************************************************/
/*                                                                          */
/*               Access functions to the rds database objects               */
/*                                                                          */
/****************************************************************************/

#include "generic.h"
#include  RDS_H
#include  MPH_H
#include  MUT_H
#include "rdsacces+.h"
#include "statistics.h"
#include  RPR_H 

rds_rec *Free_List = NULL; /* a global variable pointing to the free list
                                    of rectangles */
/*
 *rds_fig *HeadFigureRds;  a global pointer to the list of models 
 *                         replaced by HeadFigureRds from librds
 */

static ptype_list   *MODEL_CHAIN = NULL ;

view_rec(rec)
rds_rec * rec;
{
printf("\n\
rec->x            = %ld\n\
rec->y            = %ld\n\
rec->dx           = %ld\n\
rec->dy           = %ld\n\
rec->flags        = %d\n\
rec->layer        = %d\n\
rec->mbk_type     = %d\n\
rec->u_rec.name   = %s\n",
rec->x            ,
rec->y            ,
rec->dx           ,
rec->dy           ,
rec->flags        ,
rec->layer        ,
rec->mbk_type     ,
rec->u_rec.name   ,
rec->extract_type );
}

/****************************************************************************
 * rds_rectangle_in: adds the rectangle given by its data to the given list
 *                   and returns a pointer to the new list.
 *                   Before allocating memory for the new rectangle, the free
 *                   list is checked and if it is not empty : we use the memory
 *                   area pointed by the first element of this list.
 ****************************************************************************/

rds_rec *rds_rectangle_in(layer_list, x, y, dx, dy, 
                          layer_num, mbk_type, flag, name, extract_type)
rds_rec *layer_list;
long  x, y, dx, dy;
char  flag;
char  mbk_type;
char  layer_num;
char  *name;
char  extract_type;
{
  rds_rec * npr;

  STAT_RECT_REQRD++;
  if ( Free_List)
    npr = pull_free_list();
  else {
    npr = AllocRectangleRds();
    STAT_RECT_ALLOC++;
  }

  npr->next  = layer_list;
  npr->x     = x;
  npr->y     = y;
  npr->dx    = dx;
  npr->dy    = dy;
  npr->flags = flag;
  npr->layer = layer_num;
  npr->mbk_type = mbk_type;
  npr->u_rec.name = name;
  npr->extract_type = extract_type;

  return layer_list = npr;
}


/****************************************************************************
 * rds_instance_in: adds the instance given by its data to the given list
 *                   and returns a pointer to the new list.
 ****************************************************************************/

rds_ins *rds_instance_in(instance, model, name, x, y, sym)
rds_ins *instance;
char  *model, *name;
long  x, y;
char  sym;
{
  rds_ins * npr;
  char  i;

  npr = AllocInstanceRds();

  npr->next = instance;
  npr->mod_name = namealloc(model);
  npr->ins_name = namealloc(name);
  npr->x = x;
  npr->y = y;
  npr->transf = sym;
  for (i = 0; i < RDS_MAX_LAYER ; i++)
    npr->couronne[i] = NULL;

  return instance = npr;
}


/******************************************************************************
 * rds_remove_model: removes the model with the specified name from the list 
 *                   of models.
 ****************************************************************************/

void rds_remove_model( name)
char  *name;
{

  rds_figurep_typ * modelp;

  name = namealloc( name);
  for ( modelp = &(HeadFigureRds); *modelp; )
    if ( (*modelp)->name == name)
      *modelp = (*modelp)->next;
    /*****************************************************
     *** return to system the free memory would be cue ***
     *****************************************************/
    else
      modelp = &( (*modelp)->next);
}


/****************************************************************************
 * rds_create_model: creates a new model with the given name and adds it to
 *                   the list of models pointed by HeadFigureRds and returns 
 *                   a pointer to the created model.
 *                   Note: if the model exists it creates a new model and 
 *                   remove the existing one.
 ****************************************************************************/

rds_fig *rds_create_model(name)
char  *name;
{
  rds_fig * npr;
  register int  i;

  rds_remove_model( name);
  npr = AllocFigureRds();

  npr->next = HeadFigureRds;
  npr->name = namealloc(name);
  npr->instance = NULL;
  for (i = 0; i < RDS_MAX_LAYER; i++)
    npr->layertab[i] = NULL; /* the last layer is the abut */
  for (i = 0; i < RDS_MAX_LAYER; i++)
    npr->bboxtab[i] = NULL;
  for (i = 0; i < RDS_MAX_LAYER; i++)
    npr->flagtab[i] = 0;
  npr->flags = RDS_MODE_ALL;

  return HeadFigureRds = npr;
}


/******************************************************************************
 * rds_model_out: given a model name it returns a pointer to it by inspecting 
 *                the model list pointed by HeadFigureRds.
 ****************************************************************************/

rds_fig *rds_model_out(name)
char  *name;
{
  rds_fig * pt;

  name = namealloc(name);

  for (pt = HeadFigureRds; pt; pt = pt->next)
    if (pt->name == name)
      return pt;
  return NULL;
}


/*****************************************************************************
 * rds_instance_out: given an instance name and a pointer to its father model
 *                   it returns a poiner to the instance.
 ****************************************************************************/

rds_ins *rds_instance_out(rds_model, name)
rds_fig *rds_model;
char  *name;
{
  rds_ins * pt;

  name = namealloc(name);

  for (pt = rds_model->instance; pt; pt = pt->next)
    if (pt->ins_name == name)
      return pt;
  return NULL;
}


/*****************************************************************************
* rds_mbk_kill: destroies mbk database. May take time ...
*****************************************************************************/

void rds_mbk_kill()
{
  phfig_list * pt, *pt_next;

  for (pt = HEAD_PHFIG; pt; pt = pt_next) {
    pt_next = pt->NEXT;
    delphfig(pt->NAME);
  }
}


/*******************************************************************************
 * mark_remove: marks the given rectangle as to be removed
 ******************************************************************************/

void mark_remove( rectangle)
rds_rec *rectangle;
{
  if (!(is_remove(rectangle)))
    rectangle->flags += REMOVE;
}


/*******************************************************************************
 * demark_flattenres: removes the flag of the given rectangle 
 ******************************************************************************/

void demark_flattenres( rectangle)
rds_rec *rectangle;
{
  if ( is_flattenres(rectangle))
    rectangle->flags -= FLATTENRES;
}


/*******************************************************************************
 * mark_flattenres: marks the given rectangle as a result of a flatten operation
 ******************************************************************************/

void mark_flattenres( rectangle)
rds_rec *rectangle;
{
  if ( !( is_flattenres(rectangle)))
    rectangle->flags += FLATTENRES;
}


/*******************************************************************************
 * demark_used: removes the used flag 
 ******************************************************************************/

void demark_used( rectangle)
rds_rec *rectangle;
{
  if ( is_used(rectangle))
    rectangle->flags -= USED;
}


/*******************************************************************************
 * mark_used: marks the given rectangle as used
 ******************************************************************************/

void mark_used( rectangle)
rds_rec *rectangle;
{
  if ( !( is_used(rectangle)))
    rectangle->flags += USED;
}


/******************************************************************************
 * mark_scotch: marks the given rectangle as a scotch.
 ******************************************************************************/

void mark_scotch( rectangle)
rds_rec *rectangle;
{
  if ( !( is_scotch(rectangle))) {
    rectangle->flags += SCOTCH;
  }
}


/******************************************************************************
 * demark_scotch: removes the scotch flag from the rectangle. 
 ******************************************************************************/

void demark_scotch( rectangle)
rds_rec *rectangle;
{
  if ( is_scotch(rectangle))
    rectangle->flags -= SCOTCH;
}


/******************************************************************************
 * mark_old_scotch: marks the given rectangle as an old scotch
 ******************************************************************************/

void mark_old_scotch( rectangle)
rds_rec *rectangle;
{
  if ( !( is_old_scotch(rectangle)))
    rectangle->flags += OLD_SCOTCH;
}


/*****************************************************************************
 * mark_empty: marks the given model at the specified layer as empty
 ****************************************************************************/

void mark_empty(model, layer_num)
rds_fig *model;
char  layer_num;
{
  if ( !( is_empty(model, layer_num)))
    model->flagtab[ layer_num] += EMPTY;
}


/*****************************************************************************
 * mark_post: marks the given model at the specified layer as post_treated
 ****************************************************************************/

void mark_treat( model)
rds_fig *model;
{
  model->flags |= RDS_MODE_TREAT;
}


/*******************************************************************************
 * push_free_list: pushes the given pointer to a deleted rectangle into the free
 *                 list.
 ******************************************************************************/

void push_free_list( rectp)
rds_rec *rectp;
{
  rectp->next = Free_List;
  Free_List = rectp;
}


/*******************************************************************************
 * pull_free_list: pulls from the free list - if not empty - a pointer to
 *                 a ( rectangle) memory space area not used anymore.
 ******************************************************************************/

rds_rec *pull_free_list()
{
  rds_rec * temp;

  if ( Free_List) {
    temp = Free_List;
    Free_List = Free_List->next;
    temp->x  = 0;
    temp->y  = 0;
    temp->dx = 0;
    temp->dy = 0;
    temp->mbk = (void * )NULL;
    (temp->u_rec).link = (rds_rec * ) NULL;
    temp->user = (void * )NULL;
    temp->mbk_type = (char)MBK_UNKNOWN;
    temp->layer = (char)NULL;
    temp->flags = (char)NULL;
    temp->extract_type = RDS_NONE;
    return( temp);
  } else
    return(NULL);
}


/***************************************************************************
 * rds_remove_rectangle: removes from the specified list at the specified
 *              layer all the rectangles that are marked to be removed.
 *              The removed rectangles are put in a free list, to minimize
 *              memory consumption: if we need to allocate memory for a new 
 *              rectangle we check the free list first.
 ***************************************************************************/

void rds_remove_rectangle( list)
rds_rectanglep_typ *list;
{
  register rds_rectanglep_typ *rectanglep;
  register rds_rec *temp;

  for ( rectanglep = list; *rectanglep; )
    if (is_remove(*rectanglep)) {
      temp = (*rectanglep)->next;
      push_free_list( (*rectanglep));
      *rectanglep = temp;
    }
    else
      rectanglep = &( (*rectanglep)->next);
}


/***************************************************************************
 * rds_clean_layer: removes from the specified model at the specified layer 
 *              all the rectangles that have been flattened and were not used.
 *              The removed rectangles are put in a free list, to minimize
 *              memory consumption: if we need to allocate memory for a new 
 *              rectangle we check the free list first.
 ***************************************************************************/

void rds_clean_layer( list)
rds_rectanglep_typ *list;
{
  register rds_rectanglep_typ *rectanglep;
  register rds_rec *temp;

  for ( rectanglep = list; *rectanglep; ) {
    /* when a rectangle is issued from a subcell after a ring flattening
       if is to be erased if it was not used by merge */
    if ( (is_flattenres(*rectanglep)) && (!(is_used(*rectanglep)))) {
      /**** remove it ****/
      temp = (*rectanglep)->next;
      push_free_list( (*rectanglep));
      *rectanglep = temp;
    } else {
      /**** just erase using flag and flattenres ****/
      /**** the rectangle owns to the current hierararchical level ***/
      demark_flattenres( *rectanglep);
      demark_used( *rectanglep);
      rectanglep = &( (*rectanglep)->next);
    }
  }
}


/*****************************************************************************
 * side_intersec: determines if 2 rectangles could intersect in the dimension 
 *                of the specified sides.
 *                If yes it returns (1) and puts where and how long is the side
 *                of their intersection in rs and rds respectively.
 *                If not it returns (0).
 *****************************************************************************/

side_intersec( s1, s2, ds1, ds2, rs, rds)
long  s1, s2, ds1, ds2, *rs, *rds;
{
  if ( (s2 >= s1) && (s2 <= (s1 + ds1))) {
    if ( ( s2 + ds2) >= (s1 + ds1)) {
      *rs = s2;
      *rds = s1 + ds1 - s2;
    } else {
      *rs = s2;
      *rds = ds2;
    }
    return(1);
  }
  if ( ( s2 <= s1) && ( (s2 + ds2) >= s1)) {
    if ( (s2 + ds2) >= (s1 + ds1)) {
      *rs = s1;
      *rds = ds1;
    } else {
      *rs = s1;
      *rds = s2 + ds2 - s1;
    }
    return(1);
  }
  return(0);
}


/*****************************************************************************
 * rect_intersec: determines if 2 rectangles intersect. 
 *                If yes it returns (1) and puts the data of the rectangle 
 *                of their intersection in rx, ry, rdx and rdy.
 *                If not it returns (0).
 *****************************************************************************/

rect_intersec( x1, x2, dx1, dx2, y1, y2, dy1, dy2, rx, rdx, ry, rdy)
long  x1, x2, dx1, dx2, y1, y2, dy1, dy2, *rx, *rdx, *ry, *rdy;
{
  if ( side_intersec( x1, x2, dx1, dx2, rx, rdx))
    return( side_intersec( y1, y2, dy1, dy2, ry, rdy));
  else
    return(0);
}
