/****************************************************************************/ 
/*                                                                          */
/*                      ALLIANCE CAO & VLSI CAD                             */
/*                                                                          */
/*    Product : (S)ymbolic (2)to (R)eal translater                          */ 
/*    File    : postreat.c                                                  */ 
/*                                                                          */ 
/*    (c) copyright 1992 MASI laboratory CAO & VLSI team                    */ 
/*    All Right Reserved                                                    */ 
/*    Support : e-mail cao-vlsi@masi.ibp.fr                                 */ 
/*                                                                          */ 
/****************************************************************************/
/*                                                                          */
/*                     Post treatement functions                            */
/*                                                                          */
/****************************************************************************/
/* fonction void couron_layer ()                                            */
/* ATTENTION! dx et dy ne sont pas les longueurs mais les positions hautes  */
/*            et droites des couronnes dans couronne[]                      */
/****************************************************************************/

#include "generic.h"
#include  MPH_H
#include  MUT_H
#include  RDS_H
#include  RPR_H
#include "rdsacces+.h"
#include "postrat.h"

/*************************************************************************
 * resize_layer : extends the sides of all the rectangles in the specified
 *                layer in the specified model by the amount "qty" at each 
 *                extreme.
 *************************************************************************/
void resize_layer( model, layer_num)
rds_fig   *model;
char  layer_num;
{
LineOversize  * line_oversize;
rds_rec       * rectp;
long            qty;  /* given in rds_unit */

  line_oversize       = GetOversizeParam (layer_num);
  qty                 = GET_OVERSIZE( line_oversize);

  for ( rectp         = model ->layertab[layer_num];
        rectp;  
        rectp         = rectp ->next) {
    (long) rectp ->x  -= qty ;
    (long) rectp ->y  -= qty ;
    (long) rectp ->dx += 2 * qty ;
    (long) rectp ->dy += 2 * qty ;
  }
}

/*****************************************************************************
 * inv_resize_layer : minimizes the sides of all the rectangles in the specified
 *                    layer in the specified model by the amount "qty" at each 
 *                    extreme.
 *****************************************************************************/
void inv_resize_layer( model, layer_num)
rds_fig   *model;
char  layer_num;
{
LineOversize  * line_oversize;
rds_rec       * rectp;
long            qty;  /* given in rds_unit */

  line_oversize           = GetOversizeParam (layer_num);
  qty                  = GET_OVERSIZE( line_oversize);

  for ( rectp = model ->layertab[layer_num];  
        rectp ;  
        rectp = rectp ->next) {
    (long) rectp ->x  += qty ;
    (long) rectp ->y  += qty ;
    (long) rectp ->dx -= 2 * qty ;
    (long) rectp ->dy -= 2 * qty ;
  }
}

/******************************************************************************
 * adjust_rectangle: adjusts the data of the rectangle considering the symmetry
 *                   of the instance containning it.
 *                   The given symmetry might be different from the symmetry 
 *                   of the instance in its father ( the case of several 
 *                   hierarchical levels).
 *                   This function is used when copying the rectangle in higher 
 *                   hierarchical level.
 *****************************************************************************/
void adjust_rectangle( x, y, dx, dy, xins, yins, sym)
long  *x, *y, *dx, *dy;
long  xins, yins;
char  sym;
{
long  tx, ty, tdx, tdy;

tx = *x;
ty = *y;
tdx = *dx;
tdy = *dy;

  switch ( sym) {
  case NOSYM:
    *x += xins ;
    *y += yins;
    break;
  case SYM_X:
    *x  = xins - tx - tdx;
    *y += yins;
    break;
  case SYM_Y:
    *x += xins ;
    *y  = yins - ty - tdy;
    break;
  case SYMXY:
    *x  = xins - tx - tdx;
    *y  = yins - ty - tdy;
    break;
  case ROT_P:
    *x  = xins - ty - tdy;
    *y  = yins + tx;
    *dx = tdy;
    *dy = tdx;
    break;
  case ROT_M:
    *x  = xins + ty;
    *y  = yins - tx - tdx;
    *dx = tdy;
    *dy = tdx;
    break;
  case SY_RP:
    *x = xins + ty;
    *y = yins + tx;
    *dx = tdy;
    *dy = tdx;
    break;
  case SY_RM:
    *x = xins - ty - tdy;
    *y = yins - tx - tdx;
    *dx = tdy;
    *dy = tdx;
    break;
  default:
    (void) fprintf(stderr, 
           "**post_trat**adj_rect:unknown symetry number %d\n", 
           sym);
    exit(1);
  }
}

/******************************************************************************
 * readjust_rectangle: adjusts the data of the rectangle considering the 
 *                     symmetry of the instance that will contain it.
 *                     The given symmetry might be different from the symmetry 
 *                     of the instance in its father ( the case of several 
 *                     hierarchical levels).
 *                     This function is used when copying the rectangle in lower
 *                     hirerichal level.
 *****************************************************************************/
void readjust_rectangle( x, y, dx, dy, xins, yins, sym)
long  *x, *y, *dx, *dy;
long  xins, yins;
char  sym;
{
long  tx, ty, tdx, tdy;

  tx = *x;
  ty = *y;
  tdx = *dx;
  tdy = *dy;
  switch ( sym) {
  case NOSYM:
    *x -= xins ;
    *y -= yins;
    break;
  case SYM_X:
    *x  = xins - tx - tdx;
    *y -= yins;
    break;
  case SYM_Y:
    *x -= xins ;
    *y  = yins - ty - tdy;
    break;
  case SYMXY:
    *x  = xins - tx - tdx;
    *y  = yins - ty - tdy;
    break;
  case ROT_P:
    *x  = ty - yins;
    *y  = xins - tdx - tx;
    *dx = tdy;
    *dy = tdx;
    break;
  case ROT_M:
    *x = yins - tdy - ty;
    *y = tx - xins;
    *dx = tdy;
    *dy = tdx;
    break;
  case SY_RP:
    *x = ty - yins;
    *y = tx - xins;
    *dx = tdy;
    *dy = tdx;
    break;
  case SY_RM:
    *x = yins - tdy - ty;
    *y = xins - tdx - tx;
    *dx = tdy;
    *dy = tdx;
    break;
  default:
    (void) fprintf(stderr, 
           "**post_trat**readj_rect:unknown symetry number %d\n",
           sym);
    exit(1);
  }
}

/******************************************************************************
 * result_sym: gives the result of 2 successive symmetry.
 *****************************************************************************/
char  result_sym( first_sym, second_sym)
char  first_sym, second_sym;
{
  switch ( first_sym) {
  case NOSYM:
    switch ( second_sym) {
    case NOSYM: return( NOSYM);
    case SYM_X: return( SYM_X);
    case SYM_Y: return( SYM_Y);
    case SYMXY: return( SYMXY);
    case ROT_P: return( ROT_P);
    case ROT_M: return( ROT_M);
    case SY_RP: return( SY_RP);
    case SY_RM: return( SY_RM);
    default:
      (void) fprintf(stderr, 
             "**post_trat**result_sym:unknown symetry number %d\n", 
             second_sym);
      exit(1);
    }
  case SYM_X:
    switch ( second_sym) {
    case NOSYM: return( SYM_X);
    case SYM_X: return( NOSYM);
    case SYM_Y: return( SYMXY);
    case SYMXY: return( SYM_Y);
    case ROT_P: return( SY_RM);
    case ROT_M: return( SY_RP);
    case SY_RP: return( ROT_M);
    case SY_RM: return( ROT_P);
    default:
      (void) fprintf(stderr, 
             "**post_trat**result_sym:unknown symetry number %d\n", 
             second_sym);
      exit(1);
    }
  case SYM_Y:
    switch ( second_sym) {
    case NOSYM: return(SYM_Y);
    case SYM_X: return(SYMXY);
    case SYM_Y: return(NOSYM);
    case SYMXY: return(SYM_X);
    case ROT_P: return(SY_RP);
    case ROT_M: return(SY_RM);
    case SY_RP: return(ROT_P);
    case SY_RM: return(ROT_M);
    default:
      (void) fprintf(stderr, 
             "**post_trat**result_sym:unknown symetry number %d\n", 
             second_sym);
      exit(1);
    }
  case SYMXY:
    switch ( second_sym) {
    case NOSYM: return(SYMXY);
    case SYM_X: return(SYM_Y);
    case SYM_Y: return(SYM_X);
    case SYMXY: return(NOSYM);
    case ROT_P: return(ROT_M);
    case ROT_M: return(ROT_P);
    case SY_RP: return(SY_RM);
    case SY_RM: return(SY_RP);
    default:
      (void) fprintf(stderr, 
             "**post_trat**result_sym:unknown symetry number %d\n", 
             second_sym);
      exit(1);
    }
  case ROT_P:
    switch ( second_sym) {
    case NOSYM: return(ROT_P);
    case SYM_X: return(SY_RP);
    case SYM_Y: return(SY_RM);
    case SYMXY: return(ROT_M);
    case ROT_P: return(SYMXY);
    case ROT_M: return(NOSYM);
    case SY_RP: return(SYM_Y);
    case SY_RM: return(SYM_X);
    default:
      (void) fprintf(stderr, 
             "**post_trat**result_sym:unknown symetry number %d\n", 
             second_sym);
      exit(1);
    }
  case ROT_M:
    switch ( second_sym) {
    case NOSYM: return(ROT_M);
    case SYM_X: return(SY_RM);
    case SYM_Y: return(SY_RP);
    case SYMXY: return(ROT_P);
    case ROT_P: return(NOSYM);
    case ROT_M: return(SYMXY);
    case SY_RP: return(SYM_X);
    case SY_RM: return(SYM_Y);
    default:
      (void) fprintf(stderr, 
             "**post_trat**result_sym:unknown symetry number %d\n", 
             second_sym);
      exit(1);
    }
  case SY_RP:
    switch ( second_sym) {
    case NOSYM: return(SY_RP);
    case SYM_X: return(ROT_P);
    case SYM_Y: return(ROT_M);
    case SYMXY: return(SY_RM);
    case ROT_P: return(SYM_X);
    case ROT_M: return(SYM_Y);
    case SY_RP: return(NOSYM);
    case SY_RM: return(SYMXY);
    default:
      (void) fprintf(stderr, 
             "**post_trat**result_sym:unknown symetry number %d\n", 
             second_sym);
      exit(1);
    }
  case SY_RM:
    switch ( second_sym) {
    case NOSYM: return(SY_RM);
    case SYM_X: return(ROT_M);
    case SYM_Y: return(ROT_P);
    case SYMXY: return(SY_RP);
    case ROT_P: return(SYM_Y);
    case ROT_M: return(SYM_X);
    case SY_RP: return(SYMXY);
    case SY_RM: return(NOSYM);
    default:
      (void) fprintf(stderr, 
             "**post_trat**result_sym:unknown symetry number %d\n", 
             second_sym);
      exit(1);
    }
  default:
    (void) fprintf(stderr, 
           "**post_trat**result_sym:unknown symetry number %d\n", 
           first_sym);
    exit(1);
  }
  return(NOSYM);
}

/*****************************************************************************
 * flatten_instance_layer: copies to the given layer_tail, all the segments 
 *                         in the given instance that intersect the couron of
 *                         the given ancetre, using the given father_sym
 *                         to compose the resulting symmetry of the instance
 *                         in the ancetre.
 *                         The segments of the sons instances are copied also
 *                         as it is a recursive function.
 ******************************************************************************/
void flatten_instance_layer( instance, ancetre, comp_sym, origin_x, origin_y, layer_tail, layer_num)
rds_ins *instance;        /* instance where there is rectangles to flatten */
rds_ins *ancetre;         /* first instance where the ring is defined */
char  comp_sym;           /* symetry comp of instances without the first */
long  origin_x, origin_y; /* origin comp of instances without the first */
rds_rectanglep_typ *layer_tail;
char  layer_num;
{
rds_fig * modelp;
rds_ins * instancef;
long      nx, ny, ndx, ndy;
long      dummy1, dummy2;
long      new_origin_x, new_origin_y;
rds_rec * rectp;

  modelp = rds_model_out( instance ->mod_name);

  dummy1 = 0l;
  dummy2 = 0l;
  new_origin_x = origin_x;
  new_origin_y = origin_y;
  origin_x = instance->x;
  origin_y = instance->y;
  adjust_rectangle( &origin_x, &origin_y, &dummy1, &dummy2,  
                    new_origin_x, new_origin_y, comp_sym);
  comp_sym = result_sym( instance->transf, comp_sym);

  for (  instancef = modelp->instance;  
         instancef;  
         instancef = instancef->next )
    flatten_instance_layer( instancef,  ancetre,   comp_sym,  
                            origin_x, origin_y, layer_tail,  layer_num);

  for ( rectp = modelp ->layertab[layer_num];  rectp;  rectp = rectp ->next ) {
    if (rectp->extract_type == RDS_CON_EXTER) 
      continue;
    if (rectp->extract_type == RDS_REF_REF)   
      continue;
    if (rectp->extract_type == RDS_REF_CON)   
      continue;
    nx = rectp ->x;
    ny = rectp ->y;
    ndx = rectp ->dx;
    ndy = rectp ->dy;

    adjust_rectangle( &nx,  &ny,  &ndx,  &ndy,  origin_x,  origin_y,  comp_sym);
    if(( nx         <= ancetre->couronne[layer_num]->x) 
    || (  ny        <= ancetre->couronne[layer_num]->y)
    || (( nx + ndx) >= ancetre->couronne[layer_num]->dx) 
    || (( ny + ndy) >= ancetre->couronne[layer_num]->dy) ) {

      adjust_rectangle(&nx,  &ny,  &ndx,  &ndy,  
                       ancetre->x,  ancetre->y,  ancetre->transf);

      *layer_tail = rds_rectangle_in( *layer_tail,  nx,  ny,  ndx,  ndy,  
                                      layer_num, rectp->mbk_type, 0,
                                      NULL, /* a flattened rect loses name */
                                      rectp->extract_type);
      mark_flattenres( *layer_tail);
    }
  }
}

/******************************************************************************
 * flatten_layer: copies from: all the instances in the specified model in the
 *                             the specified layer any rectangle that intersects
 *                             the couron of the instance or even touches it.
 *                         to: the list of rectangles of the specified model in
 *                             th specified layer.
 *                A flattened rectangle is marked (flattened) so that if it is 
 *                not used in the merging stage, it will be removed in the 
 *                cleanning stage. 
 *                remark: if there is more than 2 hierarchical levels, there
 *                will be a call to the flatten_instance_layer() function.
 *****************************************************************************/
void flatten_layer( model, layer_num)
rds_fig *model;
char  layer_num;
{
rds_ins * instance, *instancef;
rds_rec * rectp;
rds_rectanglep_typ * layer_tail;
rds_fig * modelp;
long  nx, ny, ndx, ndy;

  /* get the tail of the list of segments of the model */
  for ( rectp = model ->layertab[ layer_num];  
        (rectp && rectp ->next);  
        rectp = rectp ->next ) 
    ;
  if ( rectp)
    layer_tail = &(rectp ->next);
  else
    layer_tail = &(model ->layertab[ layer_num]);

  for ( instance = model ->instance;  
        instance;  
        instance = instance->next) {
    modelp = rds_model_out( instance ->mod_name);
    if (incataloggds(instance ->mod_name)) 
      continue;
    /* flatten lower levels of the hierarchy */
    for ( instancef = modelp->instance;  
          instancef;  
          instancef = instancef->next)
      flatten_instance_layer( instancef,  instance,   
                              NOSYM, /* no symetry for the model */
                              0l, 0l,  layer_tail, layer_num);

    /* flatten the instances of the model */
    for ( rectp = modelp ->layertab[layer_num];  
          rectp;  
          rectp = rectp ->next ) {
      nx = rectp ->x;
      ny = rectp ->y;
      ndx = rectp ->dx;
      ndy = rectp ->dy;
      if (   ( nx <= instance->couronne[layer_num]->x) 
      || ( ny <= instance->couronne[layer_num]->y) 
      || (( nx + ndx) >= instance->couronne[layer_num]->dx) 
      || (( ny + ndy) >= instance->couronne[layer_num]->dy)) {
        adjust_rectangle( &nx,  &ny,  &ndx,  &ndy,  
                          instance->x,  instance->y,  instance->transf);
        *layer_tail = rds_rectangle_in( *layer_tail,  nx,  ny,  ndx,  ndy,  
                                         layer_num, rectp->mbk_type, 0,
                                         NULL, /* a flattened rect loses name */
                                         rectp->extract_type);
        mark_flattenres( *layer_tail);
      }
    }
  }
}

/*****************************************************************************
 * envelop_layer: defines the envelop of the specified model in the specified 
 *                layer.
 *                i.e. the minimum rectangle that contains all the rectangles
 *                and all the envelops of all instances of the specified model
 *                in the specified layer.
 *                This function as it takes into consideration the envelops
 *                of the instances of the model it must be applied to the
 *                instances before the father model; what is done implicitly
 *                by the post_treat function.
 *                This function once applied on a model at a layer, marks that
 *                model enveloped at that layer.
 *                If the model at that layer does not contain any rectangles,
 *                it is marked empty at that layer.
 ****************************************************************************/
void envelop_layer( model, layer_num)
rds_fig *model;
char  layer_num;
{
rds_fig * modelp;
rds_ins * instancep;
long  lx, ly, gx, gy;
rds_rec * rectp;
char  init = 0;
long  nx, ny, ndx, ndy;

  /* printf("envelop %d\n",layer_num); */

  /* segments of the model */
  if ( model->layertab[ layer_num]) {
    lx =    model->layertab[ layer_num]->x;
    ly =    model->layertab[ layer_num]->y;
    gx =    model->layertab[ layer_num]->x  + model->layertab[ layer_num]->dx;
    gy =    model->layertab[ layer_num]->y  + model->layertab[ layer_num]->dy;
    init = 1;
    for ( rectp = model->layertab[ layer_num];  rectp;  rectp = rectp->next) {
      if ( rectp->x < lx)
        lx = rectp->x;
      if ( rectp->y < ly)
        ly = rectp->y;
      if ( ( rectp->x + rectp->dx) > gx)
        gx = rectp->x + rectp->dx;
      if ( ( rectp->y + rectp->dy) > gy)
        gy = rectp->y + rectp->dy;
    }
  }
  /* envelops of the instances of the model */
  for ( instancep = model ->instance;  instancep;  instancep = instancep ->next) {
    modelp = rds_model_out( instancep ->mod_name);
    if (    !( is_empty(modelp, layer_num)) 
    &&      !( incataloggds(instancep ->mod_name))) {
      rectp = modelp->bboxtab[ layer_num];
      nx = rectp ->x;
      ny = rectp ->y;
      ndx = rectp ->dx;
      ndy = rectp ->dy;
      adjust_rectangle( &nx,  &ny,  &ndx,  &ndy,  
                        instancep->x,  instancep->y,  instancep->transf);
      if ( init) {
        if ( nx < lx)
          lx = nx;
        if ( ny < ly)
          ly = ny;
        if ( ( nx + ndx) > gx)
          gx = nx + ndx;
        if ( ( ny + ndy) > gy)
          gy = ny + ndy;
      } else {
        lx = nx;
        ly = ny;
        gx = nx + ndx;
        gy = ny + ndy;
        init = 1;
      }
    }
  }
  if ( init)
    model->bboxtab[ layer_num] = 
        rds_rectangle_in( model->bboxtab[ layer_num], lx,  ly,  
                          gx - lx,  gy - ly, layer_num, MBK_UNKNOWN, 0, 
                          NULL, /* no name */ RDS_NONE);
  else
    mark_empty( model, layer_num);
}

/*****************************************************************************
 * mod_ins_couron: given an instance, a layer and the rectangle of intersection
 *                 of this instance with another instance; it modifies the 
 *                 couron of this instance in that layer if the rectangle of
 *                 intersection makes the couron wider at any side.
 *****************************************************************************/
void mod_ins_couron( instance, layer_num, xi, yi, dxi, dyi)
rds_ins       *instance;
char  layer_num;
long  xi,  yi,   dxi,  dyi;
{
rds_fig       * model;
LineRingWidth    * line_ring_width;
LineLayerWidth  * line_layer_width;
long  corn_width,  xe,  ye,  dxe,  dye;

  model          = rds_model_out( instance ->mod_name);

  line_ring_width  = GetRingWidthParam (layer_num);
  line_layer_width = GetLayerWidthParam (layer_num);

  corn_width     = GET_RING_WIDTH( line_ring_width)
                 + GET_LAYER_WIDTH( line_layer_width) / 2;
  xe             = model->bboxtab[layer_num]->x;
  ye             = model->bboxtab[layer_num]->y;
  dxe            = model->bboxtab[layer_num]->dx;
  dye            = model->bboxtab[layer_num]->dy;

  readjust_rectangle ( &xi,  &yi,  &dxi,  &dyi,  
                       instance->x,  instance->y,  instance->transf );

  if ( ( xi - xe) > ( ( xe + dxe) - ( xi + dxi))) {
    if ( (xi - corn_width) < instance->couronne[ layer_num]->dx)
      instance->couronne[ layer_num]->dx = xi - corn_width;
  } else if ( (xi + dxi + corn_width) > instance->couronne[ layer_num]->x)
    instance->couronne[ layer_num]->x = xi + dxi + corn_width;

  if ( ( yi - ye) > ( ( ye + dye) - ( yi + dyi))) {
    if ( (yi - corn_width) < instance->couronne[ layer_num]->dy)
      instance->couronne[ layer_num]->dy = yi - corn_width;
  } else if ( (yi + dyi + corn_width) > instance->couronne[ layer_num]->y)
    instance->couronne[ layer_num]->y = yi + dyi + corn_width;
}

/*****************************************************************************
 * couron_layer: defines the courons of every instance in the specified model 
 *               in the specified layer.
 *               The couron is determined by 4 limits: x, y, dx 
 *               and dy beyond which is the couron area.
 * ATTENTION! dx et dy ne sont pas les longueurs mais les positions hautes 
 *            et droites des couronnes dans couronne[]
 ****************************************************************************/
void couron_layer( model, layer_num)
rds_fig *model;
char  layer_num;
{
register rds_ins *instance1, *instance2;
long  x1, x2, y1, y2, dx1, dx2, dy1, dy2, rx, rdx, ry, rdy;
register rds_fig *model1, *model2;
long  corn_width;
LineRingWidth    * line_ring_width;
LineLayerWidth   * line_layer_width;

  line_ring_width  = GetRingWidthParam (layer_num);
  line_layer_width = GetLayerWidthParam (layer_num);

  corn_width     = GET_RING_WIDTH( line_ring_width)
                 + GET_LAYER_WIDTH( line_layer_width) / 2;

/*\
 * initial couron based on the abutment box and the envelop 
\*/
  for ( instance1 = model ->instance;  instance1;  instance1 = instance1 ->next) {
    model1 = rds_model_out( instance1 ->mod_name);
    if (incataloggds(instance1 ->mod_name)) 
      continue;
/*\
 * allocation d'une structure rectancle pour la couronne[layer_num] 
\*/
    instance1->couronne[ layer_num] = (rds_rec * ) mbkalloc(sizeof(rds_rec));

    instance1->couronne[layer_num]->dy = model1->layertab[RDS_ABOX]->y  
                                       + model1->layertab[RDS_ABOX]->dy  
                                       - corn_width;

    instance1->couronne[layer_num]->y  = model1->layertab[RDS_ABOX]->y 
                                       + corn_width;

    instance1->couronne[layer_num]->dx = model1->layertab[RDS_ABOX]->x  
                                       + model1->layertab[RDS_ABOX]->dx 
                                       - corn_width;

    instance1->couronne[layer_num]->x  = model1->layertab[RDS_ABOX]->x  
                                       + corn_width;

    if ( !is_empty( model1, layer_num)) {
      if ( instance1->couronne[layer_num]->dy >   
         (   model1->bboxtab[layer_num]->y  
           + model1->bboxtab[layer_num]->dy 
           - corn_width                        ) ) {
        instance1->couronne[layer_num]->dy = model1->bboxtab[layer_num]->y  
                                           + model1->bboxtab[layer_num]->dy 
                                           - corn_width;
      }
      if ( instance1->couronne[layer_num]->y  <   
         (   model1->bboxtab[layer_num]->y  
           + corn_width                       ) ) {
        instance1->couronne[layer_num]->y = model1->bboxtab[layer_num]->y  
                                          + corn_width;
      }
      if ( instance1->couronne[layer_num]->dx >   
         (   model1->bboxtab[layer_num]->x  
           + model1->bboxtab[layer_num]->dx 
           - corn_width                        ) ) {
        instance1->couronne[layer_num]->dx = model1->bboxtab[layer_num]->x  
                                           + model1->bboxtab[layer_num]->dx 
                                           - corn_width;
      }
      if ( instance1->couronne[layer_num]->x  <   
         (   model1->bboxtab[layer_num]->x  
           + corn_width                      ) ) {
        instance1->couronne[layer_num]->x = model1->bboxtab[layer_num]->x  
                                         	+ corn_width;
      }
    }
  }

/*\
 * final couron based on instance intersections 
\*/
  for ( instance1 = model ->instance;  
        instance1;  
        instance1 = instance1 ->next) {
    model1 = rds_model_out( instance1 ->mod_name);

    if (incataloggds(instance1 ->mod_name))
      continue;

    if ( !( is_empty(model1, layer_num))) {
      x1    = model1->bboxtab[ layer_num]->x;
      y1    = model1->bboxtab[ layer_num]->y;
      dx1   = model1->bboxtab[ layer_num]->dx;
      dy1   = model1->bboxtab[ layer_num]->dy;

      adjust_rectangle( &x1,  &y1,  &dx1,  &dy1,  
                        instance1->x,  instance1->y,  instance1->transf);

      for (instance2 = instance1->next;  
           instance2;  
           instance2 = instance2->next) {
        model2 = rds_model_out( instance2 ->mod_name);

        if (incataloggds(instance2 ->mod_name)) {
          continue;
        }

        if ( !( is_empty(model2, layer_num))) {
          x2    = model2->bboxtab[ layer_num]->x;
          y2    = model2->bboxtab[ layer_num]->y;
          dx2   = model2->bboxtab[ layer_num]->dx;
          dy2   = model2->bboxtab[ layer_num]->dy;

          adjust_rectangle( &x2,  &y2,  &dx2,  &dy2,  
                            instance2->x,  instance2->y,  instance2->transf);

          if (rect_intersec( x1, x2,  dx1, dx2,   
                             y1, y2,  dy1, dy2,  &rx, &rdx,  &ry, &rdy   )) {
            mod_ins_couron( instance1,  layer_num,  rx,  ry,  rdx,  rdy);
            mod_ins_couron( instance2,  layer_num,  rx,  ry,  rdx,  rdy);
          }
        }
      }
    }
  }
}

/******************************************************************************
 * envelop: envelops the post_treatable layers of the given model.
 *****************************************************************************/
void envelop( model)
rds_fig *model;
{
char  layer_num;
LinePost     * line_postreat;

  for ( layer_num = 0;  layer_num < RDS_MAX_LAYER;  layer_num++) {
    line_postreat   = GetPostParam (layer_num);

    if ( GET_POST_Y_OR_N (line_postreat) == TREAT) {
      envelop_layer( model, layer_num);
    }
  }
}

/******************************************************************************
 * couron: courons the post_treatable layers of the given model.
 *****************************************************************************/
void couron( model)
rds_fig *model;
{
char  layer_num;
LinePost     * line_postreat;

  for ( layer_num = 0;  layer_num < RDS_MAX_LAYER;  layer_num++) {
    line_postreat   = GetPostParam (layer_num);

    if ( GET_POST_Y_OR_N (line_postreat) == TREAT) {
      couron_layer( model, layer_num);
    }
  }
}

/******************************************************************************
 * flatten: flattens the post_treatable layers of the model.
 *****************************************************************************/
void flatten( model)
rds_fig *model;
{
  char  layer_num;
  LinePost     * line_postreat;

  for ( layer_num = 0;  layer_num < RDS_MAX_LAYER;  layer_num++) {
    line_postreat   = GetPostParam (layer_num);

    if ( GET_POST_Y_OR_N (line_postreat) == TREAT) {
      flatten_layer( model, layer_num);
    }
  }
}

/******************************************************************************
 * resize: oversize all post_treatable layers of the model
 *****************************************************************************/
void resize( model)
rds_fig *model;
{
char  layer_num;
LinePost     * line_postreat;

  for ( layer_num = 0;  layer_num < RDS_MAX_LAYER;  layer_num++) {
    line_postreat   = GetPostParam (layer_num);

    if ( GET_POST_Y_OR_N (line_postreat) == TREAT) {
      resize_layer( model, layer_num);
    }
  }
}

/******************************************************************************
 * merge: merges all post_treatable layers of the model.
 *        If the layer to be merged is of the type that needs scotchs 
 *        ( indicated by having a complementary layer in its postreat table), 
 *        merge_layer_2() will be called instead of merge_layer().
 *****************************************************************************/
void merge ( model, scotch_on_flag)
rds_fig *model;
int  scotch_on_flag;
{
char  layer_num;
LinePost     * line_postreat;

  for ( layer_num = 0;  layer_num < RDS_MAX_LAYER;  layer_num++) {
    line_postreat   = GetPostParam (layer_num);

    if ( GET_POST_Y_OR_N (line_postreat) == TREAT) {
      if ( (GET_POST_COMPLEMENTARY( line_postreat) != -1) && scotch_on_flag) {
        merge_layer_2( model, layer_num);
      } else {
        merge_layer( model, layer_num);
      }
    }
  }
}

/******************************************************************************
 * clean: cleans all post_treatable layers.
 *****************************************************************************/
void clean( model)
rds_fig *model;
{
LinePost     * line_postreat;
char  layer_num;

  for ( layer_num = 0;  layer_num < RDS_MAX_LAYER;  layer_num++) {
    line_postreat   = GetPostParam (layer_num);

    if ( GET_POST_Y_OR_N (line_postreat) == TREAT) {
      rds_clean_layer( &( model->layertab[layer_num]));
    }
  }
}

/******************************************************************************
 * inv_resize: undersizes all post_treatable layers. 
 *****************************************************************************/
void inv_resize( model)
rds_fig *model;
{
char  layer_num;
LinePost     * line_postreat;

  for ( layer_num = 0;  layer_num < RDS_MAX_LAYER;  layer_num++) {
    line_postreat   = GetPostParam (layer_num);

    if ( GET_POST_Y_OR_N (line_postreat) == TREAT) {
      inv_resize_layer( model, layer_num);
    }
  }
}

/*****************************************************************************
 * post_treat: post_treats the specified model.
 *             After completion it marks this model at this layer post_treated. 
 *             This is a recursive function: if an instance of this model
 *             has its model not yet post treated at this layer, it is
 *             post treated first.
 ****************************************************************************/
void post_treat( model, scotch_on_flag, verbose)
rds_fig * model;
int       scotch_on_flag;
int       verbose;
{
  rds_ins * instance;
  rds_fig * modelp;

  for ( instance = model->instance;  instance;  instance = instance->next) {
    modelp = rds_model_out( instance->mod_name);

    if ( ! IsModeTreatRds(modelp) && ! incataloggds(modelp->name) ) {
      post_treat( modelp, scotch_on_flag, verbose);
    }
  }
  if(verbose)
    printf("\t--> post-treating model %s\n", model->name);
  envelop    ( model);
  couron     ( model);
  flatten    ( model);
  resize     ( model);
  merge      ( model, scotch_on_flag);
  clean      ( model);
  inv_resize ( model);
  mark_treat ( model);
}

/*******************************************************************************
 * replace_cells: traverses the list of models and replaces the models that are 
 *                found in the gds catalog by the corresponding models parsed 
 *                from this catalog.
 *                remarks:
 ******************************************************************************/
void replace_cells(verbose)
int verbose;
{
  rds_fig     * model;
  int  error;

  for ( model = HeadFigureRds;  model;  model = model->next) {
    if ( incataloggds( model->name)) {
      if(verbose)
        printf("\t--> replace cell %s \n", model->name);
      (void) LoadRdsFig( model->name);
    }
  }
}
