/*

Formal specification of the YASP trajectory file
================================================
 
FMP 23 Aug 90
 
1) Header
=========
 
The Header  contains information  which is stored  only  once  on  the
trajectory  file. It precedes  all individual configurations (frames).
 
- character*80 title
  write (io_unit) title
  write (io_unit) number_of_records_per_frame (i*4)
 
2) Frames
=========
 
A "frame" is the information characterising the simulated  system at a
certain  time.    Each frame contains   as many  (Fortran)  records as
specified in the header.
 
- "Frame"
  write (io_unit) configuration_number, number_of_atoms, time
	(i*4, i*4, r*8)
 
  The configuration_number is the index number of the frame as it  ap-
  pears in the simulation,  which is  in general not  the same as  the
  index number of  the frame on  the trajectory file.  The time is the
  simulated time (e.g. ps), not the cpu time.
 
- "Box"
  write (io_unit) (a(i), i = 1, 3), (b(i), i = 1, 3), (c(i), i = 1, 3)
	(all r*8)
 
  The current size and shape of the periodic box is  described by  the
  its cell unit vectors a, b and c.  Note that a, b and c refer to the
  dimensions of the entire box.  The coordinate  origin for the atomic
  positions is,  however,  in the centre of the box.  At present, YASP
  only uses rectangular boxes (all angles between cell vectors are  90
  degrees). Hence the cell vectors look like
	a = (xbox, 0,    0   )
	b = (0,    ybox, 0   )
	c = (0,    0,    zbox)
  with xbox, ybox, zbox beind the box  lengths in the  three cartesian
  directions.
 
- "Pressure"
  write (io_unit) total_isotropic_pressure,
 $		  Pxx,
 $                Pyx, Pyy,
 $                Pzx, Pyx, Pzz
	(all r*8)
  The total pressure is the isotropic pressure of the system which can
  in principle be calculated from the components of the pressure ten-
  sor that follow. This is just to avoid recalculation. All pressures
  are in kPa.
 
- "Energies"
  write (io_unit) number_of_energies, (e(i), i = 1, number_of_energies)
	(i*4, rest r*8)
  As many energy-related quantities as there are. Presently there are
	e(1) = total_energy
	e(2) = potential_energy
	e(3) = kinetic_energy
	e(4) = temperature
  YASP energies are in kJ/mole, temperatures in K.
 
- "X-coordinates"
  write (io_unit) (x(i), i = 1, natom) (all r*8)
  Coordinates (atomic positions) are in nm.
 
- "Y-coordinates"
  write (io_unit) (y(i), i = 1, natom) (all r*8)
  Coordinates (atomic positions) are in nm.
 
- "Z-coordinates"
  write (io_unit) (z(i), i = 1, natom) (all r*8)
  Coordinates (atomic positions) are in nm.
 
- "X-velocities"
  write (io_unit) (vx(i), i = 1,natom) (all r*8)
  Atomic velocities are in nm/ps.
 
- "Y-velocities"
  write (io_unit) (vy(i), i = 1,natom) (all r*8)
  Atomic velocities are in nm/ps.
 
- "Z-velocities"
  write (io_unit) (vz(i), i = 1,natom) (all r*8)
  Atomic velocities are in nm/ps.
 
- "Others"
  write (io_unit) (q(i), i = 1, natom) (all r*8)
  As many of these records as necessary, all describing order(natom)
  quantities (energy per atom and the like). These are presently
  not used.
 
>From this description it can be seen that the information in a frame
naturally decomposes into three parts.
a) system description
  "Frame", "Box", and "Pressure"
b) "Scalar" quantities. Their number does not depend on the number of
   atoms. Let's call them O(natom**0)
  "Energies"
c) "Vector" quantities. Their number depends linearly on the number of
   atoms. O(natom**1).
There is probably no need to include atom-pair-related quantities
of O(natom**2) or higher into the file.

*/

#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include <string.h>
#include "maxdefs.h"

#define MOD(a,b)  (((a+1)/b)*b - (a+1))

    extern double *dvector();
    extern float   *vector();
/* structure for the dynamics trajectory file */

   extern struct {
   char traj_file[BUFF_LEN];                 /* file name          */
   int natom;                                /* number of atoms    */
   int nstep;                                /* number of steps    */
   int time_bw_steps;                        /* time between steps */
   int time_first_frame;                     /* time of first frame */
   int first_frame;                          /* first frame to be displayed */
   int last_frame;                           /* last frame to be displayed  */
   int delta_frame;                          /* display every delta frame */
                   }    trajectory_info;     

     extern int mlist_deep;      /* stack number indicator */
     extern int mlists[MAX_MLIST]; /* First index for molecule list */
     extern int mliste[MAX_MLIST]; /* Second index for molecule list */
     extern char mnlist[MAX_MLIST][BUFF_LEN]; 
                                /* name list for molecule file names */

     extern int MAXatom;
     extern int traj_in_core;

    struct {
    int eterms;  /* number of energy terms stored in the file */
              } yasp;

/***************************************************************************/
get_frame_yasp(alt,iappend)  /* read one frame from a yasp trajectory   */


    int alt;     /* mode of operation (=0) first time in read
                                      (>0) trajectory number */
    int iappend; /* if = 0 no append , if = 1 append */
/***************************************************************************/
{
/* externals    */
    extern FILE *charmm_oc;
    extern float *x,*y,*z;
    extern int numat;
    extern char traj_file[BUFF_LEN];

   static long record_len=0;         /* the "record" length in bytes of one
                                       record containing the x,y and z
                                       coordinates plus the information
                                       in between the coordinates */
   static long ret_fseek;

   static char title[80];  /* yasp title */
   static int conf_num;    /* yasp configuration number */
   static int natom;       /* number of yasp atoms */
   static int record;
   static int nrec;
   static int kstep;
   static int i;
   static int idx1;
   static int num_yasp_energ;
   static int icount;
   static double time_ps;  /* yasp simulation time in ps */
   static double yasp_box[9]; /* yasp box */
   static double yasp_energ[20];  /* yasp energies */
   static double yasp_ppress[7];  /* yasp pressure */
   static double yasp_press;      /* total isotropic pressure */
   static float *vx,*vy,*vz;     /* pointers to yasp velocities */
   static double *util1,*util2,*util3; /* yasp utilities */

/* determine the "record length"    */
        natom = mliste[0];
        numat = mliste[0];
        record_len = 3 * sizeof(int) + sizeof(double) +
                     9 * sizeof(double) + /* cell unit */
                     7 * sizeof(double) + /* pressure and pressure tensor */
                     yasp.eterms * sizeof(double) + sizeof(int) + /* energy */
                     natom * sizeof(float) * 3 +
                     natom * sizeof(float) * 3 +
                     80;

/*  start reading  */

/* read controll record */
    icount = fread(&record,sizeof(int), 1 , charmm_oc);
/* read 80 characters   */
    icount = fread(title,sizeof(char), 80 ,charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

/* read controll record */
    icount = fread(&record,sizeof(int), 1 , charmm_oc);
/* read 1 int   */
    icount = fread(&nrec,sizeof(int), 1  ,charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

/* so far 100 bytes ... */
    if(alt > 0) {
    ret_fseek = fseek(charmm_oc,(alt*record_len),1);
    if(ret_fseek) {
    printf("?ERROR - can't read trajectory file : %s \n",traj_file);
    return(-1);}}

/* next records are specific for one frame */
/* read controll record */
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

    icount = fread(&conf_num,sizeof(int), 1 ,charmm_oc);
    icount = fread(&kstep,sizeof(int), 1 ,charmm_oc);
    icount = fread(&natom,sizeof(int), 1 , charmm_oc);
    icount = fread(&time_ps,sizeof(double), 1 , charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);

    icount = fread(yasp_box,sizeof(double), 9 ,charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);

    icount = fread(&yasp_press,sizeof(double), 1 , charmm_oc);
    icount = fread(yasp_ppress,sizeof(double), 6 , charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);


/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);

    icount = fread(&num_yasp_energ,sizeof(int), 1 , charmm_oc);
    icount = fread(yasp_energ,sizeof(double), num_yasp_energ , charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);


    if(iappend == 0) {

/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    icount = fread(x,sizeof(float),natom,charmm_oc);
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);

    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    icount = fread(y,sizeof(float),natom,charmm_oc);
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);

    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    icount = fread(z,sizeof(float),natom,charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

    for(i = 0 ; i < natom ; i++) {
       x[i] *= 10.;
       y[i] *= 10.;
       z[i] *= 10.;}
  }
    else {

    idx1 = mliste[mlist_deep - 1];

/* check dimensions */
    if((idx1+natom) > MAXatom) {
      printf("?ERROR - next frame does not fit into the coordinate array \n");
      return(-1);}

    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    icount = fread(&x[idx1],sizeof(float),natom,charmm_oc);
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);

    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    icount = fread(&y[idx1],sizeof(float),natom,charmm_oc);
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);

    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    icount = fread(&z[idx1],sizeof(float),natom,charmm_oc);
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);

    for(i = 0 ; i < natom ; i++) {
       x[i + idx1] *= 10.;
       y[i + idx1] *= 10.;
       z[i + idx1] *= 10.;}
  }



/* read control record */
    vx = vector(numat);
     vy = vector(numat);
      vz = vector(numat);

    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    icount = fread(vx,sizeof(float),natom,charmm_oc);
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);

    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    icount = fread(vy,sizeof(float),natom,charmm_oc);
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);

    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    icount = fread(vz,sizeof(float),natom,charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

     free(vx);
      free(vy);
       free(vz);

#ifdef JUNK
/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,charmm_oc);
    icount3 = icount / 3;
    icount = fread(util1,sizeof(double),icount3,charmm_oc);
    icount = fread(util2,sizeof(double),icount3,charmm_oc);
    icount = fread(util3,sizeof(double),icount3,charmm_oc);
    icount = fread(&record,sizeof(int), 1 , charmm_oc);

#endif

    update_mlist(natom);

    return(0);
/*                                  */

}
/***************************************************************************/
read_traj_yasp(alt,numset,pasback)/* read one frame from a yasp trajectory   */


    int alt;     /* mode of operation (=0) first time in read
                                      (>0) trajectory number */
    int pasback[];
    int *numset;
/***************************************************************************/
{
/* externals    */
    extern FILE *charmm_oc;
    extern float *x,*y,*z;
    extern int numat;
    extern char traj_file[BUFF_LEN];

   FILE *yasp_oc;

   static long record_len=0;         /* the "record" length in bytes of one
                                       record containing the x,y and z
                                       coordinates plus the information
                                       in between the coordinates */
   static char title[80];  /* yasp title */
   static int conf_num;    /* yasp configuration number */
   static int natom;       /* number of yasp atoms */
   static int icount;
   static int record;
   static int nrec;
   static int kstep;
   static int nstep;
   static int num_yasp_energ;
   static double time_ps;  /* yasp simulation time in ps */
   static double yasp_box[9]; /* yasp box */
   static double yasp_energ[20];  /* yasp energies */
   static double yasp_ppress[7];  /* yasp pressure */
   static double yasp_press;      /* total isotropic pressure */
   static float  *vx,*vy,*vz;     /* pointers to yasp velocities */
   static double *util1,*util2,*util3; /* yasp utilities */

   yasp_oc = fopen(traj_file,"r");
   if(yasp_oc == NULL) {
   printf(">>> Can't open input file: %s\n",traj_file);
   return(-1); }


/*  start reading  */

/* read controll record */
    icount = fread(&record,sizeof(int), 1 , yasp_oc);
/* read 80 characters   */
    icount = fread(title,sizeof(char), 80 ,yasp_oc);
    title[79] = '\0';
    icount = fread(&record,sizeof(int), 1 , yasp_oc);
   printf("Title: \n");
   printf("%s\n",title);

/* read controll record */
    icount = fread(&record,sizeof(int), 1 , yasp_oc);

/* read 1 int   */
    icount = fread(&nrec,sizeof(int), 1  ,yasp_oc);
    icount = fread(&record,sizeof(int), 1 , yasp_oc);


/* next records are specific for one frame */
/* read controll record */

    icount = fread(&record,sizeof(int), 1 , yasp_oc);

    icount = fread(&conf_num,sizeof(int), 1 ,yasp_oc);
    icount = fread(&kstep,sizeof(int), 1 ,yasp_oc);
    icount = fread(&natom,sizeof(int), 1 , yasp_oc);
    icount = fread(&time_ps,sizeof(double), 1 , yasp_oc);
    icount = fread(&record,sizeof(int), 1 , yasp_oc);


/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,yasp_oc);

    icount = fread(yasp_box,sizeof(double), 9 ,yasp_oc);
    icount = fread(&record,sizeof(int), 1 , yasp_oc);

/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,yasp_oc);

    icount = fread(&yasp_press,sizeof(double), 1 , yasp_oc);
    icount = fread(yasp_ppress,sizeof(double), 6 , yasp_oc);
    icount = fread(&record,sizeof(int), 1 , yasp_oc);


/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,yasp_oc);

    icount = fread(&num_yasp_energ,sizeof(int), 1 , yasp_oc);
    yasp.eterms = num_yasp_energ;

    if(alt == 1) {

/* check number of records */

/* go first to the end of file */
        icount = fseek(yasp_oc,0L,2);
         if(icount != 0) {
         PrintMessage("?ERROR - can't find end of trajectory file");
         return(-1);}

        record = ftell(yasp_oc);
        icount = fseek(yasp_oc, 100L ,0);

        record_len = 3 * sizeof(int) + sizeof(double) +
                     9 * sizeof(double) + /* cell unit */
                     7 * sizeof(double) + /* pressure and pressure tensor */
                     yasp.eterms * sizeof(double) + sizeof(int) + /* energy */
                     natom * sizeof(float) * 3 +
                     natom * sizeof(float) * 3 +
                     80;

        icount = ftell(yasp_oc);

        nstep = (record-icount)/record_len;

/* update trajectory info ... */
        trajectory_info.natom            = natom;
        trajectory_info.nstep            = nstep;
        *numset =                          nstep;

        pasback[0] = nstep;
        pasback[1] = 0;
        pasback[2] = 0;
/*
        trajectory_info.time_bw_steps    = icntrl[2];
        trajectory_info.time_first_frame = icntrl[1];

*/
        printf(" Info for trajectory file   : %s \n",traj_file);
	printf(" Atoms found                : %d \n",natom);
	printf(" Dynamics steps             : %d \n",nstep);

        if(trajectory_info.time_bw_steps) {
        printf(" Time between data frames   : %d \n",trajectory_info.time_bw_steps);
        pasback[2] = trajectory_info.time_bw_steps;}

/*          
        printf(" Time of the first data set : %d \n",icntrl[1]);
*/
        traj_in_core = 2; /* always out of core */
        fclose(yasp_oc);
        return(0);}





/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,yasp_oc);
    if(icount != (sizeof(double) * 3 * natom)) {
      printf("?ERROR - in 'COORDINATES' record \n");
        return(-1);}

    icount = fread(x,sizeof(float),natom,yasp_oc);
    icount = fread(y,sizeof(float),natom,yasp_oc);
    icount = fread(z,sizeof(float),natom,yasp_oc);
    icount = fread(&record,sizeof(int), 1 , yasp_oc);


/* read control record */
    vx = vector(numat);
     vy = vector(numat);
      vz = vector(numat);

    icount = fread(&record,sizeof(int), 1 ,yasp_oc);
    icount = fread(vx,sizeof(float),natom,yasp_oc);
    icount = fread(&record,sizeof(int), 1 ,yasp_oc);

    icount = fread(&record,sizeof(int), 1 ,yasp_oc);
    icount = fread(vy,sizeof(float),natom,yasp_oc);
    icount = fread(&record,sizeof(int), 1 ,yasp_oc);

    icount = fread(&record,sizeof(int), 1 ,yasp_oc);
    icount = fread(vz,sizeof(float),natom,yasp_oc);
    icount = fread(&record,sizeof(int), 1 , yasp_oc);

    free(vx);
     free(vy);
      free(vz);

#ifdef JUNK
/* read controll record */
    icount = fread(&record,sizeof(int), 1 ,yasp_oc);
    icount3 = icount / 3;
    icount = fread(util1,sizeof(double),icount3,yasp_oc);
    icount = fread(util2,sizeof(double),icount3,yasp_oc);
    icount = fread(util3,sizeof(double),icount3,yasp_oc);
    icount = fread(&record,sizeof(int), 1 , yasp_oc);

#endif

    update_mlist(natom);

    return(0);
/*                                  */

}

