/*
 * events.c
 *
 * This file is part of the basis of the Forms Library.
 *
 * It contains the routines that deal with keeping track of events.
 * This includes all the replacement routines for the user event queue.
 *
 * Written by: Mark Overmars
 *
 * Version 2.1 c
 * Date: Oct  6, 1992
 */

#include <stdio.h>
#include <gl/gl.h>
#include <gl/device.h>
#include "forms.h"

/*************** CALL-BACK ROUTINE HANDLING ***********/

static void (*fl_event_call_back)(short, short) = NULL;

void fl_set_event_call_back(FL_EVENTCALLBACKPTR call_back)
/* Sets the call_back routine for the events */
  { fl_event_call_back = call_back; }

void fl_set_call_back(FL_OBJECT *obj, FL_CALLBACKPTR call_back, long argument)
/* Sets the call_back routine for the object */
{
   if (obj == NULL)
     {fl_error("fl_set_call_back","Setting callback of NULL object."); return;}
   obj->object_call_back = call_back;
   obj->argument = argument;
}

void fl_set_form_call_back(FL_FORM *form, FL_FORMCALLBACKPTR call_back)
/* Sets the call_back routine for the form */
{
   if (form == NULL)
     {fl_error("fl_set_form_call_back","Setting callback of NULL form."); return;}
   form->form_call_back = call_back;
}

/*************** THE NORMAL EVENTS ********************/

#define QSIZE	700

static int queued[MAXSGIDEVICE];	/* devices queued by user */

static short thedev[QSIZE];  		/* The event queue */
static short theval[QSIZE];
static int head = 0 , tail = 0;

static int new_events = 0;		/* Number of not yet reported events */

static int special_event(Device dev)
/* Returns whether this is a special event */
{
  return ( dev == MOUSE1 || dev == MOUSE2 ||dev == MOUSE3 ||
	 dev == INPUTCHANGE || dev == KEYBD || dev == LEFTARROWKEY ||
	 dev == RIGHTARROWKEY || dev == UPARROWKEY || dev == DOWNARROWKEY ||
	 dev == REDRAW || dev == WINQUIT ||
         dev == WINFREEZE || dev == WINTHAW);
}

void fl_init_events()
/* initializes the event setting */
{
  int i;
  for (i=0; i<MAXSGIDEVICE; i++) queued[i] = !special_event( (Device) i);
  queued[REDRAW] = TRUE;
  queued[INPUTCHANGE] = TRUE;
}

void fl_qdevice(Device dev)
/* queues a device */
{
  queued[dev] = TRUE;
  qdevice(dev);
}

void fl_unqdevice(Device dev)
/* unqueues a device */
{
  if (special_event(dev)) queued[dev] = FALSE; else unqdevice(dev);
}

int fl_isqueued(Device dev)
/* test whether a device is queued */
{
  return (queued[dev] && isqueued(dev));
}

long fl_qtest()
/* Returns last queued event type */
{
  FL_OBJECT *obj;
  if (head == tail)
  {
    fl_treat_interaction_events(FALSE);
    fl_treat_user_events();
  }
  if (head == tail) return 0;
  return  (long) thedev[tail];
}

void fl_qenter(short qtype, short val)
/* Adds an event to the queue */
{
  if (special_event( (Device) qtype) && !queued[qtype]) return;
  new_events++;
  if (head == tail-1 || (head == QSIZE-1 && tail == 0))
    tail = (tail+1) % QSIZE;
  thedev[head] = qtype;
  theval[head] = val;
  head = (head+1) % QSIZE;
}

void fl_qreset()
/* Empties the queue. */
{ 
  fl_treat_interaction_events(FALSE); /* Get everything from the GL-queue. */
  new_events = 0;
  head = tail = 0;
}

long fl_qread(short *data)
/* reads an event from the queue. if empty, it waits for an event */
{
  short dev;
  while (head == tail)
  {
    fl_treat_interaction_events(TRUE);
    fl_treat_user_events();
  }
  dev = thedev[tail];
  *data = theval[tail];
  tail = (tail+1) % QSIZE;
  return  (long) dev;
}

long fl_blkqread(short *data,short n)
{
  long i = 0;
  if (head == tail) return 0;
  while (i < n-1 && head != tail)
  { 
    data[i++] = thedev[tail];
    data[i++] = theval[tail];
    tail = (tail+1) % QSIZE;
  }
  return i/2;
}

void fl_tie(Device action, Device dev1, Device dev2)
{
  tie(action,dev1,dev2);
}

void fl_treat_user_events()
/* Treats all new user events, i.e., either calls the callback routine
   or places FL_EVENT in the object queue. */
{
  short qtype,val;
  int i;
  if (new_events == 0) return;
  if (fl_event_call_back != NULL)
  {
    fl_restore_user_window();
    while (new_events>0)
    {
      new_events--;
      qtype = (short) fl_qread(&val);
      (*(fl_event_call_back))(qtype,val);      
    }
    fl_save_user_window();
  }
  else
  {
    for (i=0; i<new_events; i++) fl_object_qenter(FL_EVENT);
    new_events = 0;
  }
}

/*************** THE OBJECT EVENTS ********************/

FL_OBJECT *FL_EVENT = (FL_OBJECT *) (-1);	/* The special event object */

static FL_OBJECT *theobj[QSIZE]; 	 	/* The object event queue */
static int ohead = 0 , otail = 0;

void fl_object_qenter(FL_OBJECT *obj)
/* Adds an object to the queue */
{
  if (ohead == otail-1 || (ohead == QSIZE-1 && otail == 0))
    otail = (otail+1) % QSIZE;
  theobj[ohead] = obj;
  ohead = (ohead+1) % QSIZE;
}

FL_OBJECT *fl_object_qread()
/* reads an object from the queue.*/
{
  FL_OBJECT *obj;
  if (ohead == otail) return NULL;
  obj = theobj[otail];
  otail = (otail+1) % QSIZE;
  if (obj != FL_EVENT && obj->object_call_back != NULL)
  { 
    fl_restore_user_window();
    (*(obj->object_call_back))(obj,obj->argument);
    fl_save_user_window();
    return NULL;
  }
  else if (obj != FL_EVENT && obj->form->form_call_back != NULL)
  { 
    fl_restore_user_window();
    (*(obj->form->form_call_back))(obj);
    fl_save_user_window();
    return NULL;
  }
  return obj;
}
