/*
 * button.c
 *
 * Forms Object class: BUTTON, LIGHTBUTTON, ROUNDBUTTON
 *
 * Written by: Mark Overmars
 *
 * Version 2.2 a
 * Date: May 25, 1993
 */

#include <malloc.h>
#include <gl/gl.h>
#include <sys/types.h>
#include "forms.h"

/* Object specific information */
typedef struct{
   int val;		/* whether on */
   int mousebut;	/* Number of the mouse button that caused the push */
   int timdel;		/* time delay since last touch (for TOUCH buttons)*/
} SPEC;

/********** DRAWING *************/

static void draw_button(FL_OBJECT *ob)
/* Draws the object */
{
  int col;
  if (((SPEC *)(ob->spec))->val) col = ob->col2; else col = ob->col1;
  if (ob->belowmouse && col == FL_BUTTON_COL1) col = FL_BUTTON_MCOL1;
  if (ob->belowmouse && col == FL_BUTTON_COL2) col = FL_BUTTON_MCOL2;
  if (ob->boxtype == FL_UP_BOX && ((SPEC *)(ob->spec))->val)
    fl_drw_box(FL_DOWN_BOX,ob->x,ob->y,ob->w,ob->h,col,FL_BUTTON_BW);
  else
    fl_drw_box(ob->boxtype,ob->x,ob->y,ob->w,ob->h,col,FL_BUTTON_BW);
  fl_drw_text_beside(ob->align,ob->x,ob->y,ob->w,ob->h,
			ob->lcol,ob->lsize,ob->lstyle,ob->label);
  if (ob->type == FL_RETURN_BUTTON)
    fl_drw_text(0,ob->x+ob->w-0.8*ob->h,ob->y+0.2*ob->h,
			0.6*ob->h,0.6*ob->h,ob->lcol,0.0,0,"@returnarrow");
}

static void draw_lightbutton(FL_OBJECT *ob)
/* Draws the lightbutton */
{
  int c1,c2, libox;
  float xx,yy,ww,hh;
  /* Draw the box */
  if (ob->belowmouse) c1 = FL_LIGHTBUTTON_MCOL; else c1 = FL_LIGHTBUTTON_TOPCOL;
  if (((SPEC *)(ob->spec))->val) c2 = ob->col2; else c2 = ob->col1;
  fl_drw_box(ob->boxtype,ob->x,ob->y,ob->w,ob->h,c1,FL_LIGHTBUTTON_BW1);
  /* Calculate and draw the light */
  hh = ob->h - 3.0*FL_LIGHTBUTTON_BW1 - 1.0;
  if (hh < FL_LIGHTBUTTON_MINSIZE) hh = FL_LIGHTBUTTON_MINSIZE;
  ww = 0.5 * hh;
  if (ww < FL_LIGHTBUTTON_MINSIZE) ww = FL_LIGHTBUTTON_MINSIZE;
  if (ww > ob->w / 5.0) ww = ob->w / 5.0;
  xx = ob->x + 1.5*FL_LIGHTBUTTON_BW1 + 1.0;
  yy = ob->y + ob->h/2.0 - hh/2.0;
  switch (ob->boxtype)
  {
    case FL_UP_BOX:             libox = FL_DOWN_BOX; break;
    case FL_DOWN_BOX:           libox = FL_DOWN_BOX; break;
    case FL_FRAME_BOX:          libox = FL_FRAME_BOX; break;
    case FL_ROUNDED_BOX:        libox = FL_ROUNDED_BOX; break;
    case FL_RFLAT_BOX:          libox = FL_ROUNDED_BOX; break;
    case FL_RSHADOW_BOX:        libox = FL_ROUNDED_BOX; break;
    default:                    libox = FL_BORDER_BOX; break;
  }
  fl_drw_box(libox,xx,yy,ww,hh,c2,FL_LIGHTBUTTON_BW2);
  /* Draw the label */
  if (ob->align == FL_ALIGN_CENTER)
    fl_drw_text(FL_ALIGN_LEFT,xx+ww,ob->y,0.0,ob->h,
			ob->lcol,ob->lsize,ob->lstyle,ob->label);
  else
    fl_drw_text_beside(ob->align,ob->x,ob->y,ob->w,ob->h,
			ob->lcol,ob->lsize,ob->lstyle,ob->label);
  if (ob->type == FL_RETURN_BUTTON)
    fl_drw_text(0,ob->x+ob->w-0.8*ob->h,ob->y+0.2*ob->h,
			0.6*ob->h,0.6*ob->h,ob->lcol,0.0,0,"@returnarrow");
}

static void draw_roundbutton(FL_OBJECT *ob)
/* Draws a round button */
{
  int c1;
  float xx,yy,rr;
  if (ob->belowmouse) c1 = FL_ROUNDBUTTON_MCOL; else c1 = FL_ROUNDBUTTON_TOPCOL;
  fl_drw_box(ob->boxtype,ob->x,ob->y,ob->w,ob->h,c1,FL_ROUNDBUTTON_BW);
  xx = ob->x + 0.4*ob->h;
  yy = ob->y + 0.5*ob->h;
  rr = 0.4*ob->h - 1.5*FL_ROUNDBUTTON_BW;
  fl_color(ob->col1);
  circf(xx,yy,rr);
  fl_color(BLACK);
  circ(xx,yy,rr);
  if (((SPEC *)(ob->spec))->val)
  {
    fl_color(ob->col2);
    circf(xx,yy,0.8*rr);
    fl_color(BLACK);
    circ(xx,yy,0.8*rr);
  }
  if (ob->align == FL_ALIGN_CENTER)
    fl_drw_text(FL_ALIGN_LEFT,xx+rr,ob->y,0.0,ob->h,
			ob->lcol,ob->lsize,ob->lstyle,ob->label);
  else
    fl_drw_text_beside(ob->align,ob->x,ob->y,ob->w,ob->h,
			ob->lcol,ob->lsize,ob->lstyle,ob->label);
  if (ob->type == FL_RETURN_BUTTON)
    fl_drw_text(0,ob->x+ob->w-0.8*ob->h,ob->y+0.2*ob->h,
			0.6*ob->h,0.6*ob->h,ob->lcol,0.0,0,"@returnarrow");
}

static int handle_it(FL_OBJECT *ob, int event, float mx, float my, char key)
/* Handles an event */
{
  static int oldval;
  int newval;
  SPEC *sp = ((SPEC *)(ob->spec));
  switch (event)
  {
    case FL_DRAW:
	if (ob->type != FL_HIDDEN_BUTTON && ob->type != FL_HIDDEN_RET_BUTTON)
	  if (ob->objclass == FL_BUTTON)           draw_button(ob);
 	  else if (ob->objclass == FL_LIGHTBUTTON) draw_lightbutton(ob);
 	  else if (ob->objclass == FL_ROUNDBUTTON) draw_roundbutton(ob);
	return 0;
    case FL_ENTER:
    case FL_LEAVE:
        fl_redraw_object(ob);
	return 0;
    case FL_PUSH:
        oldval = sp->val;
        sp->val = ! sp->val;
	sp->mousebut = (int) key;
	sp->timdel = 1;
        fl_redraw_object(ob);
	return (ob->type == FL_INOUT_BUTTON || ob->type == FL_TOUCH_BUTTON);
    case FL_RELEASE:
	sp->mousebut = (int) key;
	if (ob->type == FL_RADIO_BUTTON)
	    return 1;
	else if (ob->type == FL_PUSH_BUTTON)
	    return (sp->val != oldval);
	else if (sp->val == 0) 
	    return 0;
	sp->val = 0;
        fl_redraw_object(ob);
	return (ob->type != FL_TOUCH_BUTTON);
    case FL_MOUSE:
	if (ob->type != FL_RADIO_BUTTON)
	{
	    if (mx<ob->x || mx>ob->x+ob->w || my<ob->y || my>ob->y+ob->h)
		newval = oldval;
	    else
		newval = !oldval;
	    if (sp->val != newval)
	        { sp->val = newval; fl_redraw_object(ob);}
	}
	return (sp->val && ob->type == FL_TOUCH_BUTTON && sp->timdel++ % 12 == 0);
    case FL_SHORTCUT:
	if (ob->type == FL_PUSH_BUTTON)
	  { sp->val = ! sp->val; fl_redraw_object(ob);}
	return (ob->type != FL_RADIO_BUTTON);
    case FL_FREEMEM:
	free(ob->spec);
	return 0;
  }
  return 0;
}

static FL_OBJECT *create_it(int objclass, int type,
			float x, float y, float w, float h, char label[])
/* Creates an object */
{
  FL_OBJECT *ob;
  ob = fl_make_object(objclass,type,x,y,w,h,label,handle_it);
  if (type == FL_RADIO_BUTTON) ob->radio = 1;
  if (type == FL_RETURN_BUTTON || type == FL_HIDDEN_RET_BUTTON)
    fl_set_object_shortcut(ob,"^M");
  ob->spec = (int *) fl_malloc(sizeof(SPEC));
  ((SPEC *)(ob->spec))->val = 0;
  ((SPEC *)(ob->spec))->timdel = 0;
  return ob;
}

/*------------------------------*/

void fl_set_button(FL_OBJECT *ob, int pushed)
/* Sets the button */
{
  ((SPEC *)(ob->spec))->val = pushed;
  if (ob->type == FL_RADIO_BUTTON) ob->pushed =pushed;
  fl_redraw_object(ob);
}

int fl_get_button(FL_OBJECT *ob)
/* Returns value of the button */
  { return ((SPEC *)(ob->spec))->val; }

int fl_get_button_numb(FL_OBJECT *ob)
/* Returns the number of the last used mouse button */
  { return ((SPEC *)(ob->spec))->mousebut; }

void fl_set_button_shortcut(FL_OBJECT *ob, char str[])
/* Sets the shortcut for the button */
{ 
  if (ob->type == FL_RADIO_BUTTON)
    fl_error("fl_set_button_shortcut", "Cannot set shortcuts for radio buttons.");
  else
    fl_set_object_shortcut(ob,str);
}

/*------------------------------*/

FL_OBJECT *fl_create_button(int type,float x,float y,float w,float h,char label[])
/* creates an object */
{
  FL_OBJECT *ob;
  ob = create_it(FL_BUTTON,type,x,y,w,h,label);
  ob->boxtype = FL_BUTTON_BOXTYPE;
  ob->col1 = FL_BUTTON_COL1;
  ob->col2 = FL_BUTTON_COL2;
  ob->align = FL_BUTTON_ALIGN;
  ob->lcol = FL_BUTTON_LCOL;
  return ob;
}

FL_OBJECT *fl_create_lightbutton(int type,float x,float y,float w,float h,char label[])
/* creates an object */
{
  FL_OBJECT *ob;
  ob = create_it(FL_LIGHTBUTTON,type,x,y,w,h,label);
  ob->boxtype = FL_LIGHTBUTTON_BOXTYPE;
  ob->col1 = FL_LIGHTBUTTON_COL1;
  ob->col2 = FL_LIGHTBUTTON_COL2;
  ob->align = FL_LIGHTBUTTON_ALIGN;
  ob->lcol = FL_LIGHTBUTTON_LCOL;
  return ob;
}

FL_OBJECT *fl_create_roundbutton(int type,float x,float y,float w,float h,char label[])
/* creates an object */
{
  FL_OBJECT *ob;
  ob = create_it(FL_ROUNDBUTTON,type,x,y,w,h,label);
  ob->boxtype = FL_ROUNDBUTTON_BOXTYPE;
  ob->col1 = FL_ROUNDBUTTON_COL1;
  ob->col2 = FL_ROUNDBUTTON_COL2;
  ob->align = FL_ROUNDBUTTON_ALIGN;
  ob->lcol = FL_ROUNDBUTTON_LCOL;
  return ob;
}

FL_OBJECT *fl_add_button(int type,float x,float y,float w,float h,char label[])
/* Adds an object */
{
  FL_OBJECT *ob;
  ob = fl_create_button(type,x,y,w,h,label);
  fl_add_object(fl_current_form,ob);
  return ob;
}

FL_OBJECT *fl_add_lightbutton(int type,float x,float y,float w,float h,char label[])
/* Adds an object */
{
  FL_OBJECT *ob;
  ob = fl_create_lightbutton(type,x,y,w,h,label);
  fl_add_object(fl_current_form,ob);
  return ob;
}

FL_OBJECT *fl_add_roundbutton(int type,float x,float y,float w,float h,char label[])
/* Adds an object */
{
  FL_OBJECT *ob;
  ob = fl_create_roundbutton(type,x,y,w,h,label);
  fl_add_object(fl_current_form,ob);
  return ob;
}
