/*
 * $Id: udunits.h,v 1.4 1992/01/24 17:12:01 steve Exp $
 *
 * Copyright (C) 1990 UCAR/Unidata
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose without fee is hereby granted, provided
 * that the above copyright notice appear in all copies, that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of UCAR/Unidata not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  UCAR makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  It is
 * provided with no support and without obligation on the part of UCAR or
 * Unidata, to assist in its use, correction, modification, or enhancement.
 *
 * Public header-file for the Unidata units(3) library.
 */

#ifndef UT_UNITS_H_INCLUDED
#define UT_UNITS_H_INCLUDED

#define UT_NAMELEN              32      /* maximum length of a unit string
					 * (including all prefixes and EOS) */

typedef float	UtOrigin;		/* origin type */
typedef double	UtFactor;               /* conversion-factor type */

#define UT_MAXNUM_BASE_QUANTITIES	10

/*
 *  Unit-structure:
 */
typedef struct {
    UtOrigin	origin;				/* origin */
    UtFactor    factor;                         /* multiplicative scaling 
						 *  factor (e.g. the "2.54" in
						 * "2.54 cm") */
    short       power[UT_MAXNUM_BASE_QUANTITIES];
						/* exponent of basic units */
}   utUnit;

/*
 *	Macro for declaring functions regardless of the availability of 
 *	function prototypes.  NB: will need double parens in actual use (e.g.
 *	"int func PROTO((int a, char *cp))").
 */
#ifndef	PROTO
#   if defined(__STDC__) || defined(__GNUC__) \
	|| defined(__cplusplus) || defined(c_plusplus)
#	define	PROTO(a)	a
#   else
#	define	PROTO(a)	()
#   endif
#endif


#define	UT_EOF		 1	/* end-of-file encountered */
#define	UT_ENOFILE	-1	/* no units-file */
#define	UT_ESYNTAX	-2	/* syntax error */
#define	UT_EUNKNOWN	-3	/* unknown specification */
#define	UT_EIO		-4	/* I/O error */
#define	UT_EINVALID	-5	/* invalid unit-structure */
#define	UT_ENOINIT	-6	/* package not initialized */
#define	UT_ECONVERT	-7	/* two units are not convertable */
#define	UT_EALLOC	-8	/* memory allocation failure */


/*
 *  Initialize the units(3) package.
 */
extern int	utInit		PROTO((
    const char	*path
));

/*
 *	Decode a formatted unit specification into a unit-structure.
 */
extern int	utScan		PROTO((
    char	*spec,
    utUnit	*up
));

/*
 *	Clear a unit structure.
 */
extern utUnit*	utClear		PROTO((
    utUnit	*unit
));

/*
 *	Copy a unit-strcture.
 */
extern utUnit*	utCopy		PROTO((
    const utUnit	*source,
    utUnit		*dest
));

/*
 *	Multiply one unit-structure by another.
 */
extern utUnit*	utMultiply	PROTO((
    utUnit	*term1,
    utUnit	*term2, 
    utUnit	*result
));

/*
 *	Divide one unit-structure by another.
 */
extern utUnit*	utDivide	PROTO((
    utUnit	*numer,
    utUnit	*denom, 
    utUnit	*result
));

/*
 *	Form the reciprocal of a unit-structure.
 */
extern utUnit*	utInvert	PROTO((
    utUnit	*source,
    utUnit	*dest
));

/*
 *	Raise a unit-structure to a power.
 */
extern utUnit*	utRaise		PROTO((
    utUnit	*source,
    int		power,
    utUnit	*result
));

/*
 *	Shift the origin of a unit-structure.
 */
extern utUnit*	utShift		PROTO((
    utUnit     *source,
    double      amount,
    utUnit	*result
));

/*
 *	Scale a unit-structure.
 */
extern utUnit*	utScale		PROTO((
    utUnit	*source,
    double	factor,
    utUnit	*result
));

/*
 *	Compute the conversion factor between two unit-structures.
 */
extern int	utConvert	PROTO((
    utUnit	*from,
    utUnit	*to,
    double	*slope,
    double	*intercept
));

/*
 *	Encode a unit-structure into a formatted unit-specification.
 */
extern int	utPrint		PROTO((
    utUnit	*unit,
    char	**buf
));

/*
 *	Terminate use of this package.
 */
extern void	utTerm		PROTO(());

#endif	/* UT_UNITS_H_INCLUDED not defined */
