/*
 * $Id: dectab.c,v 2.9 1991/11/18 16:02:48 steve Exp $
 * $__Header$
 *
 * This file contains an implementation of the functional routines which
 * support the decode-table abstract data type.
 */

/*LINTLIBRARY*/

#include "udposix.h"
#include <stddef.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include "uderrmsg.h"
#include "udalloc.h"
#include "udres.h"
#include "dectab.h"

typedef struct {
    char	*name;
    int		*count;
    VOIDP	where;
    UdConverter	*funct;
    int		nval;
    int		required;
}   KeyNode;

typedef struct {
    char	*name;
    int		*count;
    VOIDP	where;
    UdConverter	*funct;
    int		nval;
    int		nleave;
    int		required;
}   PosNode;

typedef struct DtTag {
    int		WasError;
    int		nkey;
    int		npos;
    enum {
	unknown,
	keyed,
	positional
    }		CurrItem;
    KeyNode	*CurrKey;
    PosNode	*CurrPos;
    KeyNode	*key;
    PosNode	*pos;
}   DtObject;


/*
 * WHAT:   Compare two keyed-parameter entries and indicate whether or not the 
 *	   first entry is less than, equal to, or greater than the second.  
 *	   This routine is internal to this package and is used to order the
 *	   keyed-parameter entries.
 *
 * INPUT:  Pointers to keyed-parameter entries.
 *
 * OUTPUT: A negative, zero, or positive integer dependending on whether the
 *	   first entry is lexically less-than, equal-to, or greater-than the
 *	   second.
 *
 * HOW:	   Perform a lexical comparison of the keyed-parameters names.
 */

    static int
KeyCompare(first, second)
    const VOIDP	first;
    const VOIDP	second;
{
    assert(first != NULL);
    assert(second != NULL);

    return strcmp(((KeyNode*)first)->name, ((KeyNode*)second)->name);
}


/*
 * WHAT:   Free a decode-table.
 *
 * HOW:    Free the memory allocated to the vectors in the decode table; then
 *	   free the decode-table.
 *
 * INPUT:  A pointer to a decode-table.
 */

    void
DtFree(dt)
    DtPtr	dt;
{
    if (dt != NULL) {
	if (dt->key != NULL)
	    (void)FREE(dt->key);
	if (dt->pos != NULL)
	    (void)FREE(dt->pos);
	(void)FREE(dt);
    }
}


/*
 * WHAT	   Check a decode-table entry.
 */

    static int
CheckEntry(name, count, funct, where, nval, required)
    char	*name;		/* Name of entry */
    int		*count;		/* Place to store number of values */
    UdConverter	*funct;		/* Converter function */
    VOIDP	where;		/* Place to store binary values */
    int		nval;		/* Expected number of values */
    int		required;	/* Required parameter? */
{
    int		status	= 0;		/* Routine status = error */
    static char	me[]	= "CheckEntry";	/* This routine's name */

    if (name == NULL) {
	udadvise("%s: Decode-table entry has no name.", me);
    } else if (funct == NULL && (where != NULL || count != NULL)) {
	udadvise("%s: Parameter \"%s\" %s.", me, name,
	    "has `where' or `count' field but no converter function");
    } else if (required && (funct == NULL || where == NULL)) {
	udadvise("%s: Required parameter \"%s\" %s.", 
	    me, name, "has no `converter' and/or `where' field");
    } else if (nval == UDARB && where != NULL && count == NULL) {
	udadvise("%s: Parameter \"%s\" %s.", me, name, 
	    "takes arbitrary number of values but has no `count' field");
    } else if (nval != UDARB && nval < 0) {
	udadvise("%s: Parameter \"%s\" has negative `nval' field", 
	    me, name);
    } else {
	status	= 1;
    }

    return status;
}


/*
 * WHAT:   Create a decode-table suitable for efficient argument-parsing.
 *
 * INPUT:  A (possibly nil) keyed-parameter table and a (possibly nil) 
 *	   positional-parameter table.  Each table must have a nil final
 *	   entry.
 *
 * OUTPUT: Either a pointer to the created decode-table or a nil pointer
 *	   signifying failure.
 *
 * HOW:    Allocate memory for the table structure.  For each non-nil input
 *	   table, allocate memory for the associated decode-table structure
 *	   and initialize the allocated structure from the associated input 
 *	   table.  Sort the keyed-parameter table by name for more efficient 
 *	   searching.  For each positional-parameter, determine the minimum 
 *	   number of positional parameters values that should follow it.
 */

    DtPtr
DtCreate(KeyTab, PosTab)
    UdKeyEntry	*KeyTab;	/* Keyed-parameter table */
    UdPosEntry	*PosTab;	/* Positional-parameter table */
{
    int		success	= 1;		/* Routine status */
    int		errmode	= uderrmode(UD_VERBOSE);
    DtPtr	dt	= UD_ALLOC(1, DtObject);
    UdKeyEntry	*ke;
    KeyNode	*kn;
    static char	me[]	= "DtCreate";

    if (dt == NULL) {
	udadvise("%s: Couldn't allocate decode-table.", me);
	success	= NO;

    } else {
	dt->WasError	= 0;
	dt->nkey	= 0;
	dt->npos	= 0;
	dt->key		= NULL;
	dt->pos		= NULL;
	dt->CurrItem	= unknown;
	dt->CurrKey	= NULL;
	dt->CurrPos	= NULL;

	if (KeyTab != NULL && KeyTab->name != NULL) {
	    for (dt->nkey = 0; KeyTab[dt->nkey].name != NULL; ++dt->nkey)
		;

	    if ((dt->key = UD_ALLOC((size_t)dt->nkey, KeyNode)) == NULL) {
		udadvise("%s: Couldn't allocate %u-element key-node vector.",
		    me, (unsigned)dt->nkey);
		success	= NO;

	    } else {
		ke	= KeyTab;
		kn	= dt->key;

		for (; ke < KeyTab + dt->nkey; ++ke, ++kn) {

		    if (!CheckEntry(ke->name, ke->count, ke->funct, ke->where,
			ke->nval, ke->required)) {

			success	= 0;
		    } else {
			kn->name	= ke->name;
			kn->count	= ke->count;
			kn->funct	= ke->funct;
			kn->where	= ke->where;
			kn->nval	= ke->nval;
			kn->required	= ke->required == UDREQ;

			if (kn->nval != UDARB && kn->count != NULL)
			    kn->nval	= -kn->nval;
		    }
		}

		(void)qsort((VOIDP)dt->key, dt->nkey, sizeof(KeyNode), 
		    KeyCompare);
	    }
	}

	if (success) {
	    if (PosTab != NULL && PosTab->name != NULL) {
		UdPosEntry	*pe;
		PosNode	*pn;

		for (dt->npos = 0; PosTab[dt->npos].name != NULL; ++dt->npos)
		    ;

		if ((dt->pos = UD_ALLOC((size_t)dt->npos, PosNode)) == NULL) {
		    udadvise("%s: Couldn't allocate %d-element PosNode vector.",
			me, (unsigned)dt->npos);
		    success	= NO;

		} else {
		    pe	= PosTab;
		    pn	= dt->pos;

		    for (; pe < PosTab + dt->npos; ++pe, ++pn) {

			if (!CheckEntry(pe->name, pe->count, pe->funct, 
			    pe->where, pe->nval, pe->required)) {

			    success	= 0;
			} else {
			    pn->name	= pe->name;
			    pn->count	= pe->count;
			    pn->funct	= pe->funct;
			    pn->where	= pe->where;
			    pn->nval	= pe->nval == 0 ? 1 : pe->nval;
			    pn->required	= pe->required == 0 ||
					      pe->required == UDREQ;

			    if (pn->nval != UDARB && pn->count != NULL)
				pn->nval	= -pn->nval;
			}
		    }

		    /*
		     * Set the minimum number of values that each positional 
		     * parameter should leave on the command-line for the
		     * subsequent parameters.
		     */
		    if (success) {
			int	nleave	= 0;

			for (pn = dt->pos + dt->npos; pn-- > dt->pos; ) {
			    pn->nleave	= nleave;
			    if (pn->count == NULL)	/* if fixed-number */
				nleave	+= pn->nval;
			}
		    }
		}
	    }
	}

	if (!success) {
	    DtFree(dt);
	    dt	= NULL;
	}

	(void)uderrmode(errmode);
    }

    return dt;
}


/*
 * WHAT:   Find the keyed-parameter having a given name.
 *
 * INPUT:  Pointer to a decode-table and the -- possibly abbreviated -- name of 
 *	   the desired keyed-parameter.
 *
 * OUTPUT: A success or failure flag.  On success, the current item is set
 *	   to the found keyed-parameter.
 *
 * HOW:    Perform a binary search on the keyed-parameter array.
 *	   Determine whether or not the given name uniquely
 *	   identifies a keyed-parameter.
 */

    int
DtKeyFind(dt, name)
    DtPtr	dt;		/* pointer to decode-table */
    char	*name;		/* keyed-parameter name */
{
    int		success	= 0;		/* routine status = failure */
    int		errmode	= uderrmode(UD_VERBOSE);
    static char	me[]	= "DtKeyFind";

    assert(dt != NULL);
    assert(name != NULL);

    if (dt->nkey > 0) {
	int	ikey, ileft, iright;
	int	len	= strlen(name);

	for (ikey = 0, ileft = 0, iright = dt->nkey - 1; 
	    ileft <= iright;
	    ikey = (ileft+iright)/2) {

	    KeyNode	*kn	= dt->key + ikey;
	    int	i	= strncmp(name, kn->name, len);

	    if (i < 0) {
		iright	= ikey - 1;
	    } else if (i > 0) {
		ileft	= ikey + 1;
	    } else {
		if (ikey > 0 && strncmp(name, kn[-1].name, len) == 0 ||
		    ikey < dt->nkey-1 && strncmp(name, kn[1].name, 
			len) == 0) {

		    udadvise("%s: Option \"%s\" is ambiguous.", me, name);
		} else {
		    dt->CurrItem	= keyed;
		    dt->CurrKey		= kn;
		    success		= 1;
		}
		break;
	    }
	}
    }

    (void)uderrmode(errmode);

    return success;
}


/*
 * WHAT:   Go to the next positional-parameter.
 *
 * INPUT:  Pointer to a decode-table.
 *
 * OUTPUT: A success or failure flag.  On success, the current item is set
 *	   to the next positional-parameter.
 *
 * HOW:    If another positional-parameter exists in the positional-parameter
 *	   list, set the current item to point to it.
 */

    int
DtPosFind(dt)
    DtPtr	dt;		/* pointer to decode-table */
{
    int		success	= 1;		/* routine status */

    assert(dt != NULL);

    if (dt->npos > 0) {
	if (dt->CurrPos == NULL) {
	    dt->CurrPos		= dt->pos;
	    dt->CurrItem	= positional;

	} else if (dt->CurrPos < dt->pos + dt->npos - 1) {
	    ++dt->CurrPos;
	    dt->CurrItem	= positional;

	} else {
	    success	= 0;
	}
    } else {
	success	= 0;
    }

    return success;
}


/*
 * WHAT	   Reset the decode-table for subsequent "DtNextPar" processing.
 */

    void
DtReset(dt)
    DtPtr	dt;		/* Pointer to decode-table */
{
    dt->CurrItem	= unknown;
    dt->CurrKey		= NULL;
    dt->CurrPos		= NULL;
}


/*
 * WHAT	   Go to the next keyed-parameter.
 */

    int
DtNextKey(dt)
    DtPtr	dt;		/* Pointer to decode-table */
{
    int		success	= 0;		/* Routine status = no more */

    assert(dt != NULL);

    if (dt->nkey > 0 && (dt->CurrKey == NULL || 
	dt->CurrKey < dt->key + dt->nkey - 1)) {

	if (dt->CurrKey == NULL) {
	    dt->CurrKey		= dt->key;
	    dt->CurrItem	= keyed;
	} else {
	    ++dt->CurrKey;
	}

	success	= 1;
    }

    return success;
}


/*
 * WHAT	   Go to the next positional-parameter.
 */

    int
DtNextPos(dt)
    DtPtr	dt;		/* Pointer to decode-table */
{
    int		success	= 0;		/* Routine status = no more */

    assert(dt != NULL);

    if (dt->npos > 0 && (dt->CurrPos == NULL ||
	dt->CurrPos < dt->pos + dt->npos - 1)) {

	if (dt->CurrPos == NULL) {
	    dt->CurrPos		= dt->pos;
	    dt->CurrItem	= positional;
	} else {
	    ++dt->CurrPos;
	}

	success	= 1;
    }

    return success;
}


/*
 * WHAT	   Go the the next parameter.
 */

    int
DtNextPar(dt)
    DtPtr	dt;		/* Pointer to decode-table */
{
    assert(dt != NULL);

    return DtNextKey(dt) || DtNextPos(dt);
}


/*
 * WHAT    Return the number of values for the current item.
 *
 * INPUT   Pointer to a decode-table.
 *
 * OUTPUT  Either the number of values or 0 if no current item exists.
 *
 * HOW	   Use the type of the current item to return the appropriate number
 *	   of values.
 */

    int
DtNval(dt)
    DtPtr	dt;		/* Decode-table */
{
    int		nval	= 0;		/* Number of values */

    assert(dt != NULL);
    assert(dt->CurrItem == keyed || dt->CurrItem == positional);

    if (dt->CurrItem == keyed) {
	nval	= dt->CurrKey->nval;
    } else if (dt->CurrItem == positional) {
	nval	= dt->CurrPos->nval;
    }

    return nval;
}


/*
 * WHAT	   Return the name of the current item.
 *
 * INPUT   Pointer to a decode-table.
 *
 * OUTPUT  Either the name of the current item or a nil pointer if the current
 *	   item doesn't exist.
 *
 * HOW	   Use the type of the current item to return the appropriate name.
 */

    char*
DtName(dt)
    DtPtr	dt;		/* Decode-table */
{
    char	*name	= NULL;		/* The returned name */

    assert(dt != NULL);

    if (dt->CurrItem == keyed) {
	name	= dt->CurrKey->name;
    } else if (dt->CurrItem == positional) {
	name	= dt->CurrPos->name;
    }

    assert(name != NULL);

    return name;
}


/*
 * WHAT	   Return the minimum number of positional-parameter values to leave on
 *	   the command-line.
 *
 * INPUT   Pointer to a decode-table.
 *
 * OUTPUT  Either the number to leave or 0 if the current item isn't a 
 *	   positional-parameter.
 *
 * HOW	   Return the number to leave from its structure.
 */

    int
DtNleave(dt)
    DtPtr	dt;		/* Decode-table */
{
    int		nleave;			/* The returned number-to-leave */

    assert(dt != NULL);
    assert(dt->CurrItem == keyed || dt->CurrItem == positional);

    if (dt->CurrItem == keyed) {
	nleave	= 0;
    } else if (dt->CurrItem == positional) {
	nleave	= dt->CurrPos->nleave;
    }

    assert(nleave >= 0);

    return nleave;
}


/*
 * WHAT	   Indicate whether or not the current item is required.
 *
 * INPUT   Pointer to a decode-table.
 *
 * OUTPUT  1 iff the item is required; 0 iff not.
 *
 * HOW	   Return the appropriate structure member.
 */

    int
DtRequired(dt)
    DtPtr	dt;		/* Decode-table */
{
    int		required	= 0;	/* The returned number-to-leave */

    assert(dt != NULL);
    assert(dt->CurrItem == keyed || dt->CurrItem == positional);

    if (dt->CurrItem == keyed) {
	required	= dt->CurrKey->required;
    } else if (dt->CurrItem == positional) {
	required	= dt->CurrPos->required;
    }

    return required;
}


/*
 * WHAT	   Return a pointer to the converter function of the current item.
 *
 * INPUT   Pointer to a decode-table.
 *
 * OUTPUT  Either a pointer to a converter function or a nil pointer if the 
 *	   current item doesn't exist.
 *
 * HOW	   Use the type of the current item to return the appropriate structure
 *	   member.
 */

    UdConverter*
DtFunct(dt)
    DtPtr	dt;		/* Decode-table */
{
    UdConverter	*funct	= NULL;		/* The returned function */

    assert(dt != NULL);
    assert(dt->CurrItem == keyed || dt->CurrItem == positional);

    if (dt->CurrItem == keyed) {
	funct	= dt->CurrKey->funct;
    } else if (dt->CurrItem == positional) {
	funct	= dt->CurrPos->funct;
    }

    return funct;
}


/*
 * WHAT	   Return a pointer to the immediate-storage location of the current 
 *	   item.
 *
 * INPUT   Pointer to a decode-table.
 *
 * OUTPUT  Either a pointer to a immediate-storage location or a nil pointer if
 *	   the current item doesn't exist.
 *
 * HOW	   Use the type of the current item to return the appropriate structure
 *	   member.
 */

    VOIDP
DtWhere(dt)
    DtPtr	dt;		/* Decode-table */
{
    VOIDP	where	= NULL;		/* The returned pointer */

    assert(dt != NULL);
    assert(dt->CurrItem == keyed || dt->CurrItem == positional);

    if (dt->CurrItem == keyed) {
	where	= dt->CurrKey->where;
    } else if (dt->CurrItem == positional) {
	where	= dt->CurrPos->where;
    }

    return where;
}


/*
 * WHAT	   Return a pointer to the immediate-storage count location of the 
 *	   current item.
 *
 * INPUT   Pointer to a decode-table.
 *
 * OUTPUT  Either a pointer to a immediate-storage count location or a nil 
 *	   pointer if the current item doesn't exist.
 *
 * HOW	   Use the type of the current item to return the appropriate structure
 *	   member.
 */

    int*
DtCount(dt)
    DtPtr	dt;		/* Decode-table */
{
    int		*count	= NULL;		/* The returned pointer */

    assert(dt != NULL);
    assert(dt->CurrItem == keyed || dt->CurrItem == positional);

    if (dt->CurrItem == keyed) {
	count	= dt->CurrKey->count;
    } else if (dt->CurrItem == positional) {
	count	= dt->CurrPos->count;
    }

    return count;
}


/*
 * WHAT	   Indicate whether or not an error was encountered.
 *
 * INPUT   Pointer to a decode-table.
 *
 * OUTPUT  Error flag: 1 => yes, an error occured; 0 => no, there were no
 *	   errors.
 *
 * HOW	   Return the error flag of the decode-table.
 */

    int
DtWasError(dt)
    DtPtr	dt;		/* Decode-table */
{
    assert(dt != NULL);

    return dt->WasError;
}
