/*                      Copyright (c) 1992,1993 Bellcore
 *                            All Rights Reserved
 *       Permission is granted to copy or use this program, EXCEPT that it
 *       may not be sold for profit, the copyright notice must be reproduced
 *       on copies, and credit should be given to Bellcore where it is due.
 *       BELLCORE MAKES NO WARRANTY AND ACCEPTS NO LIABILITY FOR THIS PROGRAM.
 */

/* convert 1 chunk to audio */

#include <stdio.h>
#include "synth.h"
#include	"audio.h"
#include	"libmidi.h"
#include	"libmpu.h"

static struct note *insert_note(/* struct note *list,struct note *note */);
static struct note *find_note(/* struct note *list,int key,int chan */);
static struct note *del_note(/* struct note *list,int time */);
static int find_current(/* struct list *list,int current */);
static void free_list(/* struct list *list */);
struct audio_state *init_audio(/* struct audio_state *audio */);

int
to_audio(audio,params,buff)
struct audio_state *audio;	/* progress parameters */
struct params *params;	/* performance parameters */
int *buff;					/* where to put the samples */
	{
	MCMD *mpu = (MCMD *) 0;	/* pointer to current mpu item */
	int type;				/* type of midi command */
	struct note *note;	/* current note entry */
	int sample;				/* sample # of current note */
	int n;					/* # of finished notes */
	int hunk = P(hunk);	/* cache hunk value */

	/* reset for a new file */

	if (P(state) & RESTART) {
		int state = P(state);
		dprintf('m')(stderr,"Starting for a new file\n");
		audio = init_audio(audio);
		P(state) =  state & ~(RESTART|NEW_TEMPO);
		}

	if (P(state) & NEW_TEMPO) {
		dprintf('m')(stderr,"adjusting for a new tempo %d-%d\n",
				P(old),P(tempo));
		sample = (A(now)-A(offset)) * SPS / (2 * P(old));	/* previous sample */
		A(offset) = A(now);
		adjust_time(A(list),sample,P(old),P(tempo));
		A(done) -= sample;
		A(current) -= sample;
		P(state) &= ~NEW_TEMPO;
		}

	dprintf('m')(stderr,"hunk %d - %d [%d ready]\n",
				A(done),A(done)+P(hunk),A(current));
	while (!A(no_notes) && A(done) + P(hunk) > A(current)) {
		if (!(mpu = getmcmd(P(file), A(now)))) {
			dprintf('M')(stderr," Bad MPU command\n");
			if (!feof(P(file))) {
				dprintf('M')(stderr," Bad MPU command, aborting\n");
				hunk = 0;
				}
			else {
				dprintf('M')(stderr," End of file\n");
				A(no_notes)++;
				}
			break;
			}
		A(now) = mpu->when;
		if (A(id) == 0 && A(offset)==0) A(offset) = A(now);
		sample = (A(now)-A(offset)) * SPS / (2 * P(tempo));
		type = (mpu->cmd[0] & M_CMD_MASK);
		dprintf('M')(stderr," note %d sample %d type 0x%x\n",
				A(id),sample,type);

		if (type == CH_KEY_OFF) {	/* turn into key on with velocity 0 */
			type = CH_KEY_ON;
			mpu->cmd[2] = 0;
			}

		if (type == CH_KEY_ON) {	/* key transition */
			int chan = (mpu->cmd[0] & M_CHAN_MASK);
			int key = mpu->cmd[1];
			
			P(channels) |= 1<<chan;
			if (mpu->cmd[2]) {		/* key-on, make a note entry */
				dprintf('c')(stderr," note %d/%d (%d:%d) ON\n",
						A(id),sample,chan,key);
				if (note = find_note(A(list),key,chan)) {
					dprintf('e')(stderr," NOTE %d=%d (%d:%d) re-striking\n",
							A(id),note->id,chan,key);
					note->stop = sample-1;
					A(current) = find_current(A(list),sample-1);
					}
				note = (struct note *) malloc(sizeof(struct note));
				note->chan = chan;
				note->key = key;
				note->vel = mpu->cmd[2];
				note->start = sample;
				note->stop  = sample-1;		/* always invalid! */
				note->data = NULL;
				note->id = A(id)++;
				A(list) = insert_note(A(list),note);
				}
			else if (note=find_note(A(list),key,chan)) {	/* key off, finish note */
				dprintf('c')(stderr," note %d: (%d:%d) %d->%d (%d) OFF\n",
							note->id,note->chan,note->key,
							note->start,sample,sample-note->start);
				note->stop = sample;
				A(current) = find_current(A(list),sample-1);
				}
			else {
				dprintf('e')(stderr," NOTE %d (%d:%d) off without note on\n",
						A(id), chan,key);
				}
			}
		else
			dprintf('e')(stderr,"Got a weird midi command 0x%x\n",type);
		}

	if (A(no_notes)) {
		dprintf('m')(stderr,"Ran out of midi commands, hunk: %d->%d\n",
				hunk,A(current)-A(done));
		hunk = A(current)-A(done);
		P(state) &= ~RUNNING;
		fclose(P(file));
		P(file) = NULL;
		}

	/* now we can output a hunk */

	n = output(audio,params,buff);
	A(done) += hunk;
	if (n)
		A(list)=del_note(A(list),A(done));	
	return(hunk);
	}

struct audio_state *
init_audio(audio)
struct audio_state *audio;	/* previous state (or NULL if no state) */
	{
	dprintf('m')(stderr,"Initializing audio\n");
	if (audio)
		free_list(audio->list);
	else
		audio = (struct audio_state *) malloc(sizeof(*audio));
		
	audio->list = NULL;
	audio->current = 0;
	audio->done = 0;
	audio->now = 0;
	audio->offset = 0;
	audio->id = 0;
	audio->no_notes = 0;

	return(audio);
	}

/******************************************************************
 * note list management routes
 */

/* insert a note in the front of the list */

static struct note *
insert_note(list,note)
struct note *list;		/* pointer to first note */
register struct note *note;		/* note to insert */
	{
	note -> next = list;
	dprintf('l')(stderr,"    Inserting  %d (%d:%d)\n",
			note->id,note->chan,note->key);
	return(note);
	}

/* find an unfinished note */

static struct note *
find_note(list,key,chan)
struct note *list;		/* first note in the list */
int key;						/* which key */
int chan;					/* which channel */
	{
	register struct note *note;

	for(note=list;note;note=note->next)
		if (note->chan==chan && note->key==key && note->stop<note->start)
			break;
	if (note) dprintf('l')(stderr,"   found %d:%d = %d\n",
			chan,key,note->id);
	return(note);
	}

/* finish all notes (unused) */

static int
finish_notes(list,time)
struct note *list;		/* head of note list */
int time;					/* time to finish */
	{
	register struct note *note;
	int count = 0;
	for(note=list;note;note=note->next)
		if (note->stop < note->start) {
			note->stop=time;
			count++;
			}
	return(count);
	}

/* adjust all of the times for all of the notes */

static int
adjust_time(list,time,old,new)
struct note *list;		/* head of note list */
int time;					/* time offset */
int old;						/* old tempo */
int new;						/* new tempo */
	{
	register struct note *note;
	int dur;	/* note duration */

	dprintf('f')(stderr,"Adjusting time by %d\n",time);
	for(note=list;note;note=note->next) {
		note->start -= time;
		note->stop -= time;

		/* change note duration */

		if (!note->data && (dur=note->stop-note->start) > 0) {
			dur = dur * old / new;
			note->stop = note->start + dur;
			}
		}
	}
/* find latest sample we can now render */

static int
find_current(list,current)
struct note *list;		/* head of note list */
int current;					/* current sample */
	{
	register struct note *note;
	struct note *temp=NULL;
	int result = current;
	
	dprintf('f')(stderr,"# %d:",current);
	for(note=list;note;note=note->next)
		if (note->stop < note->start) {
			if (current-note->start-1 > MAXNOTE) {
				fprintf(stderr,"Note %d too long!\n",note->id);
				note->stop = current;
				}
			else
				result = note->start-1;
			dprintf('f')(stderr," %d->%d",note->id,current);
			temp = note;
			}
	dprintf('f')(stderr,"\n");
	if (temp)
		dprintf('l')(stderr,"  current -> %d (note %d)\n",current,temp->id);
	else
		dprintf('l')(stderr,"  current -> %d\n",current);
	return(result);
	}

/* delete all notes that are done playing  ???? */

static struct note *
del_note(list,time)
struct note *list;		/* first note in the list */
int time;					/* time spent */
	{
	struct note _data, *prev=&_data;	/* previous note */
	register struct note *note;	/* current note */

	prev->next = list;
	while (prev->next) {
		note = prev->next;
		if (note->stop < time) {	/* all done, delete */
			dprintf('n')(stderr," Deleting %d, end (%d) < now (%d)\n",
				note->id, note->stop, time);
			prev->next = note->next;
			if (note->data)
				free(note->data);
			free(note);
			}
		else
			prev=prev->next;
		}
	return((&_data)->next);
	}

/* free a list (not done too often) */

static void
free_list(list)
register struct note *list;
	{
	if (list) {
		free_list(list->next);
		if (list->data)
			free(list->data);
		free(list);
		}
	}
