/*                      Copyright (c) 1992,1993 Bellcore
 *                            All Rights Reserved
 *       Permission is granted to copy or use this program, EXCEPT that it
 *       may not be sold for profit, the copyright notice must be reproduced
 *       on copies, and credit should be given to Bellcore where it is due.
 *       BELLCORE MAKES NO WARRANTY AND ACCEPTS NO LIABILITY FOR THIS PROGRAM.
 */

/*
 * Convert MPU data for SPARCstation /dev/audio
 * Permit real-time adjustment of playing parameters
 * SAU 3/93
 *
 * Strategy:
 *  Instead of using the obvious strategy of synthesizing each note as the
 *  key goes up, we wait until we need to output a sample for that note.
 *  This permits more rapid changes in
 *  the performance parameters.  As we expect this program to be called
 *  from some interactive user interface, at each call we check the current
 *  performance characteristics, and output a HUNK, reading midi events only
 *  as necessary. If we are outputting to /dev/audio (the default) we control
 *  the size of the kernel buffers so performance changes happen quickly.
 */

/* debugging flags
 * c  each midi command
 * e  error condition
 * f  time adjusting
 * l  list operations
 * m  main loop
 * n  note rendering/deleting
 * o  output
 * p  Command processing
 */

#include <fcntl.h>
#include <sys/signal.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <stropts.h>
#include	"libmidi.h"
#include	"libmpu.h"
#include	"audio.h"
#include "synth.h"
#include "reverb.h"

char debug_list[20];	/* list of debugging stuff */
int wrote;							/* number of audio sample written */
char *sine_table();

/* these are the changable performance parameters; pick some defaults */

struct params param_data = {
	-1,		/* fd for auxilliary input audio */
	NULL,		/* file */
	1000,		/* gain factor (% * 10) */
	120,		/* tempo (bpm) */
	120,		/* OLD tempo (bpm) */
	0,			/* transposition (in semi-tones) */
	110,		/* note duration (%) */ 
	4,			/* velocity factor (0-10) */
	0,			/* which tuning: 0=equal, 1=true */
	3000,		/* bytes spit-out at each iteration */
	0,			/* generate verbose information */
	0,			/* state */
	8000*5,	/* max number of samples (for broken midi files) */
	0,			/* channels used */
	0x3F,		/* channel mask (start with 7 channels */
	0,			/* which voice to change */
	NULL,		/* shapes */
	NULL		/* reverberation parameters */
	};

struct part weights[10] = {
	0, 50, 1,
	1, 40, 5, 
	2, 30, 7,
	0,  0, 0,
	};
	
struct voice builtin_voice = {
	0,			/* voice number */
	2000,		/* cutoff freq */
	1,			/* stretch */
	weights,	/* partial weights */
	NULL,		/* no shape for now */
	};

char line[MAXLINE];	/* stdin/stdout buffer */
data buff[MAXBUFF];	/* where the audio goes */
char mulaw[MAXBUFF];	/* where the converted output goes */
struct voice voices[16];	/* parameters for the 16 midi voices */

extern char *malloc();
extern char *getenv();

/* list management functions */

static int finish_notes(/* struct note *list,int time */);
static void print_list(/* struct note *list */);
char *parse_command(/* params,voice,line */);
static int catch();

/* other stuff */

int play(/* int *buff, char *out, int n,int fd, int gain */);
struct audio_state *init_audio(/* struct audio_state *audio */);

/* main program */

char *table[MIDI_MAX_CHANS];	/* pointers to partial tables */

main(argc, argv)
char *argv[];
	{
	int how = 0;
	int old_how = 0;	/* 0 is mu-law */
	int i;			/* index */
	int q;			/* bytes above high water mark */
	int n;			/* bytes of audio processed */
	int xtra = 1000;	/* extra bytes in audio Q */
	int audio_fd;		/* fd for audio device */
	FILE *file=NULL;	/* config file fd */

	char *getenv();
	struct params *params = &param_data;	/* performance data */
	struct audio_state *audio;					/* audio progress data */

	if (getenv("DEBUG")) {
		strcpy(debug_list,getenv("DEBUG"));
		fprintf(stderr,"Debug list: %s\n",debug_list);
		}

	if (getenv("EXTRA")) 
		xtra = atoi(getenv("EXTRA"));

	/* initialize the audio, send initial prompt  */

	audio = init_audio(NULL);
	signal(SIGALRM,catch);
	alarm(3);
	audio_fd = open(getenv("AUDIO")?getenv("AUDIO"):"/dev/audio",1);	
	alarm(0);
	if (audio_fd < 0) {
		fprintf(stderr,"Audio device won't open, sorry\n");
		term("/dev/audio is busy, try later",1);
		}
	wrote=audio_wrote(audio_fd);

	/* get config file (if any) */

	if (argc>1 && *argv[1] == '-') {		/* config file */
		dprintf('m')(stderr,"opening config file %s\n",argv[1]+1);
		file = fopen(argv[1]+1,"r");
		argv++, argc--;
		P(state) = 0;
		}

	/* initialize the voices (this is temporary - to alloc space) */

	for(i=0;i<MIDI_MAX_CHANS;i++) {
		int size = 10 * sizeof(struct part);
		voices[i] = builtin_voice;
		voices[i].weights = (struct part *) malloc(size);
		bcopy(weights,voices[i].weights,size);
		voices[i].id = i;
		}

	/* read voice init file */

	if (file) {
		dprintf('m')(stderr,"Reading parameter file\n");
		while(fgets(line,MAXLINE,file)) {
			parse_command(params,voices,line);
			}
		fclose(file);
		}

	for(i=0;i<MIDI_MAX_CHANS;i++)
		table[i] = sine_table(NULL,&voices[i],P(tuning));

	while(1) {

		if (!P(file) && --argc>0 && (P(file) = fopen(*++argv,"r"))) {
			dprintf('M')(stderr,"Started file %s\n",*argv);
			P(state) = RESTART|RUNNING;
			}

		if (P(state)&RUNNING && wrote >= 0) {
			if (audio_fd<0) {		/* reopen audio (if needed) */
				fprintf(stderr,"Opening audio\n");
				audio_fd = open("/dev/audio",1);	
				wrote=0;
				}
			q = wrote-audio_wrote(audio_fd)-P(hunk)+xtra;
			}
		else
			q = P(hunk);

		if (q > 0 || wrote < 0) {
			set_params(params,voices,A(id),line,MAXLINE,q);
			}

		if (q <= 0 || wrote < 0) {
			if (P(state)&(NEW_VOICE|RESPONSE)) {		/* do a voice calculation */
				int i = 0;
				dprintf('v')(stderr,"new voice 0x%x\n",P(voice));
				for(i=0;P(voice);P(voice)>>=1,i++) {
					if (P(voice)&1) {
						dprintf('v')(stderr," new table for voice %d\n",i);
						table[i] = sine_table(table[i],voices+i,P(tuning));
						sine_resp(table[i],P(shape)); /* wrong */
						}
					}
				P(state) &= ~(NEW_VOICE|RESPONSE);
				}
			n = to_audio(audio,params,buff);
			reverb(P(reverb_params),buff,buff,n);

			how = P(state)&ALAW?1:0;
			if (how != old_how) {
				fprintf(stderr,"setting ALAW to %d\n",how);
				set_alaw(audio_fd,how);
				old_how=how;
				}
			play(buff,mulaw,n,P(gain),P(state)&ALAW);
			n = write(audio_fd,mulaw,n);
			if (wrote >=0) wrote+=n;
			}

		/* see if we need to close the audio device */

		if (wrote>0 && (P(state)&RUNNING)==0 && audio_fd > 0) {
			fprintf(stderr,"Closing audio\n");
			close(audio_fd);
			audio_fd = -1;
			}
		}
	}

/* convert binary to mu-law, then out-put out */

static int old_how = 0;

int
play(buff,out,n,gain,how)
data *buff;		/* buffer of sound data */
char *out;		/* the output mu-law buffer */
int n;			/* number of samples */
int gain;		/* % of nominal gain * 10 */
int how;			/* 1 is ALAW */
	{
	char *p = out;
	int count = n;
	int clip = 0;
	int max = 0;
	unsigned char *cnvt;
	register int sample;
#ifndef NO_ALAW

	extern unsigned char B2a[];
	if (how)
		cnvt = B2a+B13ZERO;
	else
		cnvt = Ub2i+B13ZERO;
#else
	cnvt = Ub2i+B13ZERO;
#endif
	
	while(n-- > 0) {
		sample = gain * *buff++ / 4000.0;
#ifndef NOCLIP
		if (sample < -4010 || sample > 4010)  {	/* for testing */
			if (abs(sample) > max) max = abs(sample);
			sample = 0;	/* wrong */
			clip++;
			}
#endif
		*p++ = cnvt[sample];
		}
	if (clip)
		fprintf(stderr,"clip %d/%d, max value = %d\n",clip,count,max);
	return(count);
	}

/* read in a command line, for interactive use */

set_params(params,voice,id,line,max,wait)
struct params *params;	/* current performance parameters */
struct voice voice[];		/* set of voice parameters */
int id;			/* current note id (for status) */
char *line;
int max;			/* max chars to wait */
int wait;		/* number of samples to wait before reading */
	{
	char *msg;		/* message returned from parse_command */
	int n;

	if ((n=wait_read(0,line,max,wait))>0) {
		dprintf('p')(stderr,"Got command: %s\n",line);
		msg = parse_command(params,voice,line);
		printf("* %d %c %.04x {%s}\n",
				id,P(state)&RUNNING?'+':'-',P(channels),msg);
		fflush(stdout);
		P(channels) = 0;
		}
	}

/* sum a hunk.  render notes as required */

int
output(audio,params,buff)
struct audio_state *audio;
struct params *params;	/* rendering parameters */
data *buff;					/* where to put the output */
	{
	struct note *list = A(list);
	int done = A(done);
	int count = 0;			/* number of completed notes seen */
	int delta;				/* # samples from end of note */
	struct note *note;	/* current note */
	data *start_note;		/* starting sample to render in note */
	int n;					/* # of samples from this note */
	data *p;					/* starting sample to render in buff */

	dprintf('o')(stderr,"outing hunk %d-%d\n",done,done+P(hunk));
	bzero(buff,P(hunk)*sizeof(data));
	for(note=list;note;note=note->next) {

		/* skip notes not participating in this hunk */

		if (note->stop < done){					/* done with this puppy */
			dprintf('o')(stderr,"   %d (%d:%d) is done (%d->%d < %d)\n",
				note->id,note->chan,note->key,note->start,note->stop,done);
			count++;
			continue;
			}
		if (note->start >= done + P(hunk)) {
			dprintf('o')(stderr,"   %d (%d:%d) hasn't started (%d-%d > %d)\n",
				note->id,note->chan,note->key,note->start,note->stop,done);
			continue;
			}
		if (note->start > note->stop) {
			fprintf(stderr,"AARGH!, %d (%d:%d) %d->??, nothing to render\n",
				note->id,note->chan,note->key,note->start);
			print_list(list,done,A(current));
			continue;
			}

		/* skip notes for invalid channel mask */

		if (!((1<<note->chan) & P(mask)))
			continue;

		/* render the note */

		if (!note->data && note->stop>note->start) {
			int dur = (note->stop-note->start) * P(duration)/100;

			dur &= ~1;		/* need even # of samples */
			note->stop = note->start + dur-1;	/* off by 1? */
			note->data = (data *) malloc(dur*sizeof(data));
			dprintf('n') (stderr," rendering %d (%d:%d) %d->%d (%d) at 0x%x\n",
					note->id,note->chan,note->key,note->start,
					note->stop,dur,note->data);
			do_sines(table[note->chan],note->key+P(trans),dur,
						note->vel,P(factor),note->data);
			}

		/* compute boundary conditions */

		if ((delta = note->start-done) > 0) {	/* note > hunk */
			start_note = note->data;
			p = buff + delta;
			n = P(hunk) - delta;
			dprintf('o')(stderr," %d (%d:%d) adjusting start by %d, (%d bytes)\n",
					note->id, note->chan,note->key,delta,n);
			}
		else {							/* note starts before beginning of hunk */
			start_note = note->data - delta;
			p = buff;
			n = P(hunk);
			}

		if ((delta = note->stop - (done+P(hunk))) < 0) {	/* note < hunk */
			n += delta;
			dprintf('o')(stderr," %d (%d:%d) adjusting end by %d, (%d bytes)\n",
					note->id, note->chan,note->key,delta,n);
			}

		if (n<0) {
			fprintf(stderr,"done=%d, hunk=%d, note len=%d\n",
				done,P(hunk),n);
			fprintf(stderr,"AARG! note %d is too short!\n",note->id);
			print_list(list,done,A(current));
			continue;
			}
			
		/* murphy's law checks */

		if (p+n > buff+P(hunk)) {
			fprintf(stderr,"Fatal buff overrun error!!\n");
			fprintf(stderr,"0x%x + %d > 0x%x + %d\n",
					p,n,buff,P(hunk));
			print_list(list,done,A(current));
			term("Internal error: buffer overrun",2);
			}

		if (start_note+n > (note->stop-note->start)+note->data) {
			fprintf(stderr,"Fatal note-data overrun error!!\n");
			fprintf(stderr,"0x%x + %d > (%d-%d)+0x%x\n",
				start_note,n,note->stop,note->start,note->data);
			print_list(list,done,A(current));
			term("Internal error: note-data overrun",2);
			}

		/* add note to sound */

		while (n-->0)
			*p++ += *start_note++;
		}

	return(count);
	}

/* read from fd non blocking */

int
wait_read(fd,buff,n,samples)
int fd;			/* file descriptor to read from */
char *buff;		/* read into here */
int n;			/* max # to read */
int samples;	/* # of samples to wait */
	{
	int mask = 1<<fd;
	int cnt=0;
   struct timeval time;
   time.tv_sec = (samples)/8000;
   time.tv_usec = ((samples)%8000)*1000/8;

	if (select(32,&mask,0,0,&time)==1) {	/* read from stdin (if avaliable) */
		cnt = read(fd, buff, n);
		}
	return(cnt);
	}

/* return # of samples written */

int
audio_wrote(fd)
int fd;
   {
   int n;
   struct audio_info info;

   n = ioctl(fd,AUDIO_GETINFO,&info);
	if (n<0)
		return(-1);
	else
		return(info.play.samples);
   }

/* print a note list (for debugging) */

static void
print_list(list,done,current)
struct note *list;
int done, current;
	{
	fprintf(stderr,"note list: dont=%d, current=%d\n",done,current);
	for(;list;list=list->next) {
		fprintf(stderr,"%d) %d:%d (%d->%d) %s\n",
			list->id, list->chan, list->key,list->start,list->stop,
			list->data ? "rendered":"");	
		}
	}

/* catch (and ignore) a signal */

static int
catch()
	{
	fprintf(stderr,"Synth, Timeout error\n");
	}

/* die, but let user interface know first */

term(s,n)
char *s;
int n;
	{
	fprintf(stderr,"Synth aborting: %s\n",s);
	alarm(5);
	gets(line);
	alarm(0);
	printf("Xit: %s\n",s);
	exit(n);
	}
