/* generate a multi-component sine wave  (S A Uhler) */

#include <stdio.h>
#include <math.h>
#include "synth.h"
#include "bpf.h"

#define BASE	8.17579864		/* frequency for note 0 C-2 */
#define NOTES	128				/* number of notes (C-2 -> G8) */
#define SPS		8000.0			/* samples/second */
#define SCALE	3000.0				/* maximum excursion */
#define PARTIALS	10				/* max number of partials */
#define BAD_FREQ	-1.0			/* always an invalid frequency */

#define TWO_PI		6.28318530717958647692
#define PI		3.14159265358979323846
	
#define real float

static struct coef {
	real freq;		/* frequency (2PI f/fs) for debugging */
	real coef;		/* recursion coef */
	real damp;		/* damping */
	real init;		/* initial value */
	};

real get_tuning(); 
static struct coef scratch[PARTIALS];	/* place to build partials */

int
do_sines(cookie,key,dur,vel,fact,buff)
char *cookie;	/* pointer to key coef's */
int key;		/* key number */
int dur;		/* duration in samples */
int vel;		/* velocity 0-127 */
int fact;	/* velocity factor: 0=linear, 10=log */
data *buff;	/* where to put the samples */
	{
	struct coef **table = (struct coef **) cookie;
	register int count;		/* note duration in samples */
	register real c;		/* recursion coefficiant (freq) */
	register real d;		/* damping */
	register real s1,s2;	/* samples */
	register data *p;			/* buffer pointer */
	int i;			/* partial index */
	struct coef *coef;	/* pointer into the coef. array */
	real scale;	/* velocity scale */
	
	bzero(buff,dur * sizeof(*buff));

	/* compute scale factor */

	vel = (vel&127)+1;
	scale = ((10-fact)*vel+fact*pow(1.0386,vel))/1280.0;

	for(coef=table[key&0x7F]; coef->freq > BAD_FREQ; coef++) {
		c = coef->coef;
		d = coef->damp;
		s1 = 0.0;
		s2 = coef->init;
		count = dur;
		p = buff;

		dprintf('s')(stderr," %.4f,%.4f,%.4f ",c,d,s2);
		while (count > 0) {
			*p++ += s1;
			*p++ += s2;
			s1 = c*s2 - d*s1;
			s2 = c*s1 - d*s2;
			count -= 2;
			}
		}
	dprintf('s')(stderr,"\n");
	}

/* build a note coef. table */

char *
sine_table(cookie,voice,tuning)
char *cookie;				/* the existing table (or NULL) */
struct voice *voice;		/* voice characteristics */
int tuning;					/* which tuning (unused for now) */
	{
	struct coef **table = (struct coef **) cookie;
	register struct part *part;		/* parameters for current partial */
	register real damp;				/* damping parameter */
	register struct coef *coef;		/* coef pointer into scratch */
	int i,j;						/* i: which note, j: which partial */
	int size;					/* # bytes needed to represent a note */
	real norm;					/* sum of the weights */
	real freq;				/* fundamental frequency */
	real max_freq;			/* maximum permitted frequency for partials*/

	if (table)  {
		for(i=0;i<NOTES;i++)
			if (table[i]) free(table[i]);
		}
	else {
		int size = NOTES * sizeof(struct coef *);
		table = (struct coef **) malloc(size);
		}

	bzero(table,NOTES * sizeof(struct coef *));

	max_freq = TWO_PI * voice->cutoff / SPS;
	dprintf('v')(stderr,"Building sine table\n");
	dprintf('v')(stderr," Stretch=%d\n",voice->stretch);
	dprintf('v')(stderr," cutoff=%d\n",voice->cutoff);

	for (i=0;i<NOTES;i++) {
		coef = scratch;
		freq = get_tuning(i,tuning);
		part=voice->weights;

		/* create the note in "scratch" */

		dprintf('s')(stderr,"%d %.1f",i,pow(2.0,i/12.0)*BASE);
		for(norm=0.0,j=0;part->weight && j<PARTIALS;part++,j++) {
			register real f = (part->partial+1) * freq *
				 (voice->stretch*part->partial+1000)/1000.0;

			if (f >= max_freq && j)	/* always synth fundamental */
				break;
			if (f >= PI)				/* never go over SPS/2 */
				break;
			if (part->damping < 0)	/* negative damping is unstable */
				part->damping = 0;

			damp = (SPS - part->damping)/SPS;

			dprintf('s')(stderr," f%.2f w%d d%d,",
					SPS*f/TWO_PI,part->weight,part->damping);

			coef->coef = 2.0 * cos(f) * damp;
			coef->init = sin(f) * part->weight;
			coef->damp = damp*damp;
			coef->freq = f;
			norm += part->weight;
			coef++;
			}
		coef->freq = BAD_FREQ;	/* end marker */
		dprintf('s')(stderr,"\n");

		/* normalize the weights */

		for(coef=scratch;coef->freq>BAD_FREQ;coef++)
			coef->init *= SCALE/norm;

		/* copy the note into the array */

		size = sizeof(*coef)*(j+1);
		table[i] = (struct coef *) malloc(size);
		bcopy(scratch,table[i],size);
		}
	return((char *)table);
	}

/* get the fundamental frequencies */

static real false[] = {		/* "true" flat temperment */
	254.0, 285.12, 297.0, 316.8, 330.0, 352.0,
	380.16, 396.0, 422.4, 440.0, 475.2, 495.0
	};

static real true[] = {		/* "true" sharp temperment */
	254.0, 275.0, 297.0, 309.4583, 330.0, 352.0,
	366.667, 396.0, 412.5, 440.0, 458.3333, 495.0
	};

real
get_tuning(i,tuning)
register int i;		/* note number */
int tuning;	/* which tuning */
	{
	int octave = i/12;
	if (tuning == 1) {		/* use true temperment */
		return(TWO_PI * true[i%12] * pow(2.0,octave-5.0)/ SPS);
		}
	if (tuning == 2) {		/* use off temperment */
		return(TWO_PI * false[i%12] * pow(2.0,octave-5.0)/ SPS);
		}
	else
		return(TWO_PI * BASE * pow(2.0,i/12.0) / SPS);
	}

free_table(table)
struct coef **table;
	{
	register int i;

	if (table)  {
		for(i=0;i<NOTES;i++)
			if (table[i]) free(table[i]);
		free(table);
		}
	}

/* add response curve (I should use a better algorithm) */

int
sine_resp(cookie,shape)
char *cookie;				/* the existing table (or NULL) */
struct shape *shape;		/* voice characteristics */
	{
	struct coef **table = (struct coef **) cookie;
	struct coef *coef;	/* pointer into the coef. array */
	struct filt_coef filt;	/* filter coef's */
	struct shape *s;		/* current shape parameters */
	real f;				/* partial frequency */
	real response;		/* response multiplier */
	int i;

	if (!table || !shape)
		return(0);

	/* for (i=0;i<NOTES;i++) { */
	for (i=40;i<108;i++) {		/* cheat!! */
		for(coef=table[i]; coef->freq > BAD_FREQ; coef++) {
			f = coef->freq;
			response = 1.0;
			for(s = shape;s->peak>0;s++) {
				bpf(&filt,s->peak,s->width,SPS);
				response += resp(&filt,f) * s->gain;
				dprintf('r')(stderr,"  %f += %f * %f\n",
						response,resp(&filt,f,0),s->gain);
				}
			if (response < 0.0) response = 0.0;
			if (response > 500.0) response = 500.0;
			coef->init *= response;
			dprintf('r')(stderr,"> %d (%d) %.2f %.2f\n",
						i,coef-table[i],f*SPS/TWO_PI,response);
			}
		}
	return(1);
	}
