/*                      Copyright (c) 1992,1993 Bellcore
 *                            All Rights Reserved
 *       Permission is granted to copy or use this program, EXCEPT that it
 *       may not be sold for profit, the copyright notice must be reproduced
 *       on copies, and credit should be given to Bellcore where it is due.
 *       BELLCORE MAKES NO WARRANTY AND ACCEPTS NO LIABILITY FOR THIS PROGRAM.
 */

/* digital filter routines  adapted by S A Uhler from fortran code by
    J. F. Kaiser (4/18/91)
 */

#include <math.h>
#include "bpf.h"

/* Band pass filter */

int
bpf(coef,fc,fd,fs)
struct filt_coef *coef;		/* put coef's here */
double fc;					/* center frequency (HZ) */
double fd;					/* 3db width (HZ) */
double fs;					/* sampling frequency (HZ) */
	{
	double k, b1, b2;		/* outputs */
	double pp, tc, td, xd;

	/* argument checking */

	if (fd <=0 || fs <=0 || fd >= 2.0*fc || fs < 2.0*fd || fs < 2.0*fc + fd)
		return(0);

	pp = 2.0*M_PI/fs;
	tc = pp*fc;
	td = pp*fd;
	xd = cos(td);
	b2 = (1.0 - sqrt(1.0 - xd*xd))/xd;
	b1 = -(1.0 + b2)*cos(tc);
   k = (1.0 - b2)/2.0;

	coef->k  = k;
	coef->a0 = 1.0;
	coef->a1 = 0.0;
	coef->a2 = -1.0;
	coef->b1 = b1;
	coef->b2 = b2;

	return(1);
	}

/* Stop band filter */

int
bsf(coef,fc,fd,fs)
struct filt_coef *coef;		/* put coef's here */
double fc;					/* center frequency (HZ) */
double fd;					/* 3db width (HZ) */
double fs;					/* sampling frequency (HZ) */
	{
	double t1, t2, d, d1;
	double k, a1, b1, b2;		/* outputs */

	/* argument checking */

	if (fd <=0 || fs <=0 || fd >= 2.0*fc || fs < 2.0*fd || fs < 2.0*fc + fd)
		return(0);

	t1 = 2.0*M_PI*fc/fs;
	t2 = M_PI*fd/fs;
	d = sin(t2)/cos(t2);
	d1 = 1.0 + d;

   k = 1.0/d1;
	a1 = -2.0*cos(t1);
	b1 = a1/d1;
	b2 = (1.0 - d)/d1;

	coef->k  = k;
	coef->a0 = 1.0;
	coef->a1 = a1;
	coef->a2 = 1.0;
	coef->b1 = b1;
	coef->b2 = b2;

	return(1);
	}

/* butterworth low pass filter */

int
lpf(coef,f3,fs)
struct filt_coef *coef;		/* put coef's here */
double f3;					/* 3db loss frequency (hz) */
double fs;					/* sampling frequency (HZ) */
	{
	double arg, gam, gam2, gd, k, a1, b1, b2;

	/* argument checking */

	if (f3 >= 0.99*fs/2.0)
		return(0);

	arg = M_PI*f3/fs;
	gam = cos(arg)/sin(arg);
	gam2 = gam*gam;
	gd = 1.0 + M_SQRT2*gam + gam2;

	k = 1.0/gd;
	a1 = 2.0;
	b1 = 2.0*(1.0 - gam2)*k;
	b2 = k*(1.0 - M_SQRT2*gam + gam2);

	coef->k  = k;
	coef->a0 = 1.0;
	coef->a1 = a1;
	coef->a2 = 1.0;
	coef->b1 = b1;
	coef->b2 = b2;
	
	return(1);
	}

/* elliptical low pass filter */

int
lpfe(coef,f11,dd1,dd2,fs)
struct filt_coef *coef;		/* put coef's here */
double f11;					/* passband edge frequncy (HZ) */
double dd1;					/* pass band ripple */
double dd2;					/* stop band ripple */
double fs;					/* sampling frequency (HZ) */
	{
	double k, a1, a2, b1, b2;		/* outputs */
	double arg, w11, fop, fs2, fsp, dcg, d2, ga, h, zeta, ga2, gah, y1, x1;
	double wn, w0, r, r2, s, s2, d;

	if (f11 >= 0.99*fs/2.0)
		return(0);

	if (dd1 <=0.0 || dd1 >= 1.0)
		return(0);

	if (dd2 <=0.0 || dd2 > 1.0 - dd1)
		return(0);

	arg = M_PI*f11/fs;
	w11 = 2.0*fs*sin(arg)/cos(arg);
	fop = fs/M_PI;
	fs2 = 2.0*fs;
	fsp = 1.0/fs2;

	arg = f11/fop;
	w11 = fs2*sin(arg)/cos(arg);
	dcg = 1.0 - dd1;
	d2 = dd2*dd2;
	ga = dcg/dd2;
	h = dd2*dcg + sqrt(dd1*(2.0 - dd1)*(1.0 - d2));
	zeta = sqrt((1.0 - h)/2.0);
	ga2 = ga*ga;
	gah = ga*h;
	y1 = 2.0*ga*(gah - 1.0)/(ga2 - 1.0);
	x1 = sqrt(y1);
	wn = w11/x1;
	w0 = wn*sqrt(ga);
	r = w0*fsp;
	r2 = r*r;
	s = wn*fsp;
	s2 = s*s;
	d = 1.0 + 2.0*zeta*s + s2;

	k = dd2*(1.0 + r2)/d;
	a1 = -2.0*(1.0 - r2)/(1.0 + r2);
	b1 = -2.0*(1.0 - s2)/d;
	b2 = (d - 4.0*zeta*s)/d;

	coef->k  = k;
	coef->a0 = 1.0;
	coef->a1 = a1;
	coef->a2 = 1.0;
	coef->b1 = b1;
	coef->b2 = b2;
	
	return(1);
	}

/* butterworth high pass filter */

int
hpf(coef,f3,fs)
struct filt_coef *coef;		/* put coef's here */
double f3;					/* 3db loss frequency (hz) */
double fs;					/* sampling frequency (HZ) */
	{
	if (lpf(coef,fs/2.0-f3,fs)) {
		coef->a1 = -coef->a1;
		coef->b1 = -coef->b1;
		return(1);
		}
	else
		return(0);
	}

/* elliptical high pass filter */

int
hpfe(coef,f11,dd1,dd2,fs)
struct filt_coef *coef;		/* put coef's here */
double f11;					/* passband edge frequncy (HZ) */
double dd1;					/* pass band ripple */
double dd2;					/* stop band ripple */
double fs;					/* sampling frequency (HZ) */
	{
	if (lpfe(coef,fs/2.0-f11,dd1,dd2,fs)) {
		coef->a1 = -coef->a1;
		coef->b1 = -coef->b1;
		return(1);
		}
	else
		return(0);
	}

/* calculate the filter response */

double
resp(coef,f,what)
struct filt_coef *coef;		/* fiter coeficients */
double f;					/* digital frequency (2*PI*f/fs) */
int what;					/* 0==magnitude, 1==phase (degreese) */
	{
	double k, a0, a1, a2, b1, b2;		/* coefficients */
	double th2, ct, c2t, st, s2t, znr, zni, zdr, zdi, znm, zdm;
	double zm, ph;		/* resultant magnitude and phase */

	/* argument checking */

	if (f < 0.0 || f > M_PI)
		return(0.0);

	k = coef->k;
	a0 = coef->a0;
	a1 = coef->a1;
	a2 = coef->a2;
	b1 = coef->b1;
	b2 = coef->b2;

	th2 = 2.0*f*f - 1.0;
	ct = cos(f);
	c2t = 2.0*ct*ct - 1.0;
	st = sin(f);
	s2t = 2.0*st*ct;
	znr = a0 + a1*ct + a2*c2t;
	zni = -a1*st - a2*s2t;
	zdr = 1.0 + b1*ct + b2*c2t;
	zdi = -b1*st -b2*s2t;
	znm = sqrt(znr*znr + zni*zni);
	zdm = sqrt(zdr*zdr + zdi*zdi);
	if (zdm == 0.0)
		zm = HUGE_VAL;
	else
		zm = k*znm/zdm;
	ph = 2.0*M_PI*(atan2(zni,znr) - atan2(zdi,zdr))/360.0;

	if (what)
		return(ph);
	else
		return(zm);
	}
