/*********************************************************************
  
  This software module was originally developed by
  
  Eric D. Scheirer (MIT Media Laboratory)
  
  in the course of development of the MPEG-2 NBC/MPEG-4 Audio standard
  ISO/IEC 13818-7, 14496-1,2 and 3. This software module is an
  implementation of a part of one or more MPEG-2 NBC/MPEG-4 Audio tools
  as specified by the MPEG-2 NBC/MPEG-4 Audio standard.  ISO/IEC gives
  users of the MPEG-2 NBC/MPEG-4 Audio standards free license to this
  software module or modifications thereof for use in hardware or
  software products claiming conformance to the MPEG-2 NBC/ MPEG-4 Audio
  standards. Those intending to use this software module in hardware or
  software products are advised that this use may infringe existing
  patents. The original developer of this software module and his/her
  company, the subsequent editors and their companies, and ISO/IEC have
  no liability for use of this software module or modifications thereof
  in an implementation.
  
  This software module is hereby released into the public domain.
  
  ***********************************************************************/

/* saol_tables.c: Table utilities and core table generators */

#include <string.h>
#include "saol_interp.h"
#include <math.h>
#include "aifif.h"
#include "saol.h"

#ifdef _WIN32
#define DIRSEP '\\'
#else
#define DIRSEP '/'
#endif

#define NUM_CORE_TABLEGEN 18

extern double drand48();
extern struct cmdinfo cmd; /* access to cmdline parameters */

double i0(double x);
struct core_tg_struct {
  char *name;			/* the name of the generator */
  table_storage *(*func)(actparam_list *); /* the code which implements it */
};

table_storage *ctg_soundfile(actparam_list *pf);
table_storage *ctg_data(actparam_list *pf);
table_storage *ctg_random(actparam_list *pf);
table_storage *ctg_step(actparam_list *pf);
table_storage *ctg_lineseg(actparam_list *pf);
table_storage *ctg_expseg(actparam_list *pf);
table_storage *ctg_cubicseg(actparam_list *pf);
table_storage *ctg_spline(actparam_list *pf);
table_storage *ctg_polynomial(actparam_list *pf);
table_storage *ctg_window(actparam_list *pf);
table_storage *ctg_harm(actparam_list *pf);
table_storage *ctg_harm_phase(actparam_list *pf);
table_storage *ctg_harm_dc(actparam_list *pf);
table_storage *ctg_buzz(actparam_list *pf);
table_storage *ctg_logbessel(actparam_list *pf);
table_storage *ctg_cheby_poly(actparam_list *pf);
table_storage *ctg_concat(actparam_list *pf);
table_storage *ctg_empty(actparam_list *pf);

struct core_tg_struct core_tablegens[NUM_CORE_TABLEGEN] = {
  {"soundfile",ctg_soundfile},
  {"data",ctg_data},
  {"random",ctg_random},
  {"step",ctg_step},
  {"lineseg",ctg_lineseg},
  {"expseg",ctg_expseg},
  {"cubicseg",ctg_cubicseg},
  {"spline",ctg_spline},
  {"polynomial",ctg_polynomial},
  {"window",ctg_window},
  {"harm",ctg_harm},
  {"harm_phase",ctg_harm_phase},
  {"harm_dc",ctg_harm_dc},
  {"buzz",ctg_buzz},
  {"logbessel",ctg_logbessel},
  {"cheby_poly",ctg_cheby_poly},
  {"concat",ctg_concat},
  {"empty",ctg_empty}};

int get_table_size(table_storage *t) {
  return(t->size);
}

double get_table_value(table_storage *t, int idx) {
  char s[234];
  if (idx >= t->size || idx < 0) {
    sprintf(s,"Table index out of bounds: '%s', index %d, max %d.",
	    t->name,idx,t->size);
    runtime(s);
  }
  return(t->d[idx]);
}

void set_table_value(table_storage *t, int idx, double val) {
  char s[234];
  
  if (idx >= t->size || idx < 0) {
    sprintf(s,"Table index out of bounds: '%s', index %d, max %d.",
	    t->name,idx,t->size);
    runtime(s);
  }
  t->d[idx] = val;
}



table_storage *gen_table(char *gen, actparam_list *pf) {
  int i;
  char s[854];

  for (i=0;i!=NUM_CORE_TABLEGEN;i++)
    if (!strcmp(gen,core_tablegens[i].name))
      return(core_tablegens[i].func(pf));

  sprintf(s,"Unknown table generator '%s'",gen);
  runtime(s);
}

table_storage *new_table(int size) {
  table_storage *t;

  PROT_MAL(t,table_storage,new_table);
  
  if (!(t->d=(double *)calloc(sizeof(double),size))) 
    runtime("Couldn't allocate space in new_table().");

  t->size = size;
  t->srate = 0;
  t->loop = 0;
  t->loopend = 0;
  t->base = 0;

  return(t);
}

/* return the value from the byte buffer for 8/16/24/32-bit files */
/* fractional bit widths not supported */

double bufval(char *buf,int frame,int chan,int nchan,int nbits) {

  if (nbits == 16)
    return((double)*(short *)&buf[frame * nchan * 2 + chan * 2]);

  if (nbits == 32)
    return((double)*(int *)&buf[frame * nchan * 2 + chan * 2]);

  if (nbits == 8)
    return((double)*(char *)&buf[frame * nchan *2 + chan * 2]);

}

table_storage *ctg_soundfile(actparam_list *pf) {
  int size;
  char fn[800],s[800];
  AIF_STRUCT *aifin = aifNew();
  table_storage *t;
  int skip=0, chan=0, nchan, nbits, ct, done = 0, retct, i, j;
  double samp,scale;
  char *buf;
  long param[24];

  if (!pf && pf->next) {
    runtime("Bad call to 'soundfile' tablegen.\n");
  }
  
  sprintf(fn,"%s%c%s",cmd.temp,DIRSEP,pf->next->ref->d->csval);
  if (aifOpenRead(aifin,fn)) {
    sprintf(s,"Couldn't open '%s' for read.",fn);
    runtime(s);
  }

  if (pf->next->next) {
    skip = ((int)(pf->next->next->val + 0.5));
    if (pf->next->next->next)
      chan = ((int)(pf->next->next->next->val + 0.5));
  }
  
  if (pf->val == -1) {
    param[0] = AIF_P_NFRAMES;
    aifGetParams(aifin,param,2);
    size = param[1] - skip;
  }
  else
    size = ((int)(pf->val +0.5));

  t = new_table(size);
  t->size = size;

  param[0] = AIF_P_CHANNELS;
  param[2] = AIF_P_SAMPSIZE;
  param[4] = AIF_P_SAMPRATE;
  aifGetParams(aifin,param,6);
  nchan = param[1];
  nbits = param[3];
  t->srate = (int)*(float *)&param[5];

  scale = (double)(2 << (nbits-2));
  
  buf = (char *)malloc(nchan * nbits/8 * 1024); /* 1K frame buffer */

  aifFrameSeek(aifin, skip);

  ct = 0;
  while (!done) {
    retct = aifReadFrames(aifin,buf,1024); /* number actually read */
    if (retct < 1024 || ct+retct >= size) {
      done = 1;
    }
    for (i=0;i!=retct && ct < size;i++,ct++) {
      if (!chan) {		/* average all channels */
	samp = 0;
	for (j=0;j!=nchan;j++)
	  samp += bufval(buf,i,j,nchan,nbits);
	samp = samp/nchan;
      }
      else
	samp = bufval(buf,i,chan,nchan,nbits);

      set_table_value(t,ct,samp/scale);
    }
  }

  aifClose(aifin);
  aifFree(aifin);

  printf("Read %d samples from soundfile '%s'\n",ct,fn);
  return(t);
}

table_storage *ctg_data(actparam_list *pf) {
  table_storage *t;
  int ct = 0;

  if (!pf->val) 
    runtime("Invalid size for generator 'data'.");

  t = new_table((int)(pf->val + 0.5));

  for(pf=pf->next;pf;pf=pf->next,ct++) 
    set_table_value(t,ct,pf->val);

  return(t);
}

table_storage *ctg_random(actparam_list *pf) {
  table_storage *t;
  int i = 0;
  int which = 0,ct;
  double x, y, p1, p2;
  
  if (!pf->val)
    runtime("Invalid size for generator 'random'.");

  t = new_table((int)(pf->val + 0.5));
  which = (int)pf->next->val;
  switch (which) {
  case 1:			/* uniform */
    if (!pf->next->next || !pf->next->next->next)
      runtime("Need min and max for uniform random tablegen.");
    p1 = pf->next->next->val;
    p2 = pf->next->next->next->val;

    for (i=0;i!=t->size;i++) {
      x = drand48();

      set_table_value(t,i,x * (p2-p1)+p1);
    }

    break;
  case 2:			/* linear */
    if (!pf->next->next || !pf->next->next->next)
      runtime("Need min and max for linear random tablegen.");
    p1 = pf->next->next->val;
    p2 = pf->next->next->next->val;

    for (i=0;i!=t->size;i++) {
      x = drand48();
      y = drand48();

      set_table_value(t,i,MAX(x,y) * (p2-p1) + p1);
    }
    break;
  case 3:			/* exp */
    if (!pf->next->next)
      runtime("Need mean for exponential random tablegen.");
    p1 = pf->next->next->val;
    
    for (i=0;i!=t->size;i++) {
      x = drand48();

      set_table_value(t,i,-log(x)*p1);
    }
    break;
  case 4:
    if (!pf->next->next || !pf->next->next->next)
      runtime("Need mean and var for Gaussian random tablegen.");
    p1 = pf->next->next->val;
    p2 = pf->next->next->next->val;

    for (i=0;i!=t->size;i++) {
      x = drand48();
      y = drand48();

      set_table_value(t,i,sqrt(-2 * log(x)) * cos(2 * PI * y) * sqrt(p1) + p2);
    }
 
    break;
  case 5:
    if (!pf->next->next)
      runtime("Need mean for exponential random tablegen.");
    p1 = pf->next->next->val;
    
    for (i=0;i<t->size;) {
      x = drand48();

      for (ct=0;i<t->size && ct < floor(-log(drand48()) * p1);ct++,i++)
	set_table_value(t,i,0);
      set_table_value(t,i++,1);
    }
    break;
  default:
    runtime("Unknown random tablegen.");
  }

  return t;
}

table_storage *ctg_step(actparam_list *pf) {
  table_storage *t;
  int ct = 0;
  int end,lastxval,xval;
  actparam_list *p;
  double yval;

  if (!pf->val)
    runtime("Invalid size for generator 'step'.");

  t = new_table((int)(pf->val + 0.5));
  
  for (p=pf;p->next;p=p->next) ; /* find last value */
  end = (int)(floor(p->val+0.5));
  lastxval = 0;

  if (!pf->next || !pf->next->next)
    runtime("Not enough y-values for generator 'step'.");
  
  for (p=pf->next->next;p;p=p->next) { /* start with first y-value */
    yval = p->val;
    p = p->next;
    if (!p) 
      runtime("Not enough x-values for generator 'step'.");
    xval = (int)(p->val + 0.5);
    if (xval >= get_table_size(t))
      runtime("Index out-of-bounds for generator 'step'.");
    if (xval < lastxval)
      runtime("'step' x-values must be non-decreasing.");
    for (ct=lastxval;ct!=xval;ct++)
      set_table_value(t,ct,yval);
  }
      
  return(t);
}

table_storage *ctg_lineseg(actparam_list *pf) {

  table_storage *t;
  int ct = 0;
  int end,lastxval,xval;
  actparam_list *p;
  double yval,lastyval;

  if (!pf->val)
    runtime("Invalid size for generator 'lineseg'.");

  t = new_table((int)(pf->val + 0.5));
  
  for (p=pf;p->next;p=p->next,ct++) ; /* find last value */
  end = (int)floor(p->val+0.5);
  if (ct < 4)
    runtime("Need at least two x-y pairs for generator 'lineseg'");
  
  lastxval = 0;
  p=pf->next->next;
  lastyval = p->val;
  
  for (p=p->next;p;p=p->next) { /* start with second x-value */
    xval = (int)(p->val + 0.5);
    p = p->next;
    if (!p)
      runtime("Not enough y-values for generator 'lineseg'");
    yval = p->val;
    if (xval >= get_table_size(t))
      runtime("Index out-of-bounds for generator 'lineseg'.");
    if (xval < lastxval)
      runtime("'lineseg' x-values must be non-decreasing.");
    for (ct=lastxval;ct!=xval;ct++)
      set_table_value(t,ct,((yval-lastyval)/(xval-lastxval) * (ct-lastxval) +
			    lastyval));
    lastxval = xval;
    lastyval = yval;
  }
      
  return(t);

}

table_storage *ctg_expseg(actparam_list *pf) {
  interror("Table generator not implemented.\n");
}

table_storage *ctg_cubicseg(actparam_list *pf) {
  interror("Table generator not implemented.\n");
}

table_storage *ctg_spline(actparam_list *pf) {
  interror("Table generator not implemented.\n");
}

table_storage *ctg_polynomial(actparam_list *pf) {
  interror("Table generator not implemented.\n");
}

table_storage *ctg_window(actparam_list *pf) {
  table_storage *t;
  actparam_list *p;
  int i,n;
  double p1, p2,alpha,std,var,sum;

  if (!pf->val)
    runtime("Invalid size for generator 'window'.");
  t = new_table((int)(pf->val + 0.5));

  if (!pf->next)
    runtime("Must provide window type for generator 'window'.");

  switch ((int)pf->next->val) {
  case 1:			/* hamming */
    for (i=0;i!=t->size;i++) 
      set_table_value(t,i,.52 - .46*cos(2*PI*i/(t->size-1)));
    break;

  case 2:			/* hanning */
    for (i=0;i!=t->size;i++)
      set_table_value(t,i,.5*(1-cos(2*PI*(i+1)/(t->size+1))));
    break;

  case 3:			/* bartlett */
    if (t->size == (t->size/2)*2) { /* even length */
      for (i=0;i!=t->size/2;i++)
	set_table_value(t,i,2.0*i/(t->size-1.0));
      for (;i!=t->size;i++)
	set_table_value(t,i,2.0*((double)t->size-i)/(t->size-1.0));
    }
    else {
      for (i=0;i!=(int)ceil(t->size/2.0);i++)
	set_table_value(t,i,2*i/(t->size-1));
      for (;i!=t->size;i++)
	set_table_value(t,i,2.0*(t->size-i-1.0)/(t->size-1.0));
    }
    break;

  case 4:			/* gaussian */
    std = t->size/6.0;
    var = std*std;
    for (i=0;i!=t->size;i++)
      set_table_value(t,i,exp(-(i-t->size/2.0)*(i-t->size/2.0) / 2.0 / var));
    break;

  case 5:			/* kaiser */
    if (!pf->next->next)
      runtime("Must provide parameter for Kaiser window.");
    p1 = pf->next->next->val;	/* == "beta" parameter */
    alpha = t->size/2;
    for (i=0;i!=t->size;i++)
      set_table_value(t,i,i0(p1 * pow(1-pow((i-alpha)/alpha,2.0),0.5)) / i0(p1));
    break;


  case 6:			/* rectangular */
    for (i=0;i!=t->size;i++)
      set_table_value(t,i,1);
    break;
  }

	
  return(t);
}

table_storage *ctg_harm(actparam_list *pf) {
  table_storage *t;
  actparam_list *p;
  int harm,i;
  double val;
  
  if (!pf->val)
    runtime("Invalid size for generator 'harm'.");

  t = new_table((int)(pf->val +0.5));

  for (i=0;i!=get_table_size(t);i++) {
    val = 0;
    for (p=pf->next,harm=1;p;p=p->next,harm++)
      val += sin((double)i/get_table_size(t) * 2 * PI * harm) * p->val;
    set_table_value(t,i,val);
  }
  return t;
}

table_storage *ctg_harm_phase(actparam_list *pf) {
  table_storage *t;
  actparam_list *p;
  double val,wt,ph;
  int harm,i;
  
  if (!pf->val)
    runtime("Invalid size for generator 'harm_phase'.");

  t = new_table((int)(pf->val +0.5));

  for (i=0;i!=get_table_size(t);i++) {
    val = 0;
    for (p=pf->next,harm=1;p;p=p->next,harm++) {
      wt = p->val;
      p=p->next;
      if (!p)
	runtime("Not enough phase values for generator 'harm_phase'.");
      ph = p->val;
      val += sin((i/get_table_size(t)+ph/360) * 2 * PI * harm) * wt;
    }
    set_table_value(t,i,val);
  }
  return t;
}

table_storage *ctg_harm_dc(actparam_list *pf) {
  table_storage *t;
  actparam_list *p;
  double val,harm,wt,ph,dc;
  int i;
  
  if (!pf->val)
    runtime("Invalid size for generator 'harm_dc'.");

  t = new_table((int)(pf->val +0.5));

  for (i=0;i!=get_table_size(t);i++) {
    val = 0;
    for (p=pf->next;p;p=p->next) {
      harm = p->val;
      p=p->next;
      if (!p)
	runtime("Not enough values (4 per partial) for generator 'harm_dc'.");
      wt = p->val;
      p=p->next;
      if (!p)
	runtime("Not enough values (4 per partial) for generator 'harm_dc'.");
      ph = p->val;
      p=p->next;
      if (!p)
	runtime("Not enough values (4 per partial) for generator 'harm_dc'.");
      dc = p->val;
      val += sin((i/get_table_size(t)+ph/360) * 2 * PI * harm) * wt + dc;
    }
    set_table_value(t,i,val);
  }
  return t;
}

table_storage *ctg_buzz(actparam_list *pf) {
  interror("Table generator not implemented.\n");
}

table_storage *ctg_logbessel(actparam_list *pf) {
  interror("Table generator not implemented.\n");
}

table_storage *ctg_cheby_poly(actparam_list *pf) {
  interror("Table generator not implemented.\n");
}

table_storage *ctg_concat(actparam_list *pf) {
  interror("Table generator not implemented.\n");
}

table_storage *ctg_empty(actparam_list *pf) {
  table_storage *t;
  actparam_list *p;
  double val,harm,wt,ph,dc;
  int k,i;
  
  if (pf->val <= 0)
    runtime("Invalid size for generator 'empty'.");

  k = (int)(pf->val + 0.5);
  t = new_table(k);
  for (i=0;i<k;i++)
    set_table_value(t,i,0);

  return t;
}

double i0(double x) {
  /* Opp/Schaefer p. 457 */

  double ds = 1, d = 0, s = 1;

  while (fabs(ds)> 1e-12) {
    d += 2;
    ds *= x*x/d/d;
    s += ds;
  }
  return s;
}
