// rescaler.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include "rescaler.h"
#include "data.h"
#include "processfun.h"
#include "query.h"
#include "request.h"

Rescaler::Rescaler(Data* d) : DataModifier(d), preserveZero(false) {}
		
Rescaler::Rescaler(Data* d, const Range& newrange)
		: DataModifier(d), newRange(newrange), preserveZero(false) {
	initialize();
}

int
Rescaler::doApply(Data *data) {
	oldRange = data->limits();
	double factor = preserveZero ?
		newRange.absoluteMax() / oldRange.absoluteMax()
		: newRange.spread() / oldRange.spread();
	data->deferUpdate(true);
	data->scaleSelf(factor);		// first scale min-to-max range
	if(!preserveZero) {
		Range currentRange = data->limits();
		double offset = newRange.min() - currentRange.min();
		Offset o(data, offset);
		o.apply();						// then adjust offset to match new min
	}
	data->deferUpdate(false);
	return true;
}

const QueryInfo *
Rescaler::requestInfo() {
    static QueryLabelInfo labels[] = {
        { "Rescale Values to Fit New Range: " }, { nil }
    };
    static QueryValueInfo values[] = {
		{ "New Maximum Value: ", "1.0", CharCheck::numsOnly },
		{ "New Minimum Value: ", "0.0", CharCheck::numsOnly },
		{ nil }
    };
    static QueryInfo info[] = {
        { labels, "", values },
        { nil }
    };
	return info;
}

void
Rescaler::configureRequest(Request &request) {
	if(target()->limits().includesZero())
		request.appendChoice(
			"Check this to scale symmetrically around zero",
			"|Preserve zero crossings|",
			/* default state = */ true,
			/* exclusive states = */ false
		);
}

boolean
Rescaler::setValues(Request& request) {
	static const int nVals = 2;
	QueryValue v[nVals];
	request.retrieveValues(v, nVals);
	if(request.hasChoices()) {
		QueryChoice choice;
		request.retrieveChoices(choice);
		preserveZero = choice;
	}
	newRange.set(v[1], v[0]).check();
	return true;
}

Modifier *
Rescaler::createUnDo() {
	return new Rescaler(target(), oldRange);
}
