// header_config.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// methods for protected Header::ConfigRequester class and its subclasses

#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include "header.h"
#include "pvocdata.h"
#include "pvocheader.h"
#include "query.h"
#include "request.h"
#include "sound.h"
#include "soundheader.h"

Header::ConfigRequester::ConfigRequester(Header* header) : myHeader(header) {
	myHeader->setRaw();			// set to raw mode to skip all disk reads
	myHeader->setDataOffset(0);	// assume no header to start with
}

const QueryInfo *
SoundHeader::SoundConfigRequester::requestInfo() {
    static QueryLabelInfo labels[] = {
        { "Read Raw Soundfile: " }, { nil }
    };
    static QueryChoiceInfo choices[] = {
        { "Channels:", "|1|2|4|", 0x1, true },
        { nil }
    };
    static QueryInfo info[] = {
        { labels, "", nil, choices },
        { nil }
    };
	return info;
}

void
SoundHeader::SoundConfigRequester::configureRequest(Request& request) {
	request.appendValue("Sample rate: ", Sound::defaultSampleRate(),
		CharCheck::posIntsOnly);
	request.appendValue("Bytes to skip before reading samples: ", 0,
		CharCheck::posIntsOnly);
	request.appendChoice("Sample format:",
		"|8-bit linear|8-bit mu law|16-bit linear|floating point|",
		linearEnumToPowerOfTwo(Sound::defaultDataType()),
		true
	);
	request.appendChoice("Swap bytes after reading?", "|Yes|No|", 0x2, true);
}

boolean
SoundHeader::SoundConfigRequester::setValues(Request& request) {
	SoundHeader* sh = (SoundHeader *) myHeader;
    QueryValue v[2];
	request.retrieveValues(v, 2);
	sh->samprate = v[0];
	sh->setDataOffset(v[1]);
	QueryChoice c[3];
	request.retrieveChoices(c, 3);
	sh->nchans = c[0];
	sh->data_type = DataType(powerOfTwoToLinearEnum(c[1]));
	sh->forceByteSwapping(c[2] == 0x1);
	return true;
}

void
FrameDataHeader::FrameConfigRequester::configureRequest(Request &request) {
	request.appendValue("Frame rate (frames/sec):",
		Sound::defaultSampleRate()/200.0, CharCheck::posNumsOnly);
	request.appendValue("Source sample rate:", Sound::defaultSampleRate(),
		CharCheck::posIntsOnly);
}

void
LPCHeader::LPCConfigRequester::configureRequest(Request& request) {
	request.appendLabel("Read Raw LPC File:");
	request.appendValue("Number of filter poles:", 34, Range(12, 64));
	Super::configureRequest(request);
}

boolean
LPCHeader::LPCConfigRequester::setValues(Request& request) {
	QueryValue v[3];	// FIX ME:  implied knowledge of base class method
	request.retrieveValues(v, 3);
	LPCHeader* lh = (LPCHeader *) myHeader;
	lh->npoles = int(v[0]);
	lh->nchans = lh->npoles + 4;
	lh->framerate = v[1];
	lh->samprate = v[2];
	return true;
}

void
FFTHeader::FFTConfigRequester::configureRequest(Request& request) {
	request.appendLabel("Read Raw FFT File:");
	request.appendValue("Number of frequency points:", 1024, CharCheck::posIntsOnly);
	Super::configureRequest(request);
}

boolean
FFTHeader::FFTConfigRequester::setValues(Request& request) {
	QueryValue v[3];	// FIX ME:  implied knowledge of base class method
	request.retrieveValues(v, 3);
	FFTHeader* fh = (FFTHeader *) myHeader;
	fh->nchans = v[0];
	fh->framerate = v[1];
	fh->samprate = v[2];
	return true;
}

void
EnvelopeHeader::EvpConfigRequester::configureRequest(Request& request) {
	request.appendLabel("Read Raw Envelope File:");
}

const QueryInfo *
PvocHeader::PvocConfigRequester::requestInfo() {
    static QueryLabelInfo labels[] = {
        { "Read Raw Pvoc Datafile: " }, { nil }
    };
    static QueryInfo info[] = {
        { labels, "", },
        { nil }
    };
	return info;
}

void
PvocHeader::PvocConfigRequester::configureRequest(Request& request) {
	request.appendValue("Frame rate (frames per second):",
		int(PvocData::defaultFrameRate()), CharCheck::posIntsOnly);
	request.appendValue("Bytes to skip before reading samples:", 0,
		CharCheck::posIntsOnly);
	request.appendChoice("Number of frequency points (N/2):",
		"|64|128|256|512|1024|2048|4096|16384|",
			PvocData::defaultNumberOfBands() >> 6, true);
	request.appendChoice("Swap bytes after reading?", "|Yes|No|", 0x2, true);
}

boolean
PvocHeader::PvocConfigRequester::setValues(Request& request) {
	PvocHeader* pvh = (PvocHeader *) myHeader;
    QueryValue v[2];
	request.retrieveValues(v, 2);
	pvh->samprate = v[0];	// actually frame rate
	pvh->setDataOffset(v[1]);
	QueryChoice c[2];
	request.retrieveChoices(c, 2);
	pvh->nchans = ((int(c[0]) << 6) + 1) * 2;
	pvh->forceByteSwapping(c[1] == 0x1);
	return true;
}
