// controller.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/event.h>
#include <InterViews/world.h>
#include <X11/keysym.h>
#include "localdefs.h"
#include "application.h"
#include "datafile.h"
#include "query.h"
#include "controller.h"
#include "converter.h"
#include "channelview.h"
#include "frameview.h"
#include "editor.h"
#include "range.h"
#include "data.h"
#include "datawindow.h"
#include "dataview.h"
#include "dialogbox.h"
#include "dialog_ctor.h"
#include "filename.h"
#include "optionsetter.h"
#include "request.h"
#include "statusaction.h"

// global member initialization

// to avoid g++ bug
static ViewInfo Default_Controller_ViewInfo;

Controller*			Controller::head = nil;
int					Controller::numControllers = 0;
int					Controller::quitting = false;
ViewInfo&			Controller::DefaultViewInfo = Default_Controller_ViewInfo;

// this is the static public meta-constructor for use at startup time
// returns nil if file could not be opened and read

Controller *
Controller::create(DataFile* file) {
	Controller* controller = new Controller(file->name());
	if(!controller->editor()->readFile(file) || !controller->initialize()) {
		delete controller;
		controller = nil;
	}
	return controller;
}

// this is the protected ctor called by the above function
// note that it does not call initialize() on its own

Controller::Controller(const char *filename) {
	earlyInit();
	myEditor = DataEditor::create(this, filename);
	setFileName(filename);
}

// this is used for the display of newly created data objects

Controller::Controller(Data *d, const char *name, ViewInfo& info) {
	earlyInit();
	myEditor = DataEditor::create(this, d);
	setFileName(
		(name != nil) ? name : FileName::tmpName(d->fileSuffix())
	);
	initialize(info);
}

// this init is done in all cases, regardless of final init outcome

void
Controller::earlyInit() {
	prev = next = nil;
	window = nil;
	myView = nil;
	converterAction = nil;
	editDelegate = nil;
	addToList();
}

#ifdef NO_TEMPLATES
declareStatusCallback(Controller)
implementStatusCallback(Controller)
#endif

boolean
Controller::initialize(ViewInfo& info) {
	if(editor()->model() != nil) {	// DataEditor successfully initialized
#ifdef NO_TEMPLATES
		converterAction = new Controller_StatusCallback(
#else
		converterAction = new StatusCallback<Controller>(
#endif
			this, &Controller::checkForStop
		);
		converterAction->ref();
		ref();	// to make sure no one else deletes this
		createAndAttachView(info);
		return true;
	}
	return false;
}

Controller::~Controller() {
	removeFromList();
	delete myEditor;
	Resource::unref(converterAction);
	if(numControllers == 0) {		// if we are closing last controller
		Converter::destroyInstance();
		DialogConstructor::destroyInstance();
		if(world())
			world()->quit();
		else
			exit(0);
	}
	detachView();
}

void
Controller::addToList() {
	prev = next = this;
	if(!head)
		head = this;
	else {
		prev = head->prev;
		next = head;
		head->prev->next = this;
		head->prev = this;
	}
	numControllers++;
}

void
Controller::removeFromList() {
	next->prev = prev;
	prev->next = next;
	if (this == head)
		head = (prev != this) ? prev : nil;
	if (Application::isGlobalController(this))
		Application::setGlobalController(head);
	numControllers--;
}

DataView*
Controller::newView(ViewInfo& info) {
	Data* data = model();
	DataView *d = nil;
	if (data->displayAsFrames()) {
		ViewInfo frameinfo(ChannelSpecialUnit);
		d = new FrameView(this, frameinfo);
	}
	else {
		const char *a = nil;
		Range frames(-1, -1);
		if (!info.frameRange.isNegative())		// if specific range requested
			frames = info.frameRange;
		Range channels(-1, -1);
		if (!info.channelRange.isNegative())	// if specific range requested
			channels = info.channelRange;
		else if((a = Application::getGlobalResource(
				data->channelDisplayAttribute())) != nil) {
			int totalChans;
			if((totalChans = atoi(a)) > 0)
				channels.set(0, totalChans-1);
		}
		a = Application::getGlobalResource(
			data->horizontalScaleModeAttribute());
		RangeUnit units = (a != nil && (strcmp(a, "Time") == 0)) ?
			FrameTimeUnit : FrameUnit;
		ViewInfo channelinfo(units, frames, channels);
		d = new ChannelView(this, channelinfo);
	}
	return d;
}

void
Controller::createAndAttachView(ViewInfo& info) {
	myView = newView(info);
	window = new DataWindow(this, myView);
	editDelegate = myView->getEditDelegate();
	editDelegate->ref();
	editDelegate->update(model()->frameRange());
}

void
Controller::detachView() {
	myView = nil;
	Resource::unref(editDelegate);
	Resource::unref_deferred(window);
	window = nil;
}

// this method is called, via the converterAction object, in the play and
// record loops during recording and playback of sounds

int
Controller::checkForStop() {
	return window->checkForControlKeyEvent();
}

Data*
Controller::model() const { return editor()->model(); }

// called when attempting to quit application

void
Controller::closeChain() {
	register Controller *c;
	int ncontrollers = numControllers;
	for(c = next; ; c = c->next) {
		c->close();			// close and decrement no.
		if(ncontrollers == numControllers)
			break;			// user chose to stop closing
		else
			ncontrollers = numControllers;	// keep going thru chain
		if(c == this)
			break;			// until we get back current
	}
}

const char*
Controller::windowName() const {
	return FileName::stripPath(fileName());
}

void
Controller::setFileName(const char *nm) {
	if(filename != nm) {
		filename = nm;
		if(window)
			window->changeName(windowName());
	}
}

void
Controller::setEditRegion(const Range &region, int chan) {
	editor()->setEditRegion(region, chan);
	editDelegate->update(editor()->currentRegion());
}

void
Controller::setInsertPoint(int point, int chan) {
	editor()->setInsertPoint(point, chan);
	editDelegate->update(editor()->currentRegion());
}

// all subview events are handed back to the Controller for redistribution
// via this method.  Only events with the control key down are of interest.

boolean
Controller::handleEvent(Event& e) {
	if(e.eventType == KeyEvent && e.control)
		return handleKeyEvent(e);
	else {
		Application::setGlobalController(this);
		myView->Handle(e);
		return true;
	}
}

#ifndef IV_IS_PATCHED

// this code is taken from the modified and correct IV 3.2 version of
// Event::keysym()

#include <IV-X11/Xutil.h>
#include <IV-X11/xevent.h>

static unsigned long
getKeySymFrom(Event& e) {
    KeySym k = NoSymbol;
	XEvent& xe = e.rep()->xevent_;
	if (xe.type == KeyPress) {
		char buf[10];
		XLookupString(&xe.xkey, buf, sizeof(buf), &k, nil);
	}
	return k;
}
#endif

boolean
Controller::handleKeyEvent(Event& e) {
#ifndef IV_IS_PATCHED
	// this is a temporary hack until the IV source is officially patched
	return keyCommand(getKeySymFrom(e));
#else
	return keyCommand(e.keysym());
#endif
}

// examine Controller chain for ones with a selected region and return the
// selection (with the most recent timestamp) to current Controller

Data *
Controller::findSelection() {
	Data *sel = nil;
	long currentStamp = 0;
	for(Controller* c = next; c != this; c = c->next) {
		long tstamp = 0;
		Data* s = c->getSelection(tstamp);
		if(s != nil) {
			if(tstamp > currentStamp) {
				Resource::unref(sel);	// free older selection
				sel = s;
				currentStamp = tstamp;
			}
			else {
				Resource::unref(s);		// free this selection
				s = nil;
			}
		}
	}
	return sel;	// not referenced here since never used in this Ctlr
}

Data *
Controller::getSelection(long &timestamp) {
	return editor()->getSelection(timestamp);
}

void
Controller::busy(boolean b) {
	register Controller *c;
	for(c = next; ; c = c->next) {
		if(c->window)
			c->window->busy(b);
		if(c == this)
			break;
	}
}

int
Controller::handleRequest(Request& request) {
	int response = Cancel;
	DialogBox *dialog = nil;
	if(window != nil) window->makeVisible();
	dialog = DialogConstructor::getInstance()->createDialog(window, request);
	while((response = dialog->display()) != Cancel) {	
		if(request.checkValues())
			break;		// vals ok; go on
	}
	Resource::unref(dialog);
	return response;
}

void
Controller::showInsertPoint(int point, const Range &chans, boolean scroll) {
	myView->setInsertPoint(point, chans, scroll);
}

void
Controller::showEditRegion(const Range& region, const Range& chans, 
		boolean scroll) {
	myView->setEditRegion(region, chans, scroll);
}

void
Controller::resetScaleTimes() {
	myView->resetHorizontalScale();
	editDelegate->update(editor()->currentRegion());
}

void
Controller::unselectView() {
	editDelegate->update(Range(0,0));	// sets display to zeros
	myView->unselect();
}

World *
Controller::world() {
	return (window != nil) ? window->getWorld() : nil;
}

void
Controller::display(World *world) {
	if(window)
		window->display(world);
	Application::setGlobalController(this);
}

// sym is checked first by controller, then by view, then by editor
// subclass, and lastly by editor base class.  Checking stops when matched.

boolean
Controller::keyCommand(unsigned long sym) {
	boolean interested = true;
	Application::setGlobalController(this);
	busy(true);
	switch (sym) {
	case XK_N:
		newViewOfSelection();
		break;
	case XK_copyright:
		viewAsChannels();
		break;
	case XK_ordfeminine:
		viewAsFrames();
		break;
	case XK_z:
		zoomToSelection();
		break;
	case XK_Z:
		zoomToFull();
		break;
	case XK_W:
		close();
		return false;
		break;	/*NOTREACHED*/
	case XK_quoteright:
		changeName();
		break;
	case XK_yen:	// another unused keysym
		showVersion();
		break;
	case XK_Q:
		quit();
		return false;
		break;	/*NOTREACHED*/
	case XK_section:
		Converter::useNative(true);
		break;
	case XK_diaeresis:
		Converter::useNative(false);
		break;
	case XK_threesuperior:
		Converter::destroyInstance();	// functions as reset
		break;
	case XK_slash:
		Data::deferRescan(false);
		break;
	case XK_backslash:
		Data::deferRescan(true);
		break;
	case XK_macron:
		setProgramOptions();
		break;
	case XK_acute:
		{
		FileOptionSetter fos;
		editor()->applyModifier(fos);
		}
		break;
	case XK_mu:
		{
		MemoryOptionSetter mos;
		editor()->applyModifier(mos);
		}
		break;
	default:
		if((interested = myView->keyCommand(sym)) != true)
			interested = editor()->keyCommand(sym);
		break;
	}
	busy(false);
	if(interested)
		Application::inform();
	return interested;
}

// The next several methods are those called by menu commands

void
Controller::newViewOfSelection() {
	Range chansInView = myView->getChannelRange();
	// display lesser of visible range or selected range
	if(editor()->nchans() < chansInView.size())
		chansInView = editor()->currentChannels();
	ViewInfo info(
		myView->horizRangeUnits(),
		editor()->currentRegion(),		// frame range to show
		chansInView						// channel range to show
	);
	Controller *newCtlr = new Controller(model(), fileName(), info);
	newCtlr->display(world());	
}

void
Controller::viewAsFrames() {
	viewAs(AsFrames);
}

void
Controller::viewAsChannels() {
	viewAs(AsChannels);
}

void
Controller::viewAs(ViewType type) {
	boolean flag = (type == AsFrames);
	if(model()->displayAsFrames() != flag) {
		if(model()->displayAsFrames(flag)) {
			window->unDisplay();
			detachView();
			createAndAttachView();
			display(world());
		}
	}
}

void
Controller::zoomToSelection() {
	myView->setVisibleFrameRange(editor()->currentRegion());
}

void
Controller::zoomToFull() {
	myView->setVisibleFrameRange(model()->frameRange());
}

void
Controller::close() {
	if(numControllers == 1 && !quitting &&
			!confirm("Closing this window will quit mxv.", "Continue?")) {
		busy(false);
		return;
	}
	if(editor()->closeFile() != Cancel) {
		busy(false);
		window->unDisplay();
		Application::inform();
		delete this;
	}
	else {
		busy(false);
		quitting = false;
	}
}

void
Controller::showVersion() {
	extern const char MXV_version_string[];
	alert(MXV_version_string, "Copyright 1993, 1994, 1995 by Douglas Scott");
}

void
Controller::quit() {
	if(confirm("Please confirm quit.")) {
		quitting = true;
		busy(false);
		closeChain();
	}
	else {
		busy(false);
		quitting = false;
	}
}

// this method will be moved into the Editor class

void
Controller::changeName() {
	Request request("Change File Name: ");
	request.appendValue("New Name: ", fileName(), CharCheck::anyChar);
	if(handleRequest(request)) {
		QueryValue s;
		request.retrieveValues(s);
		const char* name = (const char *) s;
		String fullName = editor()->defaultDir();
		if(!FileName::isFullPathName(name)) {
			fullName += "/";
			fullName += name;
		}
		else
			fullName = name;
		if(!DataFile::exists(fullName) || Application::confirm(
				"A file with that name already exists",
				FileName::isFullPathName(name) ?
					"at the path you have specified."
					: "in the default directory for this file type.",
				"Use this file name anyway?"))
			setFileName(fullName);
	}
}

void
Controller::setProgramOptions() {
	GlobalOptionSetter options;
	editor()->applyModifier(options);
}

// The remainder of these methods handle messages that need to get back to
// the user, whether they are errors or just queries for information

void
Controller::alert(const char *msg1, const char *msg2,
		const char *msg3, const char* button) {
	AlertRequest message(msg1, msg2, msg3, button);
	handleRequest(message);
}

boolean
Controller::confirm(const char *msg1, const char *msg2,
		const char *msg3, Response r, const char* btn1, const char* btn2) {
	ConfirmRequest message(msg1, msg2, msg3, r, btn1, btn2);
	return (handleRequest(message) == Yes);
}

Response
Controller::choice(const char *msg1, const char *msg2, const char *msg3,
		Response r, const char* btn1, const char* btn2, const char* btn3) {
	ChoiceRequest message(msg1, msg2, msg3, r, btn1, btn2, btn3);
	return (Response) handleRequest(message);
}
