// request.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// a Request is sent from a Requestor to a Controller, giving it a set of
// QueryItems, which contain variables to assign values to, labels for querying
// for those variables, and a pointer to a function to check the entered chars.
// The Controller then passes the Request to the DialogConstructor, where the
// QueryItem instances are passed to the GUI components.  After the parameters
// are set by the user, the Request hands those values back to the Requester.
// Many Requests are merely to display a message for the user, or ask for
// confirmation of some operation, or allow a simple choice.  These contain
// "read-only" QueryItems which contain text strings to display.
// See query.h, requester.h, and dialog_ctor.h.

#ifndef REQUEST_H
#ifdef __GNUG__
#pragma interface
#endif
#define REQUEST_H

#include "localdefs.h"
#include <InterViews/resource.h>

class QueryItem;

struct QueryLink {
	QueryLink(QueryItem *q);
	virtual ~QueryLink();
	QueryItem *element;
	QueryLink *next;
};

class String;

class QueryList {
public:
	QueryList(const char *lbl=nil);
	virtual ~QueryList();
	void append(QueryItem *);
	void start() { current = head; }
	boolean more() { return current != nil; }
	void next() { if(more()) current = current->next; }
	const char *label();
	QueryItem* query() { return current->element; }
private:
	QueryLink *tail();
private:
	String *listLabel;
	QueryLink *current;
	QueryLink *head;
};

class QueryButton;

class ButtonList : public QueryList {
public:
	ButtonList(const char *lbl=nil, Response defaultsTo=Yes)
		: QueryList(lbl), default_response(defaultsTo) {}
	virtual ~ButtonList() {}
	QueryButton* query() { return (QueryButton *) QueryList::query(); }
	Response defaultResponse() { return default_response; }
private:
	Response default_response;
};

class QueryValue;

class ValueList : public QueryList {
public:
	ValueList(const char *lbl=nil) : QueryList(lbl) {}
	virtual ~ValueList() {}
	QueryValue* query() { return (QueryValue *) QueryList::query(); }
};

class QueryChoice;

class ChoiceList : public QueryList {
public:
	ChoiceList(const char *lbl=nil) : QueryList(lbl) {}
	virtual ~ChoiceList() {}
	QueryChoice* query() { return (QueryChoice *) QueryList::query(); }
};

enum RequestType { AlertType, ConfirmerType, ChoiceType, InputType };

struct QueryInfo;
struct QueryLabelInfo;
struct QueryValueInfo;
struct QueryChoiceInfo;
struct QueryButtonInfo;
class DialogConstructor;
class ValueArray;
class QueryReturn;
class QueryFile;
class RequestDelegate;

class Request : public Resource {
	friend DialogConstructor;
public:
	Request(const QueryInfo *, RequestDelegate *delegate=nil);
	Request(const char *label, RequestDelegate *delegate=nil);
	virtual ~Request();
	
	boolean hasLabels() { return labelList != nil; }
	boolean hasBrowser() { return browserQuery != nil; }
	boolean hasValues() { return valueList != nil; }
	boolean hasChoices() { return choiceList != nil; }
	boolean useBell() { return bell; }
	virtual boolean print(FILE *);
	virtual RequestType type() { return InputType; }	// default
	void appendLabel(const char *);
	void appendLabel(const char *, const char *);
	void addFileBrowser(const char* path, const char* suffixes=nil);
	void appendValue(const char *, const char *value, CharCheckFun);
	void appendValue(const char *, int value, CharCheckFun);
	void appendValue(const char *, double value, CharCheckFun);
	void appendChoice(const char *, const char *lbls, unsigned states,
		boolean excl);
	void appendButton(const char *, Response);
	int retrieveValues(QueryReturn &);
	int retrieveValues(QueryReturn *array, int nelements);
	int retrieveChoices(QueryReturn &);
	int retrieveChoices(QueryReturn *array, int nelements);
	boolean checkValues();
	void setBell(boolean b) { bell = b; }
protected:
	Request() : myDelegate(nil) { init(); }	// used by derived classes
	void newLabelList();
	void newValueList();
	void newButtonList();
	virtual void createLabelList(QueryLabelInfo *);
	virtual void createValueList(const char *, QueryValueInfo *);
	virtual void createChoiceList(QueryChoiceInfo *);
	virtual void createButtonList(QueryButtonInfo *, Response r=Yes);
protected:
	QueryList* labelList;
	QueryFile* browserQuery;
	ValueList* valueList;
	ChoiceList* choiceList;
	ButtonList* buttonList;
private:
	void init();
private:
	RequestDelegate *myDelegate;
	boolean bell;							// flag for bell ring
};

// subclass for requests which open browser for file selection

class FileRequest : public Request {
public:
	FileRequest(const char* title, const char* dir,
		const char* suffix=nil, RequestDelegate *delegate=nil);
};

// intermediate, abstract base class for following classes

class MessageRequest : public Request {
protected:
	MessageRequest(const char* m1, const char* m2, const char* m3);
	virtual ~MessageRequest();
};

// AlertRequest only asks for a message display and a single "ok" button

class AlertRequest : public MessageRequest {
	typedef MessageRequest Super;
	friend DialogConstructor;
public:
	AlertRequest(const char* m1, const char* m2=nil, const char* m3=nil);
	virtual ~AlertRequest();
	virtual RequestType type() { return AlertType; }
private:
	void init();
};

// ConfirmRequest asks for a message display and a confirm-cancel button pair

class ConfirmRequest : public MessageRequest {
	typedef MessageRequest Super;
	friend DialogConstructor;
public:
	ConfirmRequest(const char *m1, const char *m2=nil,
		const char *m3=nil, Response r=Yes);
	virtual ~ConfirmRequest();
	virtual RequestType type() { return ConfirmerType; }
private:
	void init(Response);
};

// ChoiceRequest asks for a message display and yes, no, and cancel buttons

class ChoiceRequest : public MessageRequest {
	typedef MessageRequest Super;
	friend DialogConstructor;
public:
	ChoiceRequest(const char *m1, const char *m2=nil,
		const char *m3=nil, Response r=Yes);
	virtual ~ChoiceRequest();
	virtual RequestType type() { return ChoiceType; }
private:
	void init(Response);
};

extern QueryButtonInfo defaultInputButtonInfo[];

#endif

