// pvsynthesizer.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "envelope.h"
#include "localdefs.h"
#include "query.h"
#include "request.h"
#include "phasevocoder.h"
#include "pipeaction.h"
#include "pvsynthesizer.h"
#include "pvocdata.h"

PVSynthesizer::PVSynthesizer(Data* data, PvocData* anal)
	: QueuedOutputFunction(data, 1024), pvoc(nil), pvdata(anal) {
	pvdata->ref();
}

PVSynthesizer::PVSynthesizer(
	Data* data,  PvocData* anal,
	int L, int I, double T, int i, int j, boolean useKaiser)
		: QueuedOutputFunction(data, 2*L),
		  pvoc(
		  	new PhaseVocoder(anal->channels() - 2, 0, anal->frameOffset(),
				L, I, T, i, j, 0.0, useKaiser, double(data->sRate()),
				(data->dataType() == FloatData) ? 1.0 : 32767.0)
		  ),
		  pvdata(anal) {
	pvdata->ref();
	if(pvoc->isGood())
		initialize();
}

Modifier *
PVSynthesizer::create(DataEditor *de) {
	return nil;
}

PVSynthesizer::~PVSynthesizer() {
	Resource::unref(pvdata);
	delete pvoc;
}

void
PVSynthesizer::restoreState() {
	Super::restoreState();
	pvoc->reset();
	currentFrame.set(0, 0);
}

const QueryInfo *
PVSynthesizer::requestInfo() {
    static QueryLabelInfo labels[] = {
        { "Phase Vocoder Resynthesis into Selected Region:" },
        { nil }
    };
    static QueryInfo info[] = {
        { labels, "All \"0\" values will be set to defaults." },
        { nil }
    };
	return info;
}

void
PVSynthesizer::configureRequest(Request& request) {
	int fftSize = pvdata->channels() - 2;
	request.appendValue("Output frame size (samples):",
		fftSize, CharCheck::posIntsOnly);
	request.appendValue("Spectral envelope warp factor:",
		0.0, CharCheck::numsOnly );
	request.appendValue("Lower band limit (hz.):",
		0, CharCheck::posNumsOnly);
	request.appendValue("Upper band limit: (hz.)",
		pvdata->sRate()/2, CharCheck::posNumsOnly);
	request.appendChoice("Window Type:", "|Hanning|Kaiser|", 0x1, true);
}

boolean
PVSynthesizer::setValues(Request& request) {
	static const int nVals = 4;
	QueryValue v[nVals];
	request.retrieveValues(v, nVals);
	int outputFrameSize = v[0];
	int outputFrameOffset = 0;
	double timeScaleFactor = 1.0;
	double warpFactor = v[1];
	int firstBand = pvdata->getBandNumber(v[2]);
	int lastBand = pvdata->getBandNumber(v[3]);
	QueryChoice choice;
	request.retrieveChoices(choice);
	boolean kaiser = (choice == 0x2);	// flag for Kaiser windowing
	int fftSize = pvdata->channels() - 2;
	double analDur = pvdata->duration();
	double synthDur = target()->length() / double(sampRate());
	printf("anal dur = %f, selected dur = %f, ratio = %f\n", analDur, synthDur, synthDur/analDur);
	printf("new timeScaleFactor = %f\n\n", timeScaleFactor /= synthDur/analDur);
	pvoc = new PhaseVocoder(
		fftSize, 0, pvdata->frameOffset(),
		outputFrameSize, outputFrameOffset, timeScaleFactor,
		firstBand, lastBand, warpFactor, kaiser,
		sampRate(), (target()->dataType() == FloatData) ? 1.0 : 32767.0);
	return pvoc->isGood();
}

boolean
PVSynthesizer::confirmValues(Controller *) {
    int status = true;
    char msg[120];
	char msg2[120];
    if(sampRate() != pvdata->sRate()) {
        sprintf(msg,
            "Warning: Phase Vocoder data samprate (%d) != selection samprate (%d).",
                pvdata->sRate(), sampRate());
		sprintf(msg2, "Resynthesized length and spectrum will be skewed by a factor of %f.", sampRate() / float(pvdata->sRate()));
        status = Application::confirm(msg, msg2, "Continue anyway?", Cancel);
    }
    return status;
}

void
PVSynthesizer::initialize() {		
	setOutQueueSize(2 * pvoc->getOutputFrameOffset());
	Super::initialize();
}

typedef int (QueuedOutputFunction::*(InQueueFunction))(double);

class PVInPipeAction : public InPipeAction {
public:
	PVInPipeAction(PVSynthesizer *pvs, InQueueFunction fun)
		: pv(pvs), pfun(fun) {}
	redefined int add(double val) { return (pv->*pfun)(val); }
private:
	PVSynthesizer* pv;
	InQueueFunction pfun;
};

int
PVSynthesizer::doProcessing() {
	BUG("PVSynthesizer::doProcessing");
	int status = (pvdata->nFrames() > currentFrame);
	if(status) {
		Data* frame = pvdata->clone(currentFrame);
		InPipeAction* action = new PVInPipeAction(
			this, &PVSynthesizer::addToOutQueue
		);
		status = pvoc->runSynthesis(frame, action);
		Resource::unref(frame);
		currentFrame += 1;
	}
	return status;
}
