// pvanalyzer.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "envelope.h"
#include "localdefs.h"
#include "query.h"
#include "request.h"
#include "pvanalyzer.h"
#include "pvocdata.h"

PVAnalyzer::PVAnalyzer(Data* data) : ArrayFunction(data), pvoc(nil) {
}

PVAnalyzer::PVAnalyzer(
	Data* data,
	int N, int F, int W, int M, int D, double T, double P,
	PhaseVocoder::Mode mode, boolean useKaiser)
		: ArrayFunction(data, M, D),
		  pvoc(new PhaseVocoder(N, F, W, M, D, T, P, mode, useKaiser,
			double(data->sRate()),
			(data->dataType()==FloatData) ? 1.0 : 1/32767.0)
		  )
	{
	if(pvoc->isGood())
		initialize();
}

PVAnalyzer::~PVAnalyzer() {
	delete pvoc;
}

const QueryInfo *
PVAnalyzer::requestInfo() {
    static QueryLabelInfo labels[] = {
        { "Phase Vocoder Analysis of Selected Region:" },
        { nil }
    };
    static QueryValueInfo values[] = {
		{ "Input frame size (samples):", "0", CharCheck::posIntsOnly },
		{ "Input frame offset (samples):", "0", CharCheck::posIntsOnly },
        { "Input frame rate (Hz):", "0", CharCheck::posNumsOnly },
        { "Time Scaling factor:", "1.0", CharCheck::posNumsOnly },
        { nil }
    };
    static QueryChoiceInfo choices[] = {
        { "FFT Size:",
            "|64|128|256|512|1024|2048|4096|8192|16384|", F_256, true },
        { "Filter Overlap Factor:", "|0|1|2|3|", 0x4, true },
        { "Window Type:", "|Hanning|Kaiser|", 0x1, true },
        { nil }
    };
    static QueryInfo info[] = {
        { labels, "All \"0\" values will be set to defaults.", values, choices },
        { nil }
    };
	return info;
}

boolean
PVAnalyzer::setValues(Request& request) {
	static const int nVals = 4;
	QueryValue v[nVals];
	request.retrieveValues(v, nVals);
	int inputFrameSize = v[0];
	int inputFrameOffset = v[1];
	if(double(v[2]) > 0.0) {			// if user entered frame rate
		setOffset(int(sampRate()/double(v[2])));
		inputFrameOffset = offset();		// reset based on frame rate
	}
	double timeScaleFactor = v[3];
	static const int nChoices = 3;
	QueryChoice choices[nChoices];
	request.retrieveChoices(choices, nChoices);
	int state = choices[0];
	int fftSize = state << 6;	// {1, 2, 4, ..} => {64, 128, 256, ...}
	int overlapFactor = -1;
	if(inputFrameSize == 0) {	// if nonzero, no need to use overlap factor
		state = choices[1];
		overlapFactor = state==0x1 ? 0 : state==0x2 ? 1 : state==0x4 ? 2 : 3;
	}
	boolean kaiser = (choices[2] == 0x2);	// flag for Kaiser windowing
	double inputScale = (target()->dataType() == FloatData) ? 1.0 : 1/32767.0;
	pvoc = new PhaseVocoder(
		fftSize, 0, overlapFactor, inputFrameSize, inputFrameOffset,
		timeScaleFactor, 1.0, PhaseVocoder::Analysis, kaiser,
		sampRate(), inputScale);
	return pvoc->isGood();
}

void
PVAnalyzer::initialize() {		
	setBaseValues(
		pvoc->roundedInputFrameSize(), 0.0, pvoc->getInputFrameOffset()
	);
	Super::initialize();
	setAnalysis(
		new PvocData(analysisLength(), pvoc->analysisChannels(), sampRate(), framerate())
	);
}

int
PVAnalyzer::operator () (double* in, Data* frame) {
	int status = pvoc->runAnalysis(in, frame);
	setOffset(pvoc->getInputFrameOffset());		// update values
	return status;
}

int
PVAnalyzer::initialOffset() { return pvoc->getStartingOffset(); };

int
PVAnalyzer::analysisLength() {
	return pvoc->calculateAnalysisLength(target()->length());
}
