// lowpassfilter.c

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <math.h>
#include "application.h"
#include "localdefs.h"
#include "lowpassfilter.h"
#include "query.h"
#include "request.h"

LowPassFilter::LowPassFilter(Data* output)
	: SimpleFunction(output), gain(1.0), g1(0), g2(0) {}

LowPassFilter::LowPassFilter(Data* output, double feedback)
		: SimpleFunction(output), gain(1), g1(1.0 - feedback), g2(feedback) {
	initialize();
}

const QueryInfo *
LowPassFilter::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Low Pass Filter Selected Region:" },
		{ nil }
	};	
	static QueryValueInfo valueInfo[] = {
		{ "Cutoff Frequency:", "1000.00", CharCheck::posNumsOnly },
		{ "Gain Factor:", "1.00", CharCheck::numsOnly },
		{ nil }
	};	
	static QueryInfo info[] = {
		{ labelInfo, "", valueInfo },
		{ nil }
	};
	return info;
}

boolean
LowPassFilter::setValues(Request& request) {
	const int nvals = 2;
	int status = true;
	QueryValue vr[nvals];
	request.retrieveValues(vr, nvals);
	cutoff = vr[0];
	if(cutoff > sampRate()/2.0) {	// cutoff freq check
		Application::alert("Cutoff frequency must be between 0 and the Nyquist.");
		status = false;
	}
	gain = vr[1];
	return status;
}

void
LowPassFilter::initialize() {
	clear();
	firstTime = true;
	setCoefficients(&g1, &g2);
	SimpleFunction::initialize();
}

void
LowPassFilter::restoreState() {
	clear();
	firstTime = true;
}

void
LowPassFilter::setCoefficients(double *cf1, double *cf2) {
	if(!*cf1 && !*cf2) {        // if these were not set in constructor
		double x = 2.0 - cos(cutoff * M_PI_2 / sampRate());
		*cf2 = x - sqrt(x * x - 1.0);	// feedback coeff.
		*cf1 = 1.0 - *cf2;		// gain coeff.
		*cf1 *= gain;			// gain multiplier
	}
}

double
LowPassFilter::operator () (double input) {
	past = firstTime ? input : past;
	firstTime = false;
	past = g1 * input + g2 * past;
	return past;
}
