// channelgraph.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include <InterViews/painter.h>
#include <InterViews/brush.h>
#include <InterViews/font.h>
#include <InterViews/perspective.h>
#include <InterViews/sensor.h>
#include <InterViews/event.h>
#include <InterViews/shape.h>
#include "channelgraph.h"
#include "data.h"
#include "selector.h"
#include "graphplot.h"
#include "controller.h"
#include "rubbertext.h"

#if defined(iv2_6_compatible)
static float defaultDashWidth = 0;
#else
static Coord defaultDashWidth = 0;
#endif

ChannelGraph::ChannelGraph(Controller *c, Data *d, int chan,
		const Range &vertRange) : Graph(c, d, vertRange),
		channel(chan), graphplot(nil), selector(nil), frameLabel(nil) {
	Init();
}

ChannelGraph::ChannelGraph(Controller *c, Data *d, int chan)
		: Graph(c, d, d->limits(chan)),
		  channel(chan), graphplot(nil), selector(nil), frameLabel(nil) {
	Init();
}

ChannelGraph::~ChannelGraph() {
	Resource::unref(graphplot);
	Resource::unref(stdbrush);
	Resource::unref(dashbrush);
	Resource::unref(selector);
	Resource::unref(frameLabel);
}

void
ChannelGraph::Init() {
	static int pattern[] = {2, 2};
	SetClassName("ChannelGraph");
	stdbrush = new Brush(0xffff, 0);
	stdbrush->ref();
	dashbrush = new Brush(pattern, 2, defaultDashWidth);
	dashbrush->ref();
	selector = new Selector(this);
	selector->ref();
}

void
ChannelGraph::Reconfig() {
	BUG("ChannelGraph::Reconfig()");
	if(!isConfigured()) {
		const char *a = GetAttribute("PlotStyle");
		useLines = (a != nil && !strcmp(a, "Line"));
	}
	Graph::Reconfig();
}

void
ChannelGraph::Redraw(Coord l, Coord b, Coord r, Coord t) {
	output->SetBrush(stdbrush);
	output->Clip(canvas, l, b, r, t);
	output->ClearRect(canvas, l, b, r, t);
	output->SetOrigin(horizOriginLocation(), vertOriginLocation());
	output->SetBrush(dashbrush);
	output->Line(canvas, 0, 0, currentWidth(), 0); // draw dotted origin
	output->SetBrush(stdbrush);
//	This cannot be used until Coords are floats
//	output->Scale(1.0, vertScaleFactor());
	graphplot->draw(canvas, output);	// draw waveform
//	output->SetTransformer(nil);		// reset to non-transformed
	output->SetOrigin(0, 0);
	selector->updateSelf(l, b, r, t);	// this should be made automatic!
	output->NoClip();
}

void
ChannelGraph::Update() {
	BUG("ChannelGraph::Update()");
	Graph::Update();
}

void
ChannelGraph::setReferenceRange() {
	Graph::setReferenceRange(graphData->limits(currentChannel()));
}

boolean
ChannelGraph::isViewChanged(Perspective &np) {
	register Perspective *p = perspective;
	return (
		np.curwidth != p->curwidth
		|| np.width != p->width
		|| np.curx != p->curx
	);
}

void
ChannelGraph::processKeyEvent(Event& ev) {
	boolean do_select = true;
	switch (*ev.keystring) {
	case 'l':
		selector->shift(1);
		break;
	case 'h':
		selector->shift(-1);
		break;
	case '+':
	case '=':
		selector->changeWidth(1);
		break;
	case '_':
	case '-':
		selector->changeWidth(-1);
		break;
	case 127:		// delete
		selector->collapseRegion(-1);
		break;
	case 32:		// space bar
		selector->collapseRegion(1);
		break;
	case 'Q':
		controller->close();
		do_select = false;
		break;
	default:
		do_select = false;
		break;
	}
	if(do_select)
		select();
}

void
ChannelGraph::grab(Event &e) {	// mouse button pressed
	if(e.button == LEFTMOUSE && !e.middle_is_down()) {
		selector->markInsertPoint(e.x);
		if(e.control_is_down()) {
			select();	// continuous update w/ ctrl key
			displayDataValue(true);
		}
	}
	else if(e.button == MIDDLEMOUSE && !e.left_is_down()) {
		selector->markEditRegion(e.x);
		if(e.control_is_down()) select();	// continuous update w/ ctrl key
	}
	else if(e.button == RIGHTMOUSE && !e.middle_is_down()) {
		if(e.control_is_down()) selectAllVisible();
		else selectAll();
	}
}

void
ChannelGraph::track(Event &e) {		// mouse button moved
	if(e.left_is_down() && !e.middle_is_down()) {
		selector->markInsertPoint(e.x);
		if(e.control_is_down()) {
			select();	// continuous update w/ ctrl key
			displayDataValue(true);
		}
		else
			displayDataValue(false);
	}
	else if(e.middle_is_down() && !e.left_is_down()) {
		selector->markEditRegion(e.x);
		if(e.control_is_down()) select();	// continuous update w/ ctrl key
	}
}

void
ChannelGraph::select() {				// mouse button released
	displayDataValue(false);		// erase value display if any
	if(selector->selectEdit()) {	// if this channel has selection
		int ch = currentChannel();
		if(selector->regionSelected())
			controller->setEditRegion(selector->editRegion(), ch);
		else
			controller->setInsertPoint(selector->insertPoint(), ch);
	}
}

void
ChannelGraph::unSelect() {
	selector->unSelect();
	controller->setInsertPoint(0, -1);	// reset
}

void
ChannelGraph::selectAll() {
	selector->selectAll();
	select();
}

void
ChannelGraph::selectAllVisible() {
	selector->selectVisible();
	select();
}

void
ChannelGraph::setInsertPoint(int point) {	// allows non-event-driven setting
	selector->setInsertLocation(point);
	select();
}

void
ChannelGraph::setEditRegion(const Range& region) {
	selector->setEditRegion(region);
	select();
}

boolean
ChannelGraph::selected() {
	return selector->isSet();
}

void
ChannelGraph::displayDataValue(boolean show) {
	static char valueText[32];
	if(show) {
		sprintf(valueText, "Value: %0.3f",
			graphData->get(selector->insertPoint(), currentChannel()));
		if(frameLabel == nil) {
			output->FillBg(false);
			// display this label in the gap created by the top border
			// if font is taller than gap, move it down to fit
			int fontHeight = output->GetFont()->Height() + 2;
			frameLabel = new RubberText(
				output, canvas, 
				leftEdge() + 2, ymax - max(fontHeight, topBorder()) + 2,
				valueText
			);
			frameLabel->ref();
		}
		frameLabel->Track(valueText, 0, 0);
	}
	else if(frameLabel != nil) {
		frameLabel->Erase();
		Resource::unref(frameLabel);
		frameLabel = nil;
		output->FillBg(true);
	}
}

void
ChannelGraph::setInsertCoord(Coord c) {	// allows non-event-driven setting
	selector->setInsert(c);
	select();
}

void
ChannelGraph::setRegionCoords(const Range& region) {
	selector->setRegion(region);
	select();
}

void
ChannelGraph::setChannel(int newchannel) {
	if(newchannel != channel) {
		channel = newchannel;
		Update();
	}
}

void
ChannelGraph::togglePlotMode() {
	useLines = !useLines;
	createPlot();
	viewIsChanged = true;
	doAdjust();
	Draw();
}

void
ChannelGraph::createPlot() {
	Resource::unref(graphplot);
	if (useLines) graphplot = new LinePlot(this, currentWidth());
	else graphplot = new BarPlot(this, currentWidth());
	graphplot->ref();
}

// Extract portion of data that will be visible in this graph

void
ChannelGraph::loadPlot() {
	register Perspective *p = GetPerspective();
	Range horiz(p->curx, p->curx + p->curwidth - 1);
	Range vert(channel, channel);
	graphplot->setPlotLength(min(p->curwidth, currentWidth()));
	Data *visible = graphData->clone(horiz, vert);
	visible->ref();
	graphplot->load(visible);
	Resource::unref(visible);
}

void
ChannelGraph::doPlot() {
	graphplot->plot();
}

const char *
ChannelGraph::verticalScaleLabel() {
	return graphData->channelName(currentChannel());
}
