/****************************************************************************** 
 *
 *  mixview - X Window System based soundfile editor and processor
 *
 *  Copyright 1989 by Douglas Scott
 *
 *  Author:     Douglas Scott 
 *  Date:       May 1, 1989
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The author makes no representations about
 *  the suitability of this software for any purpose.  
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
/* display.c -- the graphics soundfile display set and reset routines,
   including the numerical display setting routines.
*/

#include "main.h"
#include "debug.h"
#define ROUNDUP(x) ((int) ((x) + 0.5))

extern sf_struct *v;
extern cbuf_struct *c;
extern Display *dpy;
extern Window canvas_win, cwin;
extern Canvas *canvas;
extern Pixmap plot_pm, vscale_pm;
extern GC edit_gc, dbase_gc, plot_gc, vscale_gc, hscale_gc;
extern XFontStruct *scale_fs;
extern unsigned long myfg, mybg;
extern int xloc, yloc, quick_display;
extern FLAG line, box, hairmode, chmode, insert, region;
extern FLAG displaymode, chscale, rdrline, rdrbox, amps;

int xend;		          /* loc of edge of edit box */
char ampstr[2][24];
static double pkamp[2][MAXCOLS];  /* array of current peak amps for screen */
static int vmargin = 6;		  /* margin above & below displayed wave */
static int p_len = 7, s_len = 4, t_len = 2;
static int txtoffset = 10;  	  /* offset of amp & time labels */

/* low-level drawing functions:
 */

/* Draw the actual signal.  A minimum and maximum amplitude is calculated
 * for each grain, these values are scaled to fit within the available
 * graphics bandwidth.
 */

plot_sf(scalefac, readit)
double scalefac;
FLAG readit;
{
        register int grain, grain0 = 0, grain1 = 0;
        register int y0=0, y1=0, nsegs, ch, base, hbw;
	int dbase[2], start, shadow, quick;
        static double min[2][MAXSEGS], max[2][MAXSEGS];
	double *minptr, *maxptr;
        char *sfptr, *maxsfptr, *sptr;
	XSegment segments[MAXSEGS]; /* segments for drawing wave */
	XSegment *segptr;
	unsigned bytesamp;
        
/* Set some graphics constants:
 */
        v->bwidth = v->height/v->nchans;
        v->scale = (double) ((scalefac*v->bwidth*0.5-vmargin)/(v->peakamp+1.0));
	nsegs = v->width * v->nchans;
	segptr = segments;
	bytesamp = v->size * v->nchans;		/* bytes per samp  all chans */
        maxsfptr = v->sfbuff + v->bufsize;	/* last loc in array */
	maxsfptr -= (long) v->grainsize*bytesamp;	/* for added safety */
        sfptr = v->sfbuff;			/* local pointer    */
	start = (v->grainsize >= 1.0) ? v->grainsize : 1; /* counter reset */
	/* only use option when grainsize > 10 */
	quick = (v->grainsize >= 10.0) ? quick_display : 0;
	/* shadow mode for quick display */
	shadow = quick ? 1 : displaymode;

/* Do "seek" in sf buffer to the first sample to be displayed */
/* Make sure we start on first byte of first samp of first channel */

        sfptr += (long)(bytesamp*(long)(v->sgrain*v->grainsize)); 
        
/* Calculate the maximum and minimum for each channel of each grain
 * Outline of code:
 *      for each channel {
 *              for each grain {
 *                      initialize min,max
 *                      compare 'grainsize' number samples to min, max
 *                      set y0, y1
 *                      draw signal into canvas_pixrect
 *              }
 *      }
 */
	if(readit){	/* if NOT just a vertical rescaling: */
        for (ch = 0; ch < v->nchans; ch++) {
	    int choffset = ch * v->size, icount;
	    int scount = 0;
	    double *pkptr, count;
	    minptr = min[ch];
	    maxptr = max[ch];
	    if (!shadow) { 		/* line mode */
		*minptr = v->peakamp;
		*maxptr = -(v->peakamp);
	    }
	    else *minptr = *maxptr = 0.0;	/* shadow mode */
	    icount = count = start;
            for (grain = 0; grain < v->width; grain++) {
		pkptr = &pkamp[ch][grain];
		if(grain) pkamp[ch][grain] = pkamp[ch][grain-1];
		/* examine only 4 samps per grain if quick display */
	        if(quick) scount = icount - 4;
		/* offset for grain (in multiples of bytesamp) */
                sptr = sfptr+choffset+((int)(grain*v->grainsize)*bytesamp);
		while(icount > scount) {
                   if(sptr < maxsfptr) {	/* if not EOF */
		 	double samp;
		 	samp= (v->size == 4) ? *((float *) sptr) :
				(v->size == 2) ? *((short *) sptr) : *sptr; 
                        if(samp < *minptr) *minptr = samp;
                        if(samp > *maxptr) *maxptr = samp;
                        sptr += bytesamp; /* increment to next samp in chan */
                   }
                   else *minptr = *maxptr = 0.0; /* EOF */
		   ++scount;
		}
		/* set total peak pointer */
		if(*maxptr > *minptr && *maxptr > -*minptr ||
		   *maxptr < *minptr && *maxptr < -*minptr)
			*pkptr = *maxptr;
		else *pkptr = *minptr; 
		count += v->grainsize;
		icount = count;
	   	++minptr;	/* increment pointers to next grain */
		++maxptr;
		if(icount > scount) {	/* if next grain will get samp */
			if (!shadow) { 		/* line mode */
				*minptr = v->peakamp;
				*maxptr = -(v->peakamp);
			}
			else *minptr = *maxptr = 0.0;	/* shadow mode */
		}
		else {		/* if next grain on same samp as last */
			*minptr = *(minptr-1);	/* set each  		 */
			*maxptr = *(maxptr-1);	/* equal to its prev. val   */
		}
	    }
	}	
	}

        /* Calculate y0 and y1 for this grain of this channel.  Here,
         * 'base' is the Y coordinate of the zero-amp line for this channel.
         * y0 and y1 are then calculated by scaling the min/max values
         * within the bandwidth, and then subtracting from this base.
         */
	 /* but first, draw the vertical scale into its pixmap */

	if(chscale) draw_amp_scale(v->peakamp/scalefac);
	chscale = 0;

	hbw = v->bwidth/2;
        for (ch = 0; ch < v->nchans; ch++) {
	    minptr = min[ch];
	    maxptr = max[ch];
	    base = v->height - (v->bwidth * (v->nchans-1-ch));
	    base -= hbw;
	    dbase[ch] = base;
            for (grain = 0; grain < v->width; grain++) {
                if ((v->grainsize <= 1) && !shadow) /* line-style */
                {
                        grain0 = grain - 1;
                        if (grain==1) grain0 = grain; 
                        grain1 = grain;
                        y0 = y1;                 /* Old = New */
                        y1 = base - (pkamp[ch][grain]*v->scale);
                        if (grain==0) y0 = y1; 
	        	/* if(!grain) continue; */    /* get next point */
                }
                else                   /* The original shadow style */
                {
                        grain0 = grain1 = grain;
                        y0 = base - (*minptr * v->scale);
                        y1 = base - (*maxptr * v->scale);
                } 
		segptr->x1 = grain0;
		segptr->y1 = y0;
		segptr->x2 = grain1;
		segptr->y2 = y1;
		segptr++;
	   	minptr++;
		maxptr++;
           }
        }
	/* draw all segments for waveform */

	XDrawSegments(dpy, plot_pm, plot_gc, segments, nsegs);
	    
        /* draw base line for each channel and copy the vertical scale pixmap 
	   into each channel's band 					   */

	XSetFunction(dpy, edit_gc, myfg?GXxor:GXand); 
	for(ch=0; ch<v->nchans; ch++){
	   XCopyArea(dpy, vscale_pm, plot_pm, edit_gc,
		0, 0, 
		VSPM_WIDTH,  v->bwidth,
		0, v->bwidth*ch);
           XDrawLine(dpy, plot_pm, dbase_gc, 0, dbase[ch], v->width, dbase[ch]);
	}
	XSetFunction(dpy, edit_gc, GXxor);
}

draw_hair(x, y) /* prints vertical and horizontal crosshairs on screen */
int x, y;
{
        if (line && !(box&&chmode))    /* Draw over to negate previous line. */
	{    
		do_line(xloc, yloc, 0);
                if(hairmode && !chmode)  /* Erase old horiz hair */
                {
			do_line(xloc, yloc, 1);
                }
        }
	if(!(box&&chmode))
	{
		xloc= x;               /* Now draw new line */
		yloc= y;
		do_line(xloc, yloc, 0);
	}
        print_time(x, y);      /* print current insert time */
        if(hairmode || chmode)  /* Draw new horiz hair if mode = 1 */
        {
		do_line(xloc, yloc, 1);
        	print_amps(x, y);
        }                                 
        line= 1;          /* set flag for edit-point-set */
}

draw_box(x, y) /* sets reverse-video edit region */
int x, y;
{
        int xcorner;

        /* if no previous region selected, set start to hair loc */

	if((!box) || (rdrbox)) xend = xloc;

        if (x > xend){		/* increase box width */
		do_box(xend+1, x);
                xend = x; /* set current loc to edge of selected area */
        }
        else if (x < xend){		/* decrease box width */
                xcorner = (x > xloc) ? x : xloc;
		do_box(xcorner+1, xend);
                xend = (x > xloc) ? x : xloc; /* set currnt loc to ins. pt. */
        }
        box = 1;   /* set flag for region-selected */
        print_time(xend, y); /* print current time at edge of box */
	if(insert && !line) line = 1;
	if(hairmode | chmode) print_amps(xend, y);
}

undraw_box() /* erases reverse-video edit region */
{
	do_box(xloc+1, xend);
        box = region = 0;   /* unset flags for region-selected */
	set_view_display();		/* redisplay now */
	set_menus();
}

set_line_loc()  /* Determine the new position of hairline on screen */
{
        /* if insert point is now offscreen, turn off line redrawing */
        if(v->itime <= v->stime || v->itime >= v->etime) rdrline = 0;

        /* or turn on & relocate */

        else	rdrline = 1;   /* set flag to redraw it if necessary */
	xloc = screen_loc(v->itime);
}

set_box_loc()          /* Determine new position of edit box on screen */
{
	/* if both edges are off screen, turn off box drawing */

        if(c->edstart > v->etime || c->edend < v->stime) rdrbox = 0;
	else rdrbox = 1;

        /* set box coordinates according to times */ 

	xloc = screen_loc(c->edstart);
	xend = screen_loc(c->edend) - 1;
}

go_to_time(time)	/* shift screen to given time and set insert */
double time;
{
        int cv_half = v->width/2;
        
        /* set goto time to be center of screen if possible */

        v->egrain = (time / v->secgrain) + cv_half; 
	v->sgrain = v->egrain - v->width + 1;

        set_screen_times(1); /* checks for values less than first screenfull */
        
        /* if no current region selected, place hair at goto time */

        if(!region) {
                xloc = screen_loc(time);
                set_insert_point(xloc, yloc);
                rdrline = 1;
                line = amps = 0;
        }
	/* if region set, set endtime to goto time */

	else {
		double mintime = c->edstart + v->secgrain;
		c->edend = (time < mintime) ? mintime : time;
		set_box_loc();
		box = amps = 0;
	}
        set_view_display();
        display_it(1);
}

shift_insert(xinc, yinc)
int xinc, yinc;
{
	register int newx = xloc + xinc;
	register int newy = yloc + yinc;

	if(newx < 0 || newx > (v->width-1)) {	/* scroll if off screen */
		go_to_time(screen_time(newx));
		return;
	}
	draw_hair(newx, newy);
	set_insert_point(newx, newy);
}

shift_region(startinc, endinc)
int startinc, endinc;
{
	register int newstart = xloc + startinc;
	register int newend = xend + endinc;
	newend = (newend < 0) ? 0 : newend;
	if(startinc != 0) {		/* if edit start is shifting */
		/* invert current left edge */
		if(startinc > 0) do_line(newstart, yloc, 0); 
		else do_line(xloc, yloc, 0);
		draw_hair(newstart, yloc);	 /*  draw new edge */
		set_insert_point(newstart, yloc);
	}
	draw_box(newend, yloc);	/* move the right edge */
	set_region(newend, yloc);
}

do_line(x, y, horiz)
int x, y;
FLAG horiz;
{
	if(horiz)
	{
        	XDrawLine(dpy, canvas_win, edit_gc,
					0, y, v->width, y);
        	XDrawLine(dpy, plot_pm, edit_gc, 0, y, v->width, y);
	}
	else
	{
		XDrawLine(dpy, canvas_win, edit_gc,
			x, 0, x, v->height);
		XDrawLine(dpy, plot_pm, edit_gc, x, 0, x, v->height);
	}
}

do_box(x0, x1)		/* draws the inverse video box */
int x0, x1;
{
	unsigned width;
	x0 = (x0 < 0) ? 0 : x0;		/* no need to draw beyond screen */
	x1 = (x1 > v->width-1) ? v->width-1 : x1;

	width = x1 - x0 + 1; 
	XFillRectangle(dpy, canvas_win, edit_gc,
				x0, 0, width, v->height);
	XFillRectangle(dpy, plot_pm, edit_gc, 
				x0, 0, width, v->height);
}

print_time(x, y) /* prints current time value just to the right of the hair */
int x, y;
{
        static int len;
        static int tx;
        static char timestr[10];
	static double loctime;
	int margin = v->width - 150;

        /* draw over (erase) previous time label if present */

        if (line) {
                XDrawString(dpy, canvas_win, edit_gc, tx+5, 20, timestr, len);
                XDrawString(dpy, plot_pm, edit_gc, tx+5, 20, timestr, len);
        }
        /* get new time and print it.  if box drawn, get time of next grain */

        loctime = (v->sgrain + x + box) * v->secgrain;  
        sprintf(timestr, "%-10.5f", loctime);
        len = strlen(timestr);
        tx= (x < margin) ? x : margin;
        XDrawString(dpy, canvas_win, edit_gc, tx+5, 20, timestr, len);
        XDrawString(dpy, plot_pm, edit_gc, tx+5, 20, timestr, len);
}

print_amps(x, y) /* prints current max amp value just above horiz. hair */
int x, y;
{
        static int ax;
        static int len[2], point;
        int ch, vloc;
	int margin = v->width - 150;

        point = (x < v->width) ? ((x > 0) ? x : 0) : v->width - 1;
        if (amps && !chmode) {
              for(ch = 0; ch < v->nchans; ch++){
                vloc = v->bwidth * ch;
                XDrawString(dpy, canvas_win, edit_gc, ax+5, vloc+38,
                                                      ampstr[ch], len[ch]);
                XDrawString(dpy, plot_pm, edit_gc, ax+5, vloc+38,
                                                      ampstr[ch], len[ch]);
              }
        }
        ax= (x < margin) ? x : margin;
        for(ch = 0; ch < v->nchans; ch++){
		sprintf(ampstr[ch], "amp: %-11.5f", pkamp[ch][point]); 
		len[ch] = strlen(ampstr[ch]);
        	vloc = v->bwidth * ch;
        	XDrawString(dpy, canvas_win, edit_gc, ax+5, vloc+38,
                					ampstr[ch], len[ch]);
        	XDrawString(dpy, plot_pm, edit_gc, ax+5, vloc+38,
							ampstr[ch], len[ch]);
        }
        amps= 1;
}

draw_scale()    /* prints time scale under the displayed waveform */
{
        int grain, textloc, temp, base, len, toffset;
        double ntime, increment, prtime, sctime, trtime, p_inc, s_inc, t_inc;
        double timeinc, diff;
        char string[16], *pformat;

        timeinc = v->secgrain;            /* increment per grain */
        base = v->height;                 /* bottom of visible screen */
        textloc = base - txtoffset;        /* vert. loc of time chars */

        /* set time increment & format depending on current screen resolution */

        if(timeinc > 0.02){
                increment = 10.0;
                pformat = "%1.0f";
        }
        else if(timeinc > 0.0015 && timeinc <= 0.02){
                increment = 1.0;
                pformat = "%1.0f";
        }
        else if(timeinc > 0.00015 && timeinc <= 0.0015){
                increment = 0.1;
                pformat = "%5.1f";
        }
        else if(timeinc > 0.000015 && timeinc <= 0.00015){
                increment = 0.01;
                pformat = "%6.2f";
        }
        else if(timeinc > 0.0000015 && timeinc <= 0.000015){
                increment = 0.001;
                pformat = "%7.3f";
        }
        else{
                increment = 0.0001;
                pformat = "%8.4f";
        }
        p_inc = increment;
        s_inc = increment/2.0;
        t_inc = increment/10.0;

	diff = t_inc/2.0;

        temp = v->sgrain * v->secgrain; /* truncate to integer */
        prtime = (double) temp;  

	 /* set each to next highest val */

        while((prtime += p_inc) < v->stime); 
	 
        sctime = prtime - p_inc;
        while((sctime += s_inc) < v->stime);
        if(FABS(sctime-prtime) < t_inc) sctime += s_inc;

        trtime = prtime - p_inc;
        while((trtime += t_inc) < v->stime);
	if(trtime == prtime || trtime == sctime) trtime += t_inc;

        /* loop for width of current screen, check to see if next increment */
        /* is greater than primary, secondary, or tertiary hatch-times */
	 
        for(grain = 0; grain < v->width; grain++)
        {
		ntime = (v->sgrain + grain) * timeinc;
                if((ntime+timeinc) > prtime)
                {
                        XDrawLine(dpy, plot_pm, hscale_gc, grain, base,
                                        grain, base-p_len);
                        sprintf(string, pformat, prtime);
			len = strlen(string);
			toffset = XTextWidth(scale_fs, string, len)/2;
                        XDrawString(dpy, plot_pm, hscale_gc, grain-toffset, 
                                    textloc, string, len);
			trtime = prtime + t_inc;
                        prtime += p_inc;
                        continue;
                }
                if(ntime >= sctime)
                {
                        XDrawLine(dpy, plot_pm, hscale_gc, grain, base,
                                        grain, base-s_len);
			trtime = sctime + t_inc;
                        sctime += p_inc;
                        continue;
                }
                if(ntime >= trtime)
                {
                        XDrawLine(dpy, plot_pm, hscale_gc, grain, base,
                                        grain, base-t_len);
                        trtime += t_inc;
			/* temporary fix for shifty doubles */
			if((sctime-trtime) < diff) sctime = trtime;
			if((prtime-trtime) < diff) prtime = trtime;
                }
        }
}

draw_amp_scale(peak)  /* prints amp scale along the left edge of screen */
double peak;
{
        int height;
        double increment, nextamp, maxpow = .000001;
	double p_inc, s_inc, t_inc, prval, scval, trval;
	int prloc, scloc, trloc, count; 
        char string[16], *pformat;
	int n_pticks, pr_maxloc, posloc, negloc;
	double pr_maxamp;
	FLAG trflag = 1;

        height = v->bwidth/2;		/* from base to top edge    */

        /* set amp string format depending on current screen resolution */

        if(peak > 1.0) pformat = "%+3.1lg";
	else pformat = "%+.3lg";

	/* find max power of ten less than peak amp */

	while((maxpow *= 10.0) <= peak);
	maxpow /= 10.0;

	n_pticks = (int) (peak/maxpow);		/* the number of prime ticks */
	if(n_pticks*v->nchans >= 5) trflag = 0; 
	pr_maxamp = n_pticks * maxpow;		/* the max prime amp value   */
	pr_maxloc = (height-vmargin) * (pr_maxamp/peak); /* vert loc of same */
	increment = (double) pr_maxloc/n_pticks;

        p_inc = increment;
        s_inc = increment/2.0;
        t_inc = increment/10.0;
	prloc = ROUNDUP(prval = p_inc);
	scloc = ROUNDUP(scval = s_inc);
	trloc = ROUNDUP(trval = t_inc);

	/* clear all pixels in this new pixmap */

	XSetForeground(dpy, vscale_gc, myfg);
	XSetFunction(dpy, vscale_gc, myfg?GXclear:GXset);
	XFillRectangle(dpy, vscale_pm, vscale_gc,
		       0, 0, VSPM_WIDTH, (int) v->bwidth);
	XSetFunction(dpy, vscale_gc, myfg?GXxor:GXequiv); 

	/* draw prime tick mark at 0 */

	XDrawLine(dpy, vscale_pm, vscale_gc, 0, height, p_len, height);

        /* loop for height of current band, check to see if next increment */
        /* is greater than primary, secondary, or tertiary hatch-locs */
	 
	nextamp = maxpow;
	count = 0;
        for(posloc=height, negloc=height; posloc >= vmargin; posloc--, negloc++)
        {
		count++;
		if((prloc-trloc) == 1) prloc -= 1; /* correction */
                if(count == prloc)
                { 
			int len;
			int voffset = 4;
                        XDrawLine(dpy, vscale_pm, vscale_gc, 0, posloc, 
							p_len, posloc);
                        sprintf(string, pformat, nextamp);
			len = strlen(string);
                        XDrawString(dpy, vscale_pm, vscale_gc, txtoffset, 
                                    posloc+voffset, string, len);
                        XDrawLine(dpy, vscale_pm, vscale_gc, 0, negloc, 
							p_len, negloc);
                        sprintf(string, pformat, -1.0*nextamp);
			len = strlen(string);
                        XDrawString(dpy, vscale_pm, vscale_gc, txtoffset, 
                                    negloc+voffset, string, len);
			trloc = ROUNDUP(trval = prval+t_inc);
			prloc = ROUNDUP(prval += p_inc);
			nextamp += maxpow;
                        continue;
                }
                if(count == scloc)
                {
                        XDrawLine(dpy, vscale_pm, vscale_gc, 0, posloc,
                                        s_len, posloc);
                        XDrawLine(dpy, vscale_pm, vscale_gc, 0, negloc,
                                        s_len, negloc);
			trloc = ROUNDUP(trval = scval+t_inc);
			scloc = ROUNDUP(scval += p_inc);
                        continue;
                }
		if(!trflag) continue;	/* no small ticks */
                if(count == trloc)
                {
                        XDrawLine(dpy, vscale_pm, vscale_gc, 0, posloc,
                                        t_len, posloc);
                        XDrawLine(dpy, vscale_pm, vscale_gc, 0, negloc,
                                        t_len, negloc);
			trloc = ROUNDUP(trval+=t_inc);
                }
        }
}

double
screen_time(loc)	/* returns time value for given horiz location */
int loc;
{
	int grain = v->sgrain + loc;
	grain = (grain < 0) ? 0 : grain;
	return(grain * v->secgrain);
}

int 
screen_loc(time)	/* returns horiz. loc for given time */
double time;
{
	double val = time/v->secgrain;
	return (ROUND(val) - v->sgrain);
}
