/*
	databuff.c
*/

#include "main.h"
#include "dialog.h"
#include "mesg.h"
#include "databuff.h"
#include "screen.h"
#include "lp.h"
#include "fft.h"
#include "nobug.h"

Data_buff *
new_databuff()
{
	Data_buff *db;
	db = (Data_buff *) mv_alloc(sizeof(Data_buff));
/* functions */
	db->shiftData = db_shiftData;
	db->changeDataSize = db_changeDataSize;
	db->doubleValue = db_doubleValue;
	db->minValue = db_minValue;
	db->maxValue = db_maxValue;
	db->openFile = db_openFile;
	db->readFile = db_readFile;
	db->readHeader = db_readHeader;
	db->writeFile = db_writeFile;
	db->scanForPeak = db_scanForPeak;
	db->nChannels = db_nChannels;
	db->dataSize = db_dataSize;
	db->nElements = db_nElements;
	db->loadData = db_loadData;
	db->setDataFormat = db_setDataFormat;
	db->copyData = db_copyData;
	db->insertData = db_insertData;
	db->destroy = db_destroy;
	/* defaults */
	db->fd = -1;
	db->datasize = 0;
	db->nchans = 1;
	db->size = db->framesize = sizeof(float);
	return db;
}

/* private */

void
db_shiftData(b, start, nelements)
	Data_buff *b; 
	int start, nelements;
{
}

int db_changeDataSize(b, newsize)
	Data_buff *b; 
	int newsize;
{
	char *temp = b->data;
	if(b->data) temp = (char *) realloc(temp, (unsigned) newsize);
	else temp = (char *) mv_alloc(newsize);
	if(temp == CNULL) {
		mv_error(errno, "Cannot allocate data buffer.");
		return -1;
	}
	b->data = temp;
	b->datasize = newsize;
	return newsize;
}

double
db_scanForPeak(b, ch0, chn, start, end)
	Data_buff *b; 
	int ch0, chn, start, end;
{
	register int i, j;
	int nelements;
	double val;
	double *maxptr, *minptr;
	if(ch0 < 0 || chn < ch0 || ch0 >= b->nchans || chn >= b->nchans) {
		mv_alert("db_scanForPeak: invalid channels for this data.");
		ch0 = 0;
		chn = b->nchans - 1;
	}
	if((nelements = b->nElements(b, 0)) == 0) /* if zero data length */
		return 1.0;
	if(start > nelements || end > nelements) {
		mv_alert("db_scanForPeak: scan runs beyond EOF.");
		return 1.0;
	}
	for(i=ch0; i<=chn; i++) {
		maxptr = b->maxvalue+i;
		minptr = b->minvalue+i;
		*maxptr = -1.0e+60;
		*minptr = 1.0e+60;
		for(j=start; j <= end; j++) {
			val = b->doubleValue(b, i, j);
			if(val > *maxptr) *maxptr = val;
			if(val < *minptr) *minptr = val;
		}
	}
	if(FABS(*maxptr) > FABS(*minptr)) return FABS(*maxptr);
	else return FABS(*minptr);
}

double
db_doubleValue(b, chan, loc)
	Data_buff *b; 
	int chan;
	long loc;
{
	int nchans = b->nchans;
	if(chan > nchans) return 0.0;
	switch(b->size) {
	case 1:
		return *((char *)	(b->data + (1*((loc*nchans)+chan))));
	case 2:
		return *((short *)	(b->data + (2*((loc*nchans)+chan))));
	case 4:
		return *((float *)	(b->data + (4*((loc*nchans)+chan))));
	case 8:
		return *((double *)	(b->data + (8*((loc*nchans)+chan))));
	default:
		mv_die(0, "db_doubleValue: unknown data type.");
		return (double) 0;
	}
}

double db_minValue(b, chan)
	Data_buff *b; 
	int chan;
{
	return b->minvalue[chan];
}

double db_maxValue(b, chan)
	Data_buff *b; 
	int chan;
{
	return b->maxvalue[chan];
}

/* public */

void
db_destroy(b)
	Data_buff *b;
{
	if(b->dataSize(b)) cfree(b->data);
	cfree((char *) b);
	b = (Data_buff *) NULL;
}

unsigned
db_dataSize(b)	/* returns size of data buffer in bytes */
	Data_buff *b;
{
	return b->datasize;	
}

int db_nElements(b, chan)
	Data_buff *b;
	int chan;
{
	return (int) (b->datasize/b->framesize);
}

int 
db_openFile(b, filename)
	Data_buff *b;
	char *filename;
{
	struct stat st;
	unsigned filesize;
	int headersize = 0;
	if((b->fd = open(filename, O_RDWR)) < 0) {
		int code = errno;
		char string[120];
		sprintf(string, "Cannot open data file %s.", filename);
		mv_error(code, string);
		return -1;
	}
	else if(stat(filename, &st)) {
		mv_error(errno, "Cannot stat data file.");
		return -1;
	}
	else if((headersize = b->readHeader(b)) < 0)
		return -1;
	filesize = st.st_size - headersize; /* size on disk not incl. header */
	return (int) filesize;
}

int
db_readHeader(b)	/* for now this reads LPC and FFT datafile headers */
	Data_buff *b;
{
	int headdata[2], headsize = 0;
	if(read(b->fd, (char *) headdata, sizeof(headdata)) < 0) {
		mv_error(errno, "Cannot read data file.");
		return -1;
	}
	/* if header present, return size from header data */
	headsize = (headdata[1] == LP_MAGIC) ? headdata[0] 
		 : (headdata[0] == FFT_MAGIC) ? headdata[1] : 0;
	if(lseek(b->fd, (long) headsize, L_SET) < 0) {
		mv_error(errno, "Bad seek on data file.");
		return -1;
	}
	return headsize;
}

int
db_readFile(b, filename)
	Data_buff *b;
	char *filename;
{
	int size;
	if((size = b->openFile(b, filename)) < 0) return -1;
	if(!size) {
		mv_alert("db_readFile:  Empty data file.");
		close(b->fd);
		return -1;
	}
	if(b->dataSize(b) < size) {
		if(b->changeDataSize(b, size) < 0) {
			close(b->fd);
			return -1;
		}
	}
	if(read(b->fd, b->data, size) != size) {
		mv_error(errno, "db_readFile:  Error reading data file.");
		close(b->fd);
		return -1;
	}
	return 1;
}

int
db_writeFile(b)
	Data_buff *b;
{
	if(write(b->fd, b->data, (int) b->datasize) != b->datasize) {
		mv_error(errno, "db_writeFile:  unable to write to disk.");
		return -1;
	}
	return 1;
}

void
db_setDataFormat(b, nchans, size)
	Data_buff *b; 
	int nchans, size;
{
	b->nchans = nchans;
	b->size = size;
	b->framesize = size * nchans;
}

int
db_nChannels(b)
	Data_buff *b;
{
	return b->nchans;
}

Data_buff *
db_copyData(b, start, length)
	Data_buff *b; 
	int start, length;
{
	Data_buff *c = new_databuff();
	int size = length * b->framesize;
	if(start > b->nElements(b)) {
		mv_alert("db_copyData:  start is beyond end of data.");
		return (Data_buff *) NULL;
	}
	if(length > (b->nElements(b) - start)) {
		mv_alert("db_copyData:  copy goes beyond end of data.");
		return (Data_buff *) NULL;
	}
	c->setDataFormat(c, b->nchans, b->size);
	if(c->changeDataSize(c, size) < 0)
		return (Data_buff *) NULL;
	/* this is temporary until routines written */
	bcopy(b->data + (start * b->framesize), c->data, size);
	return c;
}

Data_buff *
db_insertData(b, newdata)
	Data_buff *b, *newdata; 
{
	return b;
}

int
db_loadData(b, screen)
	Data_buff *b;
	Mv_screen *screen;
{
	double value, minval, maxval;
	register int i, j, k;
	int scrnwidth, length, chanfirst;
	int xloc, xskip=1, chans, chanlast;
	Graph_points *gp = screen->graphpoints;
	Graph_adder *ga;
	DBCOM(db_loadData);
	chans = b->nChannels(b);	/* total number of channels of data */
	/* assuming the same no. of elements in each channel */
	if(!(length = b->nElements(b, 0))) { /* if no file open */
		b->changeDataSize(b, 16 * chans);
		length = b->nElements(b, 0);
	}
	/* these were set by user */
	screen->getScanChannels(screen, &chanfirst, &chanlast);
	DBPRT(length);
	chans = chanlast-chanfirst+1;
	/* scan for max and min values */
	b->scanForPeak(b, chanfirst, chanlast, 0, length-1);
	screen->setChannels(screen, chans);	/* setting display */
	screen->setVertMargine(screen, 15);	/* my default for now */
	/* entire file */
	if(screen->setHorizRange(screen, 0, length) < 0) return -1; 
	scrnwidth = screen->screenWidth(screen);
	for(i=0, j=chanfirst; j <= chanlast; i++, j++) {
		minval = b->minValue(b,i);
		maxval = b->maxValue(b,i);
		screen->setVertScale(screen, i, minval, maxval);
		ga = set_adder(gp, i, 0);
		for(k=0, xloc=0; k<length && xloc<scrnwidth; k++, xloc+=xskip) {
			/* retrieve value from data object */
			value = b->doubleValue(b, j, k);
			/* and add to graph list */
			ga->add(ga, xloc, value);
		}
		ga->destroy(ga);
	}
	screen->is_data = True;
	return 1;
}
