#include <stdio.h>
#include <ctype.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include "lxt.h"

extern boolean xp_fullredraw;
extern char xpkey_null, xpkey_tab, xpkey_cret, xpkey_backspace;
extern char xpkey_wordkill, xpkey_linekill;

extern Void *lxt_selsrc;
extern int lxt_selsrctype;
extern Time lxt_selsrctm;
extern char *lxt_selsrcbuf;
extern Panel *lxt_selsrcpanel;
extern Void *lxt_seldest;
extern int lxt_seldesttype;

static int xptext_initselpos;

int
xptext_idef_proc(p, pi)
/*
   Internal function.
   Initialize defaults for a text item.
*/
Panel *p;
Panel_item *pi;
{
	pi->xpi_item.xpi_text.xptext_max= LXPDEF_TEXTSTORE;
	pi->xpi_item.xpi_text.xptext_disp= LXPDEF_TEXTDISPLAY;
	if ((pi->xpi_item.xpi_text.xptext_val= calloc(LXPDEF_TEXTSTORE+1, sizeof(char))) == (char *) NULL) {
		(void) fprintf(stderr, "xptext_idef_proc: memory allocation error\n");
		return(LX_ERROR);
	}
	(void) strcpy(pi->xpi_item.xpi_text.xptext_val, "");
	pi->xpi_item.xpi_text.xptext_dpos= 0;
	pi->xpi_item.xpi_text.xptext_ipos= 0;
	pi->xpi_item.xpi_text.xptext_gap= LXPDEF_TEXTGAP;
	pi->xpi_item.xpi_text.xptext_kpproc= (int (*)()) NULL;
	pi->xpi_item.xpi_text.xptext_flags= 0;
	return(LX_SUCCESS);
}

int
xptext_dst_proc(pi)
/*
   Internal function.
   Destroys a text item.
*/
Panel_item *pi;
{
	void lxt_clearsel();

	if (pi == (Panel_item *) lxt_selsrc)
		lxt_clearsel(FALSE);

	if ((Panel_item *) lxt_seldest == pi) {
		lxt_seldest= (Void *) NULL;
		lxt_seldesttype= LX_NULL;
	}

	if (pi->xpi_item.xpi_text.xptext_val != (char *) NULL)
		cfree(pi->xpi_item.xpi_text.xptext_val);
	return(LX_SUCCESS);
}

int
xptext_sz_proc(p, pi)
/*
   Internal function.
   Sizes a text item.
*/
Panel *p;
Panel_item *pi;
{
	int tval_ht;

	if (p == (Panel *) NULL) {
		(void) fprintf(stderr, "xptext_sz_proc: null panel\n");
		return(LX_ERROR);
	}
	if (pi == (Panel_item *) NULL) {
		(void) fprintf(stderr, "xptext_sz_proc: null item\n");
		return(LX_ERROR);
	}
	pi->xpi_w= pi->xpi_h= 0;
	if (LX_UPWEDGEHT+1 > pi->xpi_font->max_bounds.descent)
		tval_ht= pi->xpi_font->max_bounds.ascent+LX_UPWEDGEHT+1;
	else
		tval_ht= pi->xpi_font->max_bounds.ascent+pi->xpi_font->max_bounds.descent;
	if (pi->xpi_image != (XImage *) NULL) {
		pi->xpi_w= pi->xpi_image->width;
		pi->xpi_h= pi->xpi_image->height;
	}
	else if (pi->xpi_str != (char *) NULL) {
		pi->xpi_w= XTextWidth(pi->xpi_font, pi->xpi_str, strlen(pi->xpi_str));
		pi->xpi_h= tval_ht;
	}
	else {
		pi->xpi_w= 0-pi->xpi_item.xpi_text.xptext_gap;
		pi->xpi_h= tval_ht;
	}
	pi->xpi_w+= pi->xpi_item.xpi_text.xptext_gap+(pi->xpi_item.xpi_text.xptext_disp+1)*XTextWidth(pi->xpi_font, "X", 1)+LX_UPWEDGEWD;
	if (pi->xpi_image != (XImage *) NULL) {
		if (tval_ht > pi->xpi_image->height)
			pi->xpi_h= tval_ht;
	}
	return(LX_SUCCESS);
}

void
xptext_drw_proc(p, pi)
/*
   Internal function.
   Plots a text item onto the backing store.
*/
Panel *p;
Panel_item *pi;
{
	xp_text *xpt;
	int x, y, i, charwd;
	int xs, w, h;
	char *d;
	void xptext_drw_leftwedge();
	void xptext_drw_upwedge();

	if (!xp_fullredraw)
		XFillRectangle(p->xp_dpy, p->xp_ppm, pi->xpi_cgc, pi->xpi_x, pi->xpi_y, pi->xpi_w, pi->xpi_h);
	if (pi->xpi_state == LXPI_INACTIVE)
		return;

	x= pi->xpi_x;
	y= pi->xpi_y+pi->xpi_font->max_bounds.ascent;
	xpt= (xp_text *) &(pi->xpi_item.xpi_text);

	/* attempt to draw xpi_image */
	if (pi->xpi_image != (XImage *) NULL) {
		XPutImage(p->xp_dpy, p->xp_ppm, pi->xpi_gc, pi->xpi_image, 0, 0, pi->xpi_x, pi->xpi_y, pi->xpi_image->width, pi->xpi_image->height);
		x= pi->xpi_x+pi->xpi_image->width+xpt->xptext_gap;
	}

	/* draw text label if no image present */
	else if (pi->xpi_str != (char *) NULL) {
		if (strlen(pi->xpi_str) != 0) {
			XDrawString(p->xp_dpy, p->xp_ppm, pi->xpi_gc, pi->xpi_x, y, pi->xpi_str, strlen(pi->xpi_str));
			x= pi->xpi_x+XTextWidth(pi->xpi_font, pi->xpi_str, strlen(pi->xpi_str))+xpt->xptext_gap;
		}
	}

	/* draw text value */
	if (xpt->xptext_val != (char *) NULL) {
		if (strlen(xpt->xptext_val) > 0) {
			d= xpt->xptext_val;
			for (i= 0; i < xpt->xptext_dpos; i++, d++);

			/* text scrolled out of view? */
			if (d != xpt->xptext_val) {
				xptext_drw_leftwedge(p, pi, x, y-4);
				x+= 6;
			}

			XDrawString(p->xp_dpy, p->xp_ppm, pi->xpi_gc, x, y, d, strlen(d));

			xs= x;
			charwd= XTextWidth(pi->xpi_font, "X", strlen("X"));

			/* show insertion point */
			if (p->xp_seltext == pi) {
				x+= (xpt->xptext_ipos-xpt->xptext_dpos)*charwd;
				xptext_drw_upwedge(p, pi, x, y+1);
			}

			/* highlight selection, if any */
			if ((xpt->xptext_selstop > xpt->xptext_selstart) &&
			    (xpt->xptext_flags & LXPTEXT_SELHOLDER)) {
				xs+= (xpt->xptext_selstart-xpt->xptext_dpos)*charwd;
				w= (xpt->xptext_selstop-xpt->xptext_selstart)*charwd;
				h= pi->xpi_font->max_bounds.ascent;
				y-= h;
				if (LX_UPWEDGEHT > pi->xpi_font->max_bounds.descent)
					h+= LX_UPWEDGEHT;
				else
					h+= pi->xpi_font->max_bounds.descent;
				XFillRectangle(p->xp_dpy, p->xp_ppm, pi->xpi_igc, xs, y, w, h);
			}
		}

		/* show insertion point */
		else if (p->xp_seltext == pi)
			xptext_drw_upwedge(p, pi, x, y+1);
	}

	/* show insertion point */
	else if (p->xp_seltext == pi)
		xptext_drw_upwedge(p, pi, x, y+1);

	return;
}

void
xptext_drw_leftwedge(p, pi, x, y)
/*
   Internal function.
   Draw leftward-pointing wedge indicating text out of view to left.
*/
Panel *p;
Panel_item *pi;
int x, y;
{
	XSegment xs[4];

	xs[0].x1= x;
	xs[0].y1= y;
	xs[0].x2= x;
	xs[0].y2= y;
	xs[1].x1= x+1;
	xs[1].y1= y-1;
	xs[1].x2= x+1;
	xs[1].y2= y+1;
	xs[2].x1= x+2;
	xs[2].y1= y-2;
	xs[2].x2= x+2;
	xs[2].y2= y+2;
	xs[3].x1= x+3;
	xs[3].y1= y-3;
	xs[3].x2= x+3;
	xs[3].y2= y+3;
	XDrawSegments(p->xp_dpy, p->xp_ppm, pi->xpi_gc, xs, 4);

	return;
}
void
xptext_drw_upwedge(p, pi, x, y)
/*
   Internal function.
   Draw upward-pointing wedge indicating insertion point.
*/
Panel *p;
Panel_item *pi;
int x, y;
{
	XSegment xs[4];

	xs[0].x1= x;
	xs[0].y1= y;
	xs[0].x2= x;
	xs[0].y2= y;
	xs[1].x1= x-1;
	xs[1].y1= y+1;
	xs[1].x2= x+1;
	xs[1].y2= y+1;
	xs[2].x1= x-2;
	xs[2].y1= y+2;
	xs[2].x2= x+2;
	xs[2].y2= y+2;
	xs[3].x1= x-3;
	xs[3].y1= y+3;
	xs[3].x2= x+3;
	xs[3].y2= y+3;
	XDrawSegments(p->xp_dpy, p->xp_ppm, pi->xpi_gc, xs, 4);

	return;
}

void
xptext_bp_proc(p, pi, evt)
/*
   Internal function.
   Button press on text item.
*/
Panel *p;
Panel_item *pi;
XButtonPressedEvent *evt;
{
	Panel_item *xpi;
	xp_text *xpt;
	Atom prop;
	void xptext_setipos();
	void xptext_drw_proc(), xpi_std_dsp_proc();

	p->xp_selitem= pi;

	switch (evt->button) {
	case Button1:
		if (p->xp_seltext != (Panel_item *) NULL) {
			if (p->xp_seltext != pi) {
				xpi= p->xp_seltext;
				p->xp_seltext= (Panel_item *) NULL;
				xptext_drw_proc(p, xpi);
				xpi_std_dsp_proc(p, xpi);
			}
		}
		p->xp_seltext= pi;
		xptext_setipos(p, pi, evt->x);
		xpt= &(pi->xpi_item.xpi_text);
		xpt->xptext_selstart= xpt->xptext_selstop= xpt->xptext_ipos;
		xptext_initselpos= xpt->xptext_ipos;
		xpt->xptext_flags&= ~LXPTEXT_SELHOLDER;

		/* if the insertion point is altered, don't allow
		   any pending selection transfer into a text item
		   contained in this panel */
		if (lxt_seldesttype == LX_PANELITEM) {
			for (xpi= p->xp_items; xpi != (Panel_item *) NULL; xpi= xpi->xpi_next) {
				if (xpi == (Panel_item *) lxt_seldest) {
					lxt_seldest= (Void *) NULL;
					lxt_seldesttype= LX_NULL;
					break;
				}
			}
		}

		xptext_drw_proc(p, pi);
		xpi_std_dsp_proc(p, pi);
		break;
	case Button2:
		if (p->xp_seltext != (Panel_item *) NULL) {
			if (p->xp_seltext != pi) {
				xpi= p->xp_seltext;
				p->xp_seltext= (Panel_item *) NULL;
				xptext_drw_proc(p, xpi);
				xpi_std_dsp_proc(p, xpi);
			}
		}
		p->xp_seltext= pi;
		xptext_setipos(p, pi, evt->x);
		xptext_drw_proc(p, pi);
		xpi_std_dsp_proc(p, pi);

		if (!(pi->xpi_item.xpi_text.xptext_flags & LXPTEXT_WRPROTECT)) {
			lxt_seldest= (Void *) pi;
			lxt_seldesttype= LX_PANELITEM;
			prop= XInternAtom(p->xp_dpy, LX_SELATOM, False);
			XConvertSelection(p->xp_dpy, XA_PRIMARY, XA_STRING, prop, p->xp_pwin, evt->time);
		}
		break;
	default:
		break;
	}
	return;
}

void
xptext_br_proc(p, pi, evt)
/*
   Internal function.
   Button release on text item.
*/
Panel *p;
Panel_item *pi;
XButtonReleasedEvent *evt;
{
	xp_text *xpt;
	int len, i, j;
	void xptext_drw_proc(), xpi_std_dsp_proc();

	if (evt->button != Button1) {
		p->xp_selitem= (Panel_item *) NULL;
		return;
	}
	if (p->xp_selitem == (Panel_item *) NULL)
		return;

	xpt= &(p->xp_selitem->xpi_item.xpi_text);
	if (xpt->xptext_selstart == xpt->xptext_selstop) {
		p->xp_selitem= (Panel_item *) NULL;
		return;
	}

	XSetSelectionOwner(p->xp_dpy, XA_PRIMARY, p->xp_pwin, evt->time);
	if (XGetSelectionOwner(p->xp_dpy, XA_PRIMARY) != p->xp_pwin) {
		xpt->xptext_selstart= xpt->xptext_selstop= xpt->xptext_ipos;
		xpt->xptext_flags&= ~LXPTEXT_SELHOLDER;
		xptext_drw_proc(p, p->xp_selitem);
		xpi_std_dsp_proc(p, p->xp_selitem);
		p->xp_selitem= (Panel_item *) NULL;
		return;
	}

	len= xpt->xptext_selstop-xpt->xptext_selstart;
	if ((lxt_selsrcbuf= calloc((unsigned) (len+1), sizeof(char))) == (char *) NULL) {
		(void) fprintf(stderr, "xptext_br_proc: selection memory allocation error\n");
		xpt->xptext_selstart= xpt->xptext_selstop= xpt->xptext_ipos;
		xpt->xptext_flags&= ~LXPTEXT_SELHOLDER;
		xptext_drw_proc(p, p->xp_selitem);
		xpi_std_dsp_proc(p, p->xp_selitem);
		p->xp_selitem= (Panel_item *) NULL;
		XSetSelectionOwner(p->xp_dpy, XA_PRIMARY, None, evt->time);
		return;
	}
	for (i= xpt->xptext_selstart, j= 0; i < xpt->xptext_selstop; i++, j++)
		lxt_selsrcbuf[j]= xpt->xptext_val[i];
	lxt_selsrcbuf[j]= '\0';

	lxt_selsrc= (Void *) p->xp_selitem;
	lxt_selsrctype= LX_PANELITEM;
	lxt_selsrctm= evt->time;
	lxt_selsrcpanel= p;

	p->xp_selitem= (Panel_item *) NULL;
	return;
}

void
xptext_pm_proc(p, pi, evt)
/*
   Internal function.
   Process pointer motion while a text item is selected.
*/
Panel *p;
Panel_item *pi;
XPointerMovedEvent *evt;
{
	xp_text *xpt;
	int osstart, osstop;
	void xptext_setspos();
	void lxt_clearsel();
	void xptext_drw_proc(), xpi_std_dsp_proc();

	if (!(evt->state & Button1Mask))
		return;
	if (p->xp_selitem == (Panel_item *) NULL)
		return;

	xpt= &(p->xp_selitem->xpi_item.xpi_text);
	osstart= xpt->xptext_selstart;
	osstop= xpt->xptext_selstop;
	xptext_setspos(p, p->xp_selitem, evt->x);
	if ((xpt->xptext_selstart == osstart) &&
	    (xpt->xptext_selstop == osstop))
		return;

	if (xpt->xptext_selstart != xpt->xptext_selstop) {
		if (lxt_selsrc != (Void *) NULL)
			lxt_clearsel(TRUE);
		xpt->xptext_flags|= LXPTEXT_SELHOLDER;
	}
	else
		xpt->xptext_flags&= ~LXPTEXT_SELHOLDER;

	xptext_drw_proc(p, p->xp_selitem);
	xpi_std_dsp_proc(p, p->xp_selitem);

	return;
}

void
xptext_kp_proc(p, pi, evt)
/*
   Internal function.
   Key press on text item.
*/
Panel *p;
Panel_item *pi;
XKeyEvent *evt;
{
	xp_text *xpt;
	char *c, buf[80];
	int code, i, count;
	void xptext_cycleseltext();
	void lxt_clearsel();
	void xptext_deletechar();
	void xptext_drw_proc(), xpi_std_dsp_proc();

	if (evt->state & (Button1Mask | Button2Mask | Button3Mask))
		return;

	xpt= (xp_text *) &(pi->xpi_item.xpi_text);
	bzero(buf, 80);
	count= XLookupString(evt, buf, 80, (KeySym *) NULL, (XComposeStatus *) NULL);
	buf[count]= '\0';

	/* call user function if declared */
	if (pi->xpi_item.xpi_text.xptext_kpproc != (int (*)()) NULL)
		if (!(pi->xpi_item.xpi_text.xptext_kpproc(p, pi, buf)))
			return;

	for (c= buf; (isascii(*c) && (*c != '\0')); c++) {
		code= (int) *c;
		if (code == xpkey_null)
			continue;

		/* activate next text item */
		else if ((code == xpkey_cret) || (code == xpkey_tab))
			xptext_cycleseltext(p);

		/* backspace over last character */
		else if (code == xpkey_backspace) {
			if (xpt->xptext_flags & LXPTEXT_WRPROTECT)
				continue;
			if (xpt->xptext_ipos > 0)
				xptext_deletechar(xpt);
			if (pi == (Panel_item *) lxt_selsrc)
				lxt_clearsel(FALSE);
			xptext_drw_proc(p, pi);
			xpi_std_dsp_proc(p, pi);
		}

		/* word kill */
		else if (code == xpkey_wordkill) {
			if (xpt->xptext_flags & LXPTEXT_WRPROTECT)
				continue;

			/* delete all contiguous preceding white space */
			while (xpt->xptext_ipos > 0) {
				if (isspace(xpt->xptext_val[xpt->xptext_ipos-1]))
					xptext_deletechar(xpt);
				else
					break;
			}

			/* delete preceding word */
			while (xpt->xptext_ipos > 0) {
				if (isgraph(xpt->xptext_val[xpt->xptext_ipos-1]))
					xptext_deletechar(xpt);
				else
					break;
			}
			if (pi == (Panel_item *) lxt_selsrc)
				lxt_clearsel(FALSE);
			xptext_drw_proc(p, pi);
			xpi_std_dsp_proc(p, pi);
		}

		/* line kill */
		else if (code == xpkey_linekill) {
			if (xpt->xptext_flags & LXPTEXT_WRPROTECT)
				continue;
			while (xpt->xptext_ipos > 0)
				xptext_deletechar(xpt);
			if (pi == (Panel_item *) lxt_selsrc)
				lxt_clearsel(FALSE);
			xptext_drw_proc(p, pi);
			xpi_std_dsp_proc(p, pi);
		}

		/* all other characters */
		else if (isprint(*c)) {
			if (xpt->xptext_flags & LXPTEXT_WRPROTECT)
				continue;
			if (strlen(xpt->xptext_val) < xpt->xptext_max) {
				if (xpt->xptext_ipos == strlen(xpt->xptext_val)) {
					xpt->xptext_val[xpt->xptext_ipos++]= *c;
					xpt->xptext_val[xpt->xptext_ipos]= '\0';
				}
				else {
					for (i= strlen(xpt->xptext_val); i >= xpt->xptext_ipos; i--)
						xpt->xptext_val[i+1]= xpt->xptext_val[i];
					xpt->xptext_val[xpt->xptext_ipos]= *c;
					xpt->xptext_ipos++;
				}
				if (strlen(xpt->xptext_val) > xpt->xptext_disp)
					xpt->xptext_dpos++;
			}
			if (pi == (Panel_item *) lxt_selsrc)
				lxt_clearsel(FALSE);
			xptext_drw_proc(p, pi);
			xpi_std_dsp_proc(p, pi);
		}
	}
	return;
}

void
xptext_setipos(p, pi, px)
/*
   Internal function.
   Sets the insertion point of a text item.
*/
Panel *p;
Panel_item *pi;
int px;
{
	int x;
	xp_text *xpt;

	xpt= (xp_text *) &(pi->xpi_item.xpi_text);
	x= px+p->xp_vx-pi->xpi_x;
	if (pi->xpi_image != (XImage *) NULL)
		x-= pi->xpi_image->width+xpt->xptext_gap;
	else if (pi->xpi_str != (char *) NULL) {
		if (strlen(pi->xpi_str) != 0)
			x-= XTextWidth(pi->xpi_font, pi->xpi_str, strlen(pi->xpi_str))+xpt->xptext_gap;
	}
	if (xpt->xptext_dpos > 0)
		x-= 6;

	if (x < 0)
		xpt->xptext_ipos= xpt->xptext_dpos;
	else
		xpt->xptext_ipos= xpt->xptext_dpos+(x/XTextWidth(pi->xpi_font, "X", strlen("X")));
	if (xpt->xptext_ipos > strlen(xpt->xptext_val))
		xpt->xptext_ipos= strlen(xpt->xptext_val);
	return;
}

void
xptext_setspos(p, pi, px)
/*
   Internal function.
   Sets the selection endpoints of a text item.
*/
Panel *p;
Panel_item *pi;
int px;
{
	int x, i;
	xp_text *xpt;

	xpt= (xp_text *) &(pi->xpi_item.xpi_text);
	x= px+p->xp_vx-pi->xpi_x;
	if (pi->xpi_image != (XImage *) NULL)
		x-= pi->xpi_image->width+xpt->xptext_gap;
	else if (pi->xpi_str != (char *) NULL) {
		if (strlen(pi->xpi_str) != 0)
			x-= XTextWidth(pi->xpi_font, pi->xpi_str, strlen(pi->xpi_str))+xpt->xptext_gap;
	}
	if (xpt->xptext_dpos > 0)
		x-= 6;

	if (x < 0)
		i= xpt->xptext_dpos;
	else
		i= xpt->xptext_dpos+(x/XTextWidth(pi->xpi_font, "X", strlen("X")));
	if (i > strlen(xpt->xptext_val))
		i= strlen(xpt->xptext_val);

	if (i < xptext_initselpos) {
		xpt->xptext_selstart= i;
		xpt->xptext_selstop= xptext_initselpos;
	}
	else if (i == xptext_initselpos)
		xpt->xptext_selstart= xpt->xptext_selstop= xptext_initselpos;
	else if (i > xptext_initselpos) {
		xpt->xptext_selstart= xptext_initselpos;
		xpt->xptext_selstop= i;
	}

	return;
}

void
xptext_deletechar(xpt)
/*
   Internal function.
   Delete character to the left of the insertion point.
*/
xp_text *xpt;
{
	int i;

	if (xpt->xptext_dpos > 0)
		xpt->xptext_dpos--;
	xpt->xptext_ipos--;
	for (i= xpt->xptext_ipos; xpt->xptext_val[i] != '\0'; i++)
		xpt->xptext_val[i]= xpt->xptext_val[i+1];
}

void
xptext_cycleseltext(p)
/*
   Internal function.
   Activate the next text item.
*/
Panel *p;
{
	Panel_item *pi, *oldsel;
	boolean visible;
	void xptext_drw_proc(), xpi_std_dsp_proc();

	if ((p->xp_flags & LXP_PANELVISIBLE) && (p->xp_flags & LXP_FRAMEMAPPED))
		visible= TRUE;
	else
		visible= FALSE;

	/* no currently active text item */
	if (p->xp_seltext == (Panel_item *) NULL) {
		for (pi= p->xp_items; pi != (Panel_item *) NULL; pi= pi->xpi_next) {
			if ((pi->xpi_type == LXPI_TEXT) && (pi->xpi_state == LXPI_ACTIVE))
				break;
		}
		if (pi != (Panel_item *) NULL) {
			p->xp_seltext= pi;
			if (visible) {
				xptext_drw_proc(p, pi);
				xpi_std_dsp_proc(p, pi);
			}
		}
	}

	/* currently active text item -- cycle along to next one */
	else {
		oldsel= p->xp_seltext;
		for (pi= oldsel->xpi_next; pi != (Panel_item *) NULL; pi= pi->xpi_next) {
			if ((pi->xpi_type == LXPI_TEXT) && (pi->xpi_state == LXPI_ACTIVE))
				break;
		}
		if (pi != (Panel_item *) NULL) {
			p->xp_seltext= pi;
			if (visible) {
				xptext_drw_proc(p, pi);
				xpi_std_dsp_proc(p, pi);
				xptext_drw_proc(p, oldsel);
				xpi_std_dsp_proc(p, oldsel);
			}
		}
		else {
			for (pi= p->xp_items; pi != oldsel; pi= pi->xpi_next) {
				if ((pi->xpi_type == LXPI_TEXT) && (pi->xpi_state == LXPI_ACTIVE))
					break;
			}
			if (pi != oldsel) {
				p->xp_seltext= pi;
				if (visible) {
					xptext_drw_proc(p, pi);
					xpi_std_dsp_proc(p, pi);
					xptext_drw_proc(p, oldsel);
					xpi_std_dsp_proc(p, oldsel);
				}
			}
		}
	}

	return;
}

void
xptext_sc_proc(p, evt)
Panel *p;
XSelectionClearEvent *evt;
{
	Panel_item *pi;
	xp_text *xpt;

	if (lxt_selsrc == (Void *) NULL)
		return;
	if (lxt_selsrctype != LX_PANELITEM)
		return;
	if (lxt_selsrcpanel != p)
		return;
	if (evt->time < lxt_selsrctm)
		return;

	for (pi= p->xp_items; pi != (Panel_item *) NULL; pi= pi->xpi_next) {
		if (pi == (Panel_item *) lxt_selsrc)
			break;
	}
	if (pi == (Panel_item *) NULL)
		return;

	xpt= (xp_text *) &(pi->xpi_item.xpi_text);
	xpt->xptext_selstart= xpt->xptext_selstop= xpt->xptext_ipos;
	xpt->xptext_flags&= ~LXPTEXT_SELHOLDER;
	if ((p->xp_flags & LXP_PANELVISIBLE) && (p->xp_flags & LXP_FRAMEMAPPED) &&
	    (pi->xpi_state == LXPI_ACTIVE)) {
		xptext_drw_proc(p, pi);
		xpi_std_dsp_proc(p, pi);
	}

	lxt_selsrc= (Void *) NULL;
	lxt_selsrctype= LX_NULL;
	lxt_selsrctm= (Time) 0;
	if (lxt_selsrcbuf != (char *) NULL) {
		cfree(lxt_selsrcbuf);
		lxt_selsrcbuf= (char *) NULL;
	}
	lxt_selsrcpanel= (Panel *) NULL;

	return;
}

void
xptext_sr_proc(p, evt)
Panel *p;
XSelectionRequestEvent *evt;
{
	Panel_item *pi;
	void lxt_denysel(), lxt_sendsel();

	if (lxt_selsrc == (Void *) NULL) {
		lxt_denysel(p->xp_dpy, evt);
		return;
	}
	if (lxt_selsrcbuf == (char *) NULL) {
		lxt_denysel(p->xp_dpy, evt);
		return;
	}
	if (lxt_selsrctype != LX_PANELITEM) {
		lxt_denysel(p->xp_dpy, evt);
		return;
	}
	if (lxt_selsrcpanel != p) {
		lxt_denysel(p->xp_dpy, evt);
		return;
	}
	if ((evt->time < lxt_selsrctm) && (evt->time != CurrentTime)) {
		lxt_denysel(p->xp_dpy, evt);
		return;
	}

	for (pi= p->xp_items; pi != (Panel_item *) NULL; pi= pi->xpi_next) {
		if (pi == (Panel_item *) lxt_selsrc)
			break;
	}
	if (pi == (Panel_item *) NULL) {
		lxt_denysel(p->xp_dpy, evt);
		return;
	}

	lxt_sendsel(p->xp_dpy, evt);

	return;
}

void
xptext_sn_proc(p, evt)
Panel *p;
XSelectionEvent *evt;
{
	Panel_item *pi;
	char *selbuf, *c;
	int code, i;
	xp_text *xpt;
	boolean draw;
	char *lxt_getsel();
	void lxt_clearsel();

	if (lxt_seldest == (Void *) NULL)
		return;
	if (lxt_seldesttype != LX_PANELITEM)
		return;

	for (pi= p->xp_items; pi != (Panel_item *) NULL; pi= pi->xpi_next) {
		if (pi == (Panel_item *) lxt_seldest)
			break;
	}
	if (pi == (Panel_item *) NULL)
		return;

	xpt= (xp_text *) &(pi->xpi_item.xpi_text);
	if (xpt->xptext_flags & LXPTEXT_WRPROTECT)
		return;

	if (evt->property == None)
		return;
	if (evt->target != XA_STRING)
		return;

	if ((selbuf= lxt_getsel(p->xp_dpy, p->xp_pwin)) == (char *) NULL)
		return;

	for (c= selbuf, draw= FALSE; (isascii(*c) && (*c != '\0')); c++) {
		code= (int) *c;
		if (code == xpkey_null)
			continue;

		/* replace newline or tab with a blank space */
		if ((*c == '\n') || (code == xpkey_cret) || (code == xpkey_tab)) {
			*c= ' ';
			if (strlen(xpt->xptext_val) < xpt->xptext_max) {
				if (xpt->xptext_ipos == strlen(xpt->xptext_val)) {
					xpt->xptext_val[xpt->xptext_ipos++]= *c;
					xpt->xptext_val[xpt->xptext_ipos]= '\0';
				}
				else {
					for (i= strlen(xpt->xptext_val); i >= xpt->xptext_ipos; i--)
						xpt->xptext_val[i+1]= xpt->xptext_val[i];
					xpt->xptext_val[xpt->xptext_ipos]= *c;
					xpt->xptext_ipos++;
				}
				if (strlen(xpt->xptext_val) > xpt->xptext_disp)
					xpt->xptext_dpos++;
				draw= TRUE;
			}
			else
				break;
		}

		/* backspace over last character */
		else if (code == xpkey_backspace) {
			if (xpt->xptext_ipos > 0)
				xptext_deletechar(xpt);
			draw= TRUE;
		}

		/* word kill */
		else if (code == xpkey_wordkill) {

			/* delete all contiguous preceding white space */
			while (xpt->xptext_ipos > 0) {
				if (isspace(xpt->xptext_val[xpt->xptext_ipos-1]))
					xptext_deletechar(xpt);
				else
					break;
			}

			/* delete preceding word */
			while (xpt->xptext_ipos > 0) {
				if (isgraph(xpt->xptext_val[xpt->xptext_ipos-1]))
					xptext_deletechar(xpt);
				else
					break;
			}
			draw= TRUE;
		}

		/* line kill */
		else if (code == xpkey_linekill) {
			while (xpt->xptext_ipos > 0)
				xptext_deletechar(xpt);
			draw= TRUE;
		}

		/* all other characters */
		else if (isprint(*c)) {
			if (strlen(xpt->xptext_val) < xpt->xptext_max) {
				if (xpt->xptext_ipos == strlen(xpt->xptext_val)) {
					xpt->xptext_val[xpt->xptext_ipos++]= *c;
					xpt->xptext_val[xpt->xptext_ipos]= '\0';
				}
				else {
					for (i= strlen(xpt->xptext_val); i >= xpt->xptext_ipos; i--)
						xpt->xptext_val[i+1]= xpt->xptext_val[i];
					xpt->xptext_val[xpt->xptext_ipos]= *c;
					xpt->xptext_ipos++;
				}
				if (strlen(xpt->xptext_val) > xpt->xptext_disp)
					xpt->xptext_dpos++;
				draw= TRUE;
			}
			else
				break;
		}
	}

	if (draw) {

		/* is this item a selection source? */
		if (pi == (Panel_item *) lxt_selsrc)
			lxt_clearsel(FALSE);

		xptext_drw_proc(p, pi);
		xpi_std_dsp_proc(p, pi);
	}

	cfree(selbuf);
	lxt_seldest= (Void *) NULL;
	lxt_seldesttype= LX_NULL;

	return;
}
