#include <stdio.h>
#include <varargs.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "lxt.h"
#include "menucursor.bitmap"
#include "grey.bitmap"

Menu *xm_menus= (Menu *) NULL;
static boolean xm_init= FALSE;
static boolean xm_usrhist;
static boolean xm_histsetup;
static Menu_hist *hist_active;

extern Panel *xp_blocked;

static Pixmap menucursor_pm= None;
static Pixmap stipple_pm= None;
static Pixmap clip_pm= None;
static int clip_pmw, clip_pmh;

static Cursor menu_cursor= None;

static Bool menu_release;	/* set to True upon ButtonRelease */

/* currently displayed menus (initial menu and pullright chain) */
static Menu *menu_chain= (Menu *) NULL;
static Menu *menu_active= (Menu *) NULL;	/* rightmost pullright */

/* selected items from currently displayed menus */
static Menu_itemptr *sel_chain= (Menu_itemptr *) NULL;
static Menu_itemptr *sel_active= (Menu_itemptr *) NULL;

static Menu_procexec *proc_chain= (Menu_procexec *) NULL;

/* server time at last button release */
static Time xm_tmstamp;

/* return value for menu_get() */
Void *xmret_void;

/*VARARGS*/
Menu *
menu_create(va_alist)
/*
   User-callable.
   Creates an empty menu.
*/
va_dcl
{
	va_list varg_ptr;
	int nargs;
	Menu *m;
	Display *dpy;
	Window win;
	int attr;
	char *name;
	XGCValues gcv, sgcv;
	XSetWindowAttributes xswa;
	XWindowAttributes xwa;
	XColor fg_xc, bg_xc;
	Colormap cmap;
	GC sgc;
	XImage stipple_image, menucursor_image;
	int menu_filldefaults();
	void menu_init();

	m= (Menu *) NULL;
	va_start(varg_ptr);
	for (nargs= 0;; nargs++) {

		/* get program name */
		if (nargs == 0) {
			name= va_arg(varg_ptr, char *);
			if (name == (char *) NULL) {
				(void) fprintf(stderr, "menu_create: null program name\n");
				return((Menu *) NULL);
			}
		}

		/* get display */
		else if (nargs == 1) {
			dpy= va_arg(varg_ptr, Display *);
			if (dpy == (Display *) NULL) {
				(void) fprintf(stderr, "menu_create: null display\n");
				return((Menu *) NULL);
			}
		}

		/* get parent window */
		else if (nargs == 2) {
			win= va_arg(varg_ptr, Window);
			if (win == None) {
				(void) fprintf(stderr, "menu_create: null parent window\n");
				return((Menu *) NULL);
			}
			if ((m= (Menu *) calloc(1, sizeof(Menu))) == (Menu *) NULL) {
				(void) fprintf(stderr, "menu_create: memory allocation error\n");
				return((Menu *) NULL);
			}
			m->xm_magic= LX_MENU;
			m->xm_dpy= dpy;
			m->xm_attwin= win;
			if (menu_filldefaults(name, m) != LX_SUCCESS) {
				cfree((char *) m);
				return((Menu *) NULL);
			}
			if (!xm_init)
				menu_init(name, m->xm_dpy);
		}

		else {
			char *c;
			int n;

			attr= va_arg(varg_ptr, int);
			if (attr == LXM_NULL)
				break;

			switch (attr) {
			case LXM_FONT:
				c= va_arg(varg_ptr, char *);
				if (c == (char *) NULL) {
					(void) fprintf(stderr, "menu_create: null fontname\n");
					cfree((char *) m);
					return((Menu *) NULL);
				}
				if ((m->xm_font= XLoadQueryFont(m->xm_dpy, c)) == NULL) {
					(void) fprintf(stderr, "menu_create: cannot load font %s\n", c);
					cfree((char *) m);
					return((Menu *) NULL);
				}
				cfree(m->xm_fontnm);
				if ((m->xm_fontnm= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
					(void) fprintf(stderr, "menu_create: memory allocation error\n");
					cfree((char *) m);
					return((Menu *) NULL);
				}
				(void) strcpy(m->xm_fontnm, c);
				break;
			case LXM_FOREGROUND:
				m->xm_fg= va_arg(varg_ptr, unsigned long);
				break;
			case LXM_BACKGROUND:
				m->xm_bg= va_arg(varg_ptr, unsigned long);
				break;
			case LXM_TITLE:
				c= va_arg(varg_ptr, char *);
				if (c != (char *) NULL) {
					if ((m->xm_title= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
						(void) fprintf(stderr, "menu_create: memory allocation error\n");
						cfree((char *) m);
						return((Menu *) NULL);
					}
					(void) strcpy(m->xm_title, c);
				}
				break;
			case LXM_HIGHLIGHT:
				n= va_arg(varg_ptr, int);
				if ((n != LXM_INVERT) && (n != LXM_BOX))
					(void) fprintf(stderr, "menu_create: unrecognized highlight mode\n");
				else
					m->xm_highlight= n;
				break;
			case LXM_HMARGIN:
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "menu_create: illegal margin size\n");
				else
					m->xm_hmargin= n;
				break;
			case LXM_VMARGIN:
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "menu_create: illegal margin size\n");
				else
					m->xm_vmargin= n;
				break;
			case LXM_BWIDTH:
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "menu_create: illegal border width\n");
				else
					m->xm_bwidth= n;
				break;
			case LXM_CLIENTDATA:
				m->xm_clientdata= va_arg(varg_ptr, char *);
				break;
			default:
				(void) fprintf(stderr, "menu_create: ignoring unrecognized attribute\n");
				break;
			}
		}
	}
	va_end(varg_ptr);

	m->xm_win= XCreateSimpleWindow(dpy, win, 10, 10, 10, 10, m->xm_bwidth, m->xm_fg, m->xm_bg);
	XReparentWindow(dpy, m->xm_win, DefaultRootWindow(dpy), 10, 10);

	/* set override_redirect to avoid window manager interference */
	xswa.override_redirect= TRUE;
	XChangeWindowAttributes(dpy, m->xm_win, CWOverrideRedirect, &xswa);
	XSelectInput(dpy, m->xm_win, (ExposureMask | LeaveWindowMask | ButtonPressMask | ButtonReleaseMask | ButtonMotionMask));

	/* create cursor if not already done */
	if (menu_cursor == None) {
		menucursor_pm= XCreatePixmap(dpy, DefaultRootWindow(dpy), menucursor_width, menucursor_height, 1);
		sgcv.foreground= 1;
		sgcv.background= 0;
		sgc= XCreateGC(dpy, menucursor_pm, GCForeground | GCBackground, &sgcv);
		menucursor_image.height= menucursor_height;
		menucursor_image.width= menucursor_width;
		menucursor_image.xoffset= 0;
		menucursor_image.format= XYBitmap;
		menucursor_image.data= menucursor_bits;
		menucursor_image.byte_order= LSBFirst;
		menucursor_image.bitmap_unit= 8;
		menucursor_image.bitmap_bit_order= LSBFirst;
		menucursor_image.bitmap_pad= 8;
		menucursor_image.bytes_per_line= (menucursor_width+7)>>3;
		menucursor_image.depth= 1;
		XPutImage(dpy, menucursor_pm, sgc, &menucursor_image, 0, 0, 0, 0, menucursor_width, menucursor_height);
		XGetWindowAttributes(dpy, win, &xwa);
		cmap= xwa.colormap;
		fg_xc.pixel= m->xm_fg;
		XQueryColor(dpy, cmap, &fg_xc);
		fg_xc.flags= DoRed | DoGreen | DoBlue;
		bg_xc.pixel= m->xm_bg;
		XQueryColor(dpy, cmap, &bg_xc);
		bg_xc.flags= DoRed | DoGreen | DoBlue;
		menu_cursor= XCreatePixmapCursor(dpy, menucursor_pm, menucursor_pm, &fg_xc, &bg_xc, menucursor_x_hot, menucursor_y_hot);
	}
	XDefineCursor(dpy, m->xm_win, menu_cursor);

	/* create grey stipple if not already done */
	if (stipple_pm == None) {
		stipple_pm= XCreatePixmap(dpy, DefaultRootWindow(dpy), grey_width, grey_height, 1);
		sgcv.foreground= 1;
		sgcv.background= 0;
		sgc= XCreateGC(dpy, stipple_pm, GCForeground | GCBackground, &sgcv);
		stipple_image.height= grey_height;
		stipple_image.width= grey_width;
		stipple_image.xoffset= 0;
		stipple_image.format= XYBitmap;
		stipple_image.data= grey_bits;
		stipple_image.byte_order= LSBFirst;
		stipple_image.bitmap_unit= 8;
		stipple_image.bitmap_bit_order= LSBFirst;
		stipple_image.bitmap_pad= 8;
		stipple_image.bytes_per_line= (grey_width+7)>>3;
		stipple_image.depth= 1;
		XPutImage(dpy, stipple_pm, sgc, &stipple_image, 0, 0, 0, 0, grey_width, grey_height);
	}

	/* initialize GCs */
	gcv.foreground= m->xm_fg;
	gcv.background= m->xm_bg;
	gcv.font= m->xm_font->fid;
	gcv.plane_mask= AllPlanes;
	gcv.function= GXcopy;
	m->xm_fggc= XCreateGC(dpy, m->xm_win, (GCFont | GCForeground | GCBackground | GCPlaneMask | GCFunction), &gcv);
	gcv.foreground= m->xm_bg;
	gcv.background= m->xm_fg;
	m->xm_bggc= XCreateGC(dpy, m->xm_win, (GCFont | GCForeground | GCBackground | GCFunction | GCPlaneMask), &gcv);
	sgcv.foreground= m->xm_fg;
	sgcv.background= m->xm_bg;
	sgcv.stipple= stipple_pm;
	sgcv.fill_style= FillOpaqueStippled;
	sgcv.font= m->xm_font->fid;
	sgcv.plane_mask= m->xm_fg | m->xm_bg;
	m->xm_sgc= XCreateGC(dpy, m->xm_win, (GCForeground | GCBackground | GCStipple | GCFillStyle | GCFont | GCPlaneMask), &sgcv);

	m->xm_next= xm_menus;
	xm_menus= m;

	return(m);
}

int
menu_filldefaults(name, m)
char *name;
Menu *m;
{
	char *f, fontname[LX_MAXFONTNMLEN+1];
	int screen;

	/* font */
	if ((f= XGetDefault(m->xm_dpy, name, "Font")) == (char *) NULL)
		(void) strcpy(fontname, LXMDEF_FONT);
	else
		(void) strncpy(fontname, f, LX_MAXFONTNMLEN);
	if ((m->xm_font= XLoadQueryFont(m->xm_dpy, fontname)) == NULL) {
		(void) fprintf(stderr, "menu_filldefaults: cannot load font %s\n", fontname);
		return(LX_ERROR);
	}
	if ((m->xm_fontnm= calloc((unsigned) (strlen(fontname)+1), sizeof(char))) == (char *) NULL) {
		(void) fprintf(stderr, "menu_filldefaults: memory allocation error\n");
		return(LX_ERROR);
	}
	(void) strcpy(m->xm_fontnm, fontname);

	/* foreground and background */
	screen= DefaultScreen(m->xm_dpy);
	m->xm_fg= BlackPixel(m->xm_dpy, screen);
	m->xm_bg= WhitePixel(m->xm_dpy, screen);

	m->xm_highlight= LXM_INVERT;
	m->xm_hmargin= LXMDEF_HMARGIN;
	m->xm_vmargin= LXMDEF_VMARGIN;
	m->xm_bwidth= LXMDEF_BWIDTH;

	/* null fields */
	m->xm_title= (char *) NULL;
	m->xm_items= (Menu_itemptr *) NULL;
	m->xm_caller= m->xm_pullright= (Menu *) NULL;
	m->xm_clientdata= (char *) NULL;
	m->xm_hist= (caddr_t) NULL;
	return(LX_SUCCESS);
}

void
menu_init(name, dpy)
char *name;
Display *dpy;
{
	char *c;

	if ((c= XGetDefault(dpy, name, "MenuHistory")) == (char *) NULL)
		xm_usrhist= FALSE;
	else if (strlen(c) == 0)
		xm_usrhist= FALSE;
	else if ((*c == 'y') || (*c == 'Y'))
		xm_usrhist= TRUE;
	xm_init= TRUE;
}

/*VARARGS*/
int
menu_set(va_alist)
/*
   User-callable.
   Changes an attribute of a previously created menu.
*/
va_dcl
{
	va_list varg_ptr;
	Menu *m;
	int nargs, attr;
	XFontStruct *prev_fontst;

	va_start(varg_ptr);
	for (nargs= 0;; nargs++) {

		if (nargs == 0) {
			m= va_arg(varg_ptr, Menu *);
			if (m == (Menu *) NULL) {
				(void) fprintf(stderr, "menu_set: null menu\n");
				return(LX_ERROR);
			}
			if (m->xm_magic != LX_MENU) {
				(void) fprintf(stderr, "menu_set: object is not a menu\n");
				return(LX_ERROR);
			}
		}

		else {
			char *c;
			int n;

			attr= va_arg(varg_ptr, int);
			if (attr == LXM_NULL)
				break;

			switch (attr) {
			case LXM_FONT:
				c= va_arg(varg_ptr, char *);
				if (c == (char *) NULL) {
					(void) fprintf(stderr, "menu_set: null font name\n");
					return(LX_ERROR);
				}
				prev_fontst= m->xm_font;
				if ((m->xm_font= XLoadQueryFont(m->xm_dpy, c)) == NULL) {
					(void) fprintf(stderr, "menu_set: cannot load font %s\n", c);
					m->xm_font= prev_fontst;
					return(LX_ERROR);
				}
				cfree(m->xm_fontnm);
				if ((m->xm_fontnm= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
					(void) fprintf(stderr, "menu_set: memory allocation error\n");
					return(LX_ERROR);
				}
				(void) strcpy(m->xm_fontnm, c);
				XSetFont(m->xm_dpy, m->xm_fggc, m->xm_font->fid);
				XSetFont(m->xm_dpy, m->xm_bggc, m->xm_font->fid);
				XSetFont(m->xm_dpy, m->xm_sgc, m->xm_font->fid);
				XUnloadFont(m->xm_dpy, prev_fontst->fid);
				break;
			case LXM_FOREGROUND:
				m->xm_fg= va_arg(varg_ptr, unsigned long);
				XSetForeground(m->xm_dpy, m->xm_fggc, m->xm_fg);
				XSetBackground(m->xm_dpy, m->xm_bggc, m->xm_fg);
				XSetForeground(m->xm_dpy, m->xm_sgc, m->xm_fg);
				break;
			case LXM_BACKGROUND:
				m->xm_bg= va_arg(varg_ptr, unsigned long);
				XSetBackground(m->xm_dpy, m->xm_fggc, m->xm_bg);
				XSetForeground(m->xm_dpy, m->xm_bggc, m->xm_bg);
				XSetBackground(m->xm_dpy, m->xm_sgc, m->xm_bg);
				break;
			case LXM_TITLE:
				c= va_arg(varg_ptr, char *);
				if (m->xm_title != (char *) NULL)
					cfree(m->xm_title);
				if (c == (char *) NULL)
					m->xm_title= (char *) NULL;
				else {
					if ((m->xm_title= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
						(void) fprintf(stderr, "menu_set: memory allocation error\n");
						return(LX_ERROR);
					}
					(void) strcpy(m->xm_title, c);
				}
				break;
			case LXM_HIGHLIGHT:
				n= va_arg(varg_ptr, int);
				switch (n) {
				case LXM_BOX:
				case LXM_INVERT:
					m->xm_highlight= n;
					break;
				default:
					(void) fprintf(stderr, "menu_set: unrecognized highlight mode\n");
					return(LX_ERROR);
				}
				break;
			case LXM_HMARGIN:
				n= va_arg(varg_ptr, int);
				if (n < 0) {
					(void) fprintf(stderr, "menu_set: illegal margin size\n");
					return(LX_ERROR);
				}
				m->xm_hmargin= n;
				break;
			case LXM_VMARGIN:
				n= va_arg(varg_ptr, int);
				if (n < 0) {
					(void) fprintf(stderr, "menu_set: illegal margin size\n");
					return(LX_ERROR);
				}
				m->xm_vmargin= n;
				break;
			case LXM_BWIDTH:
				n= va_arg(varg_ptr, int);
				if (n < 0) {
					(void) fprintf(stderr, "menu_set: illegal border width\n");
					return(LX_ERROR);
				}
				m->xm_bwidth= n;
				XSetWindowBorderWidth(m->xm_dpy, m->xm_win, m->xm_bwidth);
				break;
			case LXM_CLIENTDATA:
				m->xm_clientdata= va_arg(varg_ptr, char *);
				break;
			default:
				(void) fprintf(stderr, "menu_set: unrecognized menu attribute %d\n", attr);
				break;
			}
		}
	}
	va_end(varg_ptr);
	return(LX_SUCCESS);
}

Void *
menu_get(m, attr)
/*
   User-callable routine.
   Returns a pointer to the value of the
   supplied attribute or to the attribute itself.
*/
Menu *m;
int attr;
{
	xmret_void= (Void *) NULL;
	if (m == (Menu *) NULL) {
		(void) fprintf(stderr, "menu_get: null menu\n");
		return(xmret_void);
	}
	if (m->xm_magic != LX_MENU) {
		(void) fprintf(stderr, "menu_get: object is not a menu\n");
		return(xmret_void);
	}
	if (attr == LXM_NULL)
		return(xmret_void);

	switch (attr) {

	case LXM_FONT:
		xmret_void= (Void *) m->xm_fontnm;
		break;
	case LXM_FOREGROUND:
		xmret_void= (Void *) &(m->xm_fg);
		break;
	case LXM_BACKGROUND:
		xmret_void= (Void *) &(m->xm_bg);
		break;
	case LXM_TITLE:
		xmret_void= (Void *) m->xm_title;
		break;
	case LXM_HIGHLIGHT:
		xmret_void= (Void *) &(m->xm_highlight);
		break;
	case LXM_HMARGIN:
		xmret_void= (Void *) &(m->xm_hmargin);
		break;
	case LXM_VMARGIN:
		xmret_void= (Void *) &(m->xm_vmargin);
		break;
	case LXM_BWIDTH:
		xmret_void= (Void *) &(m->xm_bwidth);
		break;
	case LXM_CLIENTDATA:
		xmret_void= (Void *) m->xm_clientdata;
		break;
	default:
		(void) fprintf(stderr, "menu_get: unrecognized attribute\n");
		break;
	}
	return(xmret_void);
}

int
menu_destroy(m)
/*
   User-callable.
   Destroys a menu. Frees all Menu_itemptrs pointing
   to its items, but does not free the items themselves.
*/
Menu *m;
{
	Menu *xm, *ym;
	Menu_itemptr *xmip, *ymip;
	void menu_adjhist(), menu_dsthist();

	if (m == (Menu *) NULL) {
		(void) fprintf(stderr, "menu_destroy: null menu\n");
		return(LX_ERROR);
	}
	if (m->xm_magic != LX_MENU) {
		(void) fprintf(stderr, "menu_destroy: object is not a menu\n");
		return(LX_ERROR);
	}

	/* unlink menu from chain */
	if (m == xm_menus)
		xm_menus= xm_menus->xm_next;
	else {
		for (xm= ym= xm_menus; xm != (Menu *) NULL; ym= xm, xm= xm->xm_next) {
			if (xm == m) {
				ym->xm_next= xm->xm_next;
				break;
			}
		}
		if (xm == (Menu *) NULL) {
			(void) fprintf(stderr, "menu_destroy: cannot locate menu\n");
			return(LX_ERROR);
		}
	}

	menu_adjhist(m);
	for (xmip= m->xm_items; xmip != (Menu_itemptr *) NULL; xmip= ymip) {
		ymip= xmip->xmip_next;
		cfree((char *) xmip);
	}
	XDestroyWindow(m->xm_dpy, m->xm_win);
	if (m->xm_fggc != None)
		XFreeGC(m->xm_dpy, m->xm_fggc);
	if (m->xm_bggc != None)
		XFreeGC(m->xm_dpy, m->xm_bggc);
	if (m->xm_sgc != None)
		XFreeGC(m->xm_dpy, m->xm_sgc);
	if (m->xm_fontnm != (char *) NULL)
		cfree(m->xm_fontnm);
	if (m->xm_title != (char *) NULL)
		cfree(m->xm_title);
	if (m->xm_hist != (caddr_t) NULL)
		menu_dsthist(m);
	cfree((char *) m);
	return(LX_SUCCESS);
}

void
menu_resize(m)
/*
   Internal function.
   Determines the window size of m.
*/
Menu *m;
{
	int font_ht, maxlen;
	Menu_itemptr *mip;
	void menuitem_resize();

	if (m == (Menu *) NULL) {
		(void) fprintf(stderr, "menu_resize: null menu\n");
		return;
	}
	font_ht= m->xm_font->max_bounds.ascent+m->xm_font->max_bounds.descent;
	maxlen= 0;
	m->xm_h= m->xm_vmargin;
	if (m->xm_title != (char *) NULL) {
		m->xm_h+= font_ht+m->xm_vmargin;
		maxlen= XTextWidth(m->xm_font, m->xm_title, strlen(m->xm_title));
	}
	for (mip= m->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next) {
		menuitem_resize(m, mip, font_ht);
		mip->xmip_x= m->xm_hmargin;
		mip->xmip_y= m->xm_h;
		m->xm_h+= mip->xmip_h;
		if (mip->xmip_w > maxlen)
			maxlen= mip->xmip_w;
	}
	m->xm_w= maxlen+(2*m->xm_hmargin);
	XResizeWindow(m->xm_dpy, m->xm_win, m->xm_w, m->xm_h);
}

void
menu_show(m, evt)
/*
   User-callable.
   Displays menu, returns after invoking button is released.
*/
Menu *m;
XButtonPressedEvent *evt;
{
	void menu_display(), menu_execprocchain();

	if (m == (Menu *) NULL) {
		(void) fprintf(stderr, "menu_show: null menu\n");
		return;
	}
	if (m->xm_magic != LX_MENU) {
		(void) fprintf(stderr, "menu_show: object is not a menu\n");
		return;
	}

	if (evt->type != ButtonPress) {
		(void) fprintf(stderr, "menu_show: invalid event type\n");
		return;
	}

	xm_histsetup= xm_usrhist;
	if (evt->state & ControlMask)
		xm_histsetup= !xm_histsetup;
	if (m->xm_hist == (caddr_t) NULL)
		xm_histsetup= FALSE;
	if (xm_histsetup)
		hist_active= (Menu_hist *) m->xm_hist;

/*
	if (xp_blocked == (Panel *) NULL)
		XGrabServer(m->xm_dpy);
*/
	proc_chain= (Menu_procexec *) NULL;
	menu_display(m, evt->x, evt->y);
/*
	if (xp_blocked == (Panel *) NULL)
		XUngrabServer(m->xm_dpy);
*/
	XSync(m->xm_dpy, False);

	/* menu_execprocchain() must be the last thing called
	   in this routine, in case any item procedures result
	   in the destruction of anything referenced by (or
	   referencing) m or any of its items! */
	menu_execprocchain();
}

void
menu_display(m, x, y)
/*
   Internal function.
   Called recursively as necessary for pullrights.
*/
Menu *m;
int x, y;
{
	Menu *n;
	Window parent, child;
	int wx, wy, dw, dh, rx, ry;
	int ywarp;
	void menu_appendchain(), menu_deletechain();
	void menu_appendselchain(), menu_deleteselchain();
	void menu_appendprocchain();
	void menu_histwarp();
	void menu_resize(), menu_draw(), menu_loop();

	if (m == (Menu *) NULL) {
		(void) fprintf(stderr, "menu_show: null menu\n");
		return;
	}
	for (n= menu_chain; n != (Menu *) NULL; n= n->xm_pullright) {
		if (m == n) {
			(void) fprintf(stderr, "menu_show: pullright recursion not supported\n");
			return;
		}
	}
	menu_resize(m);
	menu_appendchain(m);
	menu_appendselchain();

	/* position menu window */
	if (menu_active != menu_chain)
		parent= menu_active->xm_caller->xm_win;
	else
		parent= m->xm_attwin;
	XTranslateCoordinates(m->xm_dpy, parent, DefaultRootWindow(m->xm_dpy), 0, 0, &rx, &ry, &child);
	wx= x+rx;
	dw= DisplayWidth(m->xm_dpy, DefaultScreen(m->xm_dpy));
	if (wx < 0)
		wx= 0;
	else if (wx > dw-m->xm_w)
		wx= dw-m->xm_w;
	wy= y+ry;
	dh= DisplayHeight(m->xm_dpy, DefaultScreen(m->xm_dpy));
	if (wy < 0)
		wy= 0;
	else if (wy > dh-m->xm_h)
		wy= dh-m->xm_h;
	XMoveWindow(m->xm_dpy, m->xm_win, wx, wy);

	XMapRaised(m->xm_dpy, m->xm_win);
	XGrabPointer(m->xm_dpy, m->xm_win, False,
		ButtonMotionMask | ButtonReleaseMask | LeaveWindowMask,
		GrabModeAsync, GrabModeAsync, None, None, CurrentTime);

	if (xm_histsetup) {
		if (hist_active->xmh_menuitem != (Menu_item *) NULL) {
			menu_histwarp(m, hist_active->xmh_menuitem, &ywarp);
			XWarpPointer(m->xm_dpy, None, m->xm_win, 0, 0, 0, 0, m->xm_w-2, ywarp);
		}
		hist_active= hist_active->xmh_next;
		if (hist_active == (Menu_hist *) NULL)
			xm_histsetup= FALSE;
	}

	/* menu will be drawn upon receiving an expose event! */
	/* menu_draw(m); */

	menu_loop(m);
	XUngrabPointer(m->xm_dpy, CurrentTime);
	XUnmapWindow(m->xm_dpy, m->xm_win);

	/* if an item from m is selected, call its associated xmi_proc() */
	if (sel_active->xmip_item != (Menu_item *) NULL)
		if (sel_active->xmip_item->xmi_proc != (void (*)()) NULL)
			menu_appendprocchain(sel_active->xmip_item->xmi_proc, m, sel_active->xmip_item);

	menu_deleteselchain();
	menu_deletechain();
	return;
}

void
menu_draw(m)
/*
   Internal function.
   Plots the menu's contents on the display.
*/
Menu *m;
{
	int i, j;
	int font_ht, asc, py;
	int max_w, max_h;
	GC gc, clip_gc;
	XGCValues gcv;
	Menu_itemptr *mip;
	void menu_highlight();

	asc= m->xm_font->max_bounds.ascent;
	font_ht= m->xm_font->max_bounds.ascent+m->xm_font->max_bounds.descent;
	py= m->xm_vmargin;

	/* the following is a kludge necessitated by the fact that since
	   stippling does not seem to apply to XPutImage(), we need to use
	   a clip mask pixmap to grey out inactive items represented by images;
	   rather than churn pixmaps for all image items, we create one and
	   reuse it repeatedly, destroying it and creating a larger one
	   whenever necessary */
	max_w= max_h= -1;
	for (mip= m->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next) {
		if (mip->xmip_item->xmi_image == (XImage *) NULL)
			continue;
		if (max_w < mip->xmip_item->xmi_image->width)
			max_w= mip->xmip_item->xmi_image->width;
		if (max_h < mip->xmip_item->xmi_image->height)
			max_h= mip->xmip_item->xmi_image->height;
	}
	if ((max_w > -1) && (max_h > -1)) {

		if ((clip_pm != None) && ((clip_pmw < max_w) || (clip_pmh < max_h))) {
			XFreePixmap(m->xm_dpy, clip_pm);
			clip_pm= None;
		}
		if (clip_pm == None) {
			clip_pmw= max_w;
			clip_pmh= max_h;
			clip_pm= XCreatePixmap(m->xm_dpy, DefaultRootWindow(m->xm_dpy), clip_pmw, clip_pmh, 1);
			gcv.foreground= 1;
			gcv.background= 0;
			clip_gc= XCreateGC(m->xm_dpy, clip_pm, GCForeground | GCBackground, &gcv);
			for (i= 0; i < clip_pmh; i++) {
				for (j= 0; j < clip_pmw; j++) {
					if (i%2 == j%2)
						XDrawPoint(m->xm_dpy, clip_pm, clip_gc, j, i);
				}
			}
			XFreeGC(m->xm_dpy, clip_gc);
		}
	}

	XClearWindow(m->xm_dpy, m->xm_win);
	if (m->xm_title != (char *) NULL) {
		XDrawString(m->xm_dpy, m->xm_win, m->xm_fggc, m->xm_hmargin, py+asc, m->xm_title, strlen(m->xm_title));
		py+= font_ht+m->xm_vmargin;
	}
	for (mip= m->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next) {
		if (mip->xmip_item->xmi_state == LXMI_ACTIVE)
			gc= m->xm_fggc;
		else
			gc= m->xm_sgc;
		if (mip->xmip_item->xmi_image != (XImage *) NULL) {
			if (mip->xmip_item->xmi_state == LXMI_INACTIVE) {
				XSetClipOrigin(m->xm_dpy, m->xm_fggc, m->xm_hmargin, py);
				XSetClipMask(m->xm_dpy, m->xm_fggc, clip_pm);
			}
			XPutImage(m->xm_dpy, m->xm_win, m->xm_fggc, mip->xmip_item->xmi_image, 0, 0,
					m->xm_hmargin, py, mip->xmip_item->xmi_image->width,
					mip->xmip_item->xmi_image->height);
			if (mip->xmip_item->xmi_state == LXMI_INACTIVE) {
				XSetClipOrigin(m->xm_dpy, m->xm_fggc, 0, 0);
				XSetClipMask(m->xm_dpy, m->xm_fggc, None);
			}
		}
		else if (mip->xmip_item->xmi_string != (char *) NULL)
			XDrawString(m->xm_dpy, m->xm_win, gc, m->xm_hmargin, py+asc,
					mip->xmip_item->xmi_string, strlen(mip->xmip_item->xmi_string));
		if (mip->xmip_item->xmi_pullright != (char *) NULL) {
			XDrawLine(m->xm_dpy, m->xm_win, gc, m->xm_w-(m->xm_hmargin+LXMI_PRARROWLEN),
					py+(mip->xmip_h/2), m->xm_w-m->xm_hmargin, py+(mip->xmip_h/2));
			XDrawLine(m->xm_dpy, m->xm_win, gc, m->xm_w-(m->xm_hmargin+6),
					py+(mip->xmip_h/2)-3, m->xm_w-m->xm_hmargin, py+(mip->xmip_h/2));
			XDrawLine(m->xm_dpy, m->xm_win, gc, m->xm_w-(m->xm_hmargin+6),
					py+(mip->xmip_h/2)+3, m->xm_w-m->xm_hmargin, py+(mip->xmip_h/2));
		}
		py+= mip->xmip_h;
	}
	if (sel_active->xmip_item != (Menu_item *) NULL)
		menu_highlight(m, sel_active->xmip_item);
}

void
menu_loop(m)
/*
   Internal function.
   Processes all input events within the menu.
*/
Menu *m;
{
	XEvent event;
	XLeaveWindowEvent *lv;
	XButtonReleasedEvent *br;
	XPointerMovedEvent *pm;
	Bool menu_event();
	void menu_exselect(), menu_mvselect();
	void menu_svhist();

	for (menu_release= False; menu_release == False;) {
		XIfEvent(m->xm_dpy, &event, menu_event, (char *) NULL);
		switch (event.type) {
		case Expose:
			menu_exselect((XExposeEvent *) &event, m);
			break;
		case LeaveNotify:
			lv= (XLeaveWindowEvent *) &event;
			if ((lv->window == m->xm_win) && (menu_active != menu_chain)) {
				sel_active->xmip_item= (Menu_item *) NULL;
				return;
			}
			break;
		case MotionNotify:

			/* the following block of code should not be necessary,
			   as the condition it reports should always be handled
			   by the LeaveNotify code immediately above --
			   unfortunately, Sun's OpenWindows 1.0 does not appear
			   to report LeaveNotify events properly */
			pm= (XPointerMovedEvent *) &event;
			if ((pm->window == m->xm_win) && (menu_active != menu_chain) && (pm->x < 0)) {
				sel_active->xmip_item= (Menu_item *) NULL;
				return;
			}

			menu_mvselect(pm, m);
			break;
		case ButtonRelease:
			br= (XButtonReleasedEvent *) &event;
			bcopy((char *) &(br->time), (char *) &xm_tmstamp , sizeof(Time));
			menu_release= True;
			menu_svhist();
			return;
			break;
		default:
			break;
		}
	}
}

Bool
menu_event(dpy, event, args)
/*
   Internal function.
   Prevents consumption of non-menu windows' expose events.
*/
Display *dpy;
XEvent *event;
char *args;
{
	Menu *m;

	if (event->type == Expose) {
		for (m= menu_chain; m != (Menu *) NULL; m= m->xm_pullright)
			if (((XExposeEvent *) event)->window == m->xm_win)
				return(True);
		return(False);
	}
	else
		return(True);
}

void
menu_exselect(event, m)
/*
   Internal function.
   Redraws menu upon exposure (e.g., by a
   pullright which has been deselected).
*/
XExposeEvent *event;
Menu *m;
{
	void menu_draw();

	if (event->window != m->xm_win)
		return;
	menu_draw(m);
}

void
menu_mvselect(event, m)
/*
   Internal function.
   Processes all motion events within the menu,
   updating the current selection and/or recursively invoking
   menu_display() upon a pullright as necessary.
*/
XPointerMovedEvent *event;
Menu *m;
{
	Menu_item *mi;
	Menu_item *menu_findsel();
	void menu_display();
	void menu_highlight(), menu_unhighlight();

	if (event->window != m->xm_win)
		return;
	if ((mi= menu_findsel(m, event->x, event->y)) == (Menu_item *) NULL) {
		if (sel_active->xmip_item != (Menu_item *) NULL) {
			menu_unhighlight(m, sel_active->xmip_item);
			sel_active->xmip_item= (Menu_item *) NULL;
		}
	}
	else {
		if (sel_active->xmip_item != (Menu_item *) NULL) {
			if (sel_active->xmip_item != mi) {
				menu_unhighlight(m, sel_active->xmip_item);
				sel_active->xmip_item= mi;
				menu_highlight(m, sel_active->xmip_item);
			}
		}
		else {
			sel_active->xmip_item= mi;
			menu_highlight(m, sel_active->xmip_item);
		}
		if ((sel_active->xmip_item->xmi_pullright != (char *) NULL) &&
			(event->x > m->xm_w-(m->xm_hmargin+LXMI_PRARROWLEN)))
			menu_display((Menu *) (sel_active->xmip_item->xmi_pullright), event->x-5, event->y-5);
	}
}

void
menu_highlight(m, mi)
/*
   Internal function.
   Highlights a Menu_item.
*/
Menu *m;
Menu_item *mi;
{
	int x, y, w, h;
	int asc, py;
	Menu_itemptr *mip;

	if (m == (Menu *) NULL)
		return;
	if (mi == (Menu_item *) NULL)
		return;
	switch (m->xm_highlight) {
	case LXM_BOX:
		XSetFunction(m->xm_dpy, m->xm_fggc, GXxor);
		break;
	case LXM_INVERT:
		break;
	default:
		break;
	}

	x= m->xm_hmargin/2;
	asc= m->xm_font->max_bounds.ascent;
	py= m->xm_vmargin;
	if (m->xm_title != (char *) NULL)
		py+= asc+m->xm_font->max_bounds.descent+m->xm_vmargin;

	for (mip= m->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next) {
		if (mip->xmip_item == mi) {
			y= mip->xmip_y;
			h= mip->xmip_h-m->xm_vmargin;
			switch (m->xm_highlight) {
			case LXM_BOX:
				XDrawLine(m->xm_dpy, m->xm_win, m->xm_fggc, x, y-1, m->xm_w-x, y-1);
				XDrawLine(m->xm_dpy, m->xm_win, m->xm_fggc, x, y+h, m->xm_w-x, y+h);
				XDrawLine(m->xm_dpy, m->xm_win, m->xm_fggc, x, y, x, y+h-1);
				XDrawLine(m->xm_dpy, m->xm_win, m->xm_fggc, m->xm_w-x, y, m->xm_w-x, y+h-1);
				break;
			case LXM_INVERT:
				w= m->xm_w-(2*x);
				if (mip->xmip_item->xmi_image != (XImage *) NULL) {
					XSetFunction(m->xm_dpy, m->xm_fggc, GXinvert);
					XSetPlaneMask(m->xm_dpy, m->xm_fggc, (unsigned long) (m->xm_fg ^ m->xm_bg));
					XFillRectangle(m->xm_dpy, m->xm_win, m->xm_fggc, x, y-1, w, h+2);
					XSetFunction(m->xm_dpy, m->xm_fggc, GXcopy);
					XSetPlaneMask(m->xm_dpy, m->xm_fggc, (unsigned long) AllPlanes);
				}
				else
					XFillRectangle(m->xm_dpy, m->xm_win, m->xm_fggc, x, y-1, w, h+2);
				if ((mip->xmip_item->xmi_image == (XImage *) NULL) &&
				    (mip->xmip_item->xmi_string != (char *) NULL)) {
					XDrawString(m->xm_dpy, m->xm_win, m->xm_bggc, m->xm_hmargin, py+asc, mip->xmip_item->xmi_string, strlen(mip->xmip_item->xmi_string));
					if (mip->xmip_item->xmi_pullright != (char *) NULL) {
						XDrawLine(m->xm_dpy, m->xm_win, m->xm_bggc, m->xm_w-(m->xm_hmargin+LXMI_PRARROWLEN),
							py+(mip->xmip_h/2), m->xm_w-m->xm_hmargin, py+(mip->xmip_h/2));
						XDrawLine(m->xm_dpy, m->xm_win, m->xm_bggc, m->xm_w-(m->xm_hmargin+6),
							py+(mip->xmip_h/2)-3, m->xm_w-m->xm_hmargin, py+(mip->xmip_h/2));
						XDrawLine(m->xm_dpy, m->xm_win, m->xm_bggc, m->xm_w-(m->xm_hmargin+6),
							py+(mip->xmip_h/2)+3, m->xm_w-m->xm_hmargin, py+(mip->xmip_h/2));
					}
				}
				break;
			default:
				break;
			}
			break;
		}
		py+= mip->xmip_h;
	}

	switch (m->xm_highlight) {
	case LXM_BOX:
		XSetFunction(m->xm_dpy, m->xm_fggc, GXcopy);
		break;
	case LXM_INVERT:
		break;
	default:
		break;
	}
	return;
}

void
menu_unhighlight(m, mi)
/*
   Internal function.
   Unhighlights a Menu_item. Funtion is currently identical
   to menu_highlight() by virtue of the fact that all
   (un)highlighting is done with GXxor. Left as a separate
   routine in case this changes in the future.
*/
Menu *m;
Menu_item *mi;
{
	int x, y, w, h;
	int asc, py;
	Menu_itemptr *mip;

	if (m == (Menu *) NULL)
		return;
	if (mi == (Menu_item *) NULL)
		return;
	switch (m->xm_highlight) {
	case LXM_BOX:
		XSetFunction(m->xm_dpy, m->xm_fggc, GXxor);
		break;
	case LXM_INVERT:
		break;
	default:
		break;
	}

	x= m->xm_hmargin/2;
	asc= m->xm_font->max_bounds.ascent;
	py= m->xm_vmargin;
	if (m->xm_title != (char *) NULL)
		py+= asc+m->xm_font->max_bounds.descent+m->xm_vmargin;

	for (mip= m->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next) {
		if (mip->xmip_item == mi) {
			y= mip->xmip_y;
			h= mip->xmip_h-m->xm_vmargin;
			switch (m->xm_highlight) {
			case LXM_BOX:
				XDrawLine(m->xm_dpy, m->xm_win, m->xm_fggc, x, y-1, m->xm_w-x, y-1);
				XDrawLine(m->xm_dpy, m->xm_win, m->xm_fggc, x, y+h, m->xm_w-x, y+h);
				XDrawLine(m->xm_dpy, m->xm_win, m->xm_fggc, x, y, x, y+h-1);
				XDrawLine(m->xm_dpy, m->xm_win, m->xm_fggc, m->xm_w-x, y, m->xm_w-x, y+h-1);
				break;
			case LXM_INVERT:
				w= m->xm_w-(2*x);
				if (mip->xmip_item->xmi_image != (XImage *) NULL) {
					XSetFunction(m->xm_dpy, m->xm_fggc, GXinvert);
					XSetPlaneMask(m->xm_dpy, m->xm_fggc, (unsigned long) (m->xm_fg ^ m->xm_bg));
					XFillRectangle(m->xm_dpy, m->xm_win, m->xm_fggc, x, y-1, w, h+2);
					XSetFunction(m->xm_dpy, m->xm_fggc, GXcopy);
					XSetPlaneMask(m->xm_dpy, m->xm_fggc, (unsigned long) AllPlanes);
				}
				else
					XFillRectangle(m->xm_dpy, m->xm_win, m->xm_bggc, x, y-1, w, h+2);
				if ((mip->xmip_item->xmi_image == (XImage *) NULL) &&
				    (mip->xmip_item->xmi_string != (char *) NULL)) {
					XDrawString(m->xm_dpy, m->xm_win, m->xm_fggc, m->xm_hmargin, py+asc, mip->xmip_item->xmi_string, strlen(mip->xmip_item->xmi_string));
					if (mip->xmip_item->xmi_pullright != (char *) NULL) {
						XDrawLine(m->xm_dpy, m->xm_win, m->xm_fggc, m->xm_w-(m->xm_hmargin+LXMI_PRARROWLEN),
							py+(mip->xmip_h/2), m->xm_w-m->xm_hmargin, py+(mip->xmip_h/2));
						XDrawLine(m->xm_dpy, m->xm_win, m->xm_fggc, m->xm_w-(m->xm_hmargin+6),
							py+(mip->xmip_h/2)-3, m->xm_w-m->xm_hmargin, py+(mip->xmip_h/2));
						XDrawLine(m->xm_dpy, m->xm_win, m->xm_fggc, m->xm_w-(m->xm_hmargin+6),
							py+(mip->xmip_h/2)+3, m->xm_w-m->xm_hmargin, py+(mip->xmip_h/2));
					}
				}
				break;
			default:
				break;
			}
			break;
		}
		py+= mip->xmip_h;
	}

	switch (m->xm_highlight) {
	case LXM_BOX:
		XSetFunction(m->xm_dpy, m->xm_fggc, GXcopy);
		break;
	case LXM_INVERT:
		break;
	default:
		break;
	}
	return;
}

void
menu_appendchain(m)
/*
   Internal function.
   Appends a menu to the list of those currently displayed
   (i.e., the initial menu and any displayed pullrights).
*/
Menu *m;
{
	if (menu_active != (Menu *) NULL) {
		XUngrabPointer(menu_active->xm_dpy, CurrentTime);
		menu_active->xm_pullright= m;
		m->xm_caller= menu_active;
		m->xm_pullright= (Menu *) NULL;
		menu_active= m;
	}
	else {
		menu_chain= menu_active= m;
		m->xm_caller= m->xm_pullright= (Menu *) NULL;
	}
}

void
menu_deletechain()
/*
   Internal function.
   Deletes the last menu from the list of those currently displayed
   (i.e., the initial menu and any displayed pullrights) after a
   pullright item is deselected or the entire chain is undisplayed
   due to button release.
*/
{
	Menu *m;

	if (menu_active == (Menu *) NULL)
		return;
	m= menu_active;
	menu_active= m->xm_caller;
	m->xm_caller= (Menu *) NULL;
	if (menu_active == (Menu *) NULL)
		menu_chain= (Menu *) NULL;
	else {
		menu_active->xm_pullright= (Menu *) NULL;
		XGrabPointer(menu_active->xm_dpy, menu_active->xm_win, False,
			ButtonMotionMask | ButtonReleaseMask | LeaveWindowMask,
			GrabModeAsync, GrabModeAsync, None, None, CurrentTime);
	}
}

void
menu_appendselchain()
/*
   Internal function.
   Appends a new cell to the list of Menu_itemptrs, one for
   each currently displayed menu, which point to the items
   which are currently selected within each displayed menu.
*/
{
	Menu_itemptr *mip;

	if ((mip= (Menu_itemptr *) calloc(1, sizeof(Menu_itemptr))) == (Menu_itemptr *) NULL) {
		(void) fprintf(stderr, "menu_appendselchain: memory allocation error\n");
		return;
	}
	if (sel_active == (Menu_itemptr *) NULL) {
		sel_chain= sel_active= mip;
		mip->xmip_prev= (Menu_itemptr *) NULL;
	}
	else {
		mip->xmip_prev= sel_active;
		sel_active->xmip_next= mip;
		sel_active= mip;
	}
	mip->xmip_item= (Menu_item *) NULL;
	mip->xmip_next= (Menu_itemptr *) NULL;
}

void
menu_deleteselchain()
/*
   Internal function.
   Deletes the last cell from the list of Menu_itemptrs
   pointing to the selected items of displayed menus upon
   pullright deselection or button release.
*/
{
	Menu_itemptr *mip;

	if (sel_active == (Menu_itemptr *) NULL)
		return;
	mip= sel_active;
	sel_active= sel_active->xmip_prev;
	cfree((char *) mip);
	if (sel_active == (Menu_itemptr *) NULL)
		sel_chain= (Menu_itemptr *) NULL;
	else
		sel_active->xmip_next= (Menu_itemptr *) NULL;
}

void
menu_appendprocchain(proc, m, mi)
/*
   Internal function.
   Adds a Menu_item xmi_proc() to the end of a chain of
   procedures to be called. The procedure at the head
   of the list (which will be called first) is associated
   with the selected item from the rightmost pullright.
   This function is called after button release.
*/
void (*proc)();
Menu *m;
Menu_item *mi;
{
	Menu_procexec *mp, *new;

	if ((new= (Menu_procexec *) calloc(1, sizeof(Menu_procexec))) == (Menu_procexec *) NULL) {
		(void) fprintf(stderr, "menu_appendprocchain: memory allocation error\n");
		return;
	}
	new->xmp_proc= proc;
	new->xmp_menu= m;
	new->xmp_menuitem= mi;
	new->xmp_next= (Menu_procexec *) NULL;
	if (proc_chain == (Menu_procexec *) NULL)
		proc_chain= new;
	else {
		for (mp= proc_chain; mp->xmp_next != (Menu_procexec *) NULL; mp= mp->xmp_next);
		mp->xmp_next= new;
	}
}

void
menu_execprocchain()
/*
   Internal function.
   Executes the xmi_proc()s contained in the
   list built by menu_appendprocchain().
*/
{
	Menu_procexec *mp, *np;

	for (mp= proc_chain; mp != (Menu_procexec *) NULL; mp= np) {

		/* set item time stamp to server time at button release */
		bcopy((char *) &xm_tmstamp, (char *) &(mp->xmp_menuitem->xmi_timestamp), sizeof(Time));

		/* call application-defined item procedure --
		   mp->xmp_menu and mp->xmp_menuitem must not be
		   referenced after the following call in case
		   */
		(*(mp->xmp_proc))(mp->xmp_menu, mp->xmp_menuitem);

		np= mp->xmp_next;
		cfree((char *) mp);
	}
	proc_chain= (Menu_procexec *) NULL;
	return;
}

void
menu_svhist()
/*
   Internal function.
   Saves the last menu chain and item chain into
   the xm_hist field of the base menu in the menu chain.
*/
{
	Menu *m;
	Menu_itemptr *mip;
	Menu_hist **mh;
	void menu_dsthist();

	if (menu_chain == (Menu *) NULL)
		return;

	if (menu_chain->xm_hist != (caddr_t) NULL)
		menu_dsthist(menu_chain);

	m= menu_chain;
	mip= sel_chain;
	mh= (Menu_hist **) &(menu_chain->xm_hist);
	while (m != (Menu *) NULL) {
		if ((*mh= (Menu_hist *) calloc(1, sizeof(Menu_hist))) == (Menu_hist *) NULL) {
			(void) fprintf(stderr, "menu_svhist: memory allocation error\n");
			return;
		}
		(*mh)->xmh_menu= m;
		(*mh)->xmh_menuitem= mip->xmip_item;
		m= m->xm_pullright;
		mip= mip->xmip_next;
		mh= &((*mh)->xmh_next);
	}
	*mh= (Menu_hist *) NULL;

	return;
}

void
menu_dsthist(m)
/*
   Internal function.
   Destroy the last history chain
   stored in the specified menu.
*/
Menu *m;
{
	Menu_hist *mh, *nh;

	for (mh= (Menu_hist *) m->xm_hist; mh != (Menu_hist *) NULL; mh= nh) {
		nh= mh->xmh_next;
		cfree((char *) mh);
	}
	m->xm_hist= (caddr_t) NULL;
	return;
}

void
menu_histwarp(m, mi, y)
/*
   Internal function.
   Return the y location within the specified menu
   of the specified item for purposes of warping
   the pointer to that location to invoke a history chain.
*/
Menu *m;
Menu_item *mi;
int *y;
{
	Menu_itemptr *mip;

	for (mip= m->xm_items; mip != (Menu_itemptr *) NULL; mip= mip->xmip_next) {
		if (mip->xmip_item == mi) {
			*y= mip->xmip_y+(mip->xmip_h/2);
			return;
		}
	}

	/* here if item cannot be located */
	*y= 0;
	xm_histsetup= FALSE;
	return;
}

void
menu_adjhist(m)
/*
   Internal function.
   Check all history chains to delete any
   references to the specified menu.
*/
Menu *m;
{
	Menu *n;
	Menu_hist *mh, *nh;
	void menu_dsthist();

	for (n= xm_menus; n != (Menu *) NULL; n= n->xm_next) {
		for (mh= (Menu_hist *) n->xm_hist; mh != (Menu_hist *) NULL; mh= mh->xmh_next) {
			if (mh->xmh_menu == m)
				break;
		}
		if (mh != (Menu_hist *) NULL) {
			if (mh == (Menu_hist *) n->xm_hist)
				menu_dsthist(n);
			else {
				for (nh= (Menu_hist *) n->xm_hist; nh->xmh_next != mh; nh= nh->xmh_next);
				nh->xmh_menuitem= (Menu_item *) NULL;
				nh->xmh_next= (Menu_hist *) NULL;
				for ( ; mh != (Menu_hist *) NULL; mh= nh) {
					nh= mh->xmh_next;
					cfree((char *) mh);
				}
			}
		}
	}
	return;
}
