#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "lxt.h"

void
canvas_clear(c)
/*
   User-callable function.
   Clears the canvas window and backing store.
*/
Canvas *c;
{
	if (c == (Canvas *) NULL) {
		(void) fprintf(stderr, "canvas_clear: null canvas\n");
		return;
	}
	if (c->xc_magic != LX_CANVAS) {
		(void) fprintf(stderr, "canvas_clear: object is not a canvas\n");
		return;
	}

	XFillRectangle(c->xc_dpy, c->xc_cpm, c->xc_cgc, 0, 0, c->xc_vcw, c->xc_vch);
	XFillRectangle(c->xc_dpy, c->xc_cwin, c->xc_cgc, 0, 0, c->xc_acw, c->xc_ach);
	XFlush(c->xc_dpy);
	return;
}

void
canvas_flush(c)
/*
   User-callable function.
   Flushes the canvas's backing store to the display.
*/
Canvas *c;
{
	if (c == (Canvas *) NULL) {
		(void) fprintf(stderr, "canvas_flush: null canvas\n");
		return;
	}
	if (c->xc_magic != LX_CANVAS) {
		(void) fprintf(stderr, "canvas_flush: object is not a canvas\n");
		return;
	}

	XCopyArea(c->xc_dpy, c->xc_cpm, c->xc_cwin, c->xc_gc, c->xc_vx, c->xc_vy, c->xc_acw, c->xc_ach, 0, 0);
	XFlush(c->xc_dpy);
	return;
}


void
canvasvscroll_draw(c)
/*
   Internal function.
   Plots the canvas's vertical scrollbar on the display.
*/
Canvas *c;
{
	int w, h, bl;
	void canvasvscroll_drawbubble();

	w= c->xc_vscroll->xs_barwidth+1;
	if (c->xc_cy == c->xc_ibw)
		h= c->xc_ach;
	else
		h= c->xc_ach+1;

	XClearWindow(c->xc_dpy, c->xc_vswin);
	XFillRectangle(c->xc_dpy, c->xc_vspm, c->xc_cgc, 0, 0, w, h);
	if (c->xc_ach < c->xc_h-(2*c->xc_ibw))
		XDrawLine(c->xc_dpy, c->xc_vspm, c->xc_gc, 0, 0, w-1, 0);
	XDrawLine(c->xc_dpy, c->xc_vspm, c->xc_gc, w-1, 0, w-1, h-1);
	if (c->xc_vscroll->xs_flags & LXS_BUTTONVIS) {
		bl= c->xc_vscroll->xs_buttonlen;
		if (c->xc_ach < c->xc_h-(2*c->xc_ibw))
			XDrawLine(c->xc_dpy, c->xc_vspm, c->xc_gc, 0, bl+1, w-1, bl+1);
		else
			XDrawLine(c->xc_dpy, c->xc_vspm, c->xc_gc, 0, bl, w-1, bl);
		XDrawLine(c->xc_dpy, c->xc_vspm, c->xc_gc, 0, h-1-bl, w-1, h-1-bl);
	}
	canvasvscroll_drawbubble(c);
	XCopyArea(c->xc_dpy, c->xc_vspm, c->xc_vswin, c->xc_gc, 0, 0, w, h, 0, 0);
}

void
canvasvscroll_drawbubble(c)
/*
   Internal function.
   Plots the canvas's vertical scrollbar bubble on the display.
*/
Canvas *c;
{
	Scrollbar *s;
	float f;

	s= c->xc_vscroll;
	f= ((float) c->xc_vy)/((float) c->xc_vch);
	s->xs_bubblemin= ((int) (f*s->xs_barlen))+s->xs_barstart;
	if (s->xs_bubblemin >= s->xs_barstart+s->xs_barlen)
		s->xs_bubblemin= s->xs_barstart+s->xs_barlen-1;
	f= ((float) c->xc_ach)/((float) c->xc_vch);
	if (c->xc_vy == c->xc_vch-c->xc_ach)
		s->xs_bubblelen= s->xs_barlen-(s->xs_bubblemin-s->xs_barstart);
	else
		s->xs_bubblelen= (int) (f*s->xs_barlen);
	if (s->xs_bubblemin+s->xs_bubblelen >= s->xs_barstart+s->xs_barlen)
		s->xs_bubblelen= s->xs_barstart+s->xs_barlen-s->xs_bubblemin;
	if (2*s->xs_bubblemargin > s->xs_barwidth)
		return;
	XFillRectangle(c->xc_dpy, c->xc_vspm, c->xc_sgc, s->xs_bubblemargin, s->xs_bubblemin, s->xs_barwidth-(2*s->xs_bubblemargin), s->xs_bubblelen);
}

void
canvashscroll_draw(c)
/*
   Internal function.
   Plots the canvas's horizontal scrollbar on the display.
*/
Canvas *c;
{
	int w, h, bl;
	void canvashscroll_drawbubble();

	if (c->xc_cx == c->xc_ibw)
		w= c->xc_acw;
	else
		w= c->xc_acw+1;
	h= c->xc_hscroll->xs_barwidth+1;

	XClearWindow(c->xc_dpy, c->xc_hswin);
	XFillRectangle(c->xc_dpy, c->xc_hspm, c->xc_cgc, 0, 0, w, h);
	if (c->xc_acw < c->xc_w-(2*c->xc_ibw))
		XDrawLine(c->xc_dpy, c->xc_hspm, c->xc_gc, 0, 0, 0, h-1);
	XDrawLine(c->xc_dpy, c->xc_hspm, c->xc_gc, 0, h-1, w-1, h-1);
	if (c->xc_hscroll->xs_flags & LXS_BUTTONVIS) {
		bl= c->xc_hscroll->xs_buttonlen;
		if (c->xc_acw < c->xc_w-(2*c->xc_ibw))
			XDrawLine(c->xc_dpy, c->xc_hspm, c->xc_gc, bl+1, 0, bl+1, h-1);
		else
			XDrawLine(c->xc_dpy, c->xc_hspm, c->xc_gc, bl, 0, bl, h-1);
		XDrawLine(c->xc_dpy, c->xc_hspm, c->xc_gc, w-1-bl, 0, w-1-bl, h-1);
	}
	canvashscroll_drawbubble(c);
	XCopyArea(c->xc_dpy, c->xc_hspm, c->xc_hswin, c->xc_gc, 0, 0, w, h, 0, 0);
}

void
canvashscroll_drawbubble(c)
/*
   Internal function.
   Plots the canvas's horizontal scrollbar bubble on the display.
*/
Canvas *c;
{
	Scrollbar *s;
	float f;

	s= c->xc_hscroll;
	f= ((float) c->xc_vx)/((float) c->xc_vcw);
	s->xs_bubblemin= ((int) (f*s->xs_barlen))+s->xs_barstart;
	if (s->xs_bubblemin >= s->xs_barstart+s->xs_barlen)
		s->xs_bubblemin= s->xs_barstart+s->xs_barlen-1;
	f= ((float) c->xc_acw)/((float) c->xc_vcw);
	if (c->xc_vx == c->xc_vcw-c->xc_acw)
		s->xs_bubblelen= s->xs_barlen-(s->xs_bubblemin-s->xs_barstart);
	else
		s->xs_bubblelen= (int) (f*s->xs_barlen);
	if (s->xs_bubblemin+s->xs_bubblelen >= s->xs_barstart+s->xs_barlen)
		s->xs_bubblelen= s->xs_barstart+s->xs_barlen-s->xs_bubblemin;
	if (2*s->xs_bubblemargin > s->xs_barwidth)
		return;
	XFillRectangle(c->xc_dpy, c->xc_hspm, c->xc_sgc, s->xs_bubblemin, s->xs_bubblemargin, s->xs_bubblelen, s->xs_barwidth-(2*s->xs_bubblemargin));
}

int
canvas_draw_string(c, x, y, string, gc)
Canvas *c;
int x, y;
char *string;
GC gc;
/* 
	User-callable.
	Draws string to the backing store and to the
	canvas if the batch flag is not on.
*/
{
	if (c == (Canvas *) NULL) {
		(void) fprintf(stderr, "canvas_draw_string: null canvas\n");
		return(LX_ERROR);
	}
	if (c->xc_magic != LX_CANVAS) {
		(void) fprintf(stderr, "canvas_draw_string: object is not a canvas\n");
		return(LX_ERROR);
	}

	if (!(c->xc_flags & LXC_BATCHSET))
		XDrawString(c->xc_dpy, c->xc_cwin, gc, x-c->xc_vx, y-c->xc_vy, string, strlen(string));
	XDrawString(c->xc_dpy, c->xc_cpm, gc, x, y, string, strlen(string));
	return(LX_SUCCESS);
}

int
canvas_draw_line(c, x1, y1, x2, y2, gc)
Canvas *c;
int x1, y1, x2, y2;
GC gc;
/* 
	User-callable.
	Draws line to the backing store and to the
	canvas if the batch flag is not on.
*/
{
	if (c == (Canvas *) NULL) {
		(void) fprintf(stderr, "canvas_draw_line: null canvas\n");
		return(LX_ERROR);
	}
	if (c->xc_magic != LX_CANVAS) {
		(void) fprintf(stderr, "canvas_draw_line: object is not a canvas\n");
		return(LX_ERROR);
	}

	if (!(c->xc_flags & LXC_BATCHSET))
		XDrawLine(c->xc_dpy, c->xc_cwin, gc, x1-c->xc_vx, y1-c->xc_vy, x2-c->xc_vx, y2-c->xc_vy);
	XDrawLine(c->xc_dpy, c->xc_cpm, gc, x1, y1, x2, y2);
	return(LX_SUCCESS);
}
