#include <stdio.h>
#include <varargs.h>
#include <sgtty.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#ifdef IRIX
#include <sys/types.h>
#include <curses.h>
#include <sys/file.h>
#endif IRIX

#include "lxt.h"

#include "xvcursor.bitmap"
#include "xvvscursor.bitmap"
#include "xvhscursor.bitmap"
extern Pixmap xvcursor_pm, xvvscursor_pm, xvhscursor_pm;
extern boolean xvcursor_init;
extern boolean xc_usrresize;

#include "grey.bitmap"

Canvas *xc_canvases= (Canvas *) NULL;
static boolean xc_init= FALSE;

/* return value for canvas_get() */
Void *xcret_void;
boolean xcret_bool;

/*VARARGS*/
Canvas *
canvas_create(va_alist)
/*
   User-callable.
   Creates an empty canvas.
*/
va_dcl
{
	va_list varg_ptr;
	int nargs;
	Canvas *c;
	Display *dpy;
	Window win;
	int screen, depth, attr;
	char *name;
	GC gc;
	boolean set_size;
	XGCValues gcv;
	XSetWindowAttributes xswa;
	XWindowAttributes xwa;
	XImage image;
	XColor fg_xc, bg_xc;
	Colormap cmap;
	int canvas_filldefaults();
	int canvas_config();
	void canvas_init();
	void canvasvscroll_draw(), canvashscroll_draw();

	c= (Canvas *) NULL;
	set_size= FALSE;
	va_start(varg_ptr);
	for (nargs= 0;; nargs++) {

		/* get program name */
		if (nargs == 0) {
			name= va_arg(varg_ptr, char *);
			if (name == (char *) NULL) {
				(void) fprintf(stderr, "canvas_create: null program name\n");
				return((Canvas *) NULL);
			}
		}

		/* get display */
		else if (nargs == 1) {
			dpy= va_arg(varg_ptr, Display *);
			if (dpy == (Display *) NULL) {
				(void) fprintf(stderr, "canvas_create: null display\n");
				return((Canvas *) NULL);
			}
			screen= DefaultScreen(dpy);
		}

		/* get enclosing window */
		else if (nargs == 2) {
			win= va_arg(varg_ptr, Window);
			if (win == None) {
				(void) fprintf(stderr, "canvas_create: null parent window\n");
				return((Canvas *) NULL);
			}
			if ((c= (Canvas *) calloc(1, sizeof(Canvas))) == (Canvas *) NULL) {
				(void) fprintf(stderr, "canvas_create: memory allocation error\n");
				return((Canvas *) NULL);
			}
			c->xc_magic= LX_CANVAS;
			c->xc_dpy= dpy;
			c->xc_win= win;
			if (canvas_filldefaults(name, c) != LX_SUCCESS) {
				cfree((char *) c);
				return((Canvas *) NULL);
			}
			if (!xc_init)
				canvas_init(c->xc_dpy);
		}

		else {
			int n;

			attr= va_arg(varg_ptr, int);
			if (attr == LXC_NULL)
				break;

			switch (attr) {
			case LXC_FOREGROUND:
				c->xc_fg= va_arg(varg_ptr, unsigned long);
				break;
			case LXC_BACKGROUND:
				c->xc_bg= va_arg(varg_ptr, unsigned long);
				break;
			case LXC_WIDTH:
				n= va_arg(varg_ptr, int);
				if (c->xc_mode != LXC_STATIC)
					(void) fprintf(stderr, "canvas_create: cannot set size of dynamic canvas\n");
				else {
					c->xc_vcw= n;
					set_size= TRUE;
				}
				break;
			case LXC_HEIGHT:
				n= va_arg(varg_ptr, int);
				if (c->xc_mode != LXC_STATIC)
					(void) fprintf(stderr, "canvas_create: cannot set size of dynamic canvas\n");
				else {
					c->xc_vch= n;
					set_size= TRUE;
				}
				break;
			case LXC_BARWIDTH:
				c->xc_vscroll->xs_barwidth= c->xc_hscroll->xs_barwidth= va_arg(varg_ptr, int);
				break;
			case LXC_BUBBLEMARGIN:
				c->xc_vscroll->xs_bubblemargin= c->xc_hscroll->xs_bubblemargin= va_arg(varg_ptr, int);
				break;
			case LXC_BUTTONLEN:
				c->xc_vscroll->xs_buttonlen= c->xc_hscroll->xs_buttonlen= va_arg(varg_ptr, int);
				break;
			case LXC_CCURSOR:
				c->xc_ccursor= va_arg(varg_ptr, Cursor);
				break;
			case LXC_VSCURSOR:
				c->xc_vscursor= va_arg(varg_ptr, Cursor);
				break;
			case LXC_HSCURSOR:
				c->xc_hscursor= va_arg(varg_ptr, Cursor);
				break;
			case LXC_VISIBLE:
				n= va_arg(varg_ptr, int);
				if (n)
					c->xc_flags|= LXC_CANVASVISIBLE;
				else
					c->xc_flags&= ~LXC_CANVASVISIBLE;
				break;
			case LXC_MODE:
				n= va_arg(varg_ptr, int);
				switch (n) {
				case LXC_STATIC:
					c->xc_mode= n;
					break;
				case LXC_DYNAMICCLR:
				case LXC_DYNAMICPRS:
					if (set_size) {
						(void) fprintf(stderr, "canvas_create: size request incompatible with dynamic mode -- setting mode to LXC_STATIC\n");
						c->xc_mode= LXC_STATIC;
					}
					else
						c->xc_mode= n;
					break;
				default:
					(void) fprintf(stderr, "canvas_create: unrecognized mode\n");
					break;
				}
				break;
			case LXC_BATCH:
				n= va_arg(varg_ptr, int);
				switch (n) {
				case LXC_BATCHON:
					c->xc_flags|= LXC_BATCHSET;
					break;
				case LXC_BATCHOFF:
					c->xc_flags&= ~LXC_BATCHSET;
					break;
				default:
					(void) fprintf(stderr, "canvas_create: unrecognized batch mode\n");
					break;
				}
				break;
			case LXC_EXPOSEPROC:
				c->xc_exproc= (void (*)()) va_arg(varg_ptr, char *);
				break;
			case LXC_ACTIVE:
				n= va_arg(varg_ptr, int);
				if (n)
					c->xc_flags|= LXC_CANVASACTIVE;
				else
					c->xc_flags&= ~LXC_CANVASACTIVE;
				break;
			case LXC_WINDOW:
			case LXC_PIXMAP:
			case LXC_XOFFSET:
			case LXC_YOFFSET:
			case LXC_VSCROLL:
			case LXC_HSCROLL:
			case LXC_DWIDTH:
			case LXC_DHEIGHT:
				(void) fprintf(stderr, "canvas_create: attribute is query-only\n");
				break;
			default:
				(void) fprintf(stderr, "canvas_create: ignoring unrecognized attribute\n");
				break;
			}
		}
	}
	va_end(varg_ptr);

	XGetWindowAttributes(c->xc_dpy, c->xc_win, &xwa);
	c->xc_cwin= XCreateSimpleWindow(c->xc_dpy, c->xc_win, 0, 0, xwa.width, xwa.height, 0, c->xc_fg, c->xc_bg);
	c->xc_vswin= XCreateSimpleWindow(c->xc_dpy, c->xc_win, 0, 0, xwa.width, xwa.height, 0, c->xc_fg, c->xc_bg);
	c->xc_hswin= XCreateSimpleWindow(c->xc_dpy, c->xc_win, 0, 0, xwa.width, xwa.height, 0, c->xc_fg, c->xc_bg);

	xswa.colormap= xwa.colormap;
	XChangeWindowAttributes(c->xc_dpy, c->xc_cwin, CWColormap, &xswa);
	XChangeWindowAttributes(c->xc_dpy, c->xc_vswin, CWColormap, &xswa);
	XChangeWindowAttributes(c->xc_dpy, c->xc_hswin, CWColormap, &xswa);

	XSelectInput(c->xc_dpy, c->xc_win, ExposureMask | StructureNotifyMask);
	XSelectInput(c->xc_dpy, c->xc_cwin, ExposureMask);
	if (c->xc_flags & LXC_CANVASACTIVE) {
		XSelectInput(c->xc_dpy, c->xc_vswin, (ExposureMask | ButtonPressMask));
		XSelectInput(c->xc_dpy, c->xc_hswin, (ExposureMask | ButtonPressMask));
	}
	else {
		XSelectInput(c->xc_dpy, c->xc_vswin, ExposureMask);
		XSelectInput(c->xc_dpy, c->xc_hswin, ExposureMask);
	}

	/* create grey stipple */
	c->xc_stipplepm= XCreatePixmap(c->xc_dpy, DefaultRootWindow(dpy), grey_width, grey_height, 1);
	gcv.foreground= c->xc_fg;
	gcv.background= c->xc_bg;
	gc= XCreateGC(dpy, c->xc_stipplepm, GCForeground | GCBackground, &gcv);
	image.height= grey_height;
	image.width= grey_width;
	image.xoffset= 0;
	image.format= XYBitmap;
	image.data= grey_bits;
	image.byte_order= LSBFirst;
	image.bitmap_unit= 8;
	image.bitmap_bit_order= LSBFirst;
	image.bitmap_pad= 8;
	image.bytes_per_line= (grey_width+7)>>3;
	image.depth= 1;
	XPutImage(dpy, c->xc_stipplepm, gc, &image, 0, 0, 0, 0, grey_width, grey_height);
	XFreeGC(dpy, gc);

	/* create cursors */
	cmap= xwa.colormap;
	fg_xc.pixel= c->xc_fg;
	XQueryColor(c->xc_dpy, cmap, &fg_xc);
	fg_xc.flags= DoRed | DoGreen | DoBlue;
	bg_xc.pixel= c->xc_bg;
	XQueryColor(c->xc_dpy, cmap, &bg_xc);
	bg_xc.flags= DoRed | DoGreen | DoBlue;
	if (c->xc_ccursor == None)
		c->xc_ccursor= XCreatePixmapCursor(c->xc_dpy, xvcursor_pm, xvcursor_pm, &fg_xc, &bg_xc, xvcursor_x_hot, xvcursor_y_hot);
	if (c->xc_vscursor == None)
		c->xc_vscursor= XCreatePixmapCursor(c->xc_dpy, xvvscursor_pm, xvvscursor_pm, &fg_xc, &bg_xc, xvvscursor_x_hot, xvvscursor_y_hot);
	if (c->xc_hscursor == None)
		c->xc_hscursor= XCreatePixmapCursor(c->xc_dpy, xvhscursor_pm, xvhscursor_pm, &fg_xc, &bg_xc, xvhscursor_x_hot, xvhscursor_y_hot);
	XDefineCursor(c->xc_dpy, c->xc_cwin, c->xc_ccursor);
	XDefineCursor(c->xc_dpy, c->xc_vswin, c->xc_vscursor);
	XDefineCursor(c->xc_dpy, c->xc_hswin, c->xc_hscursor);

	/* initialize GCs */
	gcv.foreground= c->xc_fg;
	gcv.background= c->xc_bg;
	gcv.plane_mask= AllPlanes;
	gcv.function= GXcopy;
	c->xc_gc= XCreateGC(c->xc_dpy, c->xc_cwin, (GCForeground | GCBackground | GCPlaneMask | GCFunction), &gcv);
	gcv.stipple= c->xc_stipplepm;
	gcv.fill_style= FillStippled;
	c->xc_sgc= XCreateGC(c->xc_dpy, c->xc_cwin, (GCForeground | GCBackground | GCFunction | GCStipple | GCFillStyle), &gcv);
	gcv.foreground= c->xc_bg;
        gcv.background= c->xc_fg;
	gcv.plane_mask= AllPlanes;
        gcv.function= GXcopy;
        c->xc_cgc= XCreateGC(c->xc_dpy, c->xc_cwin, (GCForeground | GCBackground | GCPlaneMask | GCFunction), &gcv);

	c->xc_next= xc_canvases;
	xc_canvases= c;

	depth= DefaultDepth(c->xc_dpy, screen);
	switch (c->xc_mode) {
	case LXC_STATIC:
		if ((c->xc_cpm= XCreatePixmap(c->xc_dpy, DefaultRootWindow(c->xc_dpy), c->xc_vcw, c->xc_vch, depth)) == None) {
			(void) fprintf(stderr, "canvas_create: memory allocation error\n");
			(void) canvas_destroy(c);
			return((Canvas *) NULL);
		}
		XFillRectangle(c->xc_dpy, c->xc_cpm, c->xc_cgc, 0, 0, c->xc_vcw, c->xc_vch);
		break;
	case LXC_DYNAMICCLR:
	case LXC_DYNAMICPRS:
		/* allocate backing store in canvas_config()
		   when the canvas window size is known */
		break;
	default:
		break;
	}

	switch (xwa.map_state) {
	case IsUnviewable:
	case IsViewable:
		c->xc_flags|= LXC_FRAMEMAPPED;
		(void) canvas_config(c);
		c->xc_flags|= LXC_CONFIGDONE;
		XMapWindow(c->xc_dpy, c->xc_cwin);
		canvas_clear(c);
		canvas_flush(c);
		if (c->xc_flags & LXC_VSCROLLVIS) {
			XMapWindow(c->xc_dpy, c->xc_vswin);
			canvasvscroll_draw(c);
		}
		if (c->xc_flags & LXC_HSCROLLVIS) {
			XMapWindow(c->xc_dpy, c->xc_hswin);
			canvashscroll_draw(c);
		}
		break;
	case IsUnmapped:
		c->xc_flags&= ~LXC_FRAMEMAPPED;
		break;
	default:
		break;
	}
	return(c);
}

int
canvas_filldefaults(name, c)
char *name;
Canvas *c;
{
	char *xdef;
	int screen;
	void canvas_cw_ex_proc();

	c->xc_vx= c->xc_vy= 0;
	c->xc_vcw= c->xc_vch= LXCDEF_SIZE;

	/* internal border width */
	if ((xdef= XGetDefault(c->xc_dpy, name, "LXTInternalBorderWidth")) == (char *) NULL)
		c->xc_ibw= 0;
	else {
		c->xc_ibw= atoi(xdef);
		if (c->xc_ibw < 0)
			c->xc_ibw= 0;
	}

	/* foreground and background */
	screen= DefaultScreen(c->xc_dpy);
	c->xc_fg= BlackPixel(c->xc_dpy, screen);
	c->xc_bg= WhitePixel(c->xc_dpy, screen);

	/* expose proc */
	c->xc_exproc= canvas_cw_ex_proc;

	/* virtual canvas will not shrink or expand to fit enclosing window */
	c->xc_mode= LXC_STATIC;

	/* scrollbars */
	if ((c->xc_vscroll= (Scrollbar *) calloc(1, sizeof(Scrollbar))) == (Scrollbar *) NULL) {
		(void) fprintf(stderr, "canvas_filldefaults: memory allocation error\n");
		return(LX_ERROR);
	}
	c->xc_vscroll->xs_type= LXS_VERTICAL;
	c->xc_vscroll->xs_barwidth= LXSDEF_BARWIDTH;
	c->xc_vscroll->xs_bubblemargin= LXSDEF_BUBBLEMARGIN;
	c->xc_vscroll->xs_buttonlen= LXSDEF_BUTTONLEN;
	if ((c->xc_hscroll= (Scrollbar *) calloc(1, sizeof(Scrollbar))) == (Scrollbar *) NULL) {
		(void) fprintf(stderr, "canvas_filldefaults: memory allocation error\n");
		return(LX_ERROR);
	}
	c->xc_hscroll->xs_type= LXS_HORIZONTAL;
	c->xc_hscroll->xs_barwidth= LXSDEF_BARWIDTH;
	c->xc_hscroll->xs_bubblemargin= LXSDEF_BUBBLEMARGIN;
	c->xc_hscroll->xs_buttonlen= LXSDEF_BUTTONLEN;

	/* flags */
	c->xc_flags= LXC_CANVASVISIBLE | LXC_CANVASACTIVE;

	/* null fields */
	c->xc_cpm= c->xc_vspm= c->xc_hspm= None;
	c->xc_ccursor= c->xc_vscursor= c->xc_hscursor= None;
	c->xc_next= (Canvas *) NULL;
	return(LX_SUCCESS);
}

void
canvas_init(dpy)
Display *dpy;
{
	void cursor_init();

	/* create cursor pixmaps */
	if (!xvcursor_init)
		cursor_init(dpy);

	xc_usrresize= FALSE;
	xc_init= TRUE;
}

/*VARARGS*/
int
canvas_set(va_alist)
/*
   User-callable.
   Changes an attribute of a previously created canvas.
*/
va_dcl
{
	va_list varg_ptr;
	int nargs, attr, depth;
	Window root;
	int ox, oy;
	unsigned int ocw, och, obw, odep;
	boolean redisplay, reconfig, undisplay;
	boolean set_size, visible;
	Canvas *c;
	Pixmap opm;
	int canvas_config();
	void canvasvscroll_draw(), canvashscroll_draw();
	void canvas_unmapsubwins();

	redisplay= reconfig= undisplay= FALSE;
	set_size= FALSE;

	va_start(varg_ptr);
	for (nargs= 0;; nargs++) {

		if (nargs == 0) {
			c= va_arg(varg_ptr, Canvas *);
			if (c == (Canvas *) NULL) {
				(void) fprintf(stderr, "canvas_set: null canvas\n");
				return(LX_ERROR);
			}
			if (c->xc_magic != LX_CANVAS) {
				(void) fprintf(stderr, "canvas_set: object is not a canvas\n");
				return(LX_ERROR);
			}
			if ((c->xc_flags & LXC_CANVASVISIBLE) && (c->xc_flags & LXC_FRAMEMAPPED))
				visible= TRUE;
			else
				visible= FALSE;
		}

		else {
			int n;

			attr= va_arg(varg_ptr, int);
			if (attr == LXC_NULL)
				break;

			switch (attr) {
			case LXC_FOREGROUND:
				c->xc_fg= va_arg(varg_ptr, unsigned long);
				XSetForeground(c->xc_dpy, c->xc_gc, c->xc_fg);
				XSetBackground(c->xc_dpy, c->xc_cgc, c->xc_fg);
				XSetForeground(c->xc_dpy, c->xc_sgc, c->xc_fg);
				if (visible)
					redisplay= TRUE;
				break;
			case LXC_BACKGROUND:
				c->xc_bg= va_arg(varg_ptr, unsigned long);
				XSetBackground(c->xc_dpy, c->xc_gc, c->xc_bg);
				XSetForeground(c->xc_dpy, c->xc_cgc, c->xc_bg);
				XSetBackground(c->xc_dpy, c->xc_sgc, c->xc_bg);
				if (visible)
					redisplay= TRUE;
				break;
			case LXC_WIDTH:
				n= va_arg(varg_ptr, int);
				if (c->xc_mode != LXC_STATIC)
					(void) fprintf(stderr, "canvas_create: cannot set size of dynamic canvas\n");
				else {
					c->xc_vcw= n;
					if (visible) {
						reconfig= TRUE;
						redisplay= TRUE;
					}
					set_size= TRUE;
				}
				break;
			case LXC_HEIGHT:
				n= va_arg(varg_ptr, int);
				if (c->xc_mode != LXC_STATIC)
					(void) fprintf(stderr, "canvas_create: cannot set size of dynamic canvas\n");
				else {
					c->xc_vch= n;
					if (visible) {
						reconfig= TRUE;
						redisplay= TRUE;
					}
					set_size= TRUE;
				}
				break;
			case LXC_BARWIDTH:
				c->xc_vscroll->xs_barwidth= c->xc_hscroll->xs_barwidth= va_arg(varg_ptr, int);
				if (visible) {
					reconfig= TRUE;
					redisplay= TRUE;
				}
				break;
			case LXC_BUBBLEMARGIN:
				c->xc_vscroll->xs_bubblemargin= c->xc_hscroll->xs_bubblemargin= va_arg(varg_ptr, int);
				if (visible) {
					reconfig= TRUE;
					redisplay= TRUE;
				}
				break;
			case LXC_BUTTONLEN:
				c->xc_vscroll->xs_buttonlen= c->xc_hscroll->xs_buttonlen= va_arg(varg_ptr, int);
				if (visible) {
					reconfig= TRUE;
					redisplay= TRUE;
				}
				break;
			case LXC_CCURSOR:
				c->xc_ccursor= va_arg(varg_ptr, Cursor);
				XDefineCursor(c->xc_dpy, c->xc_cwin, c->xc_ccursor);
				break;
			case LXC_VSCURSOR:
				c->xc_vscursor= va_arg(varg_ptr, Cursor);
				XDefineCursor(c->xc_dpy, c->xc_vswin, c->xc_vscursor);
				break;
			case LXC_HSCURSOR:
				c->xc_hscursor= va_arg(varg_ptr, Cursor);
				XDefineCursor(c->xc_dpy, c->xc_hswin, c->xc_hscursor);
				break;
			case LXC_MODE:
				n= va_arg(varg_ptr, int);
				if (n != c->xc_mode) {
					switch (n) {
					case LXC_STATIC:
						c->xc_mode= n;
						if (visible) {
							reconfig= TRUE;
							redisplay= TRUE;
						}
						break;
					case LXC_DYNAMICCLR:
					case LXC_DYNAMICPRS:
						if (set_size) {
							(void) fprintf(stderr, "canvas_create: size request incompatible with dynamic mode -- setting mode to LXC_STATIC\n");
							c->xc_mode= LXC_STATIC;
							if (visible) {
								reconfig= TRUE;
								redisplay= TRUE;
							}
						}
						else {
							if ((c->xc_mode == LXC_STATIC) && visible) {
								reconfig= TRUE;
								redisplay= TRUE;
							}
							c->xc_mode= n;
						}
						break;
					default:
						(void) fprintf(stderr, "canvas_set: unrecognized mode\n");
						break;
					}
				}
				break;
			case LXC_VISIBLE:
				n= va_arg(varg_ptr, int);
				if (n) {
					if (!(c->xc_flags & LXC_CANVASVISIBLE)) {
						c->xc_flags|= LXC_CANVASVISIBLE;
						reconfig= TRUE;
						redisplay= TRUE;
					}
				}
				else {
					if (c->xc_flags & LXC_CANVASVISIBLE) {
						c->xc_flags&= ~LXC_CANVASVISIBLE;
						undisplay= TRUE;
					}
				}
				break;
			case LXC_BATCH:
				n= va_arg(varg_ptr, int);
				switch (n) {
				case LXC_BATCHON:
					c->xc_flags|= LXC_BATCHSET;
					break;
				case LXC_BATCHOFF:
					c->xc_flags&= ~LXC_BATCHSET;
					break;
				default:
					(void) fprintf(stderr, "canvas_set: unrecognized batch mode\n");
					break;
				}
				break;
			case LXC_EXPOSEPROC:
				c->xc_exproc= (void (*)()) va_arg(varg_ptr, char *);
				break;
			case LXC_ACTIVE:
				n= va_arg(varg_ptr, int);
				if (n) {
					c->xc_flags|= LXC_CANVASACTIVE;
					XSelectInput(c->xc_dpy, c->xc_vswin, (ExposureMask | ButtonPressMask));
					XSelectInput(c->xc_dpy, c->xc_hswin, (ExposureMask | ButtonPressMask));
				}
				else {
					c->xc_flags&= ~LXC_CANVASACTIVE;
					XSelectInput(c->xc_dpy, c->xc_cwin, ExposureMask);
					XSelectInput(c->xc_dpy, c->xc_vswin, ExposureMask);
					XSelectInput(c->xc_dpy, c->xc_hswin, ExposureMask);
				}
				break;
			case LXC_WINDOW:
			case LXC_PIXMAP:
			case LXC_XOFFSET:
			case LXC_YOFFSET:
			case LXC_VSCROLL:
			case LXC_HSCROLL:
			case LXC_DWIDTH:
			case LXC_DHEIGHT:
				(void) fprintf(stderr, "canvas_set: attribute is query-only\n");
				break;
			default:
				(void) fprintf(stderr, "canvas_set: ignoring unrecognized attribute\n");
				break;
			}
		}
	}
	va_end(varg_ptr);

	if (undisplay) {
#ifdef PRE_R3
		/* this causes a server crash under X.V11R3/SunOS3.4! */
		XUnmapSubwindows(c->xc_dpy, c->xc_win);
#else
		canvas_unmapsubwins(c);
#endif PRE_R3
		return(LX_SUCCESS);
	}

	if (reconfig) {
		if (visible)
#ifdef PRE_R3
			/* this causes a server crash under X.V11R3/SunOS3.4! */
			XUnmapSubwindows(c->xc_dpy, c->xc_win);
#else
			canvas_unmapsubwins(c);
#endif PRE_R3
		if (c->xc_mode == LXC_STATIC) {
			opm= c->xc_cpm;
			depth= DefaultDepth(c->xc_dpy, DefaultScreen(c->xc_dpy));
			if (opm != None) {

				/* allocate new backing store only
				   if the canvas has changed size */
				XGetGeometry(c->xc_dpy, c->xc_cpm, &root, &ox, &oy, &ocw, &och, &obw, &odep);
				if (((int) ocw != c->xc_vcw) || ((int) och != c->xc_vch)) {
					if ((c->xc_cpm= XCreatePixmap(c->xc_dpy, DefaultRootWindow(c->xc_dpy), c->xc_vcw, c->xc_vch, depth)) == None) {
						(void) fprintf(stderr, "canvas_set: memory allocation error\n");
						return(LX_ERROR);
					}
					XFillRectangle(c->xc_dpy, c->xc_cpm, c->xc_cgc, 0, 0, c->xc_vcw, c->xc_vch);
					if ((int) ocw > c->xc_vcw)
						ocw= (unsigned int) c->xc_vcw;
					if ((int) och > c->xc_vch)
						och= (unsigned int) c->xc_vch;
					XCopyArea(c->xc_dpy, opm, c->xc_cpm, c->xc_gc, 0, 0, (int) ocw, (int) och, 0, 0);
					XFreePixmap(c->xc_dpy, opm);
				}
			}
			else {
				if ((c->xc_cpm= XCreatePixmap(c->xc_dpy, DefaultRootWindow(c->xc_dpy), c->xc_vcw, c->xc_vch, depth)) == None) {
					(void) fprintf(stderr, "canvas_set: memory allocation error\n");
					return(LX_ERROR);
				}
				XFillRectangle(c->xc_dpy, c->xc_cpm, c->xc_cgc, 0, 0, c->xc_vcw, c->xc_vch);
			}
		}
		(void) canvas_config(c);
		c->xc_flags|= LXC_CONFIGDONE;
	}

	/* canvas not yet visible, but backing store must be resized */
	else if (set_size && (c->xc_mode == LXC_STATIC)) {
		opm= c->xc_cpm;
		depth= DefaultDepth(c->xc_dpy, DefaultScreen(c->xc_dpy));
		if (opm != None) {

			/* allocate new backing store only
			   if the canvas has changed size */
			XGetGeometry(c->xc_dpy, c->xc_cpm, &root, &ox, &oy, &ocw, &och, &obw, &odep);
			if (((int) ocw != c->xc_vcw) || ((int) och != c->xc_vch)) {
				if ((c->xc_cpm= XCreatePixmap(c->xc_dpy, DefaultRootWindow(c->xc_dpy), c->xc_vcw, c->xc_vch, depth)) == None) {
					(void) fprintf(stderr, "canvas_set: memory allocation error\n");
					return(LX_ERROR);
				}
				XFillRectangle(c->xc_dpy, c->xc_cpm, c->xc_cgc, 0, 0, c->xc_vcw, c->xc_vch);
				if ((int) ocw > c->xc_vcw)
					ocw= (unsigned int) c->xc_vcw;
				if ((int) och > c->xc_vch)
					och= (unsigned int) c->xc_vch;
				XCopyArea(c->xc_dpy, opm, c->xc_cpm, c->xc_gc, 0, 0, (int) ocw, (int) och, 0, 0);
				XFreePixmap(c->xc_dpy, opm);
			}
		}
		else {
			if ((c->xc_cpm= XCreatePixmap(c->xc_dpy, DefaultRootWindow(c->xc_dpy), c->xc_vcw, c->xc_vch, depth)) == None) {
				(void) fprintf(stderr, "canvas_set: memory allocation error\n");
				return(LX_ERROR);
			}
			XFillRectangle(c->xc_dpy, c->xc_cpm, c->xc_cgc, 0, 0, c->xc_vcw, c->xc_vch);
		}
	}

	if (redisplay) {
		XMapWindow(c->xc_dpy, c->xc_cwin);
		canvas_flush(c);
		if (c->xc_flags & LXC_VSCROLLVIS) {
			XMapWindow(c->xc_dpy, c->xc_vswin);
			canvasvscroll_draw(c);
		}
		if (c->xc_flags & LXC_HSCROLLVIS) {
			XMapWindow(c->xc_dpy, c->xc_hswin);
			canvashscroll_draw(c);
		}
	}

	return(LX_SUCCESS);
}

int
canvas_destroy(c)
/*
   User-callable.
   Destroys a canvas.
*/
Canvas *c;
{
	Canvas *xc, *yc;
	void canvas_unmapsubwins();

	if (c == (Canvas *) NULL) {
		(void) fprintf(stderr, "canvas_destroy: null canvas\n");
		return(LX_ERROR);
	}
	if (c->xc_magic != LX_CANVAS) {
		(void) fprintf(stderr, "canvas_destroy: object is not a canvas\n");
		return(LX_ERROR);
	}

	/* unlink canvas from chain */
	if (c == xc_canvases)
		xc_canvases= xc_canvases->xc_next;
	else {
		for (xc= yc= xc_canvases; xc != (Canvas *) NULL; yc= xc, xc= xc->xc_next) {
			if (xc == c) {
				yc->xc_next= xc->xc_next;
				break;
			}
		}
		if (xc == (Canvas *) NULL) {
			(void) fprintf(stderr, "canvas_destroy: cannot locate canvas\n");
			return(LX_ERROR);
		}
	}

#ifdef PRE_R3
	/* this causes a server crash under X.V11R3/SunOS3.4! */
	XUnmapSubwindows(c->xc_dpy, c->xc_win);
#else
	canvas_unmapsubwins(c);
#endif PRE_R3
	XDestroySubwindows(c->xc_dpy, c->xc_win);

	if (c->xc_cpm != None)
		XFreePixmap(c->xc_dpy, c->xc_cpm);
	if (c->xc_vspm != None)
		XFreePixmap(c->xc_dpy, c->xc_vspm);
	if (c->xc_hspm != None)
		XFreePixmap(c->xc_dpy, c->xc_hspm);
	if (c->xc_stipplepm != None)
		XFreePixmap(c->xc_dpy, c->xc_stipplepm);

	if (c->xc_gc != None)
		XFreeGC(c->xc_dpy, c->xc_gc);
	if (c->xc_cgc != None)
		XFreeGC(c->xc_dpy, c->xc_cgc);
	if (c->xc_sgc != None)
		XFreeGC(c->xc_dpy, c->xc_sgc);

	if (c->xc_vscroll != (Scrollbar *) NULL)
		cfree((char *) c->xc_vscroll);
	if (c->xc_hscroll != (Scrollbar *) NULL)
		cfree((char *) c->xc_hscroll);

	cfree((char *) c);
	return(LX_SUCCESS);
}

Void *
canvas_get(c, attr)
/*
   User-callable routine.
   Returns a pointer to the value of the
   supplied attribute or to the attribute itself.
*/
Canvas *c;
int attr;
{
	xcret_void= (Void *) NULL;
	if (c == (Canvas *) NULL) {
		(void) fprintf(stderr, "canvasitem_get: null canvas\n");
		return(xcret_void);
	}
	if (c->xc_magic != LX_CANVAS) {
		(void) fprintf(stderr, "canvas_get: object is not a canvas\n");
		return(xcret_void);
	}
	if (attr == LXPI_NULL)
		return(xcret_void);

	switch (attr) {

	case LXC_FOREGROUND:
		xcret_void= (Void *) &(c->xc_fg);
		break;
	case LXC_BACKGROUND:
		xcret_void= (Void *) &(c->xc_bg);
		break;
	case LXC_WIDTH:
		xcret_void= (Void *) &(c->xc_vcw);
		break;
	case LXC_HEIGHT:
		xcret_void= (Void *) &(c->xc_vch);
		break;
	case LXC_BARWIDTH:
		xcret_void= (Void *) &(c->xc_vscroll->xs_barwidth);
		break;
	case LXC_BUBBLEMARGIN:
		xcret_void= (Void *) &(c->xc_vscroll->xs_bubblemargin);
		break;
	case LXC_BUTTONLEN:
		xcret_void= (Void *) &(c->xc_vscroll->xs_buttonlen);
		break;
	case LXC_CCURSOR:
		xcret_void= (Void *) &(c->xc_ccursor);
		break;
	case LXC_VSCURSOR:
		xcret_void= (Void *) &(c->xc_vscursor);
		break;
	case LXC_HSCURSOR:
		xcret_void= (Void *) &(c->xc_hscursor);
		break;
	case LXC_MODE:
		xcret_void= (Void *) &(c->xc_mode);
		break;
	case LXC_VISIBLE:
		if (c->xc_flags & LXC_CANVASVISIBLE)
			xcret_bool= TRUE;
		else
			xcret_bool= FALSE;
		xcret_void= (Void *) &xcret_bool;
		break;
	case LXC_BATCH:
		if (c->xc_flags & LXC_BATCHSET)
			xcret_bool= TRUE;
		else
			xcret_bool= FALSE;
		xcret_void= (Void *) &xcret_bool;
		break;
	case LXC_EXPOSEPROC:
		xcret_void= (Void *) c->xc_exproc;
	case LXC_ACTIVE:
		if (c->xc_flags & LXC_CANVASACTIVE)
			xcret_bool= TRUE;
		else
			xcret_bool= FALSE;
		xcret_void= (Void *) &xcret_bool;
		break;
	case LXC_WINDOW:
		xcret_void= (Void *) &(c->xc_cwin);
		break;
	case LXC_PIXMAP:
		xcret_void= (Void *) &(c->xc_cpm);
		break;
	case LXC_XOFFSET:
		xcret_void= (Void *) &(c->xc_vx);
		break;
	case LXC_YOFFSET:
		xcret_void= (Void *) &(c->xc_vy);
		break;
	case LXC_VSCROLL:
		if (c->xc_flags & LXC_VSCROLLVIS)
			xcret_bool= TRUE;
		else
			xcret_bool= FALSE;
		xcret_void= (Void *) &xcret_bool;
		break;
	case LXC_HSCROLL:
		if (c->xc_flags & LXC_HSCROLLVIS)
			xcret_bool= TRUE;
		else
			xcret_bool= FALSE;
		xcret_void= (Void *) &xcret_bool;
		break;
	case LXC_DWIDTH:
		xcret_void= (Void *) &(c->xc_acw);
		break;
	case LXC_DHEIGHT:
		xcret_void= (Void *) &(c->xc_ach);
		break;
	default:
		(void) fprintf(stderr, "canvas_get: unrecognized attribute\n");
		break;
	}
	return(xcret_void);
}
