/****************************************************************************** 
 *
 *  mixview - X Window System based soundfile editor and processor
 *
 *  Copyright 1989 by Douglas Scott
 *
 *  Author:     Douglas Scott 
 *  Date:       May 1, 1989
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The author makes no representations about
 *  the suitability of this software for any purpose.  
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
#include "main.h"
#include "soundfile.h"

/* Find location in header for the coded information.  
   Return a pointer to the beginning of the SFCODE structure not
   to the information structure itself. The srate, number of channels, magic
   number, number of bytes per channel are NOT coded via these routines. */

char *
getsfcode(hd,code)
	SFHEADER *hd;
	int code;
{
	char *sfc;
	SFCODE *sp;
	char *hdend = (char *) hd + sizeof(SFHEADER);

	sfc = &hd->sfinfo.sf_codes; 
	while(sfc < hdend) {
		sp = (SFCODE *) sfc;
		if(sp->code == SF_END)
			break;
		/* Catch possible wrap around on stack from bad header */
		/* or a zero struct size from bad header */
		if(sp->bsize <= 0 || sfc + sp->bsize < &hd->sfinfo.sf_codes)
			break;
		if(sp->code == code)
			return(sfc);
		sfc += sp->bsize;
	}
	return(NULL);
}

static SFCODE endcode = {
	SF_END,
	sizeof(SFCODE)
} ;

putsfcode(hd,ptr,codeptr)
	SFHEADER *hd;
	char *ptr;
	SFCODE *codeptr;
{
	char *sfc;
	SFCODE *sp = 0;
	int wasendcode = 0;
	char *hdend = (char *) hd + sizeof(SFHEADER);

	sfc = &hd->sfinfo.sf_codes; 
	while(sfc < hdend) {
		sp = (SFCODE *) sfc;
		if(sp->code == SF_END) {
			wasendcode = 1;
			break;
		}
		/* Catch possible wrap around on stack from bad header */
		if(sp->bsize <= 0 || sfc + sp->bsize < (char *) hd) {
			sp->code = SF_END; /* Force an end */
			sp->bsize = sizeof(SFCODE);
			break;
		}
		if(sp->code == codeptr->code)
			break;
		sfc += sp->bsize;
	}
	
	/* No space left */
	if(sfc + codeptr->bsize > hdend)
		return(-1);
#if 0
	if(!wasendcode)  /* Not a new one */
		if(codeptr->bsize != sp->bsize) /* Size changed */
			return(-1);
#endif

	/* copy code struct into header starting at sfinfo.sf_codes */
	/* this line is trashing the pointer to the header for some reason */
	bcopy((char *) codeptr, sfc, sizeof(SFCODE));
	/* now copy actual data struct into header right after that */
	bcopy(ptr, sfc + sizeof(SFCODE), codeptr->bsize - sizeof(SFCODE));

	/* now stick an end-code struct on to mark the end */
	if(wasendcode) 
		bcopy((char *) &endcode,sfc + codeptr->bsize,sizeof(endcode));

	return(0);
}

headerSize(header)
SFHEADER *header;
{
#if NeXT_STYLE_HEADER
	if(header->sfinfo.sf_magic == 0)
		return(header->sfinfo.lheader.data_loc);
	else return(SIZEOF_BSD_HEADER);
#else
	return(SIZEOF_BSD_HEADER);
#endif
}

#if NeXT_STYLE_HEADER

extern sf_struct *v;

/* all this remaining code is for NeXT/BSD soundfile header compatibility */

writeHeader(sf,header)
SFHEADER  *header;
int sf;
{
	long headsize;
	(header)->sfinfo.lheader.magic = HEAD_MAGIC;
	if((header)->sfinfo.sf_packmode == SF_FLOAT) 
		(header)->sfinfo.lheader.format = FLOAT_FORMAT;
	else if((header)->sfinfo.sf_packmode == SF_SHORT) 
		(header)->sfinfo.lheader.format = SHORT_FORMAT;
	else (header)->sfinfo.lheader.format = MULAW_FORMAT;
	(header)->sfinfo.lheader.samp_rate = (header)->sfinfo.sf_srate;
	(header)->sfinfo.lheader.nchans = (header)->sfinfo.sf_chans;
	/* if writing hybrid header, set data offset to end of new header */
	if(header->sfinfo.sf_magic == SF_MAGIC)
		(header)->sfinfo.lheader.data_loc = SIZEOF_BSD_HEADER;
	else { /* set first to end of struct */
		(header)->sfinfo.lheader.data_loc = HEADER_SIZE;
		addComment(header);	/* then load comment if there is one */
	}
	headsize = (header)->sfinfo.lheader.data_loc;
	/* added to complete NeXT/sparc header information */
	(header)->sfinfo.lheader.data_size = v->bufsize;
	if(write(sf, header, headsize) != headsize)
		return(-1);
	else return(1);
}

static 
SFCODE ampcode = {
	SF_MAXAMP,
	sizeof(SFMAXAMP) + sizeof(SFCODE)
};
static
SFCODE commentcode = {
	SF_COMMENT, 
	MINCOMM + sizeof(SFCODE)
};
static SFMAXAMP sfmnew;
static SFCOMMENT sfcm;

int
readHeader(sf,header)
SFHEADER *header;
int sf;
{
	int i;
	char *pointer;
	/* read in 1kb first, to see if hybrid header is present */
	if((i=read(sf,(char *) header,SIZEOF_BSD_HEADER)) != SIZEOF_BSD_HEADER)
		return(-1);
	/* if magic number is not correct for next/sparc, we cant read it */
	if((header)->sfinfo.lheader.magic != HEAD_MAGIC)
		return(-1);
	/* if hybrid header is there, we are done here */
	if((header)->sfinfo.sf_magic == SF_MAGIC)
		return(1);
	/* if file being read is native NeXT or sparc soundfile... */
	/* extract any information between header and sound data */
	/* if there is any, and zero out afterwards */
	if(moveComment(header, &sfcm, &commentcode)) {
		/* load blank peakamp code first */
		for(i=0; i<SF_MAXCHAN; i++) {
			sfmnew.value[i] = 0.0;
			sfmnew.samploc[i] = 0;
		}
		sfmnew.timetag = 0;
		/* write peakamp into bsd portion of header */
		if(putsfcode(header, (char *) &sfmnew, &ampcode) < 0) {
			mv_alert("Unable to load maxamp code into bsd header.");
			return(-1);
		}
		/* now write comment into bsd portion of header */
		if(putsfcode(header, (char *) &sfcm, &commentcode) < 0) {
			mv_alert("Unable to transfer comment into bsd header.");
			return(-1);
		}
	}
	/* if not, zero out bytes after first header */
	else {
		pointer = (char *)header;
		for(i=HEADER_SIZE+1; i<SIZEOF_BSD_HEADER; i++) *(pointer+i) = 0;
		/* and load blank peakamp code */
		for(i=0; i<SF_MAXCHAN; i++) {
			sfmnew.value[i] = 0.0;
			sfmnew.samploc[i] = 0;
		}
		sfmnew.timetag = 0;
		if(putsfcode(header, (char *) &sfmnew, &ampcode) < 0) {
			mv_alert("Unable to load maxamp code into new header.");
			return(-1);
		}
	}
	/* set file pointer to beginning of sound data */
	if(lseek(sf, (long) header->sfinfo.lheader.data_loc, L_SET) < 0) {
		mv_error(errno, "Unable to reset file pointer.");
		return(-1);
	}
	/* load all values from native header into hybrid header */
	(header)->sfinfo.sf_magic = 0; /* so we know it is native NeXT file */
	(header)->sfinfo.sf_srate = (header)->sfinfo.lheader.samp_rate;
	(header)->sfinfo.sf_chans = (header)->sfinfo.lheader.nchans;
	switch(header->sfinfo.lheader.format) {
	case FLOAT_FORMAT:
		(header)->sfinfo.sf_packmode = SF_FLOAT;
		break;
	case SHORT_FORMAT:
		(header)->sfinfo.sf_packmode = SF_SHORT;
		break;
	case MULAW_FORMAT:
		(header)->sfinfo.sf_packmode = SF_CODEC;
		break;
	default:
		mv_alert("Unknown soundfile data type for mixview.");
		return(-1);
	}
	return(1);
}

int
isNative(hd)
	SFHEADER *hd;
{
	return(hd->sfinfo.sf_magic == 0);
}

int
changeFileType(hd) /* switches type from native to hybrid and vice versa */
	SFHEADER *hd;
{
	hd->sfinfo.sf_magic = (hd->sfinfo.sf_magic) ? 0 : SF_MAGIC;
}

addComment(hd) /* transfers text information from bsd to local header */
	SFHEADER *hd;
{
	char string[MAXCOMM];
	int len;
	(void) get_comment(string, &len);
	if(len) {
		char *pt = (char *) hd + HEADER_SIZE - INFO_SIZE;
		bcopy(string, pt, len);
		hd->sfinfo.lheader.data_loc += (len-INFO_SIZE);
	}
}

moveComment(hd, cm, code) /* transfers info from local header to bsd */
	SFHEADER *hd;
	SFCOMMENT *cm;
	SFCODE *code;
{
	int i, len=0, dataloc = hd->sfinfo.lheader.data_loc;
	char *comstring = (char *) hd + HEADER_SIZE - INFO_SIZE;
	char *csptr = comstring;
	code->bsize = MINCOMM + sizeof(SFCODE);	/* default */
	/* if no extra info present */
	if(dataloc == sizeof(LOCALHEADER))
		return 0;
	if(dataloc < sizeof(LOCALHEADER))
		mv_alert("Warning:  possibly corrupted file header!");
	/* otherwise */
	len = strlen(comstring) + 1;
	if(len > MAXCOMM)
		mv_alert("Comment was too long and has been truncated.");
	for(i=0; i < len && i < MAXCOMM-1; i++)
		cm->comment[i] = *csptr++;
	cm->comment[i] = '\0';
	/* now zero out the struct from the comment to the end */
	/* show this mistake to Roger!! */
	bzero(comstring, SIZEOF_BSD_HEADER - sizeof(hd->sfinfo));
	if(len > MINCOMM) code->bsize= len + sizeof(SFCODE);
	return 1;
}

#else /* not NeXT or SPARC */

int
readHeader(sf, header)
	SFHEADER  *header;
	int sf;
{
	if(read(sf,(char *) header, sizeof(SFHEADER)) == sizeof(SFHEADER))
		return 1;
	else return -1; 
}

int writeHeader(sf, header)
	SFHEADER  *header;
	int sf;
{
	if(write(sf,(char *) header, sizeof(SFHEADER)) == sizeof(SFHEADER))
		return 1;
	else return -1; 
}

#endif
