/*
	screen.c
*/

#include "main.h"
#include "dialog.h"
#include "screen.h"
#include "databuff.h"
#include "datawin.h"
#include "nobug.h"

#define MAXSCREENS 16
#define SCRNPIX_WIDTH 1024	/* size of scrolling canvas */

extern Display *dpy;
extern unsigned long myfg, mybg;
extern XFontStruct *canvas_fs,  *scale_fs;
extern Cursor wait_curs, main_curs;
extern Cursor arrow_curs;

/*			screen object functions			*/

int nscreens = 0;
Mv_screen *screenlist[MAXSCREENS];
extern Mv_scale *new_mvscale();

static
char *screen_cmds[]= {
        "Close", "Set vert. scale", "Set horiz. scale", "Edit", 
	"Save changes", "Save to new", "Read file", 0
};

void dw_close_proc(), dw_vscale_set_proc(), dw_hscale_set_proc();
void dw_edit_proc(), dw_file_save_proc();

static
void (*screen_procs[])() = {
        dw_close_proc, dw_vscale_set_proc, dw_hscale_set_proc, 
	dw_edit_proc, dw_file_save_proc, 0, 0, 0
};

static
int screen_flags[] = {
        0, 0, 0, 0, 0, 0, 0, 0
};

Mv_screen *
new_screen(parent, width, height, xloc, yloc) 
	void *parent;
	unsigned width, height;
	int xloc, yloc;
{
	Mv_screen *s;
	int vscale_width = SCALE_WIDTH, i;
	static long r_mask= ExposureMask|StructureNotifyMask|PropertyChangeMask;
	Menu *screen_menu;
	if(nscreens == (MAXSCREENS-1)) {
		mv_alert("Maximum number of screens exceeded.");
		return (Mv_screen *) NULL;
	}
	s = (Mv_screen *) mv_alloc(sizeof(Mv_screen));
/* functions */
	s->parent = parent;	/* object contains a pointer to its owner */
	s->screenEvent = sc_screenEvent;
	s->buttonPress = sc_buttonPress;
	s->buttonMove = sc_buttonMove;
	s->buttonRelease = sc_buttonRelease;
	s->frameResize = sc_frameResize;
	s->keyPress = sc_keyPress;
	s->quantize = sc_quantize;
	s->getPix = sc_getPix;
	s->newCanvas = sc_newCanvas;
	s->setClipFrames = sc_setClipFrames;
	s->getOffsets = sc_getOffsets;
	s->vLine = sc_vLine;
	s->hLine = sc_hLine;
	s->rectangle = sc_rectangle;
	s->string = sc_string;
	s->xFormat = sc_xFormat;
	s->plotData = sc_plotData;
	s->drawLines = sc_drawLines;
	s->plotLines = sc_plotLines;
	s->plotSegments = sc_plotSegments;
	s->clear = sc_clear;
	s->plot = sc_plot;
	s->vertHair = sc_vertHair;
	s->horizHair = sc_horizHair;
	s->printLoc = sc_printLoc;
	s->plotHorizScale = sc_plotHorizScale;
	s->box = sc_box;
	s->noBox = sc_noBox;
	s->screenWidth = sc_screenWidth;
	s->screenHeight = sc_screenHeight;
	s->frameWidth = sc_frameWidth;
	s->frameHeight = sc_frameHeight;
	s->showMenu = sc_showMenu;
	s->setMenu = sc_setMenu;
	s->setCursor = sc_setCursor;
	s->setScaleTitle = sclst_setLabels;			/* inherited */
	s->setVertMargine = sc_setVertMargine;
	s->setVertScale = sc_setVertScale;
	s->getVertScale = sc_getVertScale;
	s->setHairLabel = sc_setHairLabel;
	s->setScreenFont = sc_setScreenFont;
	s->setHorizScale = sc_setHorizScale;
	s->getHorizScale = sc_getHorizScale;
	s->setHorizRange = sc_setHorizRange;
	s->getHorizRange = sc_getHorizRange;
	s->setHorizGrain = sc_setHorizGrain;
	s->setChannels = sc_setChannels;
	s->getScanChannels = sc_getScanChannels;
	s->setScanChannels = sc_setScanChannels;
	s->nChannels = sc_nChannels;
 	s->baseLocation = sc_baseLocation;
 	s->scaleFactor = sc_scaleFactor;
	s->display = sc_display;
	s->block = sc_block;
	s->unBlock = sc_unBlock;
	s->resize = sc_resize;
	s->destroy = sc_destroy;
/* now set some defaults, etc. */
	s->nchannels = 1;
	s->chanfirst = 0;
	s->chanlast = 0;
	s->pr_len = 7;
	s->sc_len = 4;
	s->tr_len = 2;
	s->hmarg = 0;
	s->npoints = 0;
	s->maxpoints = 0;
	s->hminval = 0;
	s->hmaxval = 0;
	s->horiz_scale = s->horiz_grain = 1.0;
	for(i=0; i<MAXDCHANS; i++) {
		s->maxval[i] = 1.0;
		s->minval[i] = -1.0;
	}
	s->dpy = dpy;
/* create internal objects, etc */
	s->frame = new_window(((Data_win *)(parent))->frame, width, height, xloc, yloc, 1, 0);
	s->cframe = new_window(s->frame, width-vscale_width, height, 
		vscale_width, 0, 0, 0);
	s->framewidth = width;
	s->frameheight = height;
	/* Select which events the windows are going to acknowledge */
	XSelectInput(s->dpy, s->frame, r_mask);
	XSelectInput(s->dpy, s->cframe, r_mask);
	/* this is offset by 11 due to scrollbar at top of screen */
	s->vscale= new_mvscale(s->frame, vscale_width, height-11, (long) 0, 11);
	s->vscale->addScale(s->vscale);		/* one scale for a start */
	/* create canvas and associated arrays of graphics arrays */
	s->newCanvas(s, SCRNPIX_WIDTH, height);
	/* graphics contexts */
	s->igc = newGC(s->dpy, s->win, canvas_fs, GXxor);/* inverted */
	s->wgc = newGC(s->dpy, s->win, canvas_fs, myfg?GXset:GXclear);/* reg */
	s->dgc = newGC(s->dpy, s->win, canvas_fs, GXxor);	/* dotted  */
	XSetPlaneMask(s->dpy, s->dgc, 0xffffffff);
	XSetLineAttributes(s->dpy, s->dgc, 0, LineOnOffDash, CapButt, JoinMiter);
	s->graphpoints = new_graphpoints();
	s->setVertMargine(s, 10);		/* graphics margins */
	screen_menu = (Menu *) new_menu(s->cframe, "DATA DISPLAY"); 
	add_items(screen_menu, screen_cmds, screen_procs, screen_flags);
	s->setMenu(s, screen_menu);
	s->setCursor(s, main_curs);
	s->screen_number = nscreens;
	screenlist[nscreens++] = s;		/* load into list */
	screenlist[nscreens] = NULL;	/* null-terminate list for safety */
	return s;
}

void
sc_destroy(s)
	Mv_screen *s;
{
	XUnmapSubwindows(s->dpy, s->frame);
	XUnmapWindow(s->dpy, s->frame);
	s->vscale->destroy(s->vscale);
	XFreeGC(s->dpy, s->igc);
	XFreeGC(s->dpy, s->wgc);
	XFreeGC(s->dpy, s->dgc);
	canvas_destroy(s->cnv);
	cfree((char *) s->cnv);
	destroy_menu(s->menu);
	s->graphpoints->destroy(s->graphpoints);
	cfree((char *) s->Segments);
	cfree((char *) s->Points);
	if(s->hairlabel) cfree(s->hairlabel);
	XDestroyWindow(s->dpy, s->cframe);
	XDestroyWindow(s->dpy, s->frame);
	screenlist[s->screen_number] = 0;	/* remove from list */
	cfree((char *) s);
	s = (Mv_screen *) NULL;
}

void
sc_setMenu(s, menu)
	Mv_screen *s;
	Menu *menu;
{
	s->menu = menu;
}

void
sc_setVertMargine(s, vm)
	Mv_screen *s;
	int vm;
{
/*	int scrollwidth = *((int *) canvas_get(s->cnv, LXC_BARWIDTH)); */
	s->vmarg = vm;
	s->label_y = vm/2.0;
	s->vscale->setVertMargines(s->vscale, vm, vm);
}

void
sc_setCursor(s, cursor)
	Mv_screen *s;
	Cursor cursor;
{
	if(canvas_set(s->cnv, LXC_CCURSOR, cursor, LXC_NULL) == LX_ERROR)
		mv_alert("setCursor:  unable to set cursor.");
}

void
sc_showMenu(s, event)
	Mv_screen *s;
	XButtonPressedEvent *event;
{
	menu_show(s->menu, event);
}
	
GC 
newGC(dsply, win, font, function)
	Display *dsply;
	Window win;
	XFontStruct *font;
	int function;
{
	XGCValues gcv;
	unsigned long g_mask;
	gcv.function = function; 
	gcv.foreground = myfg ^ mybg;
	gcv.background = mybg;
	gcv.font = font->fid;
	gcv.cap_style = CapNotLast;
	gcv.plane_mask = myfg ^ mybg;
	g_mask = GCFunction|GCFont|GCForeground|GCBackground
		|GCPlaneMask|GCCapStyle;
	return XCreateGC(dsply, win, g_mask, &gcv);
}

Pixmap
sc_getPix(s)
	Mv_screen *s;
{
	Pixmap pm = *((Pixmap *) canvas_get(s->cnv, LXC_PIXMAP));
	if(pm == (Pixmap) NULL) {
		mv_alert("sc_getPix: null pixmap.");
	}
	return pm;
}

void
sc_setClipFrames(s, chan)	/* clips waveform at vertical borders */
	Mv_screen *s;
	int chan;
{
	XRectangle rectangle[1];
	int height, width, i, bw;
	height = s->screenHeight(s)/s->nchannels;
	width = s->screenWidth(s);
	bw = s->vmarg/2;
	XSetClipMask(s->dpy, s->wgc, None);	/* reset */
	rectangle[0].x = 0;
	rectangle[0].y = height*chan + bw;
	rectangle[0].width = width;
	rectangle[0].height = height - s->vmarg;
	XSetClipRectangles(s->dpy, s->wgc, 0, 0, rectangle, 1, Unsorted);
}

void
sc_newCanvas(s, width, height)
	Mv_screen *s;
	unsigned width, height;
{
	static long e_mask = ExposureMask|ButtonPressMask|ButtonReleaseMask|
		ButtonMotionMask|KeyPressMask|PropertyChangeMask;
	/* destroy and recreate canvas in new height */
	if(s->cnv) {
		canvas_destroy(s->cnv);
		cfree((char *) s->cnv);
	}
	if ((s->cnv = canvas_create("mixview", s->dpy, s->cframe,
		LXC_MODE, LXC_STATIC,
		LXC_WIDTH, width <= s->frameWidth(s) ? width + 1 : width,
		LXC_HEIGHT, height-11,	
		LXC_NULL)) == (Canvas *) NULL)
			mv_die(errno, "sc_newCanvas: unable to recreate canvas.");
	s->win = *((Window *) canvas_get(s->cnv, LXC_WINDOW));
	s->pm = *((Pixmap *) canvas_get(s->cnv, LXC_PIXMAP));
	XSelectInput(s->dpy, s->win, e_mask);
	/* (re)create the graphics data arrays */
	if(width > s->maxpoints) {
		if(s->Points) cfree((char *) s->Points);
		s->maxpoints = width;
		s->Points = (XPoint *) mv_alloc(sizeof(XPoint)*s->maxpoints);
		if(s->Segments) cfree((char *) s->Segments);
		s->Segments = (XSegment *) mv_alloc(sizeof(XSegment)*s->maxpoints/2);
	}
	s->lineset = 0;
	s->stringset = 0;
	s->setCursor(s, main_curs);
}

void
sc_setScreenFont(s, fontname)
	Mv_screen *s;
	char *fontname;
{
	XFontStruct *fs;
	if ((fs = XLoadQueryFont(s->dpy, fontname)) == NULL) {
		char msg[64];
		sprintf(msg, "sc_setScreenFont: display %s can't find font %s",
			XDisplayString(s->dpy), fontname);
		mv_alert(msg);
		return;
	}
	XSetFont(s->dpy, s->wgc, fs->fid);
	XSetFont(s->dpy, s->igc, fs->fid);
	XSetFont(s->dpy, s->dgc, fs->fid);
	XFreeFont(s->dpy, fs);	/* for now we dont need to keep fontstruct */
}

unsigned
sc_screenWidth(s)
	Mv_screen *s;
{
	return (unsigned)*((int *) canvas_get(s->cnv, LXC_WIDTH));
}

unsigned
sc_screenHeight(s)
	Mv_screen *s;
{
	return (unsigned) *((int *) canvas_get(s->cnv, LXC_DHEIGHT));
}

unsigned
sc_frameWidth(s)
	Mv_screen *s;
{
	Mv_scale *vs = s->vscale;
	return (unsigned) (s->framewidth - vs->frameWidth(vs));
}

unsigned
sc_frameHeight(s)
	Mv_screen *s;
{
	return (unsigned) s->frameheight;
}

void
sc_getOffsets(s)
	Mv_screen *s;
{
	s->xoff = *((int *) canvas_get(s->cnv, LXC_XOFFSET));
	s->yoff = *((int *) canvas_get(s->cnv, LXC_YOFFSET));
}

void
sc_setChannels(s, chans)
	Mv_screen *s;
	int chans;
{	
	int n;
	Mv_scale *vs = s->vscale;
	if(chans > MAXDCHANS) {
		mv_alert("Current maximum no. of data channels exceeded.");
		chans = MAXDCHANS;
	}
	n = vs->nchans;
	while(n++ < chans)
		vs->addScale(vs);
	s->nchannels = chans;
}

int 
sc_nChannels(s)
	Mv_screen *s;
{
	return s->nchannels;
}

int
sc_setScanChannels(s, first, last)
	Mv_screen *s;
	int first, last;
{
	if(first < 0 || last < first) {
		mv_alert("sc_setScanChannels: invalid channel choices");
		return -1;
	}
	s->chanfirst = first;
	s->chanlast = last;
	return 1;
}

void
sc_getScanChannels(s, first, last)
	Mv_screen *s;
	int *first, *last;
{
	*first = s->chanfirst;
	*last = s->chanlast;
}

int
sc_setVertScale(s, chan, min, max)
	Mv_screen *s;
	int chan;
	double min, max;
{
	if(chan < 0) {
		mv_alert("sc_setVertScale: illegal channel.");
		return -1;
	}
	if(chan >= s->nChannels(s)) {
		char msg[80];
		sprintf(msg, "sc_setVertScale: no such channel.");
		mv_alert(msg);
		return -1;
	}
	if(max < min) {
		mv_alert("sc_setVertScale: max < min!");
		return -1;
	}
	if(max == 0.0 && min == 0.0) {	/* if no data all zeroes */
		max = 1.0;
		min = -1.0;
	}
	else if(max == min) { 		/* all data at same value */ 
		if(max < 0.0)  max = 0.0;
		else min = 0.0;
	}
	s->maxval[chan] = max;
	s->minval[chan] = min;
	return 1;
}

int
sc_getVertScale(s, chan, min, max)
	Mv_screen *s;
	int chan;
	double *min, *max;
{
	if(chan < 0 || chan >= s->nchannels) {
		mv_alert("sc_getVertScale: illegal channel.");
		return -1;
	}
	*max = s->maxval[chan];
	*min = s->minval[chan];
	return 1;
}

int
sc_setHorizScale(s, scale)
	Mv_screen *s;
	double scale;
{
	s->horiz_scale = scale;
	return 1;
}

double
sc_getHorizScale(s)
	Mv_screen *s;
{
	return s->horiz_scale;
}

void
sc_setHorizGrain(s)
	Mv_screen *s;
{
	int range = ABS(s->hmaxval) - ABS(s->hminval);
	double grain = (double) s->screenWidth(s)/range;
	/* set data granularity -- for now, always >= 1 */
	s->horiz_grain = grain < 1.0 ? 1.0 : grain;
}

int
sc_setHorizRange(s, min, max)
	Mv_screen *s;
	int min, max;
{
	int range;
	if(max <= min) {
		mv_alert("sc_setHorizRange: max must be > min.");
		return -1;
	}
	if((range = ABS(max-min)) > s->screenWidth(s))
		s->newCanvas(s, range, s->frameHeight(s));
	/* at this level the units are data elements */
	s->hminval = min;
	s->hmaxval = max;
	s->setHorizGrain(s);
	return 1;
}

int
sc_getHorizRange(s, min, max)
	Mv_screen *s;
	int *min, *max;
{
	*min = s->hminval;
	*max = s->hmaxval;
	return 1;
}

void
sc_setHairLabel(s, label)
	Mv_screen *s;
	char *label;
{
	s->hairlabel = (char *) mv_alloc(strlen(label)+1);
	strcpy(s->hairlabel, label);
}

int
sc_baseLocation(s, chan)
	Mv_screen *s;
	int chan;
{	
	double max, min, factor;
	int bandwidth = ROUND(s->screenHeight(s)/(double) s->nchannels), base;
	int sclbandwidth = bandwidth - (2 * s->vmarg);
	if(s->getVertScale(s, chan, &min, &max) < 0) return 0;
	factor = max/(max-min);
	base = ROUND(sclbandwidth * factor) + (bandwidth * chan) + s->vmarg;
	return base;
}

double
sc_scaleFactor(s, chan)
	Mv_screen *s;
	int chan;
{	
	double max, min, scale;
	int sclbandwidth = (s->screenHeight(s)/s->nchannels) - (2 * s->vmarg);
	s->getVertScale(s, chan, &min, &max);
	scale = (max-min)/((double) sclbandwidth);	
	return scale;
}

void
sc_plotData(s)
	Mv_screen *s;
{
	int yloc, i, chan, maxwidth, lastpoint;
	Pt_link *gpoint, *gnext;
	Graph_points *gp = s->graphpoints;
	maxwidth = s->screenWidth(s);
	lastpoint = s->maxpoints;
	for(chan=0; chan < s->nchannels; chan++) {
		double scale = s->scaleFactor(s, chan);
		int base = s->baseLocation(s, chan);
		int xloc = 0;
		s->npoints = 0;
		for(i=0, gpoint = gp->getPoint(gp, chan, s->hminval);
		    s->npoints < s->hmaxval && gpoint != (Pt_link *) NULL;
		    i++, gpoint = gnext) {
			gnext = gpoint->next;
			/* scale vertical values */
			yloc = base - gpoint->yval/scale;
			/* scale and shift horiz values */
			xloc = (gpoint->xval-s->hminval) * s->horiz_grain + .5;
			s->Points[i].x = xloc;
			s->Points[i].y = yloc;
			if(++s->npoints == lastpoint) break;
			if(xloc > maxwidth) break;
		}
		s->setClipFrames(s, chan);
		s->plotLines(s);	/* draw current chan into pixmap */
	}
}

/* lowest level graphic routines */

void
sc_vLine(s, x, y)
	Mv_screen *s;
	int x, y;
{
	unsigned height = s->screenHeight(s);
	XDrawLine(s->dpy, s->win, s->igc, x, 0, x, height);
	XDrawLine(s->dpy, s->pm, s->igc, x+s->xoff, s->yoff, x+s->xoff, height+s->yoff);
}

void
sc_hLine(s, x, y)
	Mv_screen *s;
	int x, y;
{
	unsigned width = s->screenWidth(s);
	XDrawLine(s->dpy, s->win, s->igc, 0, y, width, y);	
	XDrawLine(s->dpy, s->pm, s->igc, s->xoff, y+s->yoff, width+s->xoff, y+s->yoff);
}

void
sc_drawLines(s, points, npoints)	/* for drawing segment array directly to screen */
	Mv_screen *s;
	XPoint *points;
	int npoints;
{
	XDrawLines(s->dpy, s->win, s->igc, points, npoints, CoordModeOrigin);
	XDrawLines(s->dpy, s->getPix(s), s->igc, points, npoints, CoordModeOrigin);
}

void
sc_plotLines(s)	/* for plotting graph or waveform */
	Mv_screen *s;
{
	if(s->is_data)
		XDrawLines(s->dpy, s->getPix(s), s->wgc, s->Points, 
				s->npoints, CoordModeOrigin);
}

void
sc_plotSegments(s)	/* for plotting bar graph or waveform */
	Mv_screen *s;
{
	if(s->is_data)
		XDrawSegments(s->dpy, s->getPix(s), s->igc, s->Segments, s->nsegments);
}

void
sc_rectangle(s, x0, x1)
	Mv_screen *s;
	int x0, x1;
{
	unsigned width = x1 - x0 + 1; 
	unsigned height = s->frameHeight(s);
	XFillRectangle(s->dpy, s->win, s->igc, x0, 0, width, height);
	XFillRectangle(s->dpy, s->pm, s->igc, x0, 0, width, height);
}

void
sc_string(s, x, y, str)
	Mv_screen *s;
	int x, y;
	char *str;
{
	int len = strlen(str);
	XDrawString(s->dpy, s->win, s->igc, x, y, str, len);
	XDrawString(s->dpy, s->pm, s->igc, x+s->xoff, y+s->yoff, str, len);
}

void
sc_clear(s)
	Mv_screen *s;
{
	canvas_clear(s->cnv);
	s->lineset = s->boxset = 0;
	s->stringset = 0;
}

void
sc_box(s, x, y)
	Mv_screen *s;
	int x, y;
{
	int xcorner;
	x = s->quantize(s, x);
        /* if no previous region selected, set start to hair loc */
	if(!s->boxset || s->rdrbox) s->xend = s->xloc;
	if (x > s->xend){		/* increase box width */
		s->rectangle(s, s->xend+1, x);
		s->xend = x; /* set current loc to edge of selected area */
	}
	else if (x < s->xend){		/* decrease box width */
		xcorner = (x > s->xloc) ? x : s->xloc;
		s->rectangle(s, xcorner+1, s->xend);
		s->xend = (x > s->xloc) ? x : s->xloc; /* set currnt loc to ins. pt. */
        }
	s->boxset = 1;   	/* set flag for box drawn */
	s->lineset = 1;
}

void
sc_noBox(s)
	Mv_screen *s;
{
	s->rectangle(s, s->xloc+1, s->xend);
	s->boxset =  0;   	/* unset flags for box drawn */
}

void
sc_vertHair(s, x, y)
	Mv_screen *s;
	int x, y;
{
	static oldxoff, oldyoff;
	s->xdiff = oldxoff - s->xoff;
	s->ydiff = oldyoff - s->yoff;
	oldxoff = s->xoff;
	oldyoff = s->yoff;
	if(s->lineset) s->vLine(s, s->xloc+s->xdiff, s->yloc+s->ydiff);
	s->xloc = s->quantize(s, x);
	s->yloc = y;
	s->vLine(s, s->xloc, s->yloc);
	s->lineset = 1;
}

void
sc_horizHair(s, x, y)
	Mv_screen *s;
	int x, y;
{
}

void
sc_display(s)
	Mv_screen *s;
{
	if(!s->is_mapped) {
		XMapWindow(s->dpy, s->frame);
		s->is_mapped = True;
	}		
	XMapSubwindows(s->dpy, s->frame);
	clear_events();
	if(s->is_data)
		s->plot(s);	/* plot data curves and flush to screen */
}


void
sc_block(s)
	Mv_screen *s;
{
        canvas_set(s->cnv, LXC_ACTIVE, FALSE, LXC_CCURSOR, wait_curs, LXC_NULL);
}

void
sc_unBlock(s)
	Mv_screen *s;
{
	static long e_mask = ExposureMask|ButtonPressMask|ButtonReleaseMask|
		ButtonMotionMask|KeyPressMask|PropertyChangeMask;
	canvas_set(s->cnv, LXC_ACTIVE, TRUE, LXC_CCURSOR, main_curs, LXC_NULL);
	XSelectInput(s->dpy, s->win, e_mask);
}

void
sc_plot(s)
	Mv_screen *s;
{	
	s->clear(s);
	s->plotData(s);
	s->plotHorizScale(s);
	canvas_flush(s->cnv);
	s->vscale->display(s->vscale, s->minval, s->maxval); /* draw scales */
}

void
sc_resize(s, width, height)
	Mv_screen *s;
	unsigned width, height;
{
	Mv_scale *vs = s->vscale;
	int scrollborder = *((int *) canvas_get(s->cnv, LXC_BARWIDTH));
	int newcframewidth = width - vs->frameWidth(vs);
	int current_screenwidth = s->screenWidth(s);
	int current_frameheight = s->frameHeight(s);
	/* ignore this resize if size is same */
	if(newcframewidth == s->frameWidth(s) && height == s->frameHeight(s))
		return;
	vs->resize(vs, width, height-scrollborder);	/* resize scales */
	XResizeWindow(s->dpy, s->frame, width, height); /* resize outer frame */
	XResizeWindow(s->dpy,s->cframe,newcframewidth, height);
	s->framewidth = width;
	s->frameheight = height;
	/* if frame exceeds current dimensions of canvas, make new canvas */
	if(height!=current_frameheight || newcframewidth>current_screenwidth) {
		int new_width = (newcframewidth > current_screenwidth) ? 
			newcframewidth : current_screenwidth;
		s->newCanvas(s, new_width, height);
		s->setHorizGrain(s);
		s->display(s);
	}
}

/* higher level graphics routines */

void
sc_printLoc(s, x, y)
	Mv_screen *s;
	int x, y;
{
	static int voffset = 22, xoffset = 3, sx;
	static char string[24];
	int hlimit = s->frameWidth(s) - 100;
	/* erase old string */
	if(s->stringset) s->string(s, sx+s->xdiff, voffset+s->ydiff, string); 
	sprintf(string, "%s", s->xFormat(s, x));
	sx = s->quantize(s, (x <= hlimit ? x : hlimit)) + xoffset;
	s->string(s, sx, voffset, string); 
	s->stringset = True;
}

void
sc_plotHorizScale(s)
	Mv_screen *s;
{
	double scale = s->getHorizScale(s);
	double minval = s->hminval*scale, maxval = s->hmaxval*scale;
	double range, current_val, pr_inc, sc_inc, tr_inc;
	double maxpow = 1.0e-10,  pr_minamp, sc_minamp, tr_minamp, totalrange;
	double round = 0.999999, factor;
	int width = s->screenWidth(s)-1, height = s->frameHeight(s);
	int trflag = 1, ztick, nprimeticks, xloc, i;
	char string[24], *s_format();
	XSegment *lineptr = s->Segments;
	Pixmap pm = s->getPix(s);
	if(maxval > 0.0 && minval < 0.0) {	/* if scale crosses zero */
		 range = FABS(maxval) > FABS(minval) ? maxval : FABS(minval);
		 ztick = 1;	/* origin counts as tick mark */
	}
	else	{				/* scale all pos or neg */
		range = FABS(maxval - minval);
		ztick = 0;
	}
	totalrange = FABS(maxval - minval);
	DBPRT(range); DBPRT(totalrange);
	/* find max power of ten less than range */
	while((maxpow *= 10.0) <= totalrange);
	maxpow /= 10.0;
	DBCOM(orig maxpow); DBPRT(maxpow);
	/* make sure there are at least 4 ticks at the prime level */
	while((nprimeticks = (totalrange/maxpow)+ztick) < 4) {
		maxpow /= 2.0;
		trflag = 0;
	}
	DBCOM(recomputed maxpow); DBPRT(maxpow); DBPRT(nprimeticks);
	/* no tertiary hatchmarks if space is limited */
	if(width/nprimeticks < 60) trflag = 0;	
	pr_minamp =  (int) ((minval/maxpow)+round) * maxpow;
	sc_minamp =  (int) ((minval*2.0)/maxpow+round) * maxpow/2.0;
	if(FABS(pr_minamp-minval) < 1.0e-06) pr_minamp = minval; /* kludge */	
	if(FABS(sc_minamp - pr_minamp) < 1.0e-06) sc_minamp += maxpow/2.0;
	tr_minamp =  (int) ((minval*10.0)/maxpow+round) * maxpow/10.0;
	pr_inc = maxpow;
	sc_inc = maxpow;
	tr_inc = maxpow/10.0;
	s->nsegments = 0;		/* reset */
	factor = totalrange / (double) width;
	/* loop for width of screen, check to see if next increment */
	/* is greater than primary, secondary, or tertiary hatch-locs */
	for(xloc=0; xloc <= width; xloc++) {
		current_val = minval + (xloc * factor);
		if(current_val >= pr_minamp) {
			int len, toffset;
			lineptr->x1 = xloc;
			lineptr->y1 = 0;
			lineptr->x2 = xloc;
			lineptr->y2 = s->pr_len;
			lineptr++;
			s->nsegments++;
			if(FABS(pr_minamp)<1.0e-6) pr_minamp = 0.0; /* kludge */
			sprintf(string, "%s", s_format((double) pr_minamp));
			len = strlen(string);
			toffset = XTextWidth(scale_fs, string, len)/2;
			XDrawString(s->dpy, pm, s->igc, xloc-toffset,
				s->pr_len + 4, string, len);
			pr_minamp += pr_inc;	/* increment to next value */
			tr_minamp += tr_inc;
			continue;
		}
		if(current_val >= sc_minamp) {
			lineptr->x1 = xloc;
			lineptr->y1 = 0;
			lineptr->x2 = xloc;
			lineptr->y2 = s->sc_len;
			lineptr++;
			s->nsegments++;
			sc_minamp += sc_inc;
			tr_minamp += tr_inc;
			continue;
		}
		if(!trflag) continue;	/* skip tertiary hatchmarks */
		if(current_val >= tr_minamp) {
			lineptr->x1 = xloc;
			lineptr->y1 = 0; 
			lineptr->x2 = xloc;
			lineptr->y2 = s->tr_len;
			lineptr++;
			s->nsegments++;
			tr_minamp += tr_inc;
		}
	}
	s->plotSegments(s);
	/* draw base line if scale crosses zero */
	for(i=0; i < s->nchannels; i++) {
		GC gcback = s->igc;
		int yloc = s->baseLocation(s, i), hght = s->screenHeight(s);
		if(yloc<i*(hght/s->nchannels) || yloc>(i+1)*(hght/s->nchannels))
			continue;
		s->igc = s->dgc;
		s->hLine(s, 0, yloc);
		s->igc = gcback;
	}
}

char *
sc_xFormat(s, loc)	/* this func will be different in each datawin type */
	Mv_screen *s;
	int loc;
{
	static char string[24];
	int offset = s->xoff/s->horiz_grain + 0.5;
	loc = (int) (0.5 + loc/s->horiz_grain) + offset + s->hminval;
	sprintf(string, "%s%0.1f", s->hairlabel, loc * s->horiz_scale);
	return string;
}

int
sc_quantize(s, loc)	/* limits horiz movement to current grain size */
	Mv_screen *s;
	int loc;
{
	loc = loc < 0 ? 0 : loc;
	return (int) (0.5 + loc/s->horiz_grain) * s->horiz_grain 
		- (s->xoff % (int) s->horiz_grain);
}

int 
sc_screenEvent(s, event)
	Mv_screen *s;
	XEvent *event;
{
	switch(event->type) {
/*	case ConfigureNotify:
		if(event->xconfigure.window != s->frame) return False;
		s->frameResize(s, (XConfigureEvent *) event);
		break; */
	case MotionNotify:
		if(event->xmotion.window != s->win) return False;
		s->buttonMove(s, (XPointerMovedEvent *) event);
		break;
	case ButtonPress:
		if(event->xbutton.window != s->win) return False;
		current_datawin = (Data_win *) s->parent;
		s->getOffsets(s);
		s->buttonPress(s, (XButtonPressedEvent *) event);
		break;
	case ButtonRelease:
		if(event->xbutton.window != s->win) return False;
		s->buttonRelease(s, (XButtonReleasedEvent *) event);
		break;
	case KeyPress:
		if(event->xkey.window != s->win) return False;
		s->getOffsets(s);
		s->keyPress(s, (XKeyPressedEvent *) event);
		break;
	default:
		return False;
	}
	return True;
}

void
sc_frameResize(s, event)
	Mv_screen *s;
	XConfigureEvent *event;
{
	XEvent evt;
	/* compress resize events before calling routine */
	while(XEventsQueued(event->display, QueuedAfterReading) > 0) {
		XEvent ahead;
		XPeekEvent(event->display, &ahead);
		if(ahead.type != ConfigureNotify) break;
		if(ahead.xmotion.window != event->window) break;
		XNextEvent(event->display, &evt);
		event = (XConfigureEvent *) &evt;
	}
	s->resize(s, event->width, event->height);
}


void
sc_buttonPress(s, event)
Mv_screen *s;
XButtonPressedEvent *event;
{
	unsigned e_mask = ButtonReleaseMask | ButtonMotionMask;
	switch (event->button) {
	case Button1:
		XGrabPointer(s->dpy, s->win, False, e_mask, GrabModeAsync,
			GrabModeAsync, s->cframe, arrow_curs, event->time);
		if(s->boxset) s->noBox(s);
		s->vertHair(s, event->x, event->y);
		s->printLoc(s, event->x, event->y);
		break;
	case Button2:
		XGrabPointer(s->dpy, s->win, False, e_mask, GrabModeAsync,
			GrabModeAsync, s->cframe, arrow_curs, event->time);
		if(!s->lineset) s->vertHair(s, event->x, event->y);
		s->box(s, event->x, event->y);
		s->printLoc(s, event->x, event->y);
		break;
	case Button3:
		s->showMenu(s, event);
		break;
	default:
		break;
	}
}

void
sc_buttonMove(s, event)
	Mv_screen *s;
	XPointerMovedEvent *event;
{
	XEvent evt;
	/* compress motion events before calling routines */
	while(XEventsQueued(event->display, QueuedAfterReading) > 0) {
		XEvent ahead;
		XPeekEvent(event->display, &ahead);
		if(ahead.type != MotionNotify) break;
		if(ahead.xmotion.window != event->window) break;
		XNextEvent(event->display, &evt);
		event = (XPointerMovedEvent *) &evt;
	}
	if (event->state & Button1Mask) {
		s->vertHair(s, event->x, event->y);
		s->printLoc(s, event->x, event->y);
	}
	else if (event->state & Button2Mask) {
		s->box(s, event->x, event->y);
		s->printLoc(s, event->x, event->y);
	}
}

void
sc_buttonRelease(s, event)
Mv_screen *s;
XButtonReleasedEvent *event;
{
	switch (event->button) {
	case Button1:
		if(event->state & Button2Mask) return;
		XUngrabPointer(s->dpy, event->time);
		break;
	case Button2:
		if(event->state & Button1Mask) return;
		XSetSelectionOwner(s->dpy, XA_PRIMARY, event->window, event->time);
		XUngrabPointer(s->dpy, event->time);
		break;
        } 
}

void
sc_keyPress(s, event)
	Mv_screen *s;
	XKeyPressedEvent *event;
{
	DBCOM(keypress event received);
}

boolean
screen_event(event) /* called from main event loop */
	XEvent *event;
{
	register int i = 0;
	Mv_screen *screen;
	for(i=0; i<nscreens; i++) {
		if((screen = screenlist[i]) != (Mv_screen *) NULL)
			if(screen->screenEvent(screen, event)) return True;
	}
	return False;
}

