/*
	scale.c
*/

#include "main.h"
#include "dialog.h"
#include "screen.h"
#include "nobug.h"

extern Display *dpy;
extern unsigned long myfg, mybg;
extern char *progname;
extern XFontStruct *canvas_fs,  *scale_fs;

/*		scale object functions			*/

Scale *
new_scale(parent, width, height, xloc, yloc)
	Window parent;
	unsigned width, height;
	int xloc, yloc;
{
	Scale *scl;
	long r_mask = ExposureMask | StructureNotifyMask | PropertyChangeMask;
	scl = (Scale *) mv_alloc(sizeof(Scale));
	scl->dpy = dpy;
	scl->frame = new_window(parent, width, height, xloc, yloc, 0, 0);
	scl->framewidth = width;
	scl->frameheight = height;
	XSelectInput(scl->dpy, scl->frame, r_mask);
	if ((scl->cnv = canvas_create(progname, scl->dpy, scl->frame,
			LXC_MODE, LXC_DYNAMICCLR,
			LXC_NULL)) == (Canvas *) NULL)
				mv_die(errno, "new_scale: unable to create canvas.");
	scl->win = *((Window *) canvas_get(scl->cnv, LXC_WINDOW));
	scl->gc = newGC(scl->dpy, scl->win, scale_fs, GXxor);
	scl->maxlines = 256;
	scl->lines = (XSegment *) mv_alloc(scl->maxlines*sizeof(XSegment));
	scl->pr_len = 7;
	scl->sc_len = 4;
	scl->tr_len = 2;
	scl->txt_xloc = 5;
	scl->txt_yoffset = 4;
	scl->label1x = 2;
	scl->label2x = width/2;
	scl->label1y = 8;
	scl->label2y = 8;
	scl->topmargine = 20;
	scl->botmargine = 10;
	/* functions */
	scl->setLabels = scl_setLabels;
	scl->drawLabels = scl_drawLabels;
	scl->plotScale = scl_plotScale;
	scl->setVertMargines = scl_setVertMargines;
	scl->scaleWidth = scl_scaleWidth;
	scl->scaleHeight = scl_scaleHeight;
	scl->frameWidth = scl_frameWidth;
	scl->frameHeight = scl_frameHeight;
	scl->getPix = scl_getPix;
	scl->display = scl_display;
	scl->moveResize = scl_moveResize;
	scl->destroy = scl_destroy;
	return scl;
}

void
scl_destroy(s)
	Scale *s;
{
	cfree((char *) s->lines);
	XFreeGC(s->dpy, s->gc);
	canvas_destroy(s->cnv);
	cfree((char *) s->cnv);
	XDestroyWindow(s->dpy, s->frame);
	cfree((char *) s);
	s = (Scale *) NULL;
}

void
scl_setLabels(s, string1, string2)
	Scale *s;
	char *string1, *string2;
{
	strcpy(s->label1, string1);
	strcpy(s->label2, string2);
}

void
scl_drawLabels(s)
	Scale *s;
{
	Pixmap pm = s->getPix(s);
	XDrawString(s->dpy, pm, s->gc, s->label1x, s->label1y, 
		s->label1, strlen(s->label1));
	XDrawString(s->dpy, pm, s->gc, s->label2x, s->label2y, 
		s->label2, strlen(s->label2));
}

void
scl_plotScale(s, minval, maxval)
	Scale *s;
	double minval, maxval;
{
	double range, current_val, pr_inc, sc_inc, tr_inc;
	double maxpow = 1.0e-10,  pr_minamp, sc_minamp, tr_minamp, totalrange;
	double round = 0.999999;
	int yloc, width = s->scaleWidth(s)-1, height = s->scaleHeight(s), nprimeticks;
	int trflag = 1, ztick, i;
	char string[24], *s_format();
	XSegment *lineptr = s->lines;
	Pixmap pm = s->getPix(s);
	if(maxval > 0.0 && minval < 0.0) {	/* if scale crosses zero */
		 range = FABS(maxval) > FABS(minval) ? maxval : FABS(minval);
		 ztick = 1;	/* origin counts as tick mark */
	}
	else	{				/* scale all pos or neg */
		range = FABS(maxval - minval);
		ztick = 0;
	}
	totalrange = FABS(maxval - minval);
	DBPRT(range); DBPRT(totalrange);
	/* find max power of ten less than range */
	while((maxpow *= 10.0) <= totalrange);
	maxpow /= 10.0;
	DBCOM(orig maxpow); DBPRT(maxpow);
	/* make sure there are at least 5 ticks at the prime level */
	while((nprimeticks = (totalrange/maxpow)+ztick) < 5) {
		maxpow /= 2.0;
		trflag = 0;
	}
	DBCOM(recomputed maxpow); DBPRT(maxpow); DBPRT(nprimeticks);
	/* no tertiary hatchmarks if space is limited */
	if(height/nprimeticks < 80) trflag = 0;	
	pr_minamp =  (int) ((minval/maxpow)+round) * maxpow;
	sc_minamp =  (int) ((minval*2.0)/maxpow+round) * maxpow/2.0;
	if(FABS(pr_minamp-minval) < 1.0e-06) pr_minamp = minval; /* kludge */	
	if(FABS(sc_minamp - pr_minamp) <= (2.0*height/totalrange))
		sc_minamp += maxpow/2.0;
	tr_minamp =  (int) ((minval*10.0)/maxpow+round) * maxpow/10.0;
	pr_inc = maxpow;
	sc_inc = maxpow/2.0;
	tr_inc = maxpow/10.0;
	s->nlines = 0;		/* reset */
	/* loop for height of scale, check to see if next increment */
	/* is greater than primary, secondary, or tertiary hatch-locs */
	for(yloc=height+s->topmargine, i=0; yloc >= s->topmargine; yloc--, i++)	
	{
		current_val = minval + (i/(double) height)*totalrange;
		if(current_val >= pr_minamp) {
			lineptr->x1 = width-s->pr_len;
			lineptr->y1 = yloc;
			lineptr->x2 = width;
			lineptr->y2 = yloc;
			lineptr++;
			s->nlines++;
			if(FABS(pr_minamp)<1.0e-6) pr_minamp = 0.0; /* kludge */
			sprintf(string, "%s", s_format((double) pr_minamp));
			XDrawString(s->dpy, pm, s->gc, s->txt_xloc,
				yloc+s->txt_yoffset, string, strlen(string));
			pr_minamp += pr_inc;	/* increment to next value */
			sc_minamp += sc_inc;
			tr_minamp += tr_inc;
			continue;
		}
		if(current_val >= sc_minamp) {
			lineptr->x1 = width-s->sc_len;
			lineptr->y1 = yloc;
			lineptr->x2 = width;
			lineptr->y2 = yloc;
			lineptr++;
			s->nlines++;
			sc_minamp += sc_inc;
			tr_minamp += tr_inc;
			continue;
		}
		if(!trflag) continue;	/* skip tertiary hatchmarks */
		if(current_val >= tr_minamp) {
			lineptr->x1 = width-s->tr_len;
			lineptr->y1 = yloc;
			lineptr->x2 = width;
			lineptr->y2 = yloc;
			lineptr++;
			s->nlines++;
			tr_minamp += tr_inc;
		}
	}
			/* the right edge line */
			lineptr->x1 = width;
			lineptr->y1 = s->topmargine;
			lineptr->x2 = width;
			lineptr->y2 = height+s->topmargine+1;
			s->nlines++;
			XDrawSegments(s->dpy, pm, s->gc, s->lines, s->nlines);
}

void
scl_setVertMargines(s, tm, bm)
	Scale *s;
	int tm, bm;
{
	s->topmargine = tm;
	s->botmargine = bm;
}

unsigned
scl_scaleWidth(s)
	Scale *s;
{
	return (unsigned) s->frameWidth(s);
}

unsigned
scl_scaleHeight(s)
	Scale *s;
{
	return (unsigned) (s->frameHeight(s) -  s->topmargine - s->botmargine);
}

unsigned
scl_frameWidth(s)
	Scale *s;
{
	s->framewidth = *((int *) canvas_get(s->cnv, LXC_WIDTH));
	return (unsigned) s->framewidth;
}

unsigned
scl_frameHeight(s)
	Scale *s;
{
	s->frameheight = *((int *) canvas_get(s->cnv, LXC_HEIGHT));
	return (unsigned) s->frameheight;
}

Pixmap
scl_getPix(s)
	Scale *s;
{
	Pixmap pm = *((Pixmap *) canvas_get(s->cnv, LXC_PIXMAP));
	if(pm == (Pixmap) NULL) {
		mv_die(errno, "scl_getPix: null pixmap.");
	}
	return pm;
}

void
scl_moveResize(s, width, height, xloc, yloc)	/* called by list owning this scale */
	Scale *s;
	unsigned width, height;
	int xloc, yloc;
{
	XMoveResizeWindow(s->dpy, s->frame,  xloc, yloc, s->framewidth, height);
}
		

void
scl_display(s, min, max)
	Scale *s;
	double min, max;
{
	canvas_clear(s->cnv);
	s->drawLabels(s);
	s->plotScale(s, min, max);
	canvas_flush(s->cnv);
}


/*******************/

Mv_scale *
new_mvscale(parent, width, height, xloc, yloc)
	Window parent;
	unsigned width, height;
	int xloc, yloc;
{
	Mv_scale *scl;
	long r_mask = ExposureMask | StructureNotifyMask | PropertyChangeMask;
	scl=(Mv_scale *) mv_alloc(sizeof(Mv_scale));
	scl->dpy = dpy;
	scl->frame = new_window(parent, width, height, xloc, yloc, 0, 0);
	scl->framewidth = width;
	scl->frameheight = height;
	XSelectInput(scl->dpy, scl->frame, r_mask);
	scl->nchans = 0;
	/* functions */
	scl->setLabels = sclst_setLabels;
	scl->drawLabels = sclst_drawLabels;
	scl->plotScales = sclst_plotScales;
	scl->setVertMargines = sclst_setVertMargines;
	scl->scaleWidth = sclst_scaleWidth;
	scl->scaleHeight = sclst_scaleHeight;
	scl->frameWidth = sclst_frameWidth;
	scl->frameHeight = sclst_frameHeight;
	scl->display = sclst_display;
	scl->resize = sclst_resize;
	scl->destroy = sclst_destroy;
	scl->addScale = sclst_addScale;
	return scl;
}

void
sclst_addScale(s)
	Mv_scale *s;
{
	Scale *new_scale();
	unsigned width = s->framewidth, height;
	int yloc, i;
	height = s->frameheight/(s->nchans+1);	/* height of new scale */
	for(i=0; i < s->nchans; i++) 
		s->list[i]->moveResize(s->list[i], width, height, 0, height*i);
	yloc = height * s->nchans;		/* top of new scale */	
	s->list[s->nchans] = new_scale(s->frame, width, height, 0, yloc);
	s->nchans++;	/* increment nchans */
}

void
sclst_destroy(s)
	Mv_scale *s;
{
	int i;
	for(i=0; i<s->nchans; i++)
		s->list[i]->destroy(s->list[i]);
	XDestroyWindow(s->dpy, s->frame);
	cfree((char *) s);
	s = (Mv_scale *) NULL;
}

/* this routine could be easily used incorrectly and should be fixed */
/* it MUST be called BEFORE displaying scales! */

void
sclst_setLabels(s, chan, string1, string2)
	Mv_scale *s;
	int chan;
	char *string1, *string2;
{
	if(chan >= s->nchans) s->addScale(s);	/* add new one if necessary */
	s->list[chan]->setLabels(s->list[chan], string1, string2);
}

void
sclst_drawLabels(s)
	Mv_scale *s;
{
	int i;
	for(i=0; i<s->nchans; i++)
		s->list[i]->drawLabels(s->list[i]);
}

void
sclst_plotScales(s, chan, min, max)
	Mv_scale *s;
	int chan;
	double min, max;
{
	s->list[chan]->plotScale(s->list[chan], min, max);
}

void
sclst_setVertMargines(s, tm, bm)
	Mv_scale *s;
	int tm, bm;
{
	int i;
	for(i=0; i<s->nchans; i++)
		s->list[i]->setVertMargines(s->list[i], tm, bm);
}

void
sclst_display(s, min, max)
	Mv_scale *s;
	double *min, *max;
{
	int i;
	if(!s->is_mapped) {
		XMapWindow(s->dpy, s->frame);
		s->is_mapped = True;
	}
	XMapSubwindows(s->dpy, s->frame);
	clear_events();
	for(i=0; i<s->nchans; i++)
		s->list[i]->display(s->list[i], min[i], max[i]);
}

void
sclst_resize(s, width, height)	
	Mv_scale *s;
	unsigned width, height;
{
	int i;
	unsigned sclheight = height/s->nchans, yloc=0;
	XResizeWindow(s->dpy, s->frame, s->framewidth, height);
	for(i=0; i < s->nchans; i++) {
		s->list[i]->moveResize(s->list[i], s->framewidth, sclheight, 0,  yloc);
		yloc += sclheight;
	}
}

unsigned
sclst_scaleWidth(s)
	Mv_scale *s;
{
	s->list[0]->scaleWidth(s->list[0]);
}

unsigned
sclst_scaleHeight(s)
 	Mv_scale *s;
{
	unsigned height = 0;
	int chan = s->nchans;
	while(chan--) height += s->list[chan]->scaleHeight(s->list[chan]);
	return height;
}

unsigned 
sclst_frameWidth(s)
	Mv_scale *s;
{
	return s->framewidth;
}

unsigned 
sclst_frameHeight(s)
 	Mv_scale *s;
{
	return s->frameheight;
}

char *
s_format(number)
	double number;
{
	char f_string[12];
	static char string[24];
	int frac = (FABS(number)-ABS((int) number)) < .01 ? 0 : 1;
	if(number == 0.0) sprintf(f_string, "%s", "%+3.1f");
	else if(FABS(number) < 0.001) sprintf(f_string, "%s", "%+3.2e");
	else if(FABS(number) < 1.0) sprintf(f_string, "%s", "%+0.4f");
	else if(FABS(number) < 1000.0) {
		if(frac) sprintf(f_string, "%s", "%+0.1f");
		else sprintf(f_string, "%s", "%+4.0f");
	}
	else {
		if (frac) sprintf(f_string, "%s", "%+0.2fk");
		else sprintf(f_string, "%s", "%+4.1fk");
		number /= 1000.0;
	}
	sprintf(string, f_string, number);
	return string;
}
