/****************************************************************************** 
 *
 *  mixview - X Window System based soundfile editor and processor
 *
 *  Copyright 1989 by Douglas Scott
 *
 *  Author:     Douglas Scott 
 *  Date:       May 1, 1989
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The author makes no representations about
 *  the suitability of this software for any purpose.  
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
/* panels.c -- the setup routines for the various panels used in mixview
	Douglas Scott  3/89
*/

#include "main.h"
#include "windows.h"
#include "panels.h"
#include "nobug.h"
#include <ctype.h>

#ifdef SPARC_AUDIO
extern int volume, use_jack;
#endif
extern Display *dpy;
extern Window pframe;
extern char *progname;

Mv_panel *ctrl_panel, *view_panel, *new_panel();
VPFun default_button_proc;
XFontStruct *panel_fs;

void
pan_destroy(mp)
Mv_panel *mp;
{
	int i;
	Panel_item *pi;
	for(i = 0; i < mp->nitems; i++) {
		pi = mp->itemlist[i];
		panelitem_delete(mp->panel, pi);
		panelitem_destroy(pi);
		cfree(mp->namelist[i]);
	}
	panel_destroy(mp->panel);
	XDestroyWindow(mp->dpy, mp->frame);
	cfree((char *) mp->itemlist);
	cfree((char *) mp->namelist);
	cfree((char *) mp);
	mp = (Mv_panel *) NULL;
}

int
pan_nItems(mp)
Mv_panel *mp;
{ return mp->nitems; }

int
pan_nLabels(mp)
Mv_panel *mp;
{ return mp->nlabels; }

int
pan_nTexts(mp)
Mv_panel *mp;
{ return mp->ntexts; }

int
pan_nButtons(mp)
Mv_panel *mp;
{ return mp->nbuttons; }

int
pan_nChoices(mp)
Mv_panel *mp;
{ return mp->nchoices; }

void
pan_addLabel(mp, xpos, ypos, text, name)
Mv_panel *mp;
int xpos, ypos;
char *text, *name;
{
	Panel_item *item;
	if((item = panelitem_create(mp->panel, LXPI_LABEL, 
			LXPI_X, xpos, LXPI_Y, ypos,
			LXPI_STRING, text,
			LXPI_DISPLAY, LXPI_NOREPAINT,
			LXPI_NULL)) == (Panel_item *) NULL)
		mv_die(errno, "addLabel: unable to create item.");
	mp->addItem(mp, item, name);
	mp->nlabels++;
}

void
pan_addText(mp, xpos, ypos, text, value, proc)
Mv_panel *mp;
int xpos, ypos;
char *text, *value;
VPFun proc;
{
	Panel_item *item;
	if((item = panelitem_create(mp->panel, LXPI_TEXT, 
			LXPI_X, xpos, LXPI_Y, ypos,
			LXPI_STRING, text,
			LXPTEXT_VALUE, value,
			LXPTEXT_MAXSTORE, 128,
			LXPTEXT_MAXDISPLAY, 18,
			LXPTEXT_IPOS, strlen(value),
			LXPTEXT_PROC, proc,
			LXPI_DISPLAY, LXPI_NOREPAINT,
			LXPI_NULL)) == (Panel_item *) NULL)
		mv_die(errno, "addText: unable to create item.");
	mp->addItem(mp, item, text);
	mp->ntexts++;
}

void
pan_addButton(mp, xpos, ypos, text, proc)
Mv_panel *mp;
int xpos, ypos;
char *text;
VPFun proc;
{
	Panel_item *item;
	if((item = panelitem_create(mp->panel, LXPI_BUTTON, 
			LXPI_X, xpos, LXPI_Y, ypos,
			LXPI_STRING, text,
			LXPI_PROC, proc,
			LXPI_DISPLAY, LXPI_NOREPAINT,
			LXPI_NULL)) == (Panel_item *) NULL)
		mv_die(errno, "addButton: unable to create item.");
	mp->addItem(mp, item, text);
	mp->nbuttons++;
}

void
pan_addChoiceItem(mp, item, name)
Mv_panel *mp;
Panel_item *item;
char *name;
{
	mp->addItem(mp, item, name);
	mp->nchoices++;
}

void
pan_addChoice(mp, xpos, ypos, label, list, nchoices, proc)
	Mv_panel *mp;
	int xpos, ypos, nchoices;
	char *label;
	char **list;
	VPFun proc;
{
	Panel_item *item;
	int x, xstart, y = 0, i;
	item = panelitem_create(mp->panel, LXPI_CHOICE,
		LXPI_X, xpos, LXPI_Y, ypos, LXPI_DISPLAY, LXPI_NOREPAINT,
		LXPI_STRING, label, LXPI_NULL);
	if(!item) mv_die(errno, "addChoice: unable to create item.");
	xstart = x = XTextWidth(panel_fs, label, strlen(label)) + 20;
	for(i=0; i<nchoices; i++) {
		if(!list[i]) break;	/* to avoid trouble */
		if(panelitem_addsel(mp->panel, item, LXPENUM_SELSTRING, list[i],
			LXPENUM_MARKX, x, LXPENUM_MARKY, y,
			LXPENUM_SELX, x+20, LXPENUM_SELY, y,
			LXPI_NULL) != LX_SUCCESS) exit(-1);
		x += (XTextWidth(panel_fs, list[i], strlen(list[i])) + 30);
		if(x >= (mp->framewidth - 100)) {	/* start new row */
			x = xstart;
			y += 20;
		}
	}
	panelitem_set(mp->panel, item, 
		LXPENUM_VALUE, 0, LXPI_PROC, proc, LXPI_NULL);
	mp->addItem(mp, item, label);
	mp->nchoices++;
}

void
pan_addItem(mp, item, name)
Mv_panel *mp;
Panel_item *item;
char *name;
{
	mp->itemlist[mp->nitems] = item;
	mp->namelist[mp->nitems] = (char *) mv_alloc(strlen(name)+1);
	strcpy(mp->namelist[mp->nitems], name);
	mp->nitems++;
}

void
pan_setItem(mp, text, code, value)
Mv_panel *mp;
char *text;
int code, value;
{
	if(panelitem_set(mp->panel, mp->getItem(mp, text), 
			 code, value, LXPI_NULL) == LX_ERROR)
		mv_die(errno, "setItem: unable to set.");
}

void
pan_setText(mp, label, text)
Mv_panel *mp;
char *label, *text;
{
	if(panelitem_set(mp->panel, mp->getItem(mp, label), 
			 LXPTEXT_VALUE, text,
			 LXPTEXT_IPOS, strlen(text),
			 LXPI_NULL) == LX_ERROR)
		mv_die(errno, "setText: unable to set.");
}

void
pan_setDefaultButton(mp, label)	/* sets which button proc will be called by <return> */
Mv_panel *mp;
char *label;
{
	int len = strlen(label), i;
	char default_label[16];
	Panel_item *button;
	if((button = mp->getItem(mp, label)) == (Panel_item *) NULL)
		mv_die(errno, "setDefaultButton: unable to get button");	
	/* set this subroutine as the default */
	if((default_button_proc = (VPFun) panelitem_get(mp->panel, 
			button, LXPI_PROC)) == PROC_NULL)
		mv_die(errno, "setDefaultButton: unable to get value");	
	strcpy(default_label, label);
	/* convert button label to all caps */
	for(i = 0; i < len; i++) {
		if(isalpha(default_label[i]))
			default_label[i] = default_label[i] - 'a' + 'A';	
	}
	if(panelitem_set(mp->panel, button, 
			LXPI_STRING, default_label, LXPI_NULL) != LX_SUCCESS)
		mv_die(errno, "setDefaultButton: unable to set value");	
}

void
pan_redisplay(mp, name)
Mv_panel *mp;
char *name;
{
	panelitem_redisplay(mp->getPanel(mp), mp->getItem(mp, name));
}

void
pan_moveFrame(mp, xpos, ypos)	/* will be member of Mv_panel object */
Mv_panel *mp;
int xpos, ypos;
{
	int max_x = DisplayWidth(mp->dpy, DefaultScreen(mp->dpy));
	int max_y = DisplayHeight(mp->dpy, DefaultScreen(mp->dpy));
	if(xpos+mp->framewidth > max_x) xpos = max_x - mp->framewidth - 5;
	if(ypos+mp->frameheight > max_y) ypos = max_y - mp->frameheight - 5;
	XMoveWindow(mp->dpy, mp->frame, xpos, ypos);
}

unsigned
pan_panelWidth(mp)
Mv_panel *mp;
{
	return (unsigned) *((int *) panel_get(mp->panel, LXP_WIDTH));
}

unsigned
pan_panelHeight(mp)
Mv_panel *mp;
{
	return (unsigned) *((int *) panel_get(mp->panel, LXP_HEIGHT));
}

unsigned
pan_frameWidth(mp)
Mv_panel *mp;
{
	return (unsigned) mp->framewidth;
}

unsigned
pan_frameHeight(mp)
Mv_panel *mp;
{
	return (unsigned) mp->frameheight;
}

extern Cursor wait_curs;

void
pan_block(mp)
Mv_panel *mp;
{
	panel_set(mp->panel, LXP_ACTIVE, FALSE, 
			     LXP_PCURSOR, wait_curs, LXP_NULL);
}

extern Cursor arrow_curs;

void
pan_unBlock(mp)
Mv_panel *mp;
{
	panel_set(mp->panel, LXP_ACTIVE, TRUE, 
			     LXP_PCURSOR, arrow_curs, LXP_NULL);
}

void
pan_invisible(mp)
Mv_panel *mp;
{ panel_set(mp->panel, LXP_VISIBLE, FALSE, LXP_NULL); }

void
pan_visible(mp)
Mv_panel *mp;
{ panel_set(mp->panel, LXP_VISIBLE, TRUE, LXP_NULL); }

void
pan_active(mp)
Mv_panel *mp;
{ panel_set(mp->panel, LXP_ACTIVE, TRUE, LXP_NULL); }

void
pan_inactive(mp)
Mv_panel *mp;
{ panel_set(mp->panel, LXP_ACTIVE, FALSE, LXP_NULL); }

Panel*
pan_getPanel(mp)
Mv_panel *mp;
{ return mp->panel; }

Panel_item*
pan_getItem(mp, text)	/* returns panel item that matches text */
Mv_panel *mp;
char *text;
{
	register int i;
	char string[32];
	for(i = 0; i < mp->nitems; i++) {
		if(strcmp(mp->namelist[i], text) == 0)
			return mp->itemlist[i];
	}
	sprintf(string, "getItem: item \"%s\" not found.\n", text);
	mv_alert(string);
	return (Panel_item *) NULL;	/* item not present */
}

Panel_item *
pan_getItemN(mp, indx)	/* returns panel item for given index */
Mv_panel *mp;
int indx;
{
	if(indx < mp->nitems)
		return mp->itemlist[indx];
	else {
		mv_alert("getItemN: item not found.");
		return (Panel_item *) NULL;	/* item not present */
	}
}

Mv_panel
*new_panel(parent, width, height, xloc, yloc, bw)
Window parent;
unsigned width, height, bw;
int xloc, yloc;
{
	char *font, panel_font[128];
	Mv_panel *mp;
	mp = (Mv_panel *) mv_alloc(sizeof(Mv_panel));
	mp->dpy = dpy;
	mp->frame = new_window(parent, width, height, xloc, yloc, bw, (long) 0);
	mp->framewidth = width;
	mp->frameheight = height;
	if((font = XGetDefault(mp->dpy, progname, "PanelFont")) == CNULL)
			strcpy(panel_font, PANEL_FONT);
	else strncpy(panel_font, font, 127);
	if(!panel_fs)
		if((panel_fs = XLoadQueryFont(dpy, panel_font)) == NULL) {
			char string[80];
			sprintf(string, "new_panel: display %s cannot find font %s",
                                       XDisplayString(mp->dpy), panel_font);
			mv_die(errno, string);
		  }
	mp->panel = panel_create(progname, mp->dpy, mp->frame, 
		LXP_FONT, panel_font, 
		LXP_BOTTOMMARGIN, 5, LXP_RIGHTMARGIN, 5,
		LXP_GRAB, FALSE,
		LXP_NULL);
	mp->nitems = mp->nlabels = mp->ntexts = mp->nchoices = mp->nbuttons = 0;
	mp->ntoggles = 0;
	mp->maxitems = MAXITEMS;
	mp->itemlist = (Panel_item **) mv_alloc(MAXITEMS*sizeof(POINTER));
	mp->namelist = (char **) mv_alloc(MAXITEMS*sizeof(POINTER));
/* functions */
	mp->addLabel = pan_addLabel;
	mp->addText = pan_addText;
	mp->addButton = pan_addButton;
	mp->addChoice = pan_addChoice;
	mp->moveFrame = pan_moveFrame;
	mp->panelWidth = pan_panelWidth;
	mp->panelHeight = pan_panelHeight;
	mp->frameWidth = pan_frameWidth;
	mp->frameHeight = pan_frameHeight;
	mp->block = pan_block;
	mp->unBlock = pan_unBlock;
	mp->active = pan_active;
	mp->inactive = pan_inactive;
	mp->visible = pan_visible;
	mp->invisible = pan_invisible;
	mp->setDefaultButton = pan_setDefaultButton;
	mp->redisplay = pan_redisplay;
	mp->getPanel = pan_getPanel;
	mp->setItem = pan_setItem;
	mp->setText = pan_setText;
	mp->getItem = pan_getItem;
	mp->addItem = pan_addItem;
	mp->getItemN = pan_getItemN;
	mp->nItems = pan_nItems;
	mp->nLabels = pan_nLabels;
	mp->nTexts = pan_nTexts;
	mp->nButtons = pan_nButtons;
	mp->nChoices = pan_nChoices;
	mp->destroy = pan_destroy;
	return mp;
}

setup_view()
{
	Panel_item *scroll_bar;
	Mv_panel *vp;

        vp= view_panel= new_panel(pframe, VFRAME_WIDTH, VFRAME_HEIGHT, 0, 0, 0);
        if ((scroll_bar= panelitem_create(view_panel->getPanel(vp), 
			LXPI_SLIDER,
			LXPI_X, 50,
			LXPI_Y, 10,
			LXPI_STRING, (char *) NULL,
			LXPI_PROC, scroll_it,
			LXPSLIDER_MINVAL, 0,
			LXPSLIDER_MAXVAL, 100,
			LXPSLIDER_BARHEIGHT, 20,
			LXPSLIDER_BARLENGTH, 550,
			LXPSLIDER_VALUE, Scroll_min(),
			LXPI_NULL)) == (Panel_item *) NULL)
		exit(-1);
	view_panel->addItem(vp, scroll_bar, "scroll bar");
	view_panel->addButton(vp, 10, 8, "<", qpage_back);
	view_panel->setItem(vp, "<", LXPBUTTON_VMARGIN, 2);
	view_panel->setItem(vp, "<", LXPBUTTON_HMARGIN, 8);
	view_panel->addButton(vp, 750, 8, ">", qpage_foward);
	view_panel->setItem(vp, ">", LXPBUTTON_VMARGIN, 2);
	view_panel->setItem(vp, ">", LXPBUTTON_HMARGIN, 8);
	view_panel->addLabel(vp, 200, 45, "Start:", "start");
	view_panel->addLabel(vp, 300, 45, "", "start time");
	view_panel->addLabel(vp, 420, 45, "End:", "end");
	view_panel->addLabel(vp, 520, 45, "", "end time");
	view_panel->addButton(vp, 10, 40, "go to", go_to);
	view_panel->addText(vp, 10, 72, "time:", "0.0", check_val);
	view_panel->setItem(vp, "time:", LXPTEXT_MAXDISPLAY, 10);
	view_panel->setItem(vp, "time:", LXPTEXT_MAXSTORE, 15);
	view_panel->addLabel(vp, 200, 72, "Edit from:", "edit start");
	view_panel->addLabel(vp, 300, 72, "", "edit start time");
	view_panel->addLabel(vp, 420, 72, "To:", "edit end");
	view_panel->addLabel(vp, 520, 72, "", "edit end time");
	view_panel->addButton(vp, 625, 40, "Resize", resize_canvas);
	view_panel->addButton(vp, 710, 40, "Reset", reset_screen);
}

setup_control()
{
	Panel_item *slider;
#ifdef SPARC_AUDIO
	Panel_item *audio_choice;
	void set_audio_output(), set_volume();
#endif
	Mv_panel *cp;
        
        cp = ctrl_panel = new_panel(pframe, CPFRAME_WIDTH, CPFRAME_HEIGHT,
				  -1, VFRAME_HEIGHT, 1);
        if ((slider= panelitem_create(ctrl_panel->getPanel(cp), 
			LXPI_SLIDER,
                        LXPI_X, 380,
                        LXPI_Y, 15,
                        LXPI_STRING, "Screen Res.",
                        LXPI_PROC, set_grain,
                        LXPSLIDER_MINVAL, SLIDER_MINVAL,
                        LXPSLIDER_MAXVAL, SLIDER_MAXVAL,
                        LXPSLIDER_BARHEIGHT, 20,
                        LXPSLIDER_BARLENGTH, 150,
                        LXPSLIDER_VALUE, SLIDER_STARTVAL,
                        LXPI_NULL)) == (Panel_item *) NULL)
                exit(-1);
	ctrl_panel->addItem(ctrl_panel, slider, "slider");
#ifdef SPARC_AUDIO
        if ((slider= panelitem_create(ctrl_panel->getPanel(cp), 
			LXPI_SLIDER,
                        LXPI_X, 380,
                        LXPI_Y, 55,
                        LXPI_STRING, "Volume",
                        LXPI_PROC, set_volume,
                        LXPSLIDER_MINVAL, 0,
                        LXPSLIDER_MAXVAL, 100,
                        LXPSLIDER_BARHEIGHT, 20,
                        LXPSLIDER_BARLENGTH, 95,
                        LXPSLIDER_VALUE, unscale_gain(volume),
                        LXPI_NULL)) == (Panel_item *) NULL)
                exit(-1);
	ctrl_panel->addItem(ctrl_panel, slider, "volume");
	if((audio_choice = panelitem_create(ctrl_panel->getPanel(cp),
			LXPI_CHOICE,
			LXPI_X, 655, LXPI_Y, 50,
			LXPI_STRING, "Audio Output:",
			LXPI_PROC, set_audio_output,
			LXPENUM_SELSTRINGS, 
				"spkr.", "jack", (char *) NULL,
			LXPENUM_SELXS, 18, 88, LXPENUM_LOCTERM,
			LXPENUM_SELYS, 20, LXPENUM_LOCTERM,
			LXPENUM_MARKXS, 0, 70, LXPENUM_LOCTERM,
			LXPENUM_MARKYS, 20, LXPENUM_LOCTERM,
			LXPENUM_VALUE, use_jack,
			LXPI_NULL)) == (Panel_item *) NULL)
		exit(-1);
	ctrl_panel->addItem(ctrl_panel, audio_choice, "audio out");
#endif /* SPARC_AUDIO */
	ctrl_panel->addButton(cp, 20, 15, "x in ", xzoom);
	ctrl_panel->addButton(cp, 110, 15, "x out", xunzoom);
	ctrl_panel->addButton(cp, 200, 15, "y in ", yzoom);
	ctrl_panel->addButton(cp, 290, 15, "y out", yunzoom);
	ctrl_panel->addButton(cp, 20, 55, " Exit ", quit);
	ctrl_panel->addButton(cp, 110, 55, "Record", (VPFun) NULL);
	ctrl_panel->addButton(cp, 200, 55, " Play ", play_it);
	ctrl_panel->addButton(cp, 290, 55, " Zoom ", zoom);

}
