/*******************************************************************************
 *
 *  mixview - X Window System based soundfile editor and processor
 *
 *  Copyright 1989 by Douglas Scott
 *
 *  Author:     Douglas Scott 
 *  Date:       May 1, 1989
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The author makes no representations about
 *  the suitability of this software for any purpose.  
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
/*
	menus.c -- the menus and menu commands for mixview
*/

#include "main.h"
#include "dialog.h"
#include "mesg.h"
#include "menus.h"
#include "cmd.h"
#include "patchlevel.h"
#include "lp.h"
#include "fft.h"
#include "nobug.h"
#include <sys/dir.h>
#include <sys/stat.h>

extern Display *dpy;
extern sf_struct *v;
extern Window canvas_win, mixwin, mwin;
extern unsigned long myfg, mybg;
extern char *progname;
extern FLAG is_open, is_saved, is_new, is_owner, is_buff, yesbuff, chscale;
extern FLAG skipflag, durflag, read_only, region, box, line, insert, wasmulaw;
extern double skiptime, durtime, ampfac;

Menu *edit_menu, *alter_menu, *option_menu, *filter_menu, *analysis_menu;
Menu *file_menu, *header_menu, *canvas_menu, *canvas_edit_menu;
Menu *canvas_display_menu, *canvas_cursor_menu;
Menu *buffer_mode_menu, *peak_rescan_menu, *quick_display_menu, *sf_search_menu;

FLAG displaymode;
int peak_rescan = 1, quick_display = 0, sf_search = 0;
static FLAG top_level = 1;
static char pathname[128], menu_font[128];

static char *
file_cmds[]= {
	"New", "Change sfdir", "Open", "Save", "Save to", "Close", "Revert",
	"Print Version #", "Exit", 0
};

static void
(*file_procs[])() = {
	new_file, change_sfdir, open_file, save_file, saveto_file, close_file,
	revert_file, print_version, exit_prog, 0
};

static int
file_flags[]= {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

static char *
edit_cmds[]= {
	"Buffer mode", "Copy [^c]", "Remove [^r]", "Splice out [^x]",
	"Erase [^e]", "Zap [^z]", "Mix [^m]", "Replace [^R]",
	"Splice in [^v]", "Multiply [^M]", "Crossfade [^C]", 0
};

static void
(*edit_procs[])() = {
	0, menu_select, menu_select, menu_select, menu_select, menu_select,
	menu_select, menu_select, menu_select, menu_select, menu_select, 0
};

static int
edit_flags[]= {
	0, MC_COPY, MC_REMOVE, MC_OUTSPLICE, MC_ERASE, MC_ZAP, MC_MIX,
	MC_REPLACE, MC_INSPLICE, MC_MULT, MC_CFADE, 0
};
	
static char *
alter_cmds[]= {
        "Phrase", "Insert space", "Reverse", "Transpose", "Gravy", 0
};

static void
(*alter_procs[])() = {
	menu_select, menu_select, menu_select, menu_select, menu_select, 0
};

static int
alter_flags[]= {
	MC_PHRASE, MC_INSERT, MC_REVERSE, MC_TRANSPOSE, MC_GRAVY, 0
};
	
static char *
filter_cmds[]= {
        "Elliptical", "Resonant", "Comb", "All pass", "Tone", 0
};

static void
(*filter_procs[])() = {
	menu_select, menu_select, menu_select, menu_select, menu_select, 0
};

static int
filter_flags[]= {
	MC_MELL, MC_RESON, MC_COMB, MC_ALLPASS, MC_TONE, 0
};

static char *
option_cmds[]= {
        "Peak rescan", "Quick display", "Soundfile search",  0
};

static char *
header_cmds[]= {
        "Soundfile information", "Sample format", "Sample rate",
	"Rescale", "Change length", "Change file comment", 0
};

static void
(*header_procs[])() = {
        display_header, change_format, change_srate, rescale, change_length, 
	change_sfcomment, 0
};

static int
header_flags[]= {
	0, 0, 0, 0, 0, 0, 0
};

static char *
analysis_cmds[]= {
        "Histogram", "FFT", "LPC", "Pitch track", 0
};

static void
(*analysis_procs[])() = {
        do_hist, do_fft, do_lpc, do_ptrack, 0
};

static char *
canvas_edit_cmds[]= {
        "Select all", "Recall edit", 0
};

static char *
peak_rescan_cmds[]= {
        "On", "Off", "Query", 0
};

static char *
quick_display_cmds[]= {
        "On", "Off", 0
};

static char *
sf_search_cmds[]= {
        "Show all files",
#ifdef NeXT
	"Show .snd files only",
#else
#ifdef sparc
	"Show .au files only",
#else
	"Show .short files only",
#endif
#endif
	"Show all soundfiles", 0
};

set_menu_items(menu, itemlist, key)
Menu *menu;
char **itemlist;
unsigned short key;
{
	int nitems = get_nitems(menu);
        register int state, item = 0;
	do {
                if(key & 1) state = LXMI_ACTIVE;
                else state = LXMI_INACTIVE;
                set_item_state(menu, itemlist[item], state);
		key >>= 1;
        }
        while(++item < nitems);
}

set_item_command(item, cflag)	/* sets command flag in each menu item */
Menu_item *item;
int cflag;
{
	menuitem_info *mi;
	mi = (menuitem_info *) menuitem_get(item, LXMI_CLIENTDATA);
	if (mi == (menuitem_info *) NULL)
		mv_die(errno, "set_item_command: unable to access info struct.");
	mi->cmd = cflag;
}

int
get_item_command(item)	/* returns current command flag of menuitem */
Menu_item *item;
{
	menuitem_info *mi;
	mi = (menuitem_info *) menuitem_get(item, LXMI_CLIENTDATA);
	if (mi == (menuitem_info *) NULL)
		mv_die(errno, "get_item_command: unable to access info struct.");
	return mi->cmd;
}

Time
get_item_time(mi)
Menu_item *mi;
{
	Time time;
	if((time = *((Time *) menuitem_get(mi, LXMI_TIMESTAMP))) == (Time) 0)
		mv_die(errno, "get_item_time: unable to retrieve timestamp.");
	return time;
}

set_nitems(menu, nitems)	/* sets current number of menuitems */
Menu *menu;
int nitems;
{
	menu_info *mi;
	mi = (menu_info *) menu_get(menu, LXM_CLIENTDATA);
	if (mi == (menu_info *) NULL)
		mv_die(errno, "set_nitems: unable to access info struct.");
	mi->nitems = nitems;
}

int
get_nitems(menu)	/* returns current number of menuitems */
Menu *menu;
{
	menu_info *mi;
	mi = (menu_info *) menu_get(menu, LXM_CLIENTDATA);
	if (mi == (menu_info *) NULL)
		mv_die(errno, "get_nitems: unable to access info struct.");
	return mi->nitems;
}

add_items(menu, labels, funcs, flags)
Menu *menu;
char **labels;
void (*funcs[])();
int *flags;
{
	int nitems=0;
	while(labels[nitems] != CNULL) {
		add_menuitem(menu, labels[nitems], funcs[nitems], flags[nitems]);
		nitems++;
	}
	set_nitems(menu, nitems);
}
		
new_menu_info(menu)
Menu *menu;
{
	menu_info *info;
	info = (menu_info *) mv_alloc(sizeof(menu_info));
	info->nitems = 0;
	if(menu_set(menu, LXM_CLIENTDATA, (char *) info, LXM_NULL) == LX_ERROR)
		mv_die(errno, "new_menu_info: unable to set CLIENTDATA");
}

new_menuitem_info(item)
Menu_item *item;
{
	menuitem_info *info;
	info = (menuitem_info *) mv_alloc(sizeof(menuitem_info));
	info->cmd = 0;
	if(menuitem_set(item, LXMI_CLIENTDATA, (char *) info, LXMI_NULL) == LX_ERROR)
		mv_die(errno, "new_menuitem_info: unable to set CLIENTDATA");
}

setup_menus()
{
	char toplevel[128], *font; 
	int i=0;

	if((font = XGetDefault(dpy, progname, "MenuFont")) == CNULL)
		strcpy(menu_font, MENU_FONT);
	else strncpy(menu_font, font, 127);
	file_menu = new_menu(mwin, "FILE");
	add_items(file_menu, file_cmds, file_procs, file_flags);
	strcpy(toplevel, getenv("SFDIR")==(char *) NULL? "." : getenv("SFDIR"));
	menuitem_set(menuitem_find(file_menu, LXMI_STRING, "Open"),
		     LXMI_PULLRIGHT, make_dirmenu(toplevel),
		     LXMI_NULL);

        buffer_mode_menu = new_menu(mwin, CNULL); 
	add_menuitem(buffer_mode_menu, "Internal", set_internal, 0);
	set_item_state(buffer_mode_menu, "Internal", LXMI_INACTIVE);
	add_menuitem(buffer_mode_menu, "External", set_external, 0);

        edit_menu = new_menu(mwin, "EDIT"); 
	i = 0;
        add_pullright(edit_menu, edit_cmds[i], PROC_NULL, buffer_mode_menu);
	for(i = 1; i < NEDITITEMS; i++) 
		add_menuitem(edit_menu, edit_cmds[i], edit_procs[i],
		edit_flags[i]);
	set_nitems(edit_menu, NEDITITEMS);

        alter_menu = new_menu(mwin, "ALTER"); 
	add_items(alter_menu, alter_cmds, alter_procs, alter_flags);

        filter_menu = new_menu(mwin, "FILTER"); 
	add_items(filter_menu, filter_cmds, filter_procs, filter_flags);

	peak_rescan_menu = new_menu(mwin, CNULL);
	for(i = 0; i < 3; i++)
		add_menuitem(peak_rescan_menu, 
			peak_rescan_cmds[i], set_peak_rescan, 0);
	set_nitems(peak_rescan_menu, 3);
	set_menu_items(peak_rescan_menu, peak_rescan_cmds, 0x06);
	quick_display_menu = new_menu(mwin, CNULL);
	for(i = 0; i < 2; i++)
		add_menuitem(quick_display_menu, 
			quick_display_cmds[i], set_quick_display, 0);
	set_nitems(quick_display_menu, 2);
	set_menu_items(quick_display_menu, quick_display_cmds, 0x01);
	sf_search_menu = new_menu(mwin, CNULL);
	for(i = 0; i < 3; i++)
		add_menuitem(sf_search_menu, sf_search_cmds[i], set_sf_search, 0);
	set_nitems(sf_search_menu, 3);
	set_menu_items(sf_search_menu, sf_search_cmds, 0x06);
        option_menu = new_menu(mwin, "OPTIONS");
        add_pullright(option_menu, option_cmds[0], PROC_NULL, peak_rescan_menu);
        add_pullright(option_menu, option_cmds[1], PROC_NULL, quick_display_menu);
        add_pullright(option_menu, option_cmds[2], PROC_NULL, sf_search_menu);
	set_nitems(option_menu, 3);

        header_menu = new_menu(mwin, "HEADER"); 
        add_items(header_menu, header_cmds, header_procs, header_flags);

        analysis_menu = new_menu(mwin, "ANALYSIS");
        add_menuitem(analysis_menu, analysis_cmds[0], analysis_procs[0], 0);
        add_menuitem(analysis_menu, analysis_cmds[1], analysis_procs[1], 0);
        add_menuitem(analysis_menu, analysis_cmds[2], analysis_procs[2], 0);
        add_menuitem(analysis_menu, analysis_cmds[3], analysis_procs[3], 0);
	set_nitems(analysis_menu, 4);

        canvas_edit_menu = new_menu(mwin, CNULL);
        add_menuitem(canvas_edit_menu, canvas_edit_cmds[0], select_all, 0); 
        add_menuitem(canvas_edit_menu, canvas_edit_cmds[1], recall_edit, 0); 

        canvas_display_menu = new_menu(canvas_win, CNULL);
        add_menuitem(canvas_display_menu, "Line style", set_line_mode, 0);
        set_item_state(canvas_display_menu, "Line style", LXMI_INACTIVE);
        add_menuitem(canvas_display_menu, "Shadow style", set_shadow_mode, 0);

        canvas_cursor_menu = new_menu(canvas_win, CNULL);
        add_menuitem(canvas_cursor_menu, "Vert. line", set_cursor_line, 0);
	set_item_state(canvas_cursor_menu, "Vert. line", LXMI_INACTIVE);
        add_menuitem(canvas_cursor_menu, "Crosshairs", set_cursor_cross, 0);

        canvas_menu = new_menu(canvas_win, CNULL);
        add_pullright(canvas_menu, "Edit special", PROC_NULL, canvas_edit_menu);
        add_pullright(canvas_menu, "Display mode", PROC_NULL, canvas_display_menu);
        add_pullright(canvas_menu, "Cursor mode", PROC_NULL, canvas_cursor_menu);
        add_menuitem(canvas_menu, "Set grainsize", set_new_grainsize, 0);
        add_menuitem(canvas_menu, "Set vertical scale", set_new_scale, 0);
        add_menuitem(canvas_menu, "Set screen times", set_screentimes, 0);

	/* now set the default settings on these menus */

	set_menus();
}

void
add_menuitem(menu, name, proc, cmdflag)
Menu *menu;
char *name;
VPFun proc;
int cmdflag;
{
	Menu_item *mi;
        if ((mi= menuitem_create(LXMI_STRING, name,
			LXMI_PROC, proc,
			LXMI_STATE, proc ? LXMI_ACTIVE : LXMI_INACTIVE,
			LXMI_NULL)) == (Menu_item *) NULL)
                exit(-1);
	new_menuitem_info(mi);
	set_item_command(mi, cmdflag); /* each item contains command flag */
        if(menuitem_insert(menu, mi) == LX_ERROR)
		mv_die(errno, "add_menuitem:  unable to insert item in menu");
}

void
add_pullright(menu, name, proc, submenu)
Menu *menu, *submenu;
VPFun proc;
char *name;
{
	Menu_item *mi;

	if((mi = menuitem_create(LXMI_STRING, name,
			     LXMI_PROC, proc,
			     LXMI_PULLRIGHT, submenu,
			     LXMI_NULL)) == (Menu_item *) NULL)
		exit(-1);
	new_menuitem_info(mi); /* this sets command flag to zero */
        if(menuitem_insert(menu, mi) == LX_ERROR)
		mv_die(errno, "add_pullright:  unable to insert item in menu");
}

void
set_item_state(menu, item_name, state)
Menu *menu;
char *item_name;
int state;
{
	if(menuitem_set(menuitem_find(menu, LXMI_STRING, item_name),
			LXMI_STATE, state,
			LXMI_NULL) == LX_ERROR)
		mv_alert("set_item_state: unable to set item.");
}

Menu *
new_menu(win, title)
Window win;
char *title;
{
	Menu *menu;
        if ((menu= menu_create(progname, dpy, win,
               		LXM_TITLE, title,
			LXM_FONT, menu_font, 
			LXM_FOREGROUND, myfg, 
			LXM_BACKGROUND, mybg, 
			LXM_NULL)) == (Menu *) NULL)
                exit(-1);
	new_menu_info(menu);
	return menu;
}

void
destroy_menu(menu)
Menu *menu;
{
	Menu *pm;
	Menu_item *mi;
	char *mem;

	while((mi = menuitem_find(menu, LXMI_IMAGE, NULL)) !=
			(Menu_item *) NULL) {
		if((pm = (Menu *) menuitem_get(mi, LXMI_PULLRIGHT))
				!= (Menu *) NULL) {
			destroy_menu(pm);
		}
		if(mem = (char *) menuitem_get(mi, LXMI_CLIENTDATA))
			cfree(mem);
		menuitem_delete(menu, mi);
		menuitem_destroy(mi);
	}
	if(mem = (char *) menu_get(menu, LXM_CLIENTDATA))
		cfree(mem);
	menu_destroy(menu);
}

void 
set_menu_font(menu, font)
Menu *menu;
char *font;
{
	if(menu_set(menu, LXM_FONT, font, LXM_NULL) == LX_ERROR)
		mv_alert("set_menu_font:  unable to set font.");
}

void
set_line_mode(m, mi)	/* sets waveform display to line */
Menu *m;
Menu_item *mi;
{
        displaymode = 0;
	set_item_state(m, "Line style", LXMI_INACTIVE);
	set_item_state(m, "Shadow style", LXMI_ACTIVE);
        if(insert) set_line_loc();
        if(region) set_box_loc();
	display_it(1);        
}

void
set_shadow_mode(m, mi)	/* sets waveform display to shadow */
Menu *m;
Menu_item *mi;
{
        displaymode = 1;
	set_item_state(m, "Line style", LXMI_ACTIVE);
	set_item_state(m, "Shadow style", LXMI_INACTIVE);
        if(insert) set_line_loc();
        if(region) set_box_loc();
	display_it(1);        
}

void
new_file(m, mi)
Menu *m;
Menu_item *mi;
{
	int code;
	if(is_open) { /* if opening before closing old */
		close_file(m, mi);
		if(!is_saved) return;	/* whole operation cancelled */
	}
	if(dialog->call(dialog, D_FNEW, "") < 1) return;
	if(strlen(dialog->getText(dialog, 0)) == 0) {
		mv_alert( "You must give this file a name.");
		if(dialog->call(dialog, D_FNEW, "") < 1) return;
	}
	block_events();
	is_new = is_saved = 1;		/* flags for new (non-disk) file */
	strcpy(v->sfname, dialog->getText(dialog, 0));
	sfsrate(&v->sfh) = dialog->getValue(dialog, 1);
	code = dialog->getChoice(dialog, 0);
	sfchans(&v->sfh) = (code == 0) ? 1 : 2;
	v->sfh.sfinfo.sf_magic = SF_MAGIC;
	v->sfh.sfinfo.sf_codes = 0;
	v->is_float = dialog->getChoice(dialog, 1);
	skipflag = durflag = 0;
	(void) set_grainsize(30.0);	/* reset this first */
	reset_screen_time();
	open_sfile(v->sfname);	/* open and set menus */
	if(is_open)  {
		is_saved = 0;
		display_it(1);
	}
	unblock_events();
}

void
open_file(m, mi)
Menu *m;
Menu_item *mi;
{
	extern FLAG durflag, skipflag;
	char newsfname[256];
	int code;
	if(is_open) { /* if opening before closing old */
		close_file(m, mi);
		if(!is_saved) return;	/* whole operation cancelled */
	}
	if(dialog->call(dialog, D_FOPEN, pathname) < 1) return;
	if(strlen(dialog->getText(dialog, 0)) == 0) {
		mv_alert( "You must give this file a name.");
		if(dialog->call(dialog, D_FOPEN, pathname) < 1) return;
		open_file(m, mi);
		return;
	}
	strcpy(newsfname, dialog->getText(dialog, 0));
	if(newsfname[strlen(newsfname)-1] == '/') {
		mv_alert( "Sfname can't be a directory.");
		if(dialog->call(dialog, D_FOPEN, pathname) < 1) return;
		open_file(m, mi);
		return;
	}
	block_events();
	skipflag = durflag = 0;	   		/* default is flags off */ 
	skiptime = dialog->getValue(dialog, 1);
	durtime = dialog->getValue(dialog, 2);
	if(skiptime != 0.0) skipflag = 1;
	if(durtime > 0.0) durflag = 1;
	if(is_open && !is_new) close_sfile(); /* close prev. file if open */
	strcpy(v->sfname, newsfname);	      /* copy in new filename */
	is_new = 0;
	is_saved = 1;
	mesg->On(mesg, "Reading soundfile...");
	open_sfile(v->sfname);
	if(is_open) {		/* if successfully opened */
		chscale = 1;
		display_it(1);
	}
	mesg->Off(mesg);
	unblock_events();
}

void
save_file(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	if(read_only) {
		dialog->call(dialog, D_RONLY, v->sfname);
		return;
	}
	if(skipflag || durflag) /* warn if saving segment back to orig. */
		if(dialog->call(dialog, D_SAVWARN, "") < 0) return;
	if(v->size != 1 && wasmulaw)
		if(dialog->call(dialog, D_WASMULAW, "") < 0) return;
	block_events();
	mesg->On(mesg, "Writing soundfile...");
	if(write_sf(v->sfname) >= 0) is_saved = 1;
	unblock_events();
	mesg->Off(mesg);
}

void
saveto_file(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	char newsfname[256];
#if NeXT_STYLE_HEADER
	int changed = False;
#endif
	if(dialog->call(dialog, D_FSAVETO, v->sfname) < 1) return;
	strcpy(newsfname, dialog->getText(dialog, 0));
	if(!strcmp(newsfname, "untitled") || !strlen(newsfname)) {
		mv_alert( "You must give this file a name.");
		saveto_file(menu, menu_item);
		return;
	}
	if(newsfname[strlen(newsfname)-1] == '/') {
		mv_alert( "Sfile name can't be a directory.");
		saveto_file(menu, menu_item);
		return;
	}
	if(v->size != 1 && wasmulaw)
		if(dialog->call(dialog, D_WASMULAW, "") < 0) return;
#if NeXT_STYLE_HEADER
	if(isNative(&v->sfh) != dialog->getChoice(dialog, 0)) {
		changeFileType(&v->sfh); /* change soundfile type */
		changed = True;
	}
#endif
	block_events();
	mesg->On(mesg, "Writing soundfile...");
	if(write_sf(newsfname) >= 0) {
		is_saved = 1;
		set_file_menu();
	}
#if NeXT_STYLE_HEADER
	if(changed) changeFileType(&v->sfh); /* change file type back */
#endif
	unblock_events();
	mesg->Off(mesg);
}

void
close_file(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	if(!is_saved) {
		switch (dialog->call(dialog, D_FCLOSE, v->sfname)) {
		case YES:
			if(!read_only && !is_new) save_file(menu, menu_item);
			else saveto_file(menu, menu_item);
			if(!is_saved) return;	/* if operation cancelled */
			break;
		case NO:
			break;
		case CANCEL:
			return;
		}
	}
	set_frame_name("none");
	close_sfile();
}

void
revert_file(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	if(dialog->call(dialog, D_FREVERT, "") < 1) return;
	block_events();
	mesg->On(mesg, "Re-reading soundfile...");
	open_sfile(v->sfname);
	if(is_open) {
		is_saved = 1;
		reset_screen((Panel *) NULL, (Panel_item *) NULL);
	}
	else close_sfile();
	unblock_events();
	mesg->Off(mesg);
}

void
print_version(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	mv_alert(CURRENT_VERSION);
}

void
exit_prog(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	(void) quit((Panel *) NULL, (Panel_item *) NULL);
}

void
set_internal(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	if(yesbuff) is_buff = BUF_INT;	
	else is_buff = BUF_NONE;
	set_item_state(menu, "Internal", LXMI_INACTIVE);
	set_item_state(menu, "External", LXMI_ACTIVE);
	set_menus();
}

void
set_external(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	is_buff = BUF_EXT;
	set_item_state(menu, "Internal", LXMI_ACTIVE);
	set_item_state(menu, "External", LXMI_INACTIVE);
	set_menus();
}

reset_buff_mode()	/* called by other routines to reset buff to internal */
{
	set_internal(buffer_mode_menu, (Menu_item *) NULL);
}

void
set_peak_rescan(m, mi)
	Menu *m;
	Menu_item *mi;
{
	/* see which menu item was selected */
	if(mi == menuitem_find(m, LXMI_STRING, peak_rescan_cmds[0])) {
		set_menu_items(m, peak_rescan_cmds, 0x06);
		peak_rescan = 1;	/* On */
		mesg->On(mesg, "Now scanning for peak amp...");
		block_events();
		v->peakamp = get_peakamp(1);
		mesg->Off(mesg);
		unblock_events();
	}
	else if(mi == menuitem_find(m, LXMI_STRING, peak_rescan_cmds[1])) {
		set_menu_items(m, peak_rescan_cmds, 0x05);
		peak_rescan = 0;	/* Off */
	}	
	else if(mi == menuitem_find(m, LXMI_STRING, peak_rescan_cmds[2])) {
		set_menu_items(m, peak_rescan_cmds, 0x03);
		peak_rescan = 2;	/* Query */
	}
	else 
		mv_alert("set_peak_rescan:  menu item not found.");
	if(peak_rescan != 1 && v->size == SF_FLOAT)
		mv_alert("Warning: This setting will slow down D/A conversion.");
}

void
set_quick_display(m, mi)
	Menu *m;
	Menu_item *mi;
{
	/* see which menu item was selected */
	if(mi == menuitem_find(m, LXMI_STRING, quick_display_cmds[0])) {
		set_menu_items(m, quick_display_cmds, 0x02);
		quick_display = 1;	/* On */
	}	
	else if(mi == menuitem_find(m, LXMI_STRING, quick_display_cmds[1])) {
		set_menu_items(m, quick_display_cmds, 0x01);
		quick_display = 0;	/* Off */
	}
	else
		mv_alert("set_quick_display:  menu item not found.");
	display_it(1);        
}

void
set_sf_search(m, mi)
	Menu *m;
	Menu_item *mi;
{
	Menu *oldmenu;
	block_events();
	mesg->On(mesg, "Rescanning soundfile directory...");
	/* see which menu item was selected */
	if(mi == menuitem_find(m, LXMI_STRING, sf_search_cmds[0])) {
		set_menu_items(m, sf_search_cmds, 0x06);
		sf_search = 0;	/* All */
	}
	else if(mi == menuitem_find(m, LXMI_STRING, sf_search_cmds[1])) {
		set_menu_items(m, sf_search_cmds, 0x05);
		sf_search = 1;	/* Only files with proper suffix */
	}
	else if(mi == menuitem_find(m, LXMI_STRING, sf_search_cmds[2])) {
		set_menu_items(m, sf_search_cmds, 0x03);
		sf_search = 2;	/* Only files with valid sfheader */
	}
	else mv_alert("set_sf_search:  menu item not found.");

	oldmenu = (Menu *) menuitem_get(menuitem_find(file_menu, 
		LXMI_STRING, "Open"), LXMI_PULLRIGHT);
	destroy_menu(oldmenu);
	top_level = TRUE;
	menuitem_set(menuitem_find(file_menu, LXMI_STRING, "Open"),
		     LXMI_PULLRIGHT, make_dirmenu(v->sfdir),
		     LXMI_NULL);
	unblock_events();
	mesg->Off(mesg);
}

void
display_header(m, mi)
	Menu *m;
	Menu_item *mi;
{
	dialog->call(dialog, D_SFINFO, "");
}

void
change_format(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	int format, newsize;
	if(dialog->call(dialog, D_SFORMAT, "") < 0) return;
	format = dialog->getChoice(dialog, 0);
	newsize = format==0 ? 1 : (format==1 ? 2 : 4);
	if(newsize==v->size) {
		mv_alert("File is already in that format.");
		return;
	}
	mesg->On(mesg, "Converting...");
	block_events();
	if(newsize==2 && v->size==4) floatToShort((ebuf_struct *) v);
	else if(newsize==4 && v->size==2) shortToFloat((ebuf_struct *) v);
#if NeXT_STYLE_HEADER
	else if(newsize==2 && v->size==1) muToShort((ebuf_struct *) v);
	else if(newsize==1 && v->size==2) shortToMu((ebuf_struct *) v);
#endif
	else {
		mv_alert("This operation not yet available.");
		mesg->Off(mesg);
		unblock_events();
		return;
	}
	v->sfdur = (double) v->bufsize/(v->size * v->srate * v->nchans);
	set_screen_times(1);
	set_view_display();
	is_saved = 0;
	chscale = 1;
	if(insert) set_line_loc();
	if(region) set_box_loc();
	display_it(1);
	mesg->Off(mesg);
	unblock_events();
}

void
rescale(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	cbuf_struct cb;
	block_events();
	set_edit(&cb, v, 0.0, v->sfdur);
	mesg->On(mesg, "Rescaling soundfile...");
	mv_rescale(&cb);
	v->peakamp = (v->size==1)?256.0:32767.0;
	is_saved = 0;
	chscale = 1;
	if(insert) set_line_loc();
	if(region) set_box_loc();
	display_it(1);
	mesg->Off(mesg);
	unblock_events();
}

void
change_srate(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	if(dialog->call(dialog, D_SRATE, "") < 0) return;
	v->srate = dialog->getValue(dialog, 0);
	set_grainsize(v->grainsize);
	v->sfdur = (double) v->bufsize/(v->srate*v->nchans*v->size);
	set_screen_times(1);
	is_saved = 0;
	if(insert) set_line_loc();
	if(region) set_box_loc();
	set_view_display();
	display_it(1);
}

extern cbuf_struct *c;

void
change_length(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	int newsize;
	int remainder;
	if(dialog->call(dialog, D_SFLEN, "") < 0) return;
	newsize = dialog->getValue(dialog, 0) * v->srate * v->size;
	/* round to sample byte boundry */
	if((remainder = newsize % v->size)) newsize -= remainder;
	newsize *= v->nchans;
	if(newsize < v->bufsize) 
		if(dialog->call(dialog, D_LENWARN, "") < 0) return;
	if((newsize = setLength((unsigned) newsize)) < 0) return;
	v->sfdur = (double) newsize/(v->size * v->srate * v->nchans);
	is_saved = 0;
	set_screen_times(1);
	if(insert) set_line_loc();
	if(region) set_box_loc();
	set_view_display();
	display_it(0);
}

void 
change_sfcomment(m, mi)
Menu *m;
Menu_item *mi;
{
	char comment[MAXCOMM], tempname[36], *spacer = " ";
	int comlen, tfd;
	get_comment(comment, &comlen);
	sprintf(tempname, "/tmp/comment.%d", getpid());
	if(comlen == 0) {	/* if no previous comment in header */
		if((tfd = open(tempname,O_CREAT|O_RDWR,0644)) < 0) {
			mv_error(errno, "Cannot open temp comment file.");
			return;
		}
		write(tfd, spacer, strlen(spacer) + 1);
		close(tfd);
		if(edit_text(tempname) < 0) return;
	}
	else {	/* write previous comment out to file to be edited */
		tfd = open(tempname,O_CREAT|O_RDWR,0644);
		if(write(tfd, comment, comlen) < 0) {
			mv_error(errno, "Cannot write to temp comment file.");
			close(tfd);
		}
		close(tfd);
		if(edit_text(tempname) < 0) return;
	}
	is_saved = 0;
}

void
do_fft(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	FFTDATA fd;
	cbuf_struct cb;
	double start, end;
	int display_data = 0;
	if(dialog->call(dialog, D_QFFT, "") < 0) return;
	block_events();
	clear_events();
	fd.fftfname = dialog->getText(dialog, 0);
	fd.npoints = (double) pow(2.0, 6.0 + dialog->getChoice(dialog, 0));
	fd.srate = v->srate;
	fd.mode = dialog->getChoice(dialog, 1); /* amp display mode */
	fd.use_header = dialog->getChoice(dialog, 2); /* data file header? */
	display_data = dialog->getChoice(dialog, 3); /* display results? */
	fd.comment = "";
	start = (c->edstart < 0.0) ? 0 : c->edstart;
	end = (c->edend > v->sfdur) ? v->sfdur : c->edend;
	set_edit(&cb, v, start, end);
	if(!fftran(&cb, &fd))
		mv_alert( "Sorry...FFT failed.");
	else if(display_data)
		display_fft_data(&fd);
	unblock_events();
}

void
do_lpc(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	LPDATA ld;
	cbuf_struct cb;
	double start, end;
	int display_data = 0;
	if(dialog->call(dialog, D_QLPC, "") < 0) return;
	block_events();
	clear_events();
	ld.npoles = dialog->getValue(dialog, 0);
	ld.offset = dialog->getValue(dialog, 1);
	if(dialog->getValue(dialog, 2) > 0.0)
		ld.offset = v->srate/dialog->getValue(dialog, 2);
	ld.lpfname = dialog->getText(dialog, 3);
	ld.comment = "";
	ld.use_header = dialog->getChoice(dialog, 0);
	display_data = dialog->getChoice(dialog, 1);
	start = (c->edstart < 0.0) ? 0 : c->edstart;
	end = (c->edend > v->sfdur) ? v->sfdur : c->edend;
	set_edit(&cb, v, start, end);
	if(!anallpc(&cb, &ld))
		mv_alert( "Sorry...LPC failed.");
	else if(display_data)
		display_lpc_data(&ld);
	unblock_events();
}

void
do_ptrack(menu, menu_item)
Menu *menu;
Menu_item *menu_item;
{
	PTDATA ptd;
	cbuf_struct cb;
	double start, end;
	int display_data = 0;
	if(dialog->call(dialog, D_QPTRACK, "") < 0) return;
	block_events();
	clear_events();
	ptd.framesize = dialog->getValue(dialog, 0);
	ptd.offset = dialog->getValue(dialog, 1);
	if(dialog->getValue(dialog, 2) > 0.0)
		ptd.offset = v->srate/dialog->getValue(dialog, 2);
	ptd.highest = dialog->getValue(dialog, 3);
	ptd.lowest = dialog->getValue(dialog, 4);
	ptd.ptfname = dialog->getText(dialog, 5);
	display_data = dialog->getChoice(dialog, 0);
	start = (c->edstart < 0.0) ? 0 : c->edstart;
	end = (c->edend > v->sfdur) ? v->sfdur : c->edend;
	set_edit(&cb, v, start, end);
	if(!ptrack(&cb, &ptd))
		mv_alert( "Sorry...Pitch track failed.");
	else if(display_data)
		display_ptrack_data(&ptd);
	unblock_events();
}

extern DIR *opendir();

void
change_sfdir(m, mi)
Menu *m;
Menu_item *mi;
{
	Menu *oldmenu;
	char message[256];
	if(dialog->call(dialog, D_QSFDIR, v->sfdir) < 0) return;
	if(!strlen(dialog->getText(dialog, 0))) return;
	block_events();
	if(opendir(dialog->getText(dialog, 0)) == (DIR *) NULL) {
		sprintf(message,"Cannot open %s.",dialog->getText(dialog,0));
		mv_error(errno, message);
		unblock_events();
		return;
	}
	mesg->On(mesg, "Rescanning soundfile directory...");
	strcpy(v->sfdir, dialog->getText(dialog, 0));
	oldmenu = (Menu *) menuitem_get(menuitem_find(m, LXMI_STRING, "Open"),
			LXMI_PULLRIGHT);
	destroy_menu(oldmenu);
	top_level = TRUE;
	menuitem_set(menuitem_find(m, LXMI_STRING, "Open"),
		     LXMI_PULLRIGHT, make_dirmenu(v->sfdir),
		     LXMI_NULL);
	mesg->Off(mesg);
	unblock_events();
}

Menu *
make_dirmenu(dirname)	/* creates menu of all appropriate files is dir */
char *dirname;
{
	struct direct *dp;
	struct stat fst;
	DIR *dirp;
	Menu *dir_menu, *menu, *extension_menu, *submenu;	
	int nitems = 0, maxitems = 30;

	if((dirp = opendir(dirname)) == (DIR *) NULL) {
		perror("opendir");
		return (Menu *) NULL;
	}
	/* create menu to contain items at this level.  */

	dir_menu = new_menu(mwin, dirname);

	/* for each valid entry in dir, create menuitem.  Recursively descend
	   into subdirs to do the same, creating new submenus   */
	
	menu = dir_menu;
	for(dp = readdir(dirp); dp != (struct direct*) NULL; dp = readdir(dirp))
	{
		char path[1024];
		int no_entries = 0;
		sprintf(path, "%s/%s", dirname, dp->d_name);
		if(stat(path, &fst) < 0) 
			continue;		/* skip bad entries */
		if(strcmp(dp->d_name, ".")== 0 || strcmp(dp->d_name, "..")== 0)
			continue;		/* dont do self or parent */
		switch(fst.st_mode & S_IFMT) {
		case S_IFLNK:	/* try to avoid chasing linked dirs! */
			break;
		case S_IFDIR:	/* descend recursively into lower dirs */
			submenu = make_dirmenu(path);
			if(submenu == (Menu *) NULL) { /* dir unreadable */
				add_menuitem(menu, dp->d_name, PROC_NULL, 0);
				set_item_state(menu, dp->d_name, LXMI_INACTIVE);
				nitems++;
				break;
			}
			else if(get_nitems(submenu) == 0) no_entries = 1;
			add_pullright(menu, dp->d_name, PROC_NULL, submenu);
			if(no_entries)
				set_item_state(menu, dp->d_name, LXMI_INACTIVE);
			nitems++;
			break;
		case S_IFREG:	/* add item if file is soundfile */
			if(is_soundfile(path, sf_search)) {
				add_menuitem(menu, dp->d_name, set_sfname, 0);
				nitems++;
			}
			break;
		default:
			break;
		}
		if(nitems > maxitems) {
			add_pullright(dir_menu, "(MORE)", PROC_NULL, 
				extension_menu = new_menu(mwin, dirname));
			menu = extension_menu;
			nitems =0;
		} 
	}
	closedir(dirp);
	set_nitems(dir_menu, nitems);
	return dir_menu;
}
		
void
set_sfname(m, mi)
Menu *m;
Menu_item *mi;
{
	sprintf(pathname, "%s/%s", 
		(char *) menu_get(m, LXM_TITLE),
		(char *) menuitem_get(mi, LXMI_STRING));
}

void
set_new_grainsize(m, mi)
Menu *m;
Menu_item *mi;
{
	if(dialog->call(dialog, D_QGRAIN, "") < 0) return;
	set_grainsize(dialog->getValue(dialog, 0));
	(void) set_grain((Panel *) NULL, (Panel_item *) NULL);
}

void
set_new_scale(m, mi)
Menu *m;
Menu_item *mi;
{
	double newscale;
	if(dialog->call(dialog, D_QVSCALE, "") < 0) return;
	newscale = dialog->getValue(dialog, 0);
	ampfac = v->peakamp/newscale;
	chscale = 1;
	display_it(0);
}

void
set_screentimes(m, mi)
Menu *m;
Menu_item *mi;
{
	double grainfac; 
	if(dialog->call(dialog, D_QSTIMES, "") < 0) return;
	if(dialog->getValue(dialog, 0) < 0.0) {
		mv_alert( "Start time must be > 0.00!");
		set_screentimes(m, mi);
		return;
	}
	if(dialog->getValue(dialog, 1) <= dialog->getValue(dialog, 0)) {
		mv_alert( "End time must be > start time!");
		set_screentimes(m, mi);
		return;
	}
	grainfac = (dialog->getValue(dialog, 1) - dialog->getValue(dialog, 0))/ (v->etime - v->stime);
	v->stime = dialog->getValue(dialog, 0);
	v->etime = dialog->getValue(dialog, 1);
	set_grainsize(v->grainsize * grainfac);
	v->egrain = v->etime/v->secgrain;
	v->sgrain = v->stime/v->secgrain;
	set_screen_times(1);
	if(insert) set_line_loc();
	if(region) set_box_loc();
	set_view_display();
	display_it(1);
}

set_menus()
{
	if(region)		/* edit region has been selected */
	{
		set_menu_items(edit_menu, edit_cmds, EDIT_REGION);
		set_menu_items(alter_menu, alter_cmds, ALTER_REGION);
		set_menu_items(filter_menu, filter_cmds, FILTER_REGION);
		set_menu_items(analysis_menu, analysis_cmds, ANALYSIS_REGION);
	}
	else if(insert&is_buff) /* insert pt. set & cut buffer exists */
	{		
		set_menu_items(edit_menu, edit_cmds, EDIT_INSERT);
		set_menu_items(alter_menu, alter_cmds, ALTER_INSERT);
		set_menu_items(filter_menu, filter_cmds, FILTER_NONE);
		set_menu_items(analysis_menu, analysis_cmds, 0x0000);
	}
	else			/* default settings */
	{
		set_menu_items(edit_menu, edit_cmds, EDIT_NONE);
		if(insert) 
		   set_menu_items(alter_menu, alter_cmds, ALTER_INSERT);
		else set_menu_items(alter_menu, alter_cmds, ALTER_NONE);
		set_menu_items(filter_menu, filter_cmds, FILTER_NONE);
		set_menu_items(analysis_menu, analysis_cmds, 0x0000);
	}
}

set_file_menu()
{
	if(is_new) {
		set_menu_items(file_menu, file_cmds, FILE_NEW);
		set_menu_items(header_menu, header_cmds, HEADER_OPEN);
	}
	else if(is_open)  {
		set_menu_items(file_menu, file_cmds, FILE_OPEN);
		set_menu_items(header_menu, header_cmds, HEADER_OPEN);
	}
	else {
		set_menu_items(file_menu, file_cmds, FILE_CLOSED);
		set_menu_items(header_menu, header_cmds, HEADER_CLOSED);
	}
}

