#include <stdio.h>
#include <varargs.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <sys/file.h>

#ifdef IRIX
#include <unistd.h>
#endif IRIX

#include "lxt.h"

#include "xvcursor.bitmap"
#include "xvvscursor.bitmap"
#include "xvhscursor.bitmap"
extern Pixmap xvcursor_pm, xvvscursor_pm, xvhscursor_pm;
extern boolean xvcursor_init;
extern boolean xt_usrresize;

extern Void *lxt_selsrc;

#include "grey.bitmap"

static boolean xt_init= FALSE;
char xvt_findbuf[LXT_FINDBUFSZ];		/* text to be searched for */
char *xvt_pastebuf;				/* buffer for highlighted text */
int xvt_pastebufsz;				/* size of the paste buffer */

Textsw *xt_textsws= (Textsw *) NULL;

/*VARARGS*/
Textsw *
textsw_create(va_alist)
va_dcl
{
	va_list varg_ptr;
	int nargs, attr, depth, scr;
	Display *dpy;
	Window win;
	Textsw *t;
	char *name, *infilenm, *outfilenm;
	GC gc;
	XGCValues gcv;
	XSetWindowAttributes xswa;
	XWindowAttributes xwa;
	XImage image;
	XColor fg_xc, bg_xc;
	Colormap cmap;
	int textsw_filldefaults();
	void textsw_init();
	int textsw_config();
	void textsw_menucreate();
	void textsw_draw(), textsw_display();
	void textsw_drawscrolls();

	t= (Textsw *) NULL;
	infilenm= outfilenm= (char *) NULL;
	va_start(varg_ptr);
	for (nargs= 0;; nargs++) {

		/* get program name */
		if (nargs == 0) {
			name= va_arg(varg_ptr, char *);
			if (name == (char *) NULL) {
				(void) fprintf(stderr, "textsw_create: null program name\n");
				return((Textsw *) NULL);
			}
		}

		/* get display */
		else if (nargs == 1) {
			dpy= va_arg(varg_ptr, Display *);
			if (dpy == (Display *) NULL) {
				(void) fprintf(stderr, "textsw_create: null display\n");
				return((Textsw *) NULL);
			}
		}

		/* get enclosing window */
		else if (nargs == 2) {
			win= va_arg(varg_ptr, Window);
			if (win == None) {
				(void) fprintf(stderr, "textsw_create: null parent window\n");
				return((Textsw *) NULL);
			}
			if ((t= (Textsw *) calloc(1, sizeof(Textsw))) == (Textsw *) NULL) {
				(void) fprintf(stderr, "textsw_create: memory allocation error\n");
				return((Textsw *) NULL);
			}
			t->xt_magic= LX_TEXTSW;
			t->xt_dpy= dpy;
			t->xt_win= win;
			if (textsw_filldefaults(name, t) != LX_SUCCESS) {
				cfree((char *) t);
				return((Textsw *) NULL);
			}
			if (!xt_init)
				textsw_init(t->xt_dpy);
		}

		else {
			char *c;
			int n;

			attr= va_arg(varg_ptr, int);
			if (attr == LXT_NULL)
				break;

			switch (attr) {
			case LXT_INPUTFILE:
				infilenm= va_arg(varg_ptr, char *);
				break;
			case LXT_OUTPUTFILE:
				outfilenm= va_arg(varg_ptr, char *);
				break;
			case LXT_FOREGROUND:
				t->xt_fg= va_arg(varg_ptr, unsigned long);
				break;
			case LXT_BACKGROUND:
				t->xt_bg= va_arg(varg_ptr, unsigned long);
				break;
			case LXT_FONT:
				c= va_arg(varg_ptr, char *);
				if (c == (char *) NULL) {
					(void) fprintf(stderr, "textsw_create: null fontname\n");
					cfree((char *) t);
					return((Textsw *) NULL);
				}
				if ((t->xt_font= XLoadQueryFont(t->xt_dpy, c)) == NULL) {
					(void) fprintf(stderr, "textsw_create: cannot load font %s\n", c);
					cfree((char *) t);
					return((Textsw *) NULL);
				}
				cfree(t->xt_fontnm);
				if ((t->xt_fontnm= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
					(void) fprintf(stderr, "textsw_create: memory allocation error\n");
					cfree((char *) t);
					return((Textsw *) NULL);
				}
				(void) strcpy(t->xt_fontnm, c);
				break;
			case LXT_ALLOWINPUT:
				n= va_arg(varg_ptr, int);
				if (n)
					t->xt_allowinput= TRUE;
				else
					t->xt_allowinput= FALSE;
				break;
			case LXT_PASTEBUFFER:
				c= va_arg(varg_ptr, char *);
				if (xvt_pastebuf != NULL)
					free(xvt_pastebuf);
				if ((xvt_pastebuf= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
					(void) fprintf(stderr, "textsw_create: memory allocation error\n");
					cfree((char *) t);
					return((Textsw *) NULL);
				}
				(void) strcpy(xvt_pastebuf, c);
				xvt_pastebufsz= strlen(c);
				break;
			case LXT_CHARPROC:
				t->xt_charproc= (int (*)()) va_arg(varg_ptr, int);
				break;
			case LXT_EXITPROC:
				t->xt_exitproc= (int (*)()) va_arg(varg_ptr, int);
				break;
			case LXT_TCURSOR:
				t->xt_tcursor= va_arg(varg_ptr, Cursor);
				break;
			case LXT_VSCURSOR:
				t->xt_vscursor= va_arg(varg_ptr, Cursor);
				break;
			case LXT_HSCURSOR:
				t->xt_hscursor= va_arg(varg_ptr, Cursor);
				break;
			case LXT_BARWIDTH:
				t->xt_vscroll->xs_barwidth= t->xt_hscroll->xs_barwidth= va_arg(varg_ptr, int);
				break;
			case LXT_BUBBLEMARGIN:
				t->xt_vscroll->xs_bubblemargin= t->xt_hscroll->xs_bubblemargin= va_arg(varg_ptr, int);
				break;
			case LXT_BUTTONLEN:
				t->xt_vscroll->xs_buttonlen= t->xt_hscroll->xs_buttonlen= va_arg(varg_ptr, int);
				break;
			case LXT_CLIENTDATA:
				t->xt_clientdata= va_arg(varg_ptr, char *);
				break;
			case LXT_DRAWCURSOR:
			case LXT_DRAWHIGHLIGHT:
			case LXT_CURSORLINE:
			case LXT_CURSORCOLUMN:
			case LXT_INSERTTEXT:
			case LXT_HIGHLIGHTENDLINE:
			case LXT_HIGHLIGHTENDCOLUMN:
			case LXT_CURSORVIEWABLE:
			case LXT_NUMLINES:
			case LXT_MENU: 
			case LXT_TOPLINE: 
				(void) fprintf(stderr, "textsw_create: attribute may not be set at creation time\n");
				break;
			default:
				(void) fprintf(stderr, "textsw_create: ignoring unrecognized attribute\n");
				break;
			}
		}
	}
	va_end(varg_ptr);

	XGetWindowAttributes(t->xt_dpy, t->xt_win, &xwa);
	t->xt_twin= XCreateSimpleWindow(t->xt_dpy, t->xt_win, 0, 0, xwa.width, xwa.height, 0, t->xt_fg, t->xt_bg);
	t->xt_vswin= XCreateSimpleWindow(t->xt_dpy, t->xt_win, 0, 0, xwa.width, xwa.height, 0, t->xt_fg, t->xt_bg);
	t->xt_hswin= XCreateSimpleWindow(t->xt_dpy, t->xt_win, 0, 0, xwa.width, xwa.height, 0, t->xt_fg, t->xt_bg);

	xswa.colormap= xwa.colormap;
	XChangeWindowAttributes(t->xt_dpy, t->xt_twin, CWColormap, &xswa);
	XChangeWindowAttributes(t->xt_dpy, t->xt_vswin, CWColormap, &xswa);
	XChangeWindowAttributes(t->xt_dpy, t->xt_hswin, CWColormap, &xswa);

	XSelectInput(t->xt_dpy, t->xt_win, ExposureMask | StructureNotifyMask);
	XSelectInput(t->xt_dpy, t->xt_twin, Button1MotionMask | ExposureMask | ButtonPressMask | ButtonReleaseMask | StructureNotifyMask | KeyPressMask | PropertyChangeMask);
	XSelectInput(t->xt_dpy, t->xt_hswin, ExposureMask | ButtonPressMask | ButtonReleaseMask | EnterWindowMask);
	XSelectInput(t->xt_dpy, t->xt_vswin, ExposureMask | ButtonPressMask | ButtonReleaseMask | EnterWindowMask);

	/* create grey stipple */
	t->xt_stipplepm= XCreatePixmap(t->xt_dpy, DefaultRootWindow(t->xt_dpy), grey_width, grey_height, 1);
	gcv.foreground= t->xt_fg;
	gcv.background= t->xt_bg;
	gc= XCreateGC(dpy, t->xt_stipplepm, GCForeground | GCBackground, &gcv);
	image.height= grey_height;
	image.width= grey_width;
	image.xoffset= 0;
	image.format= XYBitmap;
	image.data= grey_bits;
	image.byte_order= LSBFirst;
	image.bitmap_unit= 8;
	image.bitmap_bit_order= LSBFirst;
	image.bitmap_pad= 8;
	image.bytes_per_line= (grey_width+7)>>3;
	image.depth= 1;
	XPutImage(dpy, t->xt_stipplepm, gc, &image, 0, 0, 0, 0, grey_width, grey_height);
	XFreeGC(dpy, gc);

	/* create cursors */
	cmap= xwa.colormap;
	fg_xc.pixel= t->xt_fg;
	XQueryColor(t->xt_dpy, cmap, &fg_xc);
	fg_xc.flags= DoRed | DoGreen | DoBlue;
	bg_xc.pixel= t->xt_fg;
	XQueryColor(t->xt_dpy, cmap, &bg_xc);
	bg_xc.flags= DoRed | DoGreen | DoBlue;
	if (t->xt_tcursor == None)
		t->xt_tcursor= XCreatePixmapCursor(t->xt_dpy, xvcursor_pm, xvcursor_pm, &fg_xc, &bg_xc, xvcursor_x_hot, xvcursor_y_hot);
	if (t->xt_vscursor == None)
		t->xt_vscursor= XCreatePixmapCursor(t->xt_dpy, xvvscursor_pm, xvvscursor_pm, &fg_xc, &bg_xc, xvvscursor_x_hot, xvvscursor_y_hot);
	if (t->xt_hscursor == None)
		t->xt_hscursor= XCreatePixmapCursor(t->xt_dpy, xvhscursor_pm, xvhscursor_pm, &fg_xc, &bg_xc, xvhscursor_x_hot, xvhscursor_y_hot);
	XDefineCursor(t->xt_dpy, t->xt_twin, t->xt_tcursor);
	XDefineCursor(t->xt_dpy, t->xt_hswin, t->xt_hscursor);
	XDefineCursor(t->xt_dpy, t->xt_vswin, t->xt_vscursor);

	/* create GCs */
	gcv.font= t->xt_font->fid;
	gcv.foreground= t->xt_fg;
	gcv.background= t->xt_bg;
	gcv.plane_mask= AllPlanes;
	gcv.function= GXcopy;
	t->xt_gc= XCreateGC(t->xt_dpy, t->xt_twin, (GCFont | GCForeground | GCBackground | GCPlaneMask | GCFunction), &gcv);
	gcv.plane_mask= t->xt_fg ^ t->xt_bg;
	gcv.function= GXinvert;
	t->xt_igc= XCreateGC(t->xt_dpy, t->xt_twin, (GCFont | GCForeground | GCBackground | GCPlaneMask | GCFunction), &gcv);
	gcv.function= GXcopy;
	gcv.stipple= t->xt_stipplepm;
	gcv.fill_style= FillStippled;
	t->xt_sgc= XCreateGC(t->xt_dpy, t->xt_twin, (GCFont | GCForeground | GCBackground | GCFunction | GCStipple | GCFillStyle), &gcv);
	gcv.function= GXxor;
	t->xt_xgc= XCreateGC(t->xt_dpy, t->xt_twin, (GCFont | GCForeground | GCBackground | GCFunction), &gcv);
	gcv.foreground= t->xt_bg;
	gcv.background= t->xt_fg;
	gcv.plane_mask= AllPlanes;
	gcv.function= GXcopy;
	t->xt_cgc= XCreateGC(t->xt_dpy, t->xt_twin, (GCFont | GCForeground | GCBackground | GCPlaneMask | GCFunction), &gcv);

	/* read file if requested */
	if (infilenm != (char *) NULL) {
		if (textsw_readfile(t, infilenm, FALSE) != LX_SUCCESS) {
			(void) fprintf(stderr, "textsw_create: could not read file '%s'\n", infilenm);
			textsw_free(t);
			return((Textsw *) NULL);
		}
	}
	else
		(void) strcpy(t->xt_rdfilenm, "");
	if (outfilenm == (char *) NULL)
		(void) strcpy(t->xt_wrfilenm, t->xt_rdfilenm);
	else
		(void) strcpy(t->xt_wrfilenm, outfilenm);

	textsw_menucreate(t);
	t->xt_next= xt_textsws;
	xt_textsws= t;

	switch (xwa.map_state) {
	case IsUnviewable:
	case IsViewable:
		t->xt_flags|= LXP_FRAMEMAPPED;
		(void) textsw_config(t);
		XMapWindow(t->xt_dpy, t->xt_twin);
		textsw_draw(t);
		textsw_display(t);
		XMapWindow(t->xt_dpy, t->xt_vswin);
		XMapWindow(t->xt_dpy, t->xt_hswin);
		textsw_drawscrolls(t);
		break;
	case IsUnmapped:
		t->xt_flags&= ~LXT_FRAMEMAPPED;
		/* allocate a temporary pixmap so that the
                   calling routine is free to draw to the
                   text window - Pixmap will be destroyed when 
		   window eventually gets mapped */
		scr= DefaultScreen(t->xt_dpy);
		depth= DefaultDepth(t->xt_dpy, scr);
		if (t->xt_tpm != (Pixmap) None)
			XFreePixmap(t->xt_dpy, t->xt_tpm);
		t->xt_tpm= XCreatePixmap(t->xt_dpy, DefaultRootWindow(t->xt_dpy), 400, 400, depth);
		break;
	default:
		break;
	}
	return(t);
}

void
textsw_init(dpy)
Display *dpy;
{
	void cursor_init();

	/* create cursor pixmaps */
	if (!xvcursor_init)
		cursor_init(dpy);

	xvt_pastebuf= (char *) NULL;
	xvt_findbuf[0]= '\0';
	xvt_pastebufsz= 0;
	xt_usrresize= FALSE;
	xt_init= TRUE;
}

int
textsw_filldefaults(name, t)
/*
   Internal routine.
   Initialize values in Textsw structure.
*/
char *name;
Textsw *t;
{
	char *xdef, fontname[LX_MAXFONTNMLEN+1];
	int textsw_exitproc();

	if ((t->xt_name= calloc((unsigned) (strlen(name)+1), sizeof(char))) == (char *) NULL) {
		(void) fprintf(stderr, "textsw_filldefaults: memory allocation error\n");
		return(LX_ERROR);
	}
	(void) strcpy(t->xt_name, name);

	/* font */
	if ((xdef= XGetDefault(t->xt_dpy, t->xt_name, "Font")) == (char *) NULL)
		(void) strcpy(fontname, LXPDEF_FONT);
	else
		(void) strncpy(fontname, xdef, LX_MAXFONTNMLEN);
	if ((t->xt_font= XLoadQueryFont(t->xt_dpy, fontname)) == NULL) {
		(void) fprintf(stderr, "textsw_filldefaults: cannot load font %s\n", fontname);
		return(LX_ERROR);
	}
	if ((t->xt_fontnm= calloc((unsigned) (strlen(fontname)+1), sizeof(char))) == (char *) NULL) {
		(void) fprintf(stderr, "textsw_filldefaults: memory allocation error\n");
		return(LX_ERROR);
	}
	(void) strcpy(t->xt_fontnm, fontname);

	/* internal border width */
	if ((xdef= XGetDefault(t->xt_dpy, t->xt_name, "LXTInternalBorderWidth")) == (char *) NULL)
		t->xt_ibw= 0;
	else {
		t->xt_ibw= atoi(xdef);
		if (t->xt_ibw < 0)
			t->xt_ibw= 0;
	}

	/* foreground and background */
	t->xt_fg= BlackPixel(t->xt_dpy, DefaultScreen(t->xt_dpy));
	t->xt_bg= WhitePixel(t->xt_dpy, DefaultScreen(t->xt_dpy));

	/* scrollbars */
	if ((t->xt_vscroll= (Scrollbar *) calloc(1, sizeof(Scrollbar))) == (Scrollbar *) NULL) {
		(void) fprintf(stderr, "textsw_filldefaults: memory allocation error\n");
		exit(-1);
	}
	t->xt_vscroll->xs_type= LXS_VERTICAL;
	t->xt_vscroll->xs_barwidth= LXSDEF_BARWIDTH;
	t->xt_vscroll->xs_bubblemargin= LXSDEF_BUBBLEMARGIN;
	t->xt_vscroll->xs_buttonlen= LXSDEF_BUTTONLEN;
	t->xt_vscroll->xs_bubblemin= 0;
	if ((t->xt_hscroll= (Scrollbar *) calloc(1, sizeof(Scrollbar))) == (Scrollbar *) NULL) {
		(void) fprintf(stderr, "textsw_filldefaults: memory allocation error\n");
		exit(-1);
	}
	t->xt_hscroll->xs_type= LXS_HORIZONTAL;
	t->xt_hscroll->xs_barwidth= LXSDEF_BARWIDTH;
	t->xt_hscroll->xs_bubblemargin= LXSDEF_BUBBLEMARGIN;
	t->xt_hscroll->xs_buttonlen= LXSDEF_BUTTONLEN;
	t->xt_hscroll->xs_bubblemin= 0;

	t->xt_w= LXTDEF_WIDTH;
	t->xt_h= LXTDEF_HEIGHT;
	t->xt_vscrolllen= t->xt_h - (2*t->xt_ibw) - t->xt_vscroll->xs_barwidth - 1 - (2*t->xt_vscroll->xs_buttonlen);
	t->xt_hscrolllen= t->xt_w - (2*t->xt_ibw) - t->xt_hscroll->xs_barwidth - 1 - (2*t->xt_hscroll->xs_buttonlen);
	t->xt_text= calloc(LXT_TEXTBUFSZ, sizeof(char));
	t->xt_linestarts= (int *) calloc(LXT_LSBUFSZ, sizeof(int));
	t->xt_linestarts[0]= 0;
	t->xt_lsbufsz= LXT_LSBUFSZ;
	t->xt_textbufsz= LXT_TEXTBUFSZ;
	t->xt_nbuflines= 0;
	t->xt_nbufchars= 0;
	t->xt_topline= 0;
	t->xt_leftpix= 0;
	t->xt_cheight= (int) (t->xt_font->max_bounds.ascent + t->xt_font->max_bounds.descent) + 2;
	t->xt_cdescent= (int) t->xt_font->max_bounds.descent;
	t->xt_cwidth=  (int) (t->xt_font->max_bounds.rbearing - t->xt_font->min_bounds.lbearing);
	t->xt_clbearing= (int) t->xt_font->min_bounds.lbearing;
	t->xt_maxlinelen= 0;
	t->xt_maxlinenum= 0;
	t->xt_irow= t->xt_icol= 0;
	t->xt_ichar= 0;
	t->xt_nspholders= 0;
	t->xt_exitproc= textsw_exitproc;

	/* flags */
	t->xt_flags= 0;

	/* the following should be incorporated into xt_flags! */
	t->xt_updatescroll= FALSE;
	t->xt_textedited= FALSE;
	t->xt_drawhlt= FALSE;
	t->xt_drawcursor= FALSE;
	t->xt_allowinput= TRUE;

	/* null fields */
	t->xt_tpm= (Pixmap) None;
	t->xt_tcursor= t->xt_vscursor= t->xt_hscursor= (Cursor) None;
	t->xt_charproc= (int (*)()) NULL;
	t->xt_clientdata= (char *) NULL;
	t->xt_next= (Textsw *) NULL;

	return(LX_SUCCESS);
}


int
textsw_readfile(t, filenm, alert_flag)
/* open file and set up arrays */
Textsw *t;
char *filenm;
int alert_flag;
{

	FILE *fopen(), *f;
	char err_string[300], str[LXT_MAXFILENM+1];
	int alert_val;

	int i, found;

	if ((f= fopen(filenm, "r")) == NULL) {
		if (alert_flag) {
			(void) strcpy(err_string, "Cannot open file '");
			(void) strcat(err_string, filenm);
			(void) strcat(err_string, "'.");
			(void) alert_prompt(t->xt_name, t->xt_dpy, &alert_val,
					LXA_LABEL, err_string,
					LXA_BUTTON, "Continue", 0,
					LXA_NULL);
		}
		return(LX_ERROR);
	}

	(void) strncpy(t->xt_rdfilenm, filenm, LXT_MAXFILENM);

	t->xt_nbuflines= 0;
	t->xt_nbufchars= 0;

	while (fgets(&t->xt_text[t->xt_nbufchars], LXT_RDBUFSZ, f) != NULL) {
		t->xt_linestarts[t->xt_nbuflines]= t->xt_nbufchars;
		t->xt_nbufchars += strlen(&t->xt_text[t->xt_nbufchars]);
		++t->xt_nbuflines;
		
		if (t->xt_lsbufsz - 10 < t->xt_nbuflines) {
			t->xt_linestarts= (int *) realloc((char *) t->xt_linestarts, (unsigned) ((t->xt_lsbufsz+LXT_LSBUFINC)*sizeof(int)));
			t->xt_lsbufsz += LXT_LSBUFINC;
		}

		if (t->xt_textbufsz - 1000 < t->xt_nbufchars) {
			t->xt_text= realloc(t->xt_text, (unsigned) ((t->xt_textbufsz+LXT_TEXTBUFINC)*sizeof(char)));
			t->xt_textbufsz +=  LXT_TEXTBUFINC;
		}

	}
	(void) fclose(f);
	t->xt_linestarts[t->xt_nbuflines]= t->xt_nbufchars;

	found= FALSE;
	for (i= 0; i < t->xt_nbufchars; i++) { 
		if(t->xt_text[i] == '\n')
			t->xt_text[i]= '\0';
		else if( t->xt_text[i] == '\t')
			continue;
		else if( (t->xt_text[i]<' ') || (t->xt_text[i]>'~') ) {
			t->xt_text[i]= LXT_SPHOLDER;
			found=TRUE;
		}
	}
	textsw_longestline(t);

	if (found) {
		(void) sprintf(str, "Warning -- unprintable characters in '%s'", filenm);
		(void) alert_prompt(t->xt_name, t->xt_dpy, &alert_val,
				LXA_LABEL, str,
				LXA_LABEL, "were removed from the text.",
				LXA_BUTTON, "Continue", 0,
				LXA_NULL);
	}

	return(LX_SUCCESS);
}

void
textsw_draw(t)
Textsw *t;
{
	int i, j, k, length, new_length, count;
	char s[512];
	int map[512];

	XFillRectangle(t->xt_dpy, t->xt_tpm, t->xt_cgc, 0, 0, t->xt_atw, t->xt_ath);

	count= 0;
	for (i= t->xt_topline; (i <= t->xt_topline+t->xt_wch) && (i < t->xt_nbuflines); i++, count++) {
		j= t->xt_linestarts[i];
		k= t->xt_linestarts[i + 1];
		/* length= k -j -1; */
		length= k-j; /* bug fix */

		textsw_expand_string(&(t->xt_text[j]), s, map, length, &new_length);
		XDrawString(t->xt_dpy, t->xt_tpm, t->xt_gc, -t->xt_leftpix, t->xt_cheight*(count+1) - t->xt_cdescent, s, new_length);
	}
}

void
textsw_display(t)
/*
   Internal function.
   Flush the text backing store to the screen.
*/
Textsw *t;
{
	XCopyArea(t->xt_dpy, t->xt_tpm, t->xt_twin, t->xt_gc, 0, 0, t->xt_atw, t->xt_ath, 0, 0);
	if (t->xt_drawcursor)
		textsw_drawcaret(t);
	if (t->xt_drawhlt)
		textsw_drawhlt(t, TRUE, (boolean *) NULL);
}

textsw_expand_string(in, out, map, length, out_length)
/*
   Internal function.
   Check for tabs and fill with blanks.
*/
char in[], out[];
int map[];
int length;
int *out_length;
{
	int c, i, j, k;

	for (i= 0; i < 512; i++)
		map[i]= -1;
	j= 0;
	for (i= 0; i < length; i++) {
		if (in[i] == '\0' ) /* bug fix */
			continue;
		if (in[i] == LXT_SPHOLDER)
			continue;
			
		if (in[i] != '\t') {
			out[j]= in[i];
			map[j]= i;
			++j;
		}
		else {
			c= LXT_TABLEN - j % LXT_TABLEN;
			for (k= 0; k < c; k++) {
				out[j]= ' ';
				map[j]= i;
				++j;
			}
		}
	}
	*out_length= j;
			
}

textsw_get_cursor_pos(evt, t, line_num, char_num)
XButtonPressedEvent *evt;
Textsw *t;
int *line_num, *char_num;
{
	char s[512];
	int line, length, new_length, map[512], i;
	int char_in_out_str, char_in_in_str;
	int full_line_length;

	line= evt->y / t->xt_cheight + t->xt_topline;

	if (t->xt_nbuflines == 0) {
		*line_num= 0;
		*char_num= 0;
		t->xt_cursorpx= -t->xt_leftpix;
		t->xt_cursorpy= -t->xt_topline * t->xt_cheight;
		return(TRUE);
	}
	else if (line + 1 > t->xt_nbuflines) {
		line= t->xt_nbuflines - 1;
	}

	length= t->xt_linestarts[line + 1] - t->xt_linestarts[line];

	if (t->xt_nbuflines != 0)
		textsw_expand_string(&(t->xt_text[t->xt_linestarts[line]]), s, map, length, &new_length);

	/* no text - gives a blank string */
	else {
		s[0]= '\0';
		new_length= 0;
	}

	full_line_length= XTextWidth(t->xt_font, s, new_length);
	if (evt->x > full_line_length - t->xt_leftpix) {
		*line_num= line;
		*char_num= t->xt_linestarts[line+1] - t->xt_linestarts[line] - 1;

		/* special case of last line*/
		if (line == t->xt_nbuflines-1)
			if (t->xt_text[t->xt_nbufchars-1] != '\0')
				(*char_num)++;

		/* update the cursor positions */
		t->xt_cursorpx= full_line_length - t->xt_leftpix;
		t->xt_cursorpy= (*line_num - t->xt_topline) * t->xt_cheight;
		return(TRUE);
	}

	char_in_out_str= -1;
	for (i= 0; i <= strlen(s)-1; i++) {
		if (i == 0) {
			if ((evt->x <= XTextWidth(t->xt_font, s, 1) - t->xt_leftpix)) {
				char_in_out_str= i;
				break;
			}
		}
		else if ((evt->x > XTextWidth(t->xt_font, s, i)-t->xt_leftpix) && 
			 (evt->x <= XTextWidth(t->xt_font, s, i+1)-t->xt_leftpix)) {
			char_in_out_str= i;
			break;
		}
	}

	if (char_in_out_str == -1)
		return(FALSE);

	char_in_in_str= map[char_in_out_str];

	*line_num= line;
	*char_num= char_in_in_str;

	/* update the cursor positions */
	t->xt_cursorpx= evt->x;
	t->xt_cursorpy= (*line_num - t->xt_topline) * t->xt_cheight;
			
	return(TRUE);
}

textsw_drawcaret(t)
Textsw *t;
/*
   Internal function.
   Draws the caret with an inverse GC but does not
   alter the caret_drawn flag! The calling routine
   must do this.
*/
{
	int xpos, ypos;
	int map[512], length, new_length;
	char s[512];
	char str[512];

	length= t->xt_linestarts[t->xt_irow+1] - t->xt_linestarts[t->xt_irow];
	bcopy(&(t->xt_text[t->xt_linestarts[t->xt_irow]]), str, length);
	str[length]='\0';

	str[t->xt_icol]= '\0';

	textsw_expand_string(str, s, map, t->xt_icol, &new_length);

	xpos= XTextWidth(t->xt_font, s, new_length) - t->xt_leftpix;
	ypos= (1 + t->xt_irow - t->xt_topline) * t->xt_cheight;

	XDrawLine(t->xt_dpy, t->xt_twin, t->xt_igc, xpos, ypos, xpos, ypos - t->xt_cheight);

	t->xt_cursorpx= xpos;
	t->xt_cursorpy= (t->xt_irow - t->xt_topline) * t->xt_cheight;
}

void
textsw_menucreate(t)
Textsw *t;
{
	Menu *m;
	Menu_item *mi;
	void textsw_copyproc();
	void textsw_pasteproc();
	void textsw_findproc();
	void textsw_findagainproc();
	void textsw_findbufproc();
	void textsw_cutproc();
	void textsw_rdproc();
	void textsw_saveproc();
	void textsw_saveasproc();
	void textsw_redspproc();
	void textsw_destroyproc();

	t->xt_menu= menu_create(t->xt_name, t->xt_dpy, t->xt_twin,
			LXM_FONT, LXTDEF_FONT,
			LXM_FOREGROUND, t->xt_fg,
			LXM_BACKGROUND, t->xt_bg,
			LXM_CLIENTDATA, (char *) t,
			LXM_NULL);

	/* edit */
	m= menu_create(t->xt_name, t->xt_dpy, t->xt_twin,
			LXM_FONT, LXTDEF_FONT,
			LXM_FOREGROUND, t->xt_fg,
			LXM_BACKGROUND, t->xt_bg,
			LXM_CLIENTDATA, (char *) t,
			LXM_NULL);
	mi= menuitem_create(LXMI_STRING, "Cut",
			LXMI_PROC, textsw_cutproc,
			LXMI_NULL);
	(void) menuitem_insert(m, mi);
	mi= menuitem_create(LXMI_STRING, "Copy",
			LXMI_PROC, textsw_copyproc,
			LXMI_NULL);
	(void) menuitem_insert(m, mi);
	mi= menuitem_create(LXMI_STRING, "Paste",
			LXMI_PROC, textsw_pasteproc,
			LXMI_NULL);
	(void) menuitem_insert(m, mi);
	mi= menuitem_create(LXMI_STRING, "Edit",
			LXMI_PULLRIGHT, m,
			LXMI_NULL);
	(void) menuitem_insert(t->xt_menu, mi);

	/* find */
	m= menu_create(t->xt_name, t->xt_dpy, t->xt_twin,
			LXM_FONT, LXTDEF_FONT,
			LXM_FOREGROUND, t->xt_fg,
			LXM_BACKGROUND, t->xt_bg,
			LXM_CLIENTDATA, (char *) t,
			LXM_NULL);
	mi= menuitem_create(LXMI_STRING, "String",
			LXMI_PROC, textsw_findproc,
			LXMI_NULL);
	(void) menuitem_insert(m, mi);
	mi= menuitem_create(LXMI_STRING, "Buffer",
			LXMI_PROC, textsw_findbufproc,
			LXMI_NULL);
	(void) menuitem_insert(m, mi);
	mi= menuitem_create(LXMI_STRING, "Again",
			LXMI_PROC, textsw_findagainproc,
			LXMI_NULL);
	(void) menuitem_insert(m, mi);
	mi= menuitem_create(LXMI_STRING, "Find",
			LXMI_PULLRIGHT, m,
			LXMI_NULL);
	(void) menuitem_insert(t->xt_menu, mi);

	/* save */
	m= menu_create(t->xt_name, t->xt_dpy, t->xt_twin,
			LXM_FONT, LXTDEF_FONT,
			LXM_FOREGROUND, t->xt_fg,
			LXM_BACKGROUND, t->xt_bg,
			LXM_CLIENTDATA, (char *) t,
			LXM_NULL);
	if (t->xt_wrfilenm[0] == '\0')
		t->xt_smenuitem= menuitem_create(LXMI_STRING, "Untitled",
						LXMI_STATE, LXMI_INACTIVE,
						LXMI_PROC, textsw_saveproc,
						LXMI_NULL);
	else
		t->xt_smenuitem= menuitem_create(LXMI_STRING, t->xt_wrfilenm,
						LXMI_PROC, textsw_saveproc,
						LXMI_NULL);
	(void) menuitem_insert(m, t->xt_smenuitem);
	mi= menuitem_create(LXMI_STRING, "Other",
			LXMI_PROC, textsw_saveasproc,
			LXMI_NULL);
	(void) menuitem_insert(m, mi);
	mi= menuitem_create(LXMI_STRING, "Save",
			LXMI_PULLRIGHT, m,
			LXMI_NULL);
	(void) menuitem_insert(t->xt_menu, mi);

	mi= menuitem_create(LXMI_STRING, "Read",
			LXMI_PROC, textsw_rdproc,
			LXMI_NULL);
	(void) menuitem_insert(t->xt_menu, mi);
	mi= menuitem_create(LXMI_STRING, "Redisplay",
			LXMI_PROC, textsw_redspproc,
			LXMI_NULL);
	(void) menuitem_insert(t->xt_menu, mi);
	mi= menuitem_create(LXMI_STRING, "Destroy",
			LXMI_PROC, textsw_destroyproc,
			LXMI_NULL);
	(void) menuitem_insert(t->xt_menu, mi);
}

void
textsw_destroyproc(menu, mi)
Menu *menu;
Menu_item *mi;
{
	Textsw *t;

	t= (Textsw *) menu->xm_clientdata;
	if ((*(t->xt_exitproc))(t, t->xt_dpy, t->xt_textedited))
		(void) textsw_destroy(t);
}

void
textsw_redspproc(menu, mi)
Menu *menu;
Menu_item *mi;
{
	Textsw *t;
	void textsw_drawscrolls();

	t= (Textsw *) menu->xm_clientdata;

	textsw_draw(t);
	textsw_display(t);
	textsw_drawscrolls(t);
}

void
textsw_rdproc(menu, mi)
Menu *menu;
Menu_item *mi;
{
	Textsw *t;
	char *s;
	int alert_val;
	void lxt_clearsel();
	void textsw_drawscrolls();

	t= (Textsw *) menu->xm_clientdata;

	if (!t->xt_allowinput) {
		(void) alert_prompt(t->xt_name, t->xt_dpy, &alert_val,
				LXA_LABEL, "Sorry, user input has been disallowed.",
				LXA_BUTTON, "Continue", 0,
				LXA_NULL);
		return;
	}

	if (t->xt_textedited) {
		(void) alert_prompt(t->xt_name, t->xt_dpy, &alert_val,
				LXA_LABEL, "Warning -- the text has been edited.",
				LXA_BUTTON, "Read", 0,
				LXA_BUTTON, "Cancel", 1,
				LXA_NULL);
		if (alert_val)
			return;
	}

	if ((s= calloc((unsigned) (LXADEF_MAXSTORE+1), sizeof(char))) == (char *) NULL) {
		(void) fprintf(stderr, "read_command: memory allocation error\n");
		return;
	}
	(void) alert_prompt(t->xt_name, t->xt_dpy, &alert_val,
			LXA_TEXT, "Enter file name:", "", s,
			LXA_NULL);
	if (textsw_readfile(t, s, TRUE) != LX_SUCCESS) {
		free(s);
		return;
	}

	cfree(s);

	t->xt_topline= 0;
	t->xt_leftpix= 0;
	t->xt_nspholders= 0;
	t->xt_drawhlt= FALSE;
	if (t == (Textsw *) lxt_selsrc)
		lxt_clearsel(FALSE);
	t->xt_textedited= FALSE;
	t->xt_drawcursor= FALSE;
	t->xt_updatescroll= FALSE;

	textsw_drawscrolls(t);
	textsw_draw(t);
	textsw_display(t);

	if (t->xt_rdfilenm[0] == '\0')
		(void) menuitem_set(t->xt_smenuitem, LXMI_STRING, "Untitled", LXMI_STATE, LXMI_INACTIVE, LXMI_NULL);
	else {
		(void) strcpy(t->xt_wrfilenm, t->xt_rdfilenm);
		(void) menuitem_set(t->xt_smenuitem, LXMI_STRING, t->xt_wrfilenm, LXMI_STATE, LXMI_ACTIVE, LXMI_NULL);
	}
}

void
textsw_saveproc(menu, mi)
Menu *menu;
Menu_item *mi;
{
	Textsw *t;
	char backup_file[LXT_MAXFILENM+1];
	char ch;
	FILE *fopen(), *f;
	int i;

	t= (Textsw *) menu->xm_clientdata;

	if (t->xt_wrfilenm[0] == '\0') {
		int val;

		(void) alert_prompt(t->xt_name, t->xt_dpy, &val,
				LXA_LABEL, "No file name. Use \"Save as ...\"",
				LXA_BUTTON, "Continue", 0,
				LXA_NULL);
		return;
	}

	(void) strcpy(backup_file, t->xt_wrfilenm);
	(void) strcat(backup_file, "%");
	(void) unlink(backup_file);
	if (t->xt_wrfilenm[0] != '\0')  
		if (access(t->xt_wrfilenm, F_OK) == 0) {
			if (link(t->xt_wrfilenm, backup_file) != 0) {
				int val;
	
				(void) alert_prompt(t->xt_name, t->xt_dpy, &val,
						LXA_LABEL, "Warning -- cannot make backup file.",
						LXA_BUTTON, "Proceed", 0,
						LXA_BUTTON, "Cancel", 1,
						LXA_NULL);
				if (val)
					return;
				else
					(void) unlink(t->xt_wrfilenm);
			}
			else
				(void) unlink(t->xt_wrfilenm);
		}
	
	if ((f= fopen(t->xt_wrfilenm, "w")) == NULL) {
		int val;

		(void) alert_prompt(t->xt_name, t->xt_dpy, &val,
				LXA_LABEL, "Cannot open file for writing.",
				LXA_BUTTON, "Continue", 0,
				LXA_NULL);
		return;
	}
	
	for (i= 0; i < t->xt_nbufchars; i++) {
		ch= t->xt_text[i];
		if (ch == LXT_SPHOLDER)
			continue;
		else if (ch == '\0')
			(void) putc('\n', f);
		else
			(void) putc(ch, f);
	}
	
	(void) fclose(f);
	t->xt_textedited= FALSE;
}

void
textsw_saveasproc(menu, mi)
Menu *menu;
Menu_item *mi;
{
	Textsw *t;
	char *s;
	char backup_file[LXT_MAXFILENM+1];
	char  ch;
	FILE *fopen(), *f;
	int i, alert_val;

	t= (Textsw *) menu->xm_clientdata;
	if ((s= calloc((unsigned) (LXADEF_MAXSTORE+1), sizeof(char))) == (char *) NULL) {
		(void) fprintf(stderr, "textsw_saveasproc: memory allocation error\n");
		return;
	}
	(void) alert_prompt(t->xt_name, t->xt_dpy, &alert_val,
			LXA_TEXT, "Enter new file name:", "", s,
			LXA_NULL);

	(void) strcpy(backup_file, s);
	(void) strcat(backup_file, "%");
	(void) unlink(backup_file);
	if (access(s, F_OK) == 0 ) {
		if (link(s, backup_file) != 0) {
			(void) alert_prompt(t->xt_name, t->xt_dpy, &alert_val,
					LXA_LABEL, "Warning -- cannot make backup file.",
					LXA_BUTTON, "Proceed", 0,
					LXA_BUTTON, "Cancel", 1,
					LXA_NULL);
			if (alert_val)
				return;
			else
				(void) unlink(s);
		}
	}
	else
		(void) unlink(s);
	
	if ((f= fopen(s, "w")) == NULL) {
		(void) alert_prompt(t->xt_name, t->xt_dpy, &alert_val,
				LXA_LABEL, "Cannot open file for writing.",
				LXA_BUTTON, "Continue", 0,
				LXA_NULL);
		return;
	}
	
	for (i= 0; i < t->xt_nbufchars; i++) {
		ch= t->xt_text[i];
		if (ch == LXT_SPHOLDER)
			continue;
		else if (ch == '\0')
			(void) putc('\n', f);
		else
			(void) putc(ch, f);
	}
	
	(void) fclose(f);
	t->xt_textedited= FALSE;
}

void
textsw_findproc(menu, mi)
Menu *menu;
Menu_item *mi;
{
	char *target;
	int alert_val;
	Textsw *t;

	t= (Textsw *) menu->xm_clientdata;

	if (!t->xt_drawcursor) {
		XBell(t->xt_dpy, 10);
		return;
	}

	if ((target= calloc((unsigned) (LXADEF_MAXSTORE+1), sizeof(char))) == (char *) NULL) {
		(void) fprintf(stderr, "textsw_findproc: memory allocation error\n");
		return;
	}
	(void) alert_prompt(t->xt_name, t->xt_dpy, &alert_val,
			LXA_TEXT, "Enter search string:", "", target,
			LXA_NULL);
	(void) strcpy(xvt_findbuf, target);

	textsw_find(t, target);
	cfree(target);

}

void
textsw_findagainproc(menu, mi)
Menu *menu;
Menu_item *mi;
{
	Textsw *t;

	t= (Textsw *) menu->xm_clientdata;

	if (!t->xt_drawcursor) {
		XBell(t->xt_dpy, 10);
		return;
	}

	textsw_find(t, xvt_findbuf);
}

void
textsw_findbufproc(menu, mi)
Menu *menu;
Menu_item *mi;
{
	char *s;
	int i, j;
	Textsw *t;

	t= (Textsw *) menu->xm_clientdata;

	if (!t->xt_drawcursor) {
		XBell(t->xt_dpy, 10);
		return;
	}

	if (xvt_pastebuf != NULL) {
		s= (char *) textsw_get(t, LXT_PASTEBUFFER);
		j= strlen(s);
		for (i= 0 ; i < j; i++) {
			if (s[i] == '\n') {
				s[i]= '\0';
				break;
			}
		}
		(void) strcpy(xvt_findbuf, s);
		textsw_find(t, xvt_findbuf);
		free(s);
	}
	else
		XBell(t->xt_dpy, 10);
}

textsw_find(t, target)
char target[];
Textsw *t;
{
	char *s;
	int i, match_column, row, col;

	/* get cursor position */
	row= *((int *) textsw_get(t, LXT_CURSORLINE));
	col= *((int *) textsw_get(t, LXT_CURSORCOLUMN));

	s= textsw_text(t, row);
	if ((match_column= textsw_ismatch(&(s[col]), target)) != -1) {
		(void) textsw_set(t, LXT_DRAWCURSOR, FALSE,
			  LXT_CURSORLINE, row,
			  LXT_CURSORCOLUMN, match_column + col + strlen(target),
			  LXT_DRAWCURSOR, TRUE,
			  LXT_DRAWHIGHLIGHT, FALSE,
			  LXT_HIGHLIGHTENDLINE, row,
			  LXT_HIGHLIGHTENDCOLUMN, match_column + col,
			  LXT_DRAWHIGHLIGHT, TRUE,
			  LXT_CURSORVIEWABLE,
			  LXT_NULL);
	}
	else {
		for (i= row+1; i < t->xt_nbuflines; i++) {
			s= textsw_text(t, i);
			if ((match_column= textsw_ismatch(s, target)) != -1) {
				(void) textsw_set(t, LXT_DRAWCURSOR, FALSE,
				  	LXT_CURSORLINE, i,
				  	LXT_CURSORCOLUMN, match_column + strlen(target),
				  	LXT_DRAWCURSOR, TRUE,
				  	LXT_DRAWHIGHLIGHT, FALSE,
				  	LXT_HIGHLIGHTENDLINE, i,
				  	LXT_HIGHLIGHTENDCOLUMN, match_column,
				  	LXT_DRAWHIGHLIGHT, TRUE,
				  	LXT_CURSORVIEWABLE,
				  	LXT_NULL);
				break;
			}
		}
	}
	if (match_column == -1)
		XBell(t->xt_dpy, 10);
	
}
		
	
int
textsw_ismatch(str, target)
char str[], target[];
{
	int c, d, i, j, ls, lt;

	c= -1;
	ls= strlen(str);
	lt= strlen(target);

	for (i= 0; i <= (ls - lt); i++) {
		d= 0;
		for (j= 0; j < lt; j++) {
			if (target[j] == str[i + j])
				++d;
			else
				break;
		}
		if (d == lt) {
			c= i;
			break;
		}
	}
	return(c);
}

boolean
textsw_compute_cursor_pos(t, line_num, char_num)
/*
   Internal function.
   Gets the current cursor position without updating anything.
*/
Textsw *t;
int *line_num, *char_num;
{
	char s[512];
	int line, length, new_length, map[512], i;
	int char_in_out_str, char_in_in_str;
	int full_line_length;

	if (t->xt_nbuflines == 0) {
		*line_num= 0;
		*char_num= 0;
		return(TRUE);
	}

	line= (t->xt_cursorpy / t->xt_cheight) + t->xt_topline;
	length= t->xt_linestarts[line + 1] - t->xt_linestarts[line];

	if (t->xt_nbuflines != 0)
		textsw_expand_string(&(t->xt_text[t->xt_linestarts[line]]), s, map, length, &new_length);

	/* no text - gives a blank string */
	else {
		s[0]= '\0';
		new_length= 0;
	}

	full_line_length= XTextWidth(t->xt_font, s, new_length);
	if (t->xt_cursorpx > full_line_length - t->xt_leftpix) {
		*line_num= line;
		*char_num= t->xt_linestarts[line+1] - t->xt_linestarts[line] - 1;
		/* *char_num= strlen(&(t->xt_text[t->xt_linestarts[line]])); */

		/*special case of last line*/
		if (line == (t->xt_nbuflines-1))
			if (t->xt_text[t->xt_nbufchars-1] != '\0')
				(*char_num)++;

		return(TRUE);
	}

	char_in_out_str= (-1);
	for (i= 0; i <= strlen(s) - 1; i++) {
		if (i == 0) {
			if (t->xt_cursorpx+t->xt_leftpix < XTextWidth(t->xt_font, s, 1)/2) {
				char_in_out_str= 0;
				break;
			}
			else if ((t->xt_cursorpx <= XTextWidth(t->xt_font, s, 1) - t->xt_leftpix)) {
				char_in_out_str= 1;
				break;
			}
		}
		else if ((t->xt_cursorpx > XTextWidth(t->xt_font, s, i)-t->xt_leftpix) && 
			 (t->xt_cursorpx <= XTextWidth(t->xt_font, s, i+1)-t->xt_leftpix)) {
			char_in_out_str= i+1;
			break;
		}
	}

	if (char_in_out_str == -1) {
		*line_num= line;
		*char_num= 0;
		return(TRUE);
	}

	char_in_in_str= map[char_in_out_str];

	*line_num= line;
	*char_num= char_in_in_str;

	return(TRUE);
}

int
textsw_active()
/*
   User-callable routine.
   Returns FALSE if the linked list is NULL and TRUE otherwise.
*/
{
	if (xt_textsws == NULL)
		return(FALSE);
	else
		return(TRUE);
}
