#include <stdio.h>
#include <varargs.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "lxt.h"

#include "grey.bitmap"
#include "buttonproc.bitmap"
#include "choicemark.bitmap"
#include "choicenomark.bitmap"
#include "cycle.bitmap"
#include "togglemark.bitmap"
#include "togglenomark.bitmap"

extern Panelitem_dispatch panelitem_proctab[];

extern Void *xpret_void;
extern int xpret_int;
extern boolean xpret_bool;

extern Void *lxt_selsrc;
extern Void *lxt_seldest;
extern int lxt_seldesttype;

/*VARARGS*/
Panel_item *
panelitem_create(va_alist)
/*
   User-callable.
   Creates a panel item.
*/
va_dcl
{
	va_list varg_ptr;
	Panel *p;
	Panel_item *pi;
	int nargs, type, attr, state, n, repaint;
	boolean set_textipos;
	XGCValues gcv;
	GC gc;
	xp_button *xpb;
	XImage image, *im;
	xp_slider *xps;
	int minlen, maxlen;
	char *c, minbuf[20], maxbuf[20], barbuf[20];
	xp_enum *xpe;
	xp_esel *es, *fs, *gs;
	boolean slpos_set;
	boolean eval_set, eloc_set;
	boolean emarks_set, enmarks_set;
	int val, i, depth;
	xp_esel *xpesel_alloc();
	int panelitem_filldefaults();
	void xpenum_elocinit(), xpenum_eszinit();
	void xpenum_bld_menu(), xpenum_adj_menu();

	p= (Panel *) NULL;
	pi= (Panel_item *) NULL;
	va_start(varg_ptr);
	for (nargs= 0;; nargs++) {

		/* get panel */
		if (nargs == 0) {
			p= va_arg(varg_ptr, Panel *);
			if (p == (Panel *) NULL) {
				(void) fprintf(stderr, "panelitem_create: null panel\n");
				return((Panel_item *) NULL);
			}
		}

		/* get item type */
		else if (nargs == 1) {
			type= va_arg(varg_ptr, int);
			switch (type) {
			case LXPI_LABEL:
			case LXPI_TEXT:
			case LXPI_BUTTON:
			case LXPI_SLIDER:
			case LXPI_CHOICE:
			case LXPI_CYCLE:
			case LXPI_TOGGLE:
				break;
			default:
				(void) fprintf(stderr, "panelitem_create: unrecognized item type\n");
				return((Panel_item *) NULL);
			}

			if ((pi= (Panel_item *) calloc(1, sizeof(Panel_item))) == (Panel_item *) NULL) {
				(void) fprintf(stderr, "panelitem_create: memory allocation error\n");
				return((Panel_item *) NULL);
			}
			pi->xpi_magic= LX_PANELITEM;
			pi->xpi_type= type;
			if (panelitem_filldefaults(p, pi) != LX_SUCCESS) {
				cfree((char *) pi);
				return((Panel_item *) NULL);
			}
			set_textipos= slpos_set= FALSE;
			eval_set= eloc_set= FALSE;
			emarks_set= enmarks_set= FALSE;
		}

		/* attribute list */
		else {
			char *c;

			attr= va_arg(varg_ptr, int);
			if (attr == LXPI_NULL)
				break;

			switch (attr) {

			case LXPI_X:
				pi->xpi_x= va_arg(varg_ptr, int);
				break;
			case LXPI_Y:
				pi->xpi_y= va_arg(varg_ptr, int);
				break;
			case LXPI_PROC:
				pi->xpi_proc= (void (*)()) va_arg(varg_ptr, char *);
				break;
			case LXPI_FONT:
				c= va_arg(varg_ptr, char *);
				if (c == (char *) NULL)
					(void) fprintf(stderr, "panelitem_create: null fontname\n");
				else if ((pi->xpi_font= XLoadQueryFont(pi->xpi_dpy, c)) == NULL)
					(void) fprintf(stderr, "panelitem_create: cannot load font %s\n", c);
				break;
			case LXPI_FOREGROUND:
				pi->xpi_fg= va_arg(varg_ptr, unsigned long);
				break;
			case LXPI_BACKGROUND:
				pi->xpi_bg= va_arg(varg_ptr, unsigned long);
				break;
			case LXPI_STATE:
				state= va_arg(varg_ptr, int);
				switch (state) {
				case LXPI_ACTIVE:
				case LXPI_INACTIVE:
					pi->xpi_state= state;
					break;
				default:
					(void) fprintf(stderr, "panelitem_create: invalid item state\n");
					break;
				}
				break;
			case LXPI_DISPLAY:
				repaint= va_arg(varg_ptr, int);
				switch (repaint) {
				case LXPI_REPAINT:
				case LXPI_NOREPAINT:
					pi->xpi_display= repaint;
					break;
				default:
					(void) fprintf(stderr, "panelitem_create: invalid item display value\n");
					break;
				}
				break;
			case LXPI_STRING:
				c= va_arg(varg_ptr, char *);
				if (pi->xpi_str != (char *) NULL)
					cfree(pi->xpi_str);
				pi->xpi_str= (char *) NULL;
				if (c != (char *) NULL) {
					if (strlen(c) != 0) {
						if ((pi->xpi_str= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
							(void) fprintf(stderr, "panelitem_create: memory allocation error\n");
							(void) panelitem_destroy(pi);
							return((Panel_item *) NULL);
						}
						(void) strcpy(pi->xpi_str, c);
					}
				}
				break;
			case LXPI_IMAGE:
				pi->xpi_image= va_arg(varg_ptr, XImage *);
				break;
			case LXPI_CLIENTDATA:
				pi->xpi_clientdata= va_arg(varg_ptr, char *);
				break;
			case LXPTEXT_VALUE:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				c= va_arg(varg_ptr, char *);
				if (c == (char *) NULL)
					(void) strcpy(pi->xpi_item.xpi_text.xptext_val, "");
				else if (strlen(c) == 0)
					(void) strcpy(pi->xpi_item.xpi_text.xptext_val, "");
				else if (strlen(c) <= pi->xpi_item.xpi_text.xptext_max)
					(void) strcpy(pi->xpi_item.xpi_text.xptext_val, c);
				else {
					cfree(pi->xpi_item.xpi_text.xptext_val);
					pi->xpi_item.xpi_text.xptext_max= strlen(c);
					if ((pi->xpi_item.xpi_text.xptext_val= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
						(void) fprintf(stderr, "panelitem_create: memory allocation error\n");
						(void) panelitem_destroy(pi);
						return((Panel_item *) NULL);
					}
					(void) strcpy(pi->xpi_item.xpi_text.xptext_val, c);
				}
				break;
			case LXPTEXT_MAXSTORE:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panelitem_create: invalid text storage length\n");
				else {
					char *oldval;

					oldval= pi->xpi_item.xpi_text.xptext_val;
					pi->xpi_item.xpi_text.xptext_max= n;
					if ((pi->xpi_item.xpi_text.xptext_val= calloc((unsigned) (n+1), sizeof(char))) == (char *) NULL) {
						(void) fprintf(stderr, "panelitem_create: memory allocation error\n");
						(void) panelitem_destroy(pi);
						return((Panel_item *) NULL);
					}
					(void) strncpy(pi->xpi_item.xpi_text.xptext_val, oldval, n);
					cfree(oldval);
				}
				break;
			case LXPTEXT_MAXDISPLAY:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n >= 0) {
					pi->xpi_item.xpi_text.xptext_disp= n;
					if (pi->xpi_item.xpi_text.xptext_dpos < strlen(pi->xpi_item.xpi_text.xptext_val)-n)
						pi->xpi_item.xpi_text.xptext_dpos= strlen(pi->xpi_item.xpi_text.xptext_val)-n;
				}
				else
					(void) fprintf(stderr, "panelitem_create: invalid text display length\n");
				break;
			case LXPTEXT_IPOS:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if ((n < 0) || (n > strlen(pi->xpi_item.xpi_text.xptext_val)))
					(void) fprintf(stderr, "panelitem_create: invalid text position\n");
				else {
					pi->xpi_item.xpi_text.xptext_ipos= n;
					set_textipos= TRUE;
				}
				break;
			case LXPTEXT_GAP:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panelitem_create: invalid text gap\n");
				else
					pi->xpi_item.xpi_text.xptext_gap= n;
				break;
			case LXPTEXT_PROC:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				pi->xpi_item.xpi_text.xptext_kpproc= (int (*)()) va_arg(varg_ptr, char *);
				break;
			case LXPTEXT_RDONLY:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n)
					pi->xpi_item.xpi_text.xptext_flags|= LXPTEXT_WRPROTECT;
				else
					pi->xpi_item.xpi_text.xptext_flags&= ~LXPTEXT_WRPROTECT;
				break;
			case LXPBUTTON_HMARGIN:
				if (pi->xpi_type != LXPI_BUTTON)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panelitem_create: invalid button margin\n");
				else
					pi->xpi_item.xpi_button.xpbutton_hmargin= n;
				break;
			case LXPBUTTON_VMARGIN:
				if (pi->xpi_type != LXPI_BUTTON)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panelitem_create: invalid button margin\n");
				else
					pi->xpi_item.xpi_button.xpbutton_vmargin= n;
				break;
			case LXPSLIDER_MINVAL:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_minval= va_arg(varg_ptr, int);
				break;
			case LXPSLIDER_MAXVAL:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_maxval= va_arg(varg_ptr, int);
				break;
			case LXPSLIDER_VALUE:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_val= va_arg(varg_ptr, int);
				break;
			case LXPSLIDER_VALX:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_valx= va_arg(varg_ptr, int);
				slpos_set= TRUE;
				break;
			case LXPSLIDER_VALY:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_valy= va_arg(varg_ptr, int);
				slpos_set= TRUE;
				break;
			case LXPSLIDER_BARHEIGHT:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panelitem_create: invalid slider bar width\n");
				else
					pi->xpi_item.xpi_slider.xpslider_barh= n;
				break;
			case LXPSLIDER_BARLENGTH:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panelitem_create: invalid slider bar width\n");
				else
					pi->xpi_item.xpi_slider.xpslider_barl= n;
				break;
			case LXPSLIDER_BARX:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_barx= va_arg(varg_ptr, int);
				slpos_set= TRUE;
				break;
			case LXPSLIDER_BARY:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_bary= va_arg(varg_ptr, int);
				slpos_set= TRUE;
				break;
			case LXPENUM_VALUE:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_CYCLE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				switch (pi->xpi_type) {
				case LXPI_CHOICE:
				case LXPI_CYCLE:
					pi->xpi_item.xpi_enum.xpenum_val= (0x1 << n);
					break;
				case LXPI_TOGGLE:
					pi->xpi_item.xpi_enum.xpenum_val= n;
					break;
				default:
					break;
				}
				eval_set= TRUE;
				break;
			case LXPENUM_DISPLAY:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if ((n != LXPENUM_DISPALL) && (n != LXPENUM_DISPCURRSEL)) {
					(void) fprintf(stderr, "panelitem_create: unrecognized enum display level\n");
					break;
				}
				pi->xpi_item.xpi_enum.xpenum_flags&= ~(LXPENUM_DISPALL | LXPENUM_DISPCURRSEL);
				pi->xpi_item.xpi_enum.xpenum_flags|= n;
				break;
			case LXPENUM_SELSTRINGS:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_CYCLE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				es= pi->xpi_item.xpi_enum.xpenum_list;
				while ((c= va_arg(varg_ptr, char *)) != (char *) NULL) {
					if (es == (xp_esel *) NULL) {
						if ((es= xpesel_alloc(pi)) == (xp_esel *) NULL) {
							(void) fprintf(stderr, "panelitem_create: memory allocation error\n");
							(void) panelitem_destroy(pi);
							return((Panel_item *) NULL);
						}
					}
					if (es->xpesel_str != (char *) NULL)
						cfree(es->xpesel_str);
					if ((es->xpesel_str= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
						(void) fprintf(stderr, "panelitem_create: memory allocation error\n");
						(void) panelitem_destroy(pi);
						return((Panel_item *) NULL);
					}
					(void) strcpy(es->xpesel_str, c);
					es= es->xpesel_next;
				}
				break;
			case LXPENUM_SELIMAGES:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_CYCLE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				es= pi->xpi_item.xpi_enum.xpenum_list;
				while ((im= va_arg(varg_ptr, XImage *)) != (XImage *) NULL) {
					if (es == (xp_esel *) NULL) {
						if ((es= xpesel_alloc(pi)) == (xp_esel *) NULL) {
							(void) fprintf(stderr, "panelitem_create: memory allocation error\n");
							(void) panelitem_destroy(pi);
							return((Panel_item *) NULL);
						}
					}
					es->xpesel_image= im;
					es= es->xpesel_next;
				}
				break;
			case LXPENUM_MARKIMAGES:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				es= pi->xpi_item.xpi_enum.xpenum_list;
				while ((im= va_arg(varg_ptr, XImage *)) != (XImage *) NULL) {
					if (es == (xp_esel *) NULL) {
						if ((es= xpesel_alloc(pi)) == (xp_esel *) NULL) {
							(void) fprintf(stderr, "panelitem_create: memory allocation error\n");
							(void) panelitem_destroy(pi);
							return((Panel_item *) NULL);
						}
					}
					es->xpesel_mimage= im;
					fs= es;
					es= es->xpesel_next;
				}
				if (fs != (xp_esel *) NULL) {
					for (gs= fs->xpesel_next; gs != (xp_esel *) NULL; gs= gs->xpesel_next)
						gs->xpesel_mimage= fs->xpesel_mimage;
				}
				emarks_set= TRUE;
				break;
			case LXPENUM_NOMARKIMAGES:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				es= pi->xpi_item.xpi_enum.xpenum_list;
				while ((im= va_arg(varg_ptr, XImage *)) != (XImage *) NULL) {
					if (es == (xp_esel *) NULL) {
						if ((es= xpesel_alloc(pi)) == (xp_esel *) NULL) {
							(void) fprintf(stderr, "panelitem_create: memory allocation error\n");
							(void) panelitem_destroy(pi);
							return((Panel_item *) NULL);
						}
					}
					es->xpesel_nmimage= im;
					fs= es;
					es= es->xpesel_next;
				}
				if (fs != (xp_esel *) NULL) {
					for (gs= fs->xpesel_next; gs != (xp_esel *) NULL; gs= gs->xpesel_next)
						gs->xpesel_nmimage= fs->xpesel_nmimage;
				}
				enmarks_set= TRUE;
				break;
			case LXPENUM_SELXS:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				es= pi->xpi_item.xpi_enum.xpenum_list;
				while ((n= va_arg(varg_ptr, int)) != LXPENUM_LOCTERM) {
					if (es == (xp_esel *) NULL) {
						if ((es= xpesel_alloc(pi)) == (xp_esel *) NULL) {
							(void) fprintf(stderr, "panelitem_create: memory allocation error\n");
							(void) panelitem_destroy(pi);
							return((Panel_item *) NULL);
						}
					}
					es->xpesel_lx= n;
					fs= es;
					es= es->xpesel_next;
				}
				if (fs != (xp_esel *) NULL) {
					for (gs= fs->xpesel_next; gs != (xp_esel *) NULL; gs= gs->xpesel_next)
						gs->xpesel_lx= fs->xpesel_lx;
				}
				eloc_set= TRUE;
				break;
			case LXPENUM_SELYS:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				es= pi->xpi_item.xpi_enum.xpenum_list;
				while ((n= va_arg(varg_ptr, int)) != LXPENUM_LOCTERM) {
					if (es == (xp_esel *) NULL) {
						if ((es= xpesel_alloc(pi)) == (xp_esel *) NULL) {
							(void) fprintf(stderr, "panelitem_create: memory allocation error\n");
							(void) panelitem_destroy(pi);
							return((Panel_item *) NULL);
						}
					}
					es->xpesel_ly= n;
					fs= es;
					es= es->xpesel_next;
				}
				if (fs != (xp_esel *) NULL) {
					for (gs= fs->xpesel_next; gs != (xp_esel *) NULL; gs= gs->xpesel_next)
						gs->xpesel_ly= fs->xpesel_ly;
				}
				eloc_set= TRUE;
				break;
			case LXPENUM_MARKXS:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				es= pi->xpi_item.xpi_enum.xpenum_list;
				while ((n= va_arg(varg_ptr, int)) != LXPENUM_LOCTERM) {
					if (es == (xp_esel *) NULL) {
						if ((es= xpesel_alloc(pi)) == (xp_esel *) NULL) {
							(void) fprintf(stderr, "panelitem_create: memory allocation error\n");
							(void) panelitem_destroy(pi);
							return((Panel_item *) NULL);
						}
					}
					es->xpesel_mx= n;
					fs= es;
					es= es->xpesel_next;
				}
				if (fs != (xp_esel *) NULL) {
					for (gs= fs->xpesel_next; gs != (xp_esel *) NULL; gs= gs->xpesel_next)
						gs->xpesel_mx= fs->xpesel_mx;
				}
				eloc_set= TRUE;
				break;
			case LXPENUM_MARKYS:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				es= pi->xpi_item.xpi_enum.xpenum_list;
				while ((n= va_arg(varg_ptr, int)) != LXPENUM_LOCTERM) {
					if (es == (xp_esel *) NULL) {
						if ((es= xpesel_alloc(pi)) == (xp_esel *) NULL) {
							(void) fprintf(stderr, "panelitem_create: memory allocation error\n");
							(void) panelitem_destroy(pi);
							return((Panel_item *) NULL);
						}
					}
					es->xpesel_my= n;
					fs= es;
					es= es->xpesel_next;
				}
				if (fs != (xp_esel *) NULL) {
					for (gs= fs->xpesel_next; gs != (xp_esel *) NULL; gs= gs->xpesel_next)
						gs->xpesel_my= fs->xpesel_my;
				}
				eloc_set= TRUE;
				break;
			case LXPENUM_INVERT:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n)
					pi->xpi_item.xpi_enum.xpenum_flags|= LXPENUM_INVERTLABEL;
				else
					pi->xpi_item.xpi_enum.xpenum_flags&= ~LXPENUM_INVERTLABEL;
				break;
			case LXPENUM_LABELX:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				pi->xpi_item.xpi_enum.xpenum_lx= va_arg(varg_ptr, int);
				eloc_set= TRUE;
				break;
			case LXPENUM_LABELY:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
				pi->xpi_item.xpi_enum.xpenum_ly= va_arg(varg_ptr, int);
				eloc_set= TRUE;
				break;
			default:
				(void) fprintf(stderr, "panelitem_create: ignoring unrecognized attribute\n");
				break;
			}
		}
	}
	va_end(varg_ptr);

	/* initialize GCs */
	gcv.font= pi->xpi_font->fid;
	gcv.foreground= pi->xpi_fg;
	gcv.background= pi->xpi_bg;
	gcv.function= GXcopy;
	pi->xpi_gc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCFont | GCForeground | GCBackground | GCFunction), &gcv);
	gcv.plane_mask= pi->xpi_fg ^ pi->xpi_bg;
	gcv.function= GXinvert;
	pi->xpi_igc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCFont | GCForeground | GCBackground | GCFunction | GCPlaneMask), &gcv);
	gcv.function= gcv.foreground ? GXor : GXand;
	gcv.stipple= p->xp_stipplepm;
	gcv.fill_style= FillStippled;
	pi->xpi_sgc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCFont | GCForeground | GCBackground | GCFunction | GCStipple | GCFillStyle), &gcv);
	gcv.foreground= pi->xpi_bg;
	gcv.background= pi->xpi_fg;
	gcv.plane_mask= pi->xpi_fg | pi->xpi_bg;
	gcv.function= GXcopy;
	pi->xpi_cgc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCFont | GCForeground | GCBackground | GCPlaneMask | GCFunction), &gcv);

	depth= DefaultDepth(p->xp_dpy, DefaultScreen(p->xp_dpy));

	/* item-specific stuff */
	switch (pi->xpi_type) {
	case LXPI_TEXT:

		/* set text insertion point to end
		   of string if not explicitly set */
		if (!set_textipos)
			pi->xpi_item.xpi_text.xptext_ipos= strlen(pi->xpi_item.xpi_text.xptext_val);

		/* has display length been exceeded? */
		if (strlen(pi->xpi_item.xpi_text.xptext_val) > pi->xpi_item.xpi_text.xptext_disp) {
			pi->xpi_item.xpi_text.xptext_dpos= strlen(pi->xpi_item.xpi_text.xptext_val)-pi->xpi_item.xpi_text.xptext_disp;
			if (pi->xpi_item.xpi_text.xptext_ipos < pi->xpi_item.xpi_text.xptext_dpos)
				pi->xpi_item.xpi_text.xptext_ipos= pi->xpi_item.xpi_text.xptext_dpos;
		}

		if ((p->xp_seltext == (Panel_item *) NULL) && (pi->xpi_state == LXPI_ACTIVE))
			p->xp_seltext= pi;

		break;

	case LXPI_BUTTON:

		xpb= (xp_button *) &(pi->xpi_item.xpi_button);

		/* create button proc hash */
		xpb->xpbutton_procpm= XCreatePixmap(pi->xpi_dpy, DefaultRootWindow(pi->xpi_dpy), buttonproc_width, buttonproc_height, 1);
		gcv.foreground= pi->xpi_fg;
		gcv.background= pi->xpi_bg;
		gc= XCreateGC(pi->xpi_dpy, xpb->xpbutton_procpm, GCForeground | GCBackground, &gcv);
		image.height= buttonproc_height;
		image.width= buttonproc_width;
		image.xoffset= 0;
		image.format= XYBitmap;
		image.data= buttonproc_bits;
		image.byte_order= LSBFirst;
		image.bitmap_unit= 8;
		image.bitmap_bit_order= LSBFirst;
		image.bitmap_pad= 8;
		image.bytes_per_line= (buttonproc_width+7)>>3;
		image.depth= 1;
		XPutImage(pi->xpi_dpy, xpb->xpbutton_procpm, gc, &image, 0, 0, 0, 0, buttonproc_width, buttonproc_height);
		XFreeGC(pi->xpi_dpy, gc);

		/* set GC for button proc activation */
		gcv.foreground= pi->xpi_fg;
		gcv.background= pi->xpi_bg;
/*
		gcv.function= gcv.foreground ? GXor : GXand;
*/
		gcv.function= GXcopy;
		gcv.stipple= xpb->xpbutton_procpm;
		gcv.fill_style= FillStippled;
		pi->xpi_item.xpi_button.xpbutton_bgc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCForeground | GCBackground | GCFunction | GCStipple | GCFillStyle), &gcv);
		break;

	case LXPI_SLIDER:

		xps= (xp_slider *) &(pi->xpi_item.xpi_slider);

		/* create grey stipple */
		xps->xpslider_bpm= XCreatePixmap(pi->xpi_dpy, DefaultRootWindow(pi->xpi_dpy), grey_width, grey_height, 1);
		gcv.foreground= pi->xpi_fg;
		gcv.background= pi->xpi_bg;
		gc= XCreateGC(pi->xpi_dpy, xps->xpslider_bpm, GCForeground | GCBackground, &gcv);
		image.height= grey_height;
		image.width= grey_width;
		image.xoffset= 0;
		image.format= XYBitmap;
		image.data= grey_bits;
		image.byte_order= LSBFirst;
		image.bitmap_unit= 8;
		image.bitmap_bit_order= LSBFirst;
		image.bitmap_pad= 8;
		image.bytes_per_line= (grey_width+7)>>3;
		image.depth= 1;
		XPutImage(pi->xpi_dpy, xps->xpslider_bpm, gc, &image, 0, 0, 0, 0, grey_width, grey_height);
		XFreeGC(pi->xpi_dpy, gc);

		/* set GC for slider bubble */
		gcv.foreground= pi->xpi_fg;
		gcv.background= pi->xpi_bg;
		gcv.function= gcv.foreground ? GXor : GXand;
		gcv.stipple= xps->xpslider_bpm;
		gcv.fill_style= FillStippled;
		xps->xpslider_bgc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCForeground | GCBackground | GCFunction | GCStipple | GCFillStyle), &gcv);

		/* set position of value and bar */
		if (!slpos_set) {
			xps->xpslider_valx= 0;
			xps->xpslider_valy= pi->xpi_font->max_bounds.ascent;
			if (pi->xpi_image != (XImage *) NULL) {
				xps->xpslider_valx= pi->xpi_image->width+4;
				if (pi->xpi_font->max_bounds.ascent < pi->xpi_image->height)
					xps->xpslider_valy= pi->xpi_image->height;
			}
			else if (pi->xpi_str != (char *) NULL) {
				if (strlen(pi->xpi_str) != 0)
					xps->xpslider_valx= XTextWidth(pi->xpi_font, pi->xpi_str, strlen(pi->xpi_str))+4;
			}
			(void) sprintf(minbuf, "[%1d]", xps->xpslider_minval);
			minlen= strlen(minbuf);
			(void) sprintf(maxbuf, "[%1d]", xps->xpslider_maxval);
			maxlen= strlen(maxbuf);
			if (minlen < maxlen)
				c= maxbuf;
			else
				c= minbuf;
			(void) sprintf(barbuf, " %1d ", xps->xpslider_minval);
			xps->xpslider_barx= xps->xpslider_valx+XTextWidth(pi->xpi_font, c, strlen(c))+XTextWidth(pi->xpi_font, barbuf, strlen(barbuf));
			xps->xpslider_bary= xps->xpslider_valy-(pi->xpi_font->max_bounds.ascent/2)-(xps->xpslider_barh/2);
			if (xps->xpslider_bary < 0) {
				xps->xpslider_valy-= xps->xpslider_bary;
				xps->xpslider_bary= 0;
			}
		}
		break;

	case LXPI_CHOICE:

		xpe= (xp_enum *) &(pi->xpi_item.xpi_enum);

		/* make sure that a legal value was specified */
		if (eval_set) {
			xpe->xpenum_sel= xpe->xpenum_list;
			if ((val= xpenum_value(pi)) >= 0) {
				for (i= 0; i < val; i++) {
					if (xpe->xpenum_sel == (xp_esel *) NULL)
						break;
					else
						xpe->xpenum_sel= xpe->xpenum_sel->xpesel_next;
				}
				if (xpe->xpenum_sel == (xp_esel *) NULL)
					xpe->xpenum_val= 0;
			}
		}

		/* if no value (or an invalid one) was specified and the
		   list is non-empty, set the value to the first xp_esel */
		if (xpe->xpenum_list != (xp_esel *) NULL) {
			if (xpe->xpenum_sel == (xp_esel *) NULL) {
				xpe->xpenum_sel= xpe->xpenum_list;
				xpe->xpenum_val= 0x1;
			}
		}
		else
			xpe->xpenum_val= 0;

		if (!emarks_set) {
			if ((xpe->xpenum_choicemark= image_create(pi->xpi_dpy, pi->xpi_fg, pi->xpi_bg, choicemark_bits, choicemark_width, choicemark_height, depth)) == (XImage *) NULL)
				exit(-1);
			for (es= xpe->xpenum_list; es != (xp_esel *) NULL; es= es->xpesel_next)
				es->xpesel_mimage= xpe->xpenum_choicemark;
		}
		if (!enmarks_set) {
			if ((xpe->xpenum_choicenomark= image_create(pi->xpi_dpy, pi->xpi_fg, pi->xpi_bg, choicenomark_bits, choicenomark_width, choicenomark_height, depth)) == (XImage *) NULL)
				exit(-1);
			for (es= xpe->xpenum_list; es != (xp_esel *) NULL; es= es->xpesel_next)
				es->xpesel_nmimage= xpe->xpenum_choicenomark;
		}
		if (!eloc_set)
			xpenum_elocinit(pi);
		else
			xpenum_eszinit(pi);

		xpenum_bld_menu(p, pi);
		xpenum_adj_menu(pi);

		break;

	case LXPI_CYCLE:

		xpe= (xp_enum *) &(pi->xpi_item.xpi_enum);

		/* make sure that a legal value was specified */
		if (eval_set) {
			xpe->xpenum_sel= xpe->xpenum_list;
			if ((val= xpenum_value(pi)) >= 0) {
				for (i= 0; i < val; i++) {
					if (xpe->xpenum_sel == (xp_esel *) NULL)
						break;
					else
						xpe->xpenum_sel= xpe->xpenum_sel->xpesel_next;
				}
				if (xpe->xpenum_sel == (xp_esel *) NULL)
					xpe->xpenum_val= 0;
			}
		}

		/* if no value (or an invalid one) was specified and the
		   list is non-empty, set the value to the first xp_esel */
		if (xpe->xpenum_list != (xp_esel *) NULL) {
			if (xpe->xpenum_sel == (xp_esel *) NULL) {
				xpe->xpenum_sel= xpe->xpenum_list;
				xpe->xpenum_val= 0x1;
			}
		}
		else
			xpe->xpenum_val= 0;

		/* create cycle image */
		if ((xpe->xpenum_cycle= image_create(pi->xpi_dpy, pi->xpi_fg, pi->xpi_bg, cycle_bits, cycle_width, cycle_height, depth)) == (XImage *) NULL)
			exit(-1);

		xpenum_bld_menu(p, pi);
		xpenum_adj_menu(pi);

		break;

	case LXPI_TOGGLE:

		xpe= (xp_enum *) &(pi->xpi_item.xpi_enum);

		/* make sure that a legal value was specified */
		if (eval_set) {
			for (n= 0, es= xpe->xpenum_list; es != (xp_esel *) NULL; n++, es= es->xpesel_next);
			for (i= 0; i < LXPENUM_MAXESELS; i++) {
				if ((xpe->xpenum_val & (0x1 << i)) && (i >= n))
					xpe->xpenum_val&= ~(0x1 << i);
			}
		}

		/* enable all xp_sel structs if no value specified */
		else {
			xpe->xpenum_val= 0;
			for (i= 0, es= xpe->xpenum_list; es != (xp_esel *) NULL; i++, es= es->xpesel_next)
				xpe->xpenum_val|= (0x1 << i);
		}

		if (!emarks_set) {
			if ((xpe->xpenum_togglemark= image_create(pi->xpi_dpy, pi->xpi_fg, pi->xpi_bg, togglemark_bits, togglemark_width, togglemark_height, depth)) == (XImage *) NULL)
				exit(-1);
			for (es= xpe->xpenum_list; es != (xp_esel *) NULL; es= es->xpesel_next)
				es->xpesel_mimage= xpe->xpenum_togglemark;
		}
		if (!enmarks_set) {
			if ((xpe->xpenum_togglenomark= image_create(pi->xpi_dpy, pi->xpi_fg, pi->xpi_bg, togglenomark_bits, togglenomark_width, togglenomark_height, depth)) == (XImage *) NULL)
				exit(-1);
			for (es= xpe->xpenum_list; es != (xp_esel *) NULL; es= es->xpesel_next)
				es->xpesel_nmimage= xpe->xpenum_togglenomark;
		}

		if (!eloc_set)
			xpenum_elocinit(pi);
		else
			xpenum_eszinit(pi);

		xpenum_bld_menu(p, pi);

		break;

	default:
		break;
	}

	/* link new item into panel */
	if (panelitem_insert(p, pi) != LX_SUCCESS) {
		(void) panelitem_destroy(pi);
		return((Panel_item *) NULL);
	}
	(void) (panelitem_proctab[pi->xpi_type].xpi_sz_proc)(p, pi);

	return(pi);
}

int
panelitem_filldefaults(p, pi)
/*
   Internal function.
   Initialize defaults for a newly-created item.
*/
Panel *p;
Panel_item *pi;
{
	pi->xpi_dpy= p->xp_dpy;
	pi->xpi_x= pi->xpi_y= 0;
	pi->xpi_fg= p->xp_fg;
	pi->xpi_bg= p->xp_bg;
	pi->xpi_font= p->xp_font;
	pi->xpi_proc= (void (*)()) NULL;
	pi->xpi_state= LXPI_ACTIVE;
	pi->xpi_display= LXPI_REPAINT;
	pi->xpi_str= (char *) NULL;
	pi->xpi_image= (XImage *) NULL;
	pi->xpi_clientdata= (char *) NULL;
	pi->xpi_gc= pi->xpi_igc= None;
	pi->xpi_cgc= pi->xpi_sgc= None;
	pi->xpi_menu= (Menu *) NULL;
	pi->xpi_next= (Panel_item *) NULL;
	return((panelitem_proctab[pi->xpi_type].xpi_idef_proc)(p, pi));
}

/*VARARGS*/
int
panelitem_set(va_alist)
/*
   User-callable.
   Changes an attribute of a previously created Panel_item.
*/
va_dcl
{
	va_list varg_ptr;
	Panel *p;
	Panel_item *pi, *xpi;
	int attr, nargs, n, state, i, repaint, textlen;
	boolean change_loc, change_sz, change_gcs;
	boolean change_state, change_val;
	boolean visible;
	boolean panel_redraw, panelitem_redraw;
	boolean valid;
	XImage image;
	XGCValues gcv;
	GC gc;
	char *c;
	xp_button *xpb;
	xp_slider *xps;
	xp_esel *es;
	void xptext_cycleseltext(), lxt_clearsel();
	int panel_resize(), panel_config();
	void panel_draw(), panel_display();
	void panelitem_draw(), panelitem_display();
	void panelvscroll_draw(), panelhscroll_draw();
	void xpenum_adj_menu();
	void panel_unmapsubwins();

	p= (Panel *) NULL;
	pi= (Panel_item *) NULL;
	change_loc= change_sz= change_gcs= FALSE;
	change_state= change_val= FALSE;
	panel_redraw= panelitem_redraw= FALSE;

	va_start(varg_ptr);
	for (nargs= 0;; nargs++) {

		/* get panel */
		if (nargs == 0) {
			p= va_arg(varg_ptr, Panel *);
			if (p == (Panel *) NULL) {
				(void) fprintf(stderr, "panelitem_set: null panel\n");
				return(LX_ERROR);
			}
			if (p->xp_magic != LX_PANEL) {
				(void) fprintf(stderr, "panelitem_set: object is not a panel\n");
				return(LX_ERROR);
			}
			if ((p->xp_flags & LXP_PANELVISIBLE) && (p->xp_flags & LXP_FRAMEMAPPED))
				visible= TRUE;
			else
				visible= FALSE;
		}

		/* get item */
		else if (nargs == 1) {
			pi= va_arg(varg_ptr, Panel_item *);
			if (pi == (Panel_item *) NULL) {
				(void) fprintf(stderr, "panelitem_set: null panel item\n");
				return(LX_ERROR);
			}
			if (pi->xpi_magic != LX_PANELITEM) {
				(void) fprintf(stderr, "panelitem_set: object is not a panel item\n");
				return(LX_ERROR);
			}
			for (xpi= p->xp_items; xpi != (Panel_item *) NULL; xpi= xpi->xpi_next) {
				if (xpi == pi)
					break;
			}
			if (xpi == (Panel_item *) NULL) {
				(void) fprintf(stderr, "panelitem_set: cannot locate panel item within given panel\n");
				return(LX_ERROR);
			}
		}

		/* get attribute */
		else {
			attr= va_arg(varg_ptr, int);
			if (attr == LXPI_NULL)
				break;

			switch (attr) {

			case LXPI_X:
				pi->xpi_x= va_arg(varg_ptr, int);
				change_loc= TRUE;
				break;
			case LXPI_Y:
				pi->xpi_y= va_arg(varg_ptr, int);
				change_loc= TRUE;
				break;
			case LXPI_PROC:
				pi->xpi_proc= (void (*)()) va_arg(varg_ptr, char *);
				break;
			case LXPI_FONT:
				c= va_arg(varg_ptr, char *);
				if (c == (char *) NULL)
					(void) fprintf(stderr, "panelitem_set: null fontname\n");
				else if ((pi->xpi_font= XLoadQueryFont(pi->xpi_dpy, c)) == NULL)
					(void) fprintf(stderr, "panelitem_set: cannot load font %s\n", c);
				else {
					if (pi->xpi_menu != (Menu *) NULL)
						(void) menu_set(pi->xpi_menu, LXM_FONT, c, LXM_NULL);
					change_sz= TRUE;
					change_gcs= TRUE;
				}
				break;
			case LXPI_FOREGROUND:
				pi->xpi_fg= va_arg(varg_ptr, unsigned long);
				change_gcs= TRUE;
				break;
			case LXPI_BACKGROUND:
				pi->xpi_bg= va_arg(varg_ptr, unsigned long);
				change_gcs= TRUE;
				break;
			case LXPI_STATE:
				state= va_arg(varg_ptr, int);
				switch (state) {
				case LXPI_ACTIVE:
				case LXPI_INACTIVE:
					if (pi->xpi_state != state)
						change_state= TRUE;
					pi->xpi_state= state;
					break;
				default:
					(void) fprintf(stderr, "panelitem_set: invalid item state\n");
					break;
				}
				break;
			case LXPI_DISPLAY:
				repaint= va_arg(varg_ptr, int);
				switch (repaint) {
				case LXPI_REPAINT:
				case LXPI_NOREPAINT:
					pi->xpi_display= repaint;
					break;
				default:
					(void) fprintf(stderr, "panelitem_set: invalid item display value\n");
					break;
				}
				break;
			case LXPI_STRING:
				c= va_arg(varg_ptr, char *);
				if (pi->xpi_str != (char *) NULL)
					cfree(pi->xpi_str);
				pi->xpi_str= (char *) NULL;
				if (c != (char *) NULL) {
					if (strlen(c) != 0) {
						if ((pi->xpi_str= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
							(void) fprintf(stderr, "panelitem_set: memory allocation error\n");
							return(LX_ERROR);
						}
						(void) strcpy(pi->xpi_str, c);
						change_sz= TRUE;
					}
				}
				break;
			case LXPI_IMAGE:
				pi->xpi_image= va_arg(varg_ptr, XImage *);
				change_sz= TRUE;
				break;
			case LXPI_CLIENTDATA:
				pi->xpi_clientdata= va_arg(varg_ptr, char *);
				break;
			case LXPTEXT_VALUE:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				c= va_arg(varg_ptr, char *);
				if (c == (char *) NULL) {
					(void) strcpy(pi->xpi_item.xpi_text.xptext_val, "");
					textlen= 0;
				}
				else if ((textlen= strlen(c)) == 0)
					(void) strcpy(pi->xpi_item.xpi_text.xptext_val, "");
				else if (textlen <= pi->xpi_item.xpi_text.xptext_max)
					(void) strcpy(pi->xpi_item.xpi_text.xptext_val, c);
				else {
					cfree(pi->xpi_item.xpi_text.xptext_val);
					pi->xpi_item.xpi_text.xptext_max= textlen;
					if ((pi->xpi_item.xpi_text.xptext_val= calloc((unsigned) (textlen+1), sizeof(char))) == (char *) NULL) {
						(void) fprintf(stderr, "panelitem_set: memory allocation error\n");
						return(LX_ERROR);
					}
					(void) strcpy(pi->xpi_item.xpi_text.xptext_val, c);
				}
				pi->xpi_item.xpi_text.xptext_ipos= textlen;
				if (pi->xpi_item.xpi_text.xptext_disp < textlen)
					pi->xpi_item.xpi_text.xptext_dpos= textlen-pi->xpi_item.xpi_text.xptext_disp;
				else
					pi->xpi_item.xpi_text.xptext_dpos= 0;
				change_val= TRUE;
				break;
			case LXPTEXT_MAXSTORE:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panelitem_set: invalid text storage length\n");
				else {
					char *oldval;

					oldval= pi->xpi_item.xpi_text.xptext_val;
					pi->xpi_item.xpi_text.xptext_max= n;
					if ((pi->xpi_item.xpi_text.xptext_val= calloc((unsigned) (n+1), sizeof(char))) == (char *) NULL) {
						(void) fprintf(stderr, "panelitem_set: memory allocation error\n");
						(void) panelitem_destroy(pi);
						return(LX_ERROR);
					}
					(void) strncpy(pi->xpi_item.xpi_text.xptext_val, oldval, n);
					cfree(oldval);
					change_val= TRUE;
				}
				break;
			case LXPTEXT_MAXDISPLAY:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n >= 0) {
					pi->xpi_item.xpi_text.xptext_disp= n;
					if (pi->xpi_item.xpi_text.xptext_dpos < strlen(pi->xpi_item.xpi_text.xptext_val)-n)
						pi->xpi_item.xpi_text.xptext_dpos= strlen(pi->xpi_item.xpi_text.xptext_val)-n;
					change_sz= TRUE;
				}
				else
					(void) fprintf(stderr, "panelitem_set: invalid text display length\n");
				break;
			case LXPTEXT_IPOS:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if ((n < 0) || (n > strlen(pi->xpi_item.xpi_text.xptext_val)))
					(void) fprintf(stderr, "panelitem_set: invalid text position\n");
				else {
					pi->xpi_item.xpi_text.xptext_ipos= n;
					change_val= TRUE;
				}
				break;
			case LXPTEXT_GAP:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panelitem_set: invalid text gap\n");
				else {
					pi->xpi_item.xpi_text.xptext_gap= n;
					change_sz= TRUE;
				}
				break;
			case LXPTEXT_PROC:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				pi->xpi_item.xpi_text.xptext_kpproc= (int (*)()) va_arg(varg_ptr, char *);
				break;
			case LXPTEXT_RDONLY:
				if (pi->xpi_type != LXPI_TEXT)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n)
					pi->xpi_item.xpi_text.xptext_flags|= LXPTEXT_WRPROTECT;
				else
					pi->xpi_item.xpi_text.xptext_flags&= ~LXPTEXT_WRPROTECT;
				break;
			case LXPBUTTON_HMARGIN:
				if (pi->xpi_type != LXPI_BUTTON)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panelitem_set: invalid button margin\n");
				else {
					pi->xpi_item.xpi_button.xpbutton_hmargin= n;
					change_sz= TRUE;
				}
				break;
			case LXPBUTTON_VMARGIN:
				if (pi->xpi_type != LXPI_BUTTON)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panelitem_set: invalid button margin\n");
				else {
					pi->xpi_item.xpi_button.xpbutton_vmargin= n;
					change_sz= TRUE;
				}
				break;
			case LXPSLIDER_MINVAL:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_minval= va_arg(varg_ptr, int);
				change_sz= TRUE;
				break;
			case LXPSLIDER_MAXVAL:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_maxval= va_arg(varg_ptr, int);
				change_sz= TRUE;
				break;
			case LXPSLIDER_VALUE:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_val= va_arg(varg_ptr, int);
				change_val= TRUE;
				break;
			case LXPSLIDER_VALX:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_valx= va_arg(varg_ptr, int);
				change_sz= TRUE;
				break;
			case LXPSLIDER_VALY:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_valy= va_arg(varg_ptr, int);
				change_sz= TRUE;
				break;
			case LXPSLIDER_BARHEIGHT:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panelitem_set: invalid slider bar width\n");
				else {
					pi->xpi_item.xpi_slider.xpslider_barh= n;
					change_sz= TRUE;
				}
				break;
			case LXPSLIDER_BARLENGTH:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n < 0)
					(void) fprintf(stderr, "panelitem_set: invalid slider bar width\n");
				else {
					pi->xpi_item.xpi_slider.xpslider_barl= n;
					change_sz= TRUE;
				}
				break;
			case LXPSLIDER_BARX:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_barx= va_arg(varg_ptr, int);
				change_sz= TRUE;
				break;
			case LXPSLIDER_BARY:
				if (pi->xpi_type != LXPI_SLIDER)
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				pi->xpi_item.xpi_slider.xpslider_bary= va_arg(varg_ptr, int);
				change_sz= TRUE;
				break;
			case LXPENUM_VALUE:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_CYCLE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				switch (pi->xpi_type) {
				case LXPI_CHOICE:
				case LXPI_CYCLE:
					es= pi->xpi_item.xpi_enum.xpenum_list;
					for (i= 0; es != (xp_esel *) NULL; i++, es= es->xpesel_next) {
						if (i == n)
							break;
					}
					if (es == (xp_esel *) NULL) {
						valid= FALSE;
						(void) fprintf(stderr, "panelitem_set: illegal enum value\n");
					}
					else {
						valid= TRUE;
						pi->xpi_item.xpi_enum.xpenum_sel= es;
						pi->xpi_item.xpi_enum.xpenum_val= (0x1 << n);
						xpenum_adj_menu(pi);
					}
					break;
				case LXPI_TOGGLE:
					pi->xpi_item.xpi_enum.xpenum_val= n;
					for (n= 0, es= pi->xpi_item.xpi_enum.xpenum_list; es != (xp_esel *) NULL; n++, es= es->xpesel_next);
					for (i= 0; i < LXPENUM_MAXESELS; i++) {
						if ((pi->xpi_item.xpi_enum.xpenum_val & (0x1 << i)) && (i >= n))
							pi->xpi_item.xpi_enum.xpenum_val&= ~(0x1 << i);
					}
					break;
				default:
					break;
				}
				if (valid)
					change_val= TRUE;
				break;
			case LXPENUM_DISPLAY:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if ((n != LXPENUM_DISPALL) && (n != LXPENUM_DISPCURRSEL)) {
					(void) fprintf(stderr, "panelitem_set: unrecognized enum display level\n");
					break;
				}
				pi->xpi_item.xpi_enum.xpenum_flags&= ~(LXPENUM_DISPALL | LXPENUM_DISPCURRSEL);
				pi->xpi_item.xpi_enum.xpenum_flags|= n;
				break;
			case LXPENUM_INVERT:
				if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
					(void) fprintf(stderr, "panelitem_set: warning -- item type mismatch\n");
				n= va_arg(varg_ptr, int);
				if (n) {
					if (!(pi->xpi_item.xpi_enum.xpenum_flags & LXPENUM_INVERTLABEL)) {
						pi->xpi_item.xpi_enum.xpenum_flags|= LXPENUM_INVERTLABEL;
						change_sz= TRUE;
					}
				}
				else {
					if (pi->xpi_item.xpi_enum.xpenum_flags & LXPENUM_INVERTLABEL) {
						pi->xpi_item.xpi_enum.xpenum_flags&= ~LXPENUM_INVERTLABEL;
						change_sz= TRUE;
					}
				}
				break;
			default:
				(void) fprintf(stderr, "panelitem_set: unrecognized attribute\n");
				return(LX_ERROR);
				break;
			}
		}
	}
	va_end(varg_ptr);

	if (pi->xpi_display == LXPI_NOREPAINT)
		visible= FALSE;

	if (visible) {

		/* redo entire panel */
		if (change_loc || change_sz)
			panel_redraw= TRUE;

		/* redisplay single item */
		if (change_gcs || change_state || change_val)
			panelitem_redraw= TRUE;
	}

	if (change_state && (pi->xpi_type == LXPI_TEXT)) {
		switch (pi->xpi_state) {
		case LXPI_INACTIVE:
			if (p->xp_seltext == pi) {
				p->xp_seltext= (Panel_item *) NULL;
				xptext_cycleseltext(p);
			}
			if (pi == (Panel_item *) lxt_selsrc)
				lxt_clearsel(FALSE);
			if (pi == (Panel_item *) lxt_seldest) {
				lxt_seldest= (Void *) NULL;
				lxt_seldesttype= LX_NULL;
			}
			break;
		case LXPI_ACTIVE:
			if (p->xp_seltext == (Panel_item *) NULL)
				p->xp_seltext= pi;
			break;
		}
	}

	if (change_sz)
		(void) (panelitem_proctab[pi->xpi_type].xpi_sz_proc)(p, pi);

	if (change_gcs) {
		switch (pi->xpi_type) {
	
		case LXPI_BUTTON:
	
			xpb= (xp_button *) &(pi->xpi_item.xpi_button);
	
			/* rebuild button proc hash */
			if (xpb->xpbutton_procpm != None)
				XFreePixmap(pi->xpi_dpy, xpb->xpbutton_procpm);
			if (xpb->xpbutton_bgc != None)
				XFreeGC(pi->xpi_dpy, xpb->xpbutton_bgc);
			xpb->xpbutton_procpm= XCreatePixmap(pi->xpi_dpy, DefaultRootWindow(pi->xpi_dpy), buttonproc_width, buttonproc_height, 1);
			gcv.foreground= pi->xpi_fg;
			gcv.background= pi->xpi_bg;
			gc= XCreateGC(pi->xpi_dpy, xpb->xpbutton_procpm, GCForeground | GCBackground, &gcv);
			image.height= buttonproc_height;
			image.width= buttonproc_width;
			image.xoffset= 0;
			image.format= XYBitmap;
			image.data= buttonproc_bits;
			image.byte_order= LSBFirst;
			image.bitmap_unit= 8;
			image.bitmap_bit_order= LSBFirst;
			image.bitmap_pad= 8;
			image.bytes_per_line= (buttonproc_width+7)>>3;
			image.depth= 1;
			XPutImage(pi->xpi_dpy, xpb->xpbutton_procpm, gc, &image, 0, 0, 0, 0, buttonproc_width, buttonproc_height);
			XFreeGC(pi->xpi_dpy, gc);
	
			/* set GC for button proc activation */
			gcv.foreground= pi->xpi_fg;
			gcv.background= pi->xpi_bg;
			gcv.function= gcv.foreground ? GXor : GXand;
			gcv.stipple= xpb->xpbutton_procpm;
			gcv.fill_style= FillStippled;
			pi->xpi_item.xpi_button.xpbutton_bgc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCForeground | GCBackground | GCFunction | GCStipple | GCFillStyle), &gcv);
			break;
	
		case LXPI_SLIDER:
	
			xps= (xp_slider *) &(pi->xpi_item.xpi_slider);
	
			/* rebuild grey stipple */
			if (xps->xpslider_bpm != None)
				XFreePixmap(pi->xpi_dpy, xps->xpslider_bpm);
			if (xps->xpslider_bgc != None)
				XFreeGC(pi->xpi_dpy, xps->xpslider_bgc);
			xps->xpslider_bpm= XCreatePixmap(pi->xpi_dpy, DefaultRootWindow(pi->xpi_dpy), grey_width, grey_height, 1);
			gcv.foreground= pi->xpi_fg;
			gcv.background= pi->xpi_bg;
			gc= XCreateGC(pi->xpi_dpy, xps->xpslider_bpm, GCForeground | GCBackground, &gcv);
			image.height= grey_height;
			image.width= grey_width;
			image.xoffset= 0;
			image.format= XYBitmap;
			image.data= grey_bits;
			image.byte_order= LSBFirst;
			image.bitmap_unit= 8;
			image.bitmap_bit_order= LSBFirst;
			image.bitmap_pad= 8;
			image.bytes_per_line= (grey_width+7)>>3;
			image.depth= 1;
			XPutImage(pi->xpi_dpy, xps->xpslider_bpm, gc, &image, 0, 0, 0, 0, grey_width, grey_height);
			XFreeGC(pi->xpi_dpy, gc);
	
			/* set GC for slider bubble */
			gcv.foreground= pi->xpi_fg;
			gcv.background= pi->xpi_bg;
			gcv.function= gcv.foreground ? GXor : GXand;
			gcv.stipple= xps->xpslider_bpm;
			gcv.fill_style= FillStippled;
			xps->xpslider_bgc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCForeground | GCBackground | GCFunction | GCStipple | GCFillStyle), &gcv);
			break;
	
		default:
			break;
		}
	
		/* recreate GCs */
		if (pi->xpi_gc != None)
			XFreeGC(pi->xpi_dpy, pi->xpi_gc);
		if (pi->xpi_igc != None)
			XFreeGC(pi->xpi_dpy, pi->xpi_igc);
		if (pi->xpi_sgc != None)
			XFreeGC(pi->xpi_dpy, pi->xpi_sgc);
		if (pi->xpi_cgc != None)
			XFreeGC(pi->xpi_dpy, pi->xpi_cgc);
		gcv.font= pi->xpi_font->fid;
		gcv.foreground= pi->xpi_fg;
		gcv.background= pi->xpi_bg;
		gcv.plane_mask= pi->xpi_fg | pi->xpi_bg;
		gcv.function= GXcopy;
		pi->xpi_gc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCFont | GCForeground | GCBackground | GCPlaneMask | GCFunction), &gcv);
		gcv.plane_mask= pi->xpi_fg ^ pi->xpi_bg;
		gcv.function= GXinvert;
		pi->xpi_igc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCFont | GCForeground | GCBackground | GCFunction | GCPlaneMask), &gcv);
		gcv.function= gcv.foreground ? GXor : GXand;
		gcv.stipple= p->xp_stipplepm;
		gcv.fill_style= FillStippled;
		pi->xpi_sgc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCFont | GCForeground | GCBackground | GCFunction | GCStipple | GCFillStyle), &gcv);
		gcv.foreground= pi->xpi_bg;
        	gcv.background= pi->xpi_fg;
		gcv.plane_mask= pi->xpi_fg | pi->xpi_bg;
        	gcv.function= GXcopy;
        	pi->xpi_cgc= XCreateGC(p->xp_dpy, p->xp_pwin, (GCFont | GCForeground | GCBackground | GCPlaneMask | GCFunction), &gcv);
	}

	if (panel_redraw && (pi->xpi_display == LXPI_REPAINT)) {
#ifdef PRE_R3
		/* this causes a server crash under X.V11R3/SunOS3.4! */
		XUnmapSubwindows(p->xp_dpy, p->xp_win);
#else
		panel_unmapsubwins(p);
#endif PRE_R3
		(void) panel_resize(p);
		(void) panel_config(p);
		p->xp_flags|= LXP_CONFIGDONE;
		XMapWindow(p->xp_dpy, p->xp_pwin);
		panel_draw(p);
		panel_display(p);
		if (p->xp_flags & LXP_VSCROLLVIS) {
			XMapWindow(p->xp_dpy, p->xp_vswin);
			panelvscroll_draw(p);
		}
		if (p->xp_flags & LXP_HSCROLLVIS) {
			XMapWindow(p->xp_dpy, p->xp_hswin);
			panelhscroll_draw(p);
		}
	}
	else if (panelitem_redraw) {
		panelitem_draw(p, pi);
		panelitem_display(p, pi);
	}

	return(LX_SUCCESS);
}

int
panelitem_insert(p, pi)
/*
   Internal function.
   Inserts a Panel_item into a previously created panel.
*/
Panel *p;
Panel_item *pi;
{
	Panel_item *qi;
	int panel_resize();

	if (p == (Panel *) NULL) {
		(void) fprintf(stderr, "panelitem_insert: null panel\n");
		return(LX_ERROR);
	}
	if (pi == (Panel_item *) NULL) {
		(void) fprintf(stderr, "panelitem_insert: null panel item\n");
		return(LX_ERROR);
	}

	pi->xpi_next= (Panel_item *) NULL;
	if (p->xp_items == (Panel_item *) NULL)
		p->xp_items= pi;
	else {
		for (qi= p->xp_items; qi->xpi_next != (Panel_item *) NULL; qi= qi->xpi_next);	qi->xpi_next= pi;
	}

	/* resize panel */
	if (panel_resize(p) != LX_SUCCESS)
		return(LX_ERROR);

	return(LX_SUCCESS);
}

int
panelitem_delete(p, pi)
/*
   User-callable.
   Removes a Panel_item from a panel.
*/
Panel *p;
Panel_item *pi;
{
	Panel_item *xpi, *ypi;
	int panel_resize();
	void lxt_clearsel();

	if (p == (Panel *) NULL) {
		(void) fprintf(stderr, "panelitem_delete: null panel\n");
		return(LX_ERROR);
	}
	if (p->xp_magic != LX_PANEL) {
		(void) fprintf(stderr, "panelitem_delete: object is not a panel\n");
		return(LX_ERROR);
	}
	if (pi == (Panel_item *) NULL) {
		(void) fprintf(stderr, "panelitem_delete: null panel item\n");
		return(LX_ERROR);
	}
	if (pi->xpi_magic != LX_PANELITEM) {
		(void) fprintf(stderr, "panelitem_delete: object is not a panel item\n");
		return(LX_ERROR);
	}

	for (xpi= ypi= p->xp_items; xpi != (Panel_item *) NULL; xpi= xpi->xpi_next) {
		if (xpi == pi)
			break;
		else
			ypi= xpi;
	}
	if (xpi == (Panel_item *) NULL) {
		(void) fprintf(stderr, "panelitem_delete: item not found within panel\n");
		return(LX_ERROR);
	}

	if (pi == p->xp_selitem)
		p->xp_selitem= (Panel_item *) NULL;

	/* make sure that p->xp_seltext != pi */
	if (pi == p->xp_seltext) {
		xptext_cycleseltext(p);
		if (pi == p->xp_seltext)
			p->xp_seltext= (Panel_item *) NULL;
	}

	/* make sure that lxt_selsrc != pi */
	if (pi == (Panel_item *) lxt_selsrc)
		lxt_clearsel(FALSE);

	/* make sure that lxt_seldest != pi */
	if (pi == (Panel_item *) lxt_seldest) {
		lxt_seldest= (Void *) NULL;
		lxt_seldesttype= LX_NULL;
	}

	/* unlink item from panel */
	if (pi == p->xp_items)
		p->xp_items= pi->xpi_next;
	else
		ypi->xpi_next= pi->xpi_next;

	/* resize panel */
	if (panel_resize(p) != LX_SUCCESS)
		return(LX_ERROR);

	return(LX_SUCCESS);
}

int
panelitem_destroy(pi)
/*
   User-callable.
   Destroys a Panel_item. Does not check to see
   if the item is still linked into a panel!
*/
Panel_item *pi;
{
	int code;

	if (pi == (Panel_item *) NULL) {
		(void) fprintf(stderr, "panelitem_destroy: null panel item\n");
		return(LX_ERROR);
	}
	if (pi->xpi_magic != LX_PANELITEM) {
		(void) fprintf(stderr, "panelitem_destroy: object is not a panel item\n");
		return(LX_ERROR);
	}

	if (pi->xpi_str != (char *) NULL)
		cfree(pi->xpi_str);
	if (pi->xpi_gc != None)
		XFreeGC(pi->xpi_dpy, pi->xpi_gc);
	if (pi->xpi_igc != None)
		XFreeGC(pi->xpi_dpy, pi->xpi_igc);
	if (pi->xpi_cgc != None)
		XFreeGC(pi->xpi_dpy, pi->xpi_cgc);
	if (pi->xpi_sgc != None)
		XFreeGC(pi->xpi_dpy, pi->xpi_sgc);

	code= (panelitem_proctab[pi->xpi_type].xpi_dst_proc)(pi);
	cfree((char *) pi);
	return(code);
}

Void *
panelitem_get(p, pi, attr)
/*
   User-callable.
   Returns the value of an attribute or a pointer thereto.
*/
Panel *p;
Panel_item *pi;
int attr;
{
	Panel_item *xpi;
	int xpenum_value();

	xpret_void= (Void *) NULL;
	if (p == (Panel *) NULL) {
		(void) fprintf(stderr, "panelitem_get: null panel\n");
		return(xpret_void);
	}
	if (p->xp_magic != LX_PANEL) {
		(void) fprintf(stderr, "panelitem_get: object is not a panel\n");
		return(xpret_void);
	}
	if (pi == (Panel_item *) NULL) {
		(void) fprintf(stderr, "panelitem_get: null panel item\n");
		return(xpret_void);
	}
	if (pi->xpi_magic != LX_PANELITEM) {
		(void) fprintf(stderr, "panelitem_get: object is not a panel item\n");
		return(xpret_void);
	}

	for (xpi= p->xp_items; xpi != (Panel_item *) NULL; xpi= xpi->xpi_next) {
		if (xpi == pi)
			break;
	}
	if (xpi == (Panel_item *) NULL) {
		(void) fprintf(stderr, "panelitem_get: item not found within panel\n");
		return(xpret_void);
	}
	if (attr == LXPI_NULL)
		return(xpret_void);

	switch (attr) {

	case LXPI_X:
		xpret_void= (Void *) &(pi->xpi_x);
		break;
	case LXPI_Y:
		xpret_void= (Void *) &(pi->xpi_y);
		break;
	case LXPI_PROC:
		xpret_void= (Void *) pi->xpi_proc;
		break;
	case LXPI_FONT:
		xpret_void= (Void *) pi->xpi_font;
		break;
	case LXPI_FOREGROUND:
		xpret_void= (Void *) &(pi->xpi_fg);
		break;
	case LXPI_BACKGROUND:
		xpret_void= (Void *) &(pi->xpi_bg);
		break;
	case LXPI_STATE:
		xpret_void= (Void *) &(pi->xpi_state);
		break;
	case LXPI_DISPLAY:
		xpret_void= (Void *) &(pi->xpi_display);
		break;
	case LXPI_STRING:
		xpret_void= (Void *) pi->xpi_str;
		break;
	case LXPI_IMAGE:
		xpret_void= (Void *) pi->xpi_image;
		break;
	case LXPI_CLIENTDATA:
		xpret_void= (Void *) pi->xpi_clientdata;
		break;
	case LXPI_TIMESTAMP:
		xpret_void= (Void *) &(pi->xpi_timestamp);
		break;
	case LXPI_TYPE:
		xpret_void= (Void *) &(pi->xpi_type);
		break;
	case LXPTEXT_VALUE:
		if (pi->xpi_type != LXPI_TEXT)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) pi->xpi_item.xpi_text.xptext_val;
		break;
	case LXPTEXT_MAXSTORE:
		if (pi->xpi_type != LXPI_TEXT)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_text.xptext_max);
		break;
	case LXPTEXT_MAXDISPLAY:
		if (pi->xpi_type != LXPI_TEXT)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_text.xptext_disp);
		break;
	case LXPTEXT_IPOS:
		if (pi->xpi_type != LXPI_TEXT)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_text.xptext_ipos); 
		break;
	case LXPTEXT_GAP:
		if (pi->xpi_type != LXPI_TEXT)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_text.xptext_gap);
		break;
	case LXPTEXT_PROC:
		if (pi->xpi_type != LXPI_TEXT)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) pi->xpi_item.xpi_text.xptext_kpproc;
		break;
	case LXPTEXT_RDONLY:
		if (pi->xpi_type != LXPI_TEXT)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else {
			if (pi->xpi_item.xpi_text.xptext_flags & LXPTEXT_WRPROTECT)
				xpret_bool= TRUE;
			else
				xpret_bool= FALSE;
			xpret_void= (Void *) &xpret_bool;
		}
		break;
	case LXPBUTTON_HMARGIN:
		if (pi->xpi_type != LXPI_BUTTON)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_button.xpbutton_hmargin);
		break;
	case LXPBUTTON_VMARGIN:
		if (pi->xpi_type != LXPI_BUTTON)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_button.xpbutton_vmargin);
		break;
	case LXPSLIDER_MINVAL:
		if (pi->xpi_type != LXPI_SLIDER)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_slider.xpslider_minval);
		break;
	case LXPSLIDER_MAXVAL:
		if (pi->xpi_type != LXPI_SLIDER)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_slider.xpslider_maxval);
		break;
	case LXPSLIDER_VALUE:
		if (pi->xpi_type != LXPI_SLIDER)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_slider.xpslider_val);
		break;
	case LXPSLIDER_VALX:
		if (pi->xpi_type != LXPI_SLIDER)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_slider.xpslider_valx);
		break;
	case LXPSLIDER_VALY:
		if (pi->xpi_type != LXPI_SLIDER)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_slider.xpslider_valy);
		break;
	case LXPSLIDER_BARHEIGHT:
		if (pi->xpi_type != LXPI_SLIDER)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_slider.xpslider_barh);
		break;
	case LXPSLIDER_BARLENGTH:
		if (pi->xpi_type != LXPI_SLIDER)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_slider.xpslider_barl);
		break;
	case LXPSLIDER_BARX:
		if (pi->xpi_type != LXPI_SLIDER)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_slider.xpslider_barx);
		break;
	case LXPSLIDER_BARY:
		if (pi->xpi_type != LXPI_SLIDER)
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
		else
			xpret_void= (Void *) &(pi->xpi_item.xpi_slider.xpslider_bary);
		break;
	case LXPENUM_VALUE:
		switch (pi->xpi_type) {
		case LXPI_CHOICE:
		case LXPI_CYCLE:
			xpret_int= xpenum_value(pi);
			xpret_void= (Void *) &xpret_int;
			break;
		case LXPI_TOGGLE:
			xpret_void= (Void *) &(pi->xpi_item.xpi_enum.xpenum_val);
			break;
		default:
			(void) fprintf(stderr, "panelitem_get: item type mismatch\n");
			break;
		}
		break;
	case LXPENUM_DISPLAY:
		if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
			(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
		else {
			if (pi->xpi_item.xpi_enum.xpenum_flags & LXPENUM_DISPALL)
				xpret_int= LXPENUM_DISPALL;
			else
				xpret_int= LXPENUM_DISPCURRSEL;
			xpret_void= (Void *) &xpret_int;
		}
		break;
	case LXPENUM_INVERT:
		if ((pi->xpi_type != LXPI_CHOICE) && (pi->xpi_type != LXPI_TOGGLE))
			(void) fprintf(stderr, "panelitem_create: warning -- item type mismatch\n");
		else {
			if (pi->xpi_item.xpi_enum.xpenum_flags & LXPENUM_INVERTLABEL)
				xpret_bool= TRUE;
			else
				xpret_bool= FALSE;
			xpret_void= (Void *) &xpret_bool;
		}
		break;
	default:
		(void) fprintf(stderr, "panelitem_get: unrecognized attribute\n");
		break;
	}
	return(xpret_void);
}

void
xpi_null_proc(p, pi)
Panel *p;
Panel_item *pi;
{
}

/*VARARGS*/
int
panelitem_addsel(va_alist)
/*
   User-callable.
   Adds a selection to a previously created enum item.
*/
va_dcl
{
	va_list varg_ptr;
	Panel *p;
	Panel_item *pi, *xpi;
	int attr, nargs, depth;
	boolean visible, emark_set, enmark_set;
	char *c;
	Menu_item *mi;
	xp_esel *es;
	xp_enum *xpe;
	void xpenum_eszinit();
	int panel_resize(), panel_config();
	void panel_draw(), panel_display();
	void panelitem_draw(), panelitem_display();
	void panelvscroll_draw(), panelhscroll_draw();
	void xpenum_adj_menu();
	void xpenum_menu_proc();
	void panel_unmapsubwins();

	p= (Panel *) NULL;
	pi= (Panel_item *) NULL;
	emark_set= enmark_set= FALSE;

	va_start(varg_ptr);
	for (nargs= 0;; nargs++) {

		/* get panel */
		if (nargs == 0) {
			p= va_arg(varg_ptr, Panel *);
			if (p == (Panel *) NULL) {
				(void) fprintf(stderr, "panelitem_addsel: null panel\n");
				return(LX_ERROR);
			}
			if (p->xp_magic != LX_PANEL) {
				(void) fprintf(stderr, "panelitem_addsel: object is not a panel\n");
				return(LX_ERROR);
			}
			if ((p->xp_flags & LXP_PANELVISIBLE) && (p->xp_flags & LXP_FRAMEMAPPED))
				visible= TRUE;
			else
				visible= FALSE;
		}

		/* get item */
		else if (nargs == 1) {
			pi= va_arg(varg_ptr, Panel_item *);
			if (pi == (Panel_item *) NULL) {
				(void) fprintf(stderr, "panelitem_addsel: null panel item\n");
				return(LX_ERROR);
			}
			if (pi->xpi_magic != LX_PANELITEM) {
				(void) fprintf(stderr, "panelitem_addsel: object is not a panel item\n");
				return(LX_ERROR);
			}
			for (xpi= p->xp_items; xpi != (Panel_item *) NULL; xpi= xpi->xpi_next) {
				if (xpi == pi)
					break;
			}
			if (xpi == (Panel_item *) NULL) {
				(void) fprintf(stderr, "panelitem_addsel: cannot locate panel item within given panel\n");
				return(LX_ERROR);
			}
			switch (pi->xpi_type) {
			case LXPI_CHOICE:
			case LXPI_CYCLE:
			case LXPI_TOGGLE:
				break;
			default:
				(void) fprintf(stderr, "panelitem_addsel: item type mismatch\n");
				return(LX_ERROR);
				break;
			}
			if ((es= xpesel_alloc(pi)) == (xp_esel *) NULL) {
				(void) fprintf(stderr, "panelitem_addsel: memory allocation error\n");
				return(LX_ERROR);
			}
		}

		/* get attribute */
		else {
			attr= va_arg(varg_ptr, int);
			if (attr == LXPI_NULL)
				break;

			switch (attr) {

			case LXPENUM_SELSTRING:
				c= va_arg(varg_ptr, char *);
				if (es->xpesel_str != (char *) NULL)
					cfree(es->xpesel_str);
				es->xpesel_str= (char *) NULL;
				if (c != (char *) NULL) {
					if (strlen(c) != 0) {
						if ((es->xpesel_str= calloc((unsigned) (strlen(c)+1), sizeof(char))) == (char *) NULL) {
							(void) fprintf(stderr, "panelitem_addsel: memory allocation error\n");
							return(LX_ERROR);
						}
						(void) strcpy(es->xpesel_str, c);
					}
				}
				break;
			case LXPENUM_SELIMAGE:
				es->xpesel_image= va_arg(varg_ptr, XImage *);
				break;
			case LXPENUM_MARKIMAGE:
				es->xpesel_mimage= va_arg(varg_ptr, XImage *);
				emark_set= TRUE;
				break;
			case LXPENUM_NOMARKIMAGE:
				es->xpesel_nmimage= va_arg(varg_ptr, XImage *);
				enmark_set= TRUE;
				break;
			case LXPENUM_SELX:
				es->xpesel_lx= va_arg(varg_ptr, int);
				break;
			case LXPENUM_SELY:
				es->xpesel_ly= va_arg(varg_ptr, int);
				break;
			case LXPENUM_MARKX:
				es->xpesel_mx= va_arg(varg_ptr, int);
				break;
			case LXPENUM_MARKY:
				es->xpesel_my= va_arg(varg_ptr, int);
				break;
			default:
				(void) fprintf(stderr, "panelitem_addsel: unrecognized attribute\n");
				return(LX_ERROR);
				break;
			}
		}
	}

	depth= DefaultDepth(p->xp_dpy, DefaultScreen(p->xp_dpy));

	switch (pi->xpi_type) {
	case LXPI_CHOICE:
		xpe= (xp_enum *) &(pi->xpi_item.xpi_enum);

		if (!emark_set) {
			if ((xpe->xpenum_choicemark= image_create(pi->xpi_dpy, pi->xpi_fg, pi->xpi_bg, choicemark_bits, choicemark_width, choicemark_height, depth)) == (XImage *) NULL)
				exit(-1);
			es->xpesel_mimage= xpe->xpenum_choicemark;
		}
		if (!enmark_set) {
			if ((xpe->xpenum_choicenomark= image_create(pi->xpi_dpy, pi->xpi_fg, pi->xpi_bg, choicenomark_bits, choicenomark_width, choicenomark_height, depth)) == (XImage *) NULL)
				exit(-1);
			es->xpesel_nmimage= xpe->xpenum_choicenomark;
		}

		xpenum_eszinit(pi);

		if (xpe->xpenum_sel == (xp_esel *) NULL) {
			xpe->xpenum_val= (0x1 << es->xpesel_ord);
			xpe->xpenum_sel= es;
		}

		break;

	case LXPI_CYCLE:
		xpe= (xp_enum *) &(pi->xpi_item.xpi_enum);

		if (xpe->xpenum_sel == (xp_esel *) NULL) {
			xpe->xpenum_val= (0x1 << es->xpesel_ord);
			xpe->xpenum_sel= es;
		}

		break;

	case LXPI_TOGGLE:
		xpe= (xp_enum *) &(pi->xpi_item.xpi_enum);
		if (!emark_set) {
			if ((xpe->xpenum_togglemark= image_create(pi->xpi_dpy, pi->xpi_fg, pi->xpi_bg, togglemark_bits, togglemark_width, togglemark_height, depth)) == (XImage *) NULL)
				exit(-1);
			es->xpesel_mimage= xpe->xpenum_togglemark;
		}
		if (!enmark_set) {
			if ((xpe->xpenum_togglenomark= image_create(pi->xpi_dpy, pi->xpi_fg, pi->xpi_bg, togglenomark_bits, togglenomark_width, togglenomark_height, depth)) == (XImage *) NULL)
				exit(-1);
			es->xpesel_nmimage= xpe->xpenum_togglenomark;
		}
		xpenum_eszinit(pi);
		break;

	default:
		break;
	}

	if ((mi= menuitem_create(LXMI_STRING, es->xpesel_str,
				LXMI_IMAGE, es->xpesel_image,
				LXMI_PROC, xpenum_menu_proc,
				LXMI_NULL)) == (Menu_item *) NULL) {
		pi->xpi_menu= (Menu *) NULL;
		return(LX_ERROR);
	}
	(void) menuitem_insert(pi->xpi_menu, mi);
	xpenum_adj_menu(pi);

	(void) (panelitem_proctab[pi->xpi_type].xpi_sz_proc)(p, pi);

	if (visible) {
		if (pi->xpi_display == LXPI_REPAINT) {
#ifdef PRE_R3
			/* this causes a server crash under X.V11R3/SunOS3.4! */
			XUnmapSubwindows(p->xp_dpy, p->xp_win);
#else
			panel_unmapsubwins(p);
#endif PRE_R3
			(void) panel_resize(p);
			(void) panel_config(p);
			p->xp_flags|= LXP_CONFIGDONE;
				XMapWindow(p->xp_dpy, p->xp_pwin);
			panel_draw(p);
			panel_display(p);
			if (p->xp_flags & LXP_VSCROLLVIS) {
				XMapWindow(p->xp_dpy, p->xp_vswin);
				panelvscroll_draw(p);
			}
			if (p->xp_flags & LXP_HSCROLLVIS) {
				XMapWindow(p->xp_dpy, p->xp_hswin);
				panelhscroll_draw(p);
			}
		}
		else {
			panelitem_draw(p, pi);
			panelitem_display(p, pi);
		}
	}

	return(LX_SUCCESS);
}
