/****************************************************************************** 
 *
 *  mixview - X Window System based soundfile editor and processor
 *
 *  Copyright 1989 by Douglas Scott
 *
 *  Author:     Douglas Scott 
 *  Date:       May 1, 1989
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The author makes no representations about
 *  the suitability of this software for any purpose.  
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
/* io.c -- the soundfile and soundfile buffer i/o routines for mixview 
   Douglas Scott 1/89
*/

#include "main.h"
#include "debug.h"
#include "dialog.h"
#include "mesg.h"

extern Display *dpy;
extern Window mixwin;
extern FLAG is_new, skipflag, durflag, read_only;
extern sf_struct *v;
extern Canvas *canvas;
extern cbuf_struct *c;
extern ebuf_struct *e;


static float maxamp[SF_MAXCHAN];
static long maxloc[SF_MAXCHAN];
static SFMAXAMP sfm;
static SFCODE ampcode = {
	SF_MAXAMP,
	sizeof(SFMAXAMP) + sizeof(SFCODE)
};
FLAG ismapped;
FLAG wasmulaw;

/* declaration of pointers to functions used for low-level i/o */

int _mgetsamp(), _sgetsamp(), _fgetsamp(), _saddsamp(), _faddsamp();
int _maddsamp(), _mputsamp(), _sputsamp(), _fputsamp();
int (*addsamp[])() = {0, _maddsamp, _saddsamp, 0, _faddsamp};
int (*getsamp[])() = {0, _mgetsamp, _sgetsamp, 0, _fgetsamp};
int (*putsamp[])() = {0, _mputsamp, _sputsamp, 0, _fputsamp};

/* get_peakamp -- a little routine to query the peak amp of sfile */

double
get_peakamp(readsf)
FLAG readsf;		/* flag for rereading soundfile for maxamps */
{
	char *cptr;
	double peakamp = 1.0;
	register int i;

	if(!readsf) {	/* check existing HEADER for peak amp listing */
	     /* if info IS in header, copy into sfm structure */
	     if ((cptr=getsfcode(&v->sfh, SF_MAXAMP)) != NULL) {
		  bcopy(cptr + sizeof(SFCODE), (char *) &sfm, sizeof(SFMAXAMP));
	     }
	     if((cptr==NULL)||(sfmaxamp(&sfm,0)==0.0)) { /* no listing */
		  if(ismapped) mesg->On(mesg, "Scanning for peak amp...");
	 	  else fprintf(stderr,"No peak amp listing for sfile: scanning...\n");
		  if(v->size==4) ffindamp();
		  else if(v->size==2) sfindamp();
		  else mfindamp();
		  for(i=0; i<v->nchans; i++) {
			sfmaxamploc(&sfm,i) = maxloc[i];
			sfmaxamp(&sfm,i) = maxamp[i];
		  }
		  /* load into header*/
		  if(putsfcode(&v->sfh,(char*)&sfm,&ampcode) < 0)
			mv_alert("Unable to load peak amp value into header.");
		  mesg->Off(mesg);
	     }
	}
	else {		/* do NOT read header; scan BUFFER for new peak amp */
		for(i=0; i<v->nchans; i++) {
			maxamp[i] = 0.0; /* reset */
			maxloc[i] = 0; /* reset */
		}
		if(v->size==4) ffindamp();
		else if(v->size==2) sfindamp();
		else mfindamp();
		for(i=0; i<v->nchans; i++) {
			sfmaxamploc(&sfm,i) = maxloc[i];
			sfmaxamp(&sfm,i) = maxamp[i];
		}
		/* load into header*/
		if(putsfcode(&v->sfh,(char*)&sfm,&ampcode) < 0)
			mv_alert("Unable to load peak amp value into header.");
	}
	/* determine overall peak for graphics rescaling */

	for(i=0; i<v->nchans; i++)
		if(sfmaxamp(&sfm,i)>peakamp) peakamp = (double)sfmaxamp(&sfm,i);
	return(peakamp);
}

char *
get_comment(string, length)
char *string;
int *length;
{
	SFCOMMENT sfcm;
	SFCODE *sfc;
	char *cp = getsfcode(&v->sfh, SF_COMMENT);
	if(cp == NULL) { /* if no comment present */
		*length = 0;
		return (char *) 0;
	}
	sfc = (SFCODE *) cp;
	bcopy(cp + sizeof(SFCODE) , (char *) &sfcm, 
		(int) sfc->bsize - sizeof(SFCODE));
	*length = strlen(sfcm.comment) + 1;
	bcopy(sfcm.comment, string, *length);
	return string;
}

set_comment(string, length)
char *string;
int length;
{
	int i, nchars = (length > MAXCOMM) ? MAXCOMM : length;
	int oldsize;
	SFCOMMENT sfcm;
	SFCODE *sfc;
	static SFCODE	commentcode = {
		SF_COMMENT,
		MINCOMM + sizeof(SFCODE)
	};

	char *cp = getsfcode(&v->sfh,SF_COMMENT);
	if(cp == NULL) {		/*  if no pre-existing comment */
		for(i=0; i < nchars; i++) 	/* copy chars into struct */
			sfcm.comment[i] = string[i];
		if(length > MAXCOMM)
			mv_alert("Comment truncated to fit into header.");
		sfcm.comment[i] = '\0';
		if(nchars > MINCOMM) commentcode.bsize= nchars + sizeof(SFCODE);
		if(putsfcode(&v->sfh, (char *) &sfcm, &commentcode) < 0) {
			mv_alert("Unable to write comment into header.");
			return -1;
		}
	}
	else {
		sfc = (SFCODE *) cp;
		oldsize = sfc->bsize - sizeof(SFCODE);
		bcopy(cp+sizeof(SFCODE) , (char *) &sfcm, oldsize);
		for(i=0; i < nchars; i++) 
			sfcm.comment[i] = string[i];
		if(nchars < oldsize) {
			for(i=nchars; i < oldsize; i++) 
				sfcm.comment[i] = '\0';
		}
		else
			sfcm.comment[i] = '\0';
		if(putsfcode(&v->sfh, (char *) &sfcm, sfc) < 0) {
			mv_alert("Unable to write comment into header.");
			return -1;
		}
	}
	return 1;
}

static char sfname[128];

/* read_sf is called before the current mixwindow is mapped to the screen,
   and is not called again until a new file is opened
*/

read_sf(filename, inskip, dur)
char *filename;
double inskip, dur;
{
	int result;
	unsigned sfsize, sfminsize;
	int fdtmp, readbytes;
	XWindowAttributes attr;

/* Flush the event buffer first */

	XFlush(dpy);

/* Open the sf for read/write operations if possible. Create it if necessary.
*/
	getsfname(filename, sfname); 

/* Check to see if windows are mapped before using dialog boxes */

	XGetWindowAttributes(dpy, mixwin, &attr);
	if(attr.map_state == IsUnmapped) ismapped = 0;
	else ismapped = 1;

	/* check for existing file - if no, open temp "untitled" file */

	read_only = 0;
	if(!is_new && (fdtmp= open(sfname, O_RDONLY)) < 0 && (errno==ENOENT)) {
		mv_alert("File not found. Opening as untitled.");
		getsfname("untitled", sfname); 
		strcpy(v->sfname, "untitled");
		is_new = 1;
	}
	close(fdtmp); 

	if(!is_new){			/* if opening a file on disk */
		if ((v->sffd = open(sfname, O_RDWR))  < 0) {
			if(ismapped) {
			    mesg->On(mesg, "Opening file read-only...");
			    sleep(1);
			}
			else fprintf(stderr, "mixview: Attempting to open %s read-only...\n", sfname);
			/* if no good, try opening for read_only */
			if ((v->sffd = open(sfname, O_RDONLY))  < 0) {
			    if(ismapped) dialog->call(dialog, D_NOPEN, sfname);
			    else fprintf(stderr,"mixview: cannot open file %s .\n", sfname);
			    close(v->sffd);	/* no good at all */
			    return (-1);
			}
			else read_only = 1;
		}
		if(stat(sfname,&v->sfst)) {
			mv_error(errno, "Cannot get status on soundfile.");
			close(v->sffd);	/* no good at all */
			return (-1);
		}
		else if(readHeader(v->sffd,&v->sfh) < 0) {
			if(ismapped) dialog->call(dialog, D_NHEADER, sfname);
			else fprintf(stderr, "mixview: cannot read header from %s .\n", sfname);
			close(v->sffd);	/* no good at all */
			return (-1);
		}
		else if(checkHeader(&v->sfh) < 0) {
			close(v->sffd);
			return (-1);
		}
#if !NeXT_STYLE_HEADER
		else if (!ismagic(&v->sfh)){
			mv_alert("Error:  file has wrong magic number.");
			close(v->sffd);	/* no good at all */
			return (-1);
		}
#endif
	}

/* The pointer to this soundfile will be cast as short or float depending on
   the soundfile class, so set a flag for which pointer to use */

	v->size = sfclass(&v->sfh);
	v->is_float = (v->size == SF_FLOAT);

/* Set some soundfile constants
 */
	v->srate = sfsrate(&v->sfh);
	v->nchans = sfchans(&v->sfh);
	v->secgrain = v->grainsize / v->srate;

	if(!is_new && skipflag) {	/* if skip set, do lseek */
		long byteskip = inskip*v->nchans*v->size*v->srate;
		if(lseek(v->sffd, byteskip, L_INCR) < 0) {
			mv_error(errno, "Bad seek on soundfile.");
			close(v->sffd);
			return(-1);
		}
	}
	readbytes = 0;

	sfsize = is_new ? 0 : v->sfst.st_size - headerSize(&v->sfh);
	if(durflag && !is_new)			/* read dur from disk */
		readbytes = dur*v->srate*v->nchans*v->size;
	/* else read entire file   */
	else readbytes = sfsize;
	readbytes = readbytes > sfsize ? sfsize : readbytes; /* max dur */
	if(set_property((ebuf_struct *) v) < 0) /* set cutproperty */
		return -1;
/* Set same atom for cut buffer, IF it is empty */

	if(c->bufsize == 0) c->cutproperty = v->cutproperty;

/* Allocate the buffer for the entire soundfile--cast as char* 		*/
/* Make sure the buffer is at least as large as the current screen res. */
	
	sfminsize = (int) (v->width * v->grainsize) * v->nchans * v->size;
	sfsize = (sfsize > sfminsize) ? sfsize : sfminsize;
	if ((v->sfbuff = (char *) calloc(sfsize, sizeof(char))) == NULL) {
		if(ismapped) dialog->call(dialog, D_NBMEM, sfname);
		else {
			fprintf(stderr, "Not enough memory to load file.\n");
			perror("calloc");
		}
		if ((v->sfbuff = (char *) calloc(sfminsize, sizeof(char))) == NULL)
			exit(-1);
		close(v->sffd);
		return(-1);
	}
	wasmulaw = False;
/* if disk file is open, read soundfile into buffer */

	if(!is_new) {
		FLAG scan_it = False;
		if(!ismapped) fprintf(stderr, "\nReading soundfile...   ");
		v->bufsize = read(v->sffd, v->sfbuff, readbytes);
#if NeXT_STYLE_HEADER
		if(v->size == 1) {
			if(!ismapped) fprintf(stderr, "converting to 16 bit...\n");
			else mesg->On(mesg, "Converting to 16 bit...");
			wasmulaw = True;	/* flag */
			muToShort((ebuf_struct *) v);
			scan_it = True;
		}
#endif
		v->sfdur = ((double) v->bufsize)/(v->srate*v->nchans*v->size);
		v->peakamp = get_peakamp(scan_it);  /* scan for peak amp */
		if(!ismapped) fprintf(stderr, "\n");
	}
	else {
		v->bufsize = sfsize;
		v->sfdur = ((double) v->bufsize)/(v->srate*v->nchans*v->size);
	}
	return(1);
}

/* write_sf is called to save the soundfile buffer to disk */

write_sf(filename)
char *filename;
{
	char newsfname[128], tmpcom[MAXCOMM];
	int len, newfd;
	FLAG new_name = 1;

	/* Flush the event buffer first and wait till clear */

	XFlush(dpy);

	/* update the soundfile header before writing to disk */

	sfsrate(&v->sfh) = v->srate;
	sfclass(&v->sfh) = v->size;
	sfchans(&v->sfh) = v->nchans;
	putsfcode(&v->sfh, (char *) &sfm, &ampcode);
	getsfname(filename, newsfname);
#ifdef HAS_SFCOMMENT
	/* set the comment to the filename if no other comment present */
	if(get_comment(tmpcom, &len) == (char *) 0)
		set_comment(filename, strlen(filename));
#endif
	if(strcmp(newsfname, sfname) == 0) new_name = False;

	/* saving existing file to same name */
	if(!new_name && !is_new) {
		long offset = 0;
		lseek(v->sffd,offset,0); /* go to beginning of file */
		newfd = v->sffd;
		if(v->bufsize < v->sfst.st_size-headerSize(&v->sfh))
			ftruncate(newfd, (long) v->bufsize+headerSize(&v->sfh));
	}	
	/* if saving to new name, or saving new file, check for existing file */
	else if((newfd= open(newsfname, O_EXCL|O_CREAT|O_WRONLY, 0644)) == -1) {
		char msg[128];
		switch(errno) {
		case EEXIST:
			/* replace it? */
			switch(dialog->call(dialog, D_SFEXISTS, newsfname)) {
			case YES:
				break;
			case CANCEL:
				close(newfd);
				return(-1);
			}
			if((newfd = open(newsfname, O_WRONLY)) == -1) {
				/* can't open it! */
				dialog->call(dialog, D_NOPEN, newsfname);
				return(-1);
			}
			break;
		default:
			sprintf(msg, "Cannot create %s.", newsfname);
			mv_error(errno, msg);
			return(-1);
			break;
		}
	}
	if(writeHeader(newfd, &v->sfh) < 0)	/* write new header out */ {
		dialog->call(dialog, D_NWHEADER, filename);
		close(newfd);
		if(is_new) unlink(filename);
		return(-1);
	}
	/* write buffer to disk */

	if(write(newfd, v->sfbuff, (int) v->bufsize) != v->bufsize) {
		mv_error(errno, "Unable to write file to disk.");
		close(newfd);
		if(is_new) unlink(filename);
		return(-1);
	}
	/* if file is new and saved to its orig. new name */
	if(is_new && !new_name) {
		is_new = 0;
		v->sffd = newfd;
		strcpy(v->sfname, filename);
		return(0);
	}
	/* close new disk file if file has been saved to new name */	
	else if(new_name) close(newfd);  
	return(0);
}

/* getsfname -- build a full soundfile pathname
 * Takes pointer to short name and a pointer to buffer for result
 */

char *
getsfname(name,nambuf)
char *name,*nambuf;
{
	char *sfdir;

/* If a full path is present, don't add the SFDIR */

	if(strlen(v->sfdir) == 0) {	/* if not previously set */
		if((sfdir = getenv("SFDIR")) == NULL) {
			fprintf(stderr,
				"Warning: environmental variable SFDIR not found.\n");
			sfdir = ".";
		}
	}
	else sfdir = v->sfdir;

	if(*name == '/' || *name == '.') 
		strcpy(nambuf,name);
	else {
		strcpy(v->sfdir, sfdir); 
		sprintf(nambuf,"%s/%s",sfdir,name);
	}
	return nambuf;
}

/************************************************************************/
/************************************************************************/

/*	The soundfile BUFFER reading and writing routines */

sfreset()		/* resets the pointer incrementors to zero */
{
	c->inptr = c->outptr = 0;
	e->inptr = e->outptr = 0;
}

_mgetsamp(cs, inbox)
cbuf_struct *cs;
double *inbox;
{
	register int i;
	register char *sptr;
	for(i=0, sptr=cs->sfbuff + cs->inptr; i<cs->nchans; i++)
		*(inbox+i) = *(sptr+i);
	cs->inptr += cs->nchans;
}

_sgetsamp(cs, inbox)
cbuf_struct *cs;
double *inbox;
{
	register int i;
	register short *sptr;
	for(i=0, sptr=(short *)cs->sfbuff + cs->inptr; i<cs->nchans; i++)
		*(inbox+i) = *(sptr+i);
	cs->inptr += cs->nchans;
}

_fgetsamp(cs, inbox)
cbuf_struct *cs;
double *inbox;
{
	register int i;
	register float *fptr;

	for(i=0, fptr=(float *)cs->sfbuff + cs->inptr; i<cs->nchans; i++)
		*(inbox+i) = *(fptr+i);
	cs->inptr += cs->nchans;
}

process(src, dest, f, arg)
cbuf_struct *src, *dest;
double (*f)(/*int, double, double, void **/);
void *arg;
{
    if (src->size == 4) {
	if (dest->size == 4) {
	    float *s, *d;
	    int nc, i;
	    for (s= (float *)src->sfbuff, d =(float *)dest->sfbuff, i=0, nc=0;
		 ++nc < src->nchans || (nc = 0, ++i < src->nsamps);
		 ++s, ++d) {
		*d = (*f)(i, *s, *d, arg);	
	    }
	} else if (dest->size == 2) {
	    float *s;
	    short *d;
	    int nc, i;
	    for (s= (float *)src->sfbuff, d =(short *)dest->sfbuff, i=0, nc=0;
		 ++nc < src->nchans || (nc = 0, ++i < src->nsamps);
		 ++s, ++d) {
		*d = (*f)(i, *s, (double)*d, arg) + 0.5;	
	    }
	} else {
	    float *s;
	    char *d;
	    int nc, i;
	    for (s= (float *)src->sfbuff, d = dest->sfbuff, i=0, nc=0;
		 ++nc < src->nchans || (nc = 0, ++i < src->nsamps);
		 ++s, ++d) {
		*d = (*f)(i, *s, (double)*d, arg) + 0.5;	
	    }
	}
    } else if(src->size == 2) {
	if (dest->size == 4) {
	    short *s;
	    float *d;
	    int nc, i;
	    for (s= (short *)src->sfbuff, d =(float *)dest->sfbuff, i=0, nc=0;
		 ++nc < src->nchans || (nc = 0, ++i < src->nsamps);
		 ++s, ++d) {
		*d = (*f)(i, (double)*s, *d, arg);
	    }
	} else if(dest->size == 2) {
	    short *s;
	    short *d;
	    int nc, i;
	    for (s= (short *)src->sfbuff, d =(short *)dest->sfbuff, i=0, nc=0;
		 ++nc < src->nchans || (nc = 0, ++i < src->nsamps);
		 ++s, ++d) {
		*d = (*f)(i, (double)*s, (double)*d, arg) + 0.5;
	    }
	} else {
	    short *s;
	    char *d;
	    int nc, i;
	    for (s= (short *)src->sfbuff, d = dest->sfbuff, i=0, nc=0;
		 ++nc < src->nchans || (nc = 0, ++i < src->nsamps);
		 ++s, ++d) {
		*d = (*f)(i, (double)*s, (double)*d, arg) + 0.5;
	    }
	}
    } else {
	if (dest->size == 4) {
	    char *s;
	    float *d;
	    int nc, i;
	    for (s= src->sfbuff, d =(float *)dest->sfbuff, i=0, nc=0;
		 ++nc < src->nchans || (nc = 0, ++i < src->nsamps);
		 ++s, ++d) {
		*d = (*f)(i, (double)*s, *d, arg);
	    }
	} else if(dest->size == 2) {
	    char *s;
	    short *d;
	    int nc, i;
	    for (s= src->sfbuff, d =(short *)dest->sfbuff, i=0, nc=0;
		 ++nc < src->nchans || (nc = 0, ++i < src->nsamps);
		 ++s, ++d) {
		*d = (*f)(i, (double)*s, (double)*d, arg) + 0.5;
	    }
	} else {
	    char *s;
	    char *d;
	    int nc, i;
	    for (s= src->sfbuff, d = dest->sfbuff, i=0, nc=0;
		 ++nc < src->nchans || (nc = 0, ++i < src->nsamps);
		 ++s, ++d) {
		*d = (*f)(i, (double)*s, (double)*d, arg) + 0.5;
	    }
	}
    }
}

_maddsamp(cs, outbox)
cbuf_struct *cs;
double *outbox;
{
	register int i;
	register char *sptr;

	for(i=0, sptr=cs->sfbuff + cs->outptr; i<cs->nchans; i++)
		*(sptr+i) += (*(outbox+i) + 0.5);
	cs->outptr += cs->nchans;
}

_saddsamp(cs, outbox)
cbuf_struct *cs;
double *outbox;
{
	register int i;
	register short *sptr;

	for(i=0, sptr=(short *)cs->sfbuff + cs->outptr; i<cs->nchans; i++)
		*(sptr+i) += (*(outbox+i) + 0.5);
	cs->outptr += cs->nchans;
}

_faddsamp(cs, outbox)
cbuf_struct *cs;
double *outbox;
{
	register int i;
	register float *fptr;

	for(i=0, fptr=(float *)cs->sfbuff + cs->outptr; i<cs->nchans; i++)
		*(fptr+i) += *(outbox+i);
	cs->outptr += cs->nchans;
}

_mputsamp(cs, outbox)
cbuf_struct *cs;
double *outbox;
{
	register int i;
	register char *sptr;
	for(i=0, sptr=cs->sfbuff + cs->outptr; i<cs->nchans; i++)
		*(sptr+i) = *(outbox+i) + 0.5; 
	cs->outptr += cs->nchans;
}

_sputsamp(cs, outbox)
cbuf_struct *cs;
double *outbox;
{
	register int i;
	register short *sptr;
	for(i=0, sptr=(short *)cs->sfbuff + cs->outptr; i<cs->nchans; i++)
		*(sptr+i) = *(outbox+i) + 0.5; 
	cs->outptr += cs->nchans;
}

_fputsamp(cs, outbox)
cbuf_struct *cs;
double *outbox;
{
	register int i;
	register float *fptr;

	for(i=0, fptr=(float *)cs->sfbuff + cs->outptr; i<cs->nchans; i++)
		*(fptr+i) = *(outbox+i);
	cs->outptr += cs->nchans;
}

/* these peakamp locators are adapted from cmix routines */

ffindamp()
{
	register float *fbuffer = (float *) v->sfbuff;
	register float val;
	register int i, j;
	register long tsamps = v->bufsize/v->size;

	for(i = 0; i < tsamps; i += v->nchans) {
		for(j = 0; j < v->nchans; j++) {
			val = FABS(*fbuffer);
			if(val > maxamp[j]) {
				maxamp[j] = val;
		/*		maxloc[j] = i/2;	*/
			}
			fbuffer++;
		}
	}

}

sfindamp()
{
	register short *sbuffer = (short *) v->sfbuff;
	register float val;
	register int i, j;
	register long tsamps = v->bufsize/v->size;

	for(i = 0; i < tsamps; i += v->nchans) {
		for(j = 0; j < v->nchans; j++) {
			val = ABS(*sbuffer);
			if(val > maxamp[j]) {
				maxamp[j] = val;
		/*		maxloc[j] = i/2;	*/
			}
			sbuffer++;
		}
	}
}

mfindamp()
{
	register char *sbuffer = v->sfbuff;
	register float val;
	register int i, j;
	register long tsamps = v->bufsize/v->size;

	for(i = 0; i < tsamps; i += v->nchans) {
		for(j = 0; j < v->nchans; j++) {
			val = ABS(*sbuffer);
			if(val > maxamp[j]) {
				maxamp[j] = val;
		/*		maxloc[j] = i/2;	*/
			}
			sbuffer++;
		}
	}
}

/* Set the property Atom for this file type */
int
set_property(buff)
ebuf_struct *buff;
{
	static char *propstr[2][3] = {
		{"MV_MONOCHAR", "MV_MONOSHORT", "MV_MONOFLOAT"},
		{"MV_STEREOCHAR", "MV_STEREOSHORT", "MV_STEREOFLOAT"},
	};
	int pt=0;
	switch(buff->size) {
	case 1:
		pt = 0;
		break;
	case 2:
		pt = 1;
		break;
	case 4:
		pt = 2;
		break;
	default:
		mv_alert("Unknown sample size.");
		return(-1);
	}
	buff->cutproperty= XInternAtom(dpy, propstr[v->nchans-1][pt], False);
	return(1);
}

checkHeader(hd)	/* check for corrupted values in header */
SFHEADER *hd;
{
	int retcode = 1, chans = hd->sfinfo.sf_chans;
	float rate = hd->sfinfo.sf_srate;
	char msg[60];
	if(chans == 3 || chans == 4) {
		sprintf(msg, "Sorry...cannot display %d channel files.", chans);
		retcode = -1;
	}
	else if(chans > 4) {
		sprintf(msg, "Error: possibly corrupt header: nchans= %d", chans);
		retcode = -1;
	}
	else if(rate < 4000.0 || rate > 64000.0) {
		sprintf(msg, "Error: possibly corrupt header: samp rate= %0.1f",
			rate);
		retcode = -1;
	}
	if(retcode < 0) mv_alert(msg);
	return retcode;
}
