/****************************************************************************** 
 *
 *  mixview - X Window System based soundfile editor and processor
 *
 *  Copyright 1989 by Douglas Scott
 *
 *  Author:     Douglas Scott 
 *  Date:       May 1, 1989
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The author makes no representations about
 *  the suitability of this software for any purpose.  
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
/*
	 draw.c 
*/

#include "main.h"
#include "windows.h"
#include "dialog.h"
#include "menus.h"
#include "nobug.h"

#define NPTS	512
#define MAXPT	511
#define S_HEIGHT  460
#define S_WIDTH   572

extern Display *dpy;
extern Window mixwin, lwin, lcwin;
extern Cursor arrow_curs, wait_curs, draw_curs, cross_curs;
extern sf_struct *v;
extern char *progname;
extern unsigned long myfg, mybg;
extern double array[SIZE];
extern int lineset;

Window sc_win;
Pixmap scrn_pm;
Mv_panel *c_panel;
Canvas *screen;
Menu *curve_menu;
GC s_gc, s_dgc;
Time evt_time;

int ret_code;
static FLAG locflag = 0;
static int ybase;
static double vscale, maxval, minval;
static double list[NPTS*2+2];	/* array of {loc0,val0, loc1,val1, ...} */
static XPoint epnt[30], dpnt[3], points[S_WIDTH], old, new;

static
char *menu_cmds[]= { "set range", "blank", "blank", 0
};

setup_screen()
{
}

setup_curve_window()
{
        void  reset_curve(), set_range(), cancel_curve(), confirm_curve();
	Mv_panel *new_panel();

	/* Create the panel and panelitems for this window */

        c_panel = new_panel(lwin, LFRAME_WIDTH, 100, 0, 0, 0);
        c_panel->addLabel(c_panel, 20, 10, 
		"Draw continuous and/or segmented curve for values:",
		"label 1");
        c_panel->addLabel(c_panel, 20, 32,
		"Left button: segmented; Center button: continuous.",
		"label 2");
        c_panel->addButton(c_panel, 20, 60, "confirm", confirm_curve);
        c_panel->addButton(c_panel, 125, 60, "cancel", cancel_curve);
        c_panel->addButton(c_panel, 220, 60, " reset ", reset_curve);

	/* create and initialize canvas for screen */

	setup_screen();

	/* create menu and menuitems for screen */

        curve_menu = new_menu(sc_win, CNULL);
        add_menuitem(curve_menu, menu_cmds[0], set_range);
        add_menuitem(curve_menu, menu_cmds[1], PROC_NULL);
        add_menuitem(curve_menu, menu_cmds[2], PROC_NULL);
}

FLAG again;
unsigned e_mask = ButtonPressMask | ButtonMotionMask |
	ButtonReleaseMask | ExposureMask | KeyPressMask;

get_curve()	/* displays window/panel/canvas for entering data curve */
{
	XSetWindowAttributes xswa;
	XEvent evt;
	unsigned wait_mask = ExposureMask;
	void reset_curve();
	int x, y;
	static int first_time = 1;
	get_win_coords(mixwin, &x, &y); /* get position relative to root */
	move_window(lwin, x+150, y+50); /* locate dialog box on root */
	xswa.override_redirect= TRUE;
	XChangeWindowAttributes(dpy, lwin, CWOverrideRedirect, &xswa);
	XMapRaised(dpy, lwin);
	XMapSubwindows(dpy, lwin);
	XGrabPointer(dpy, lwin, True, e_mask,
			GrabModeAsync, GrabModeAsync,
			lwin, arrow_curs, CurrentTime);


	/* reset, draw scale and base line, etc. */

	if(first_time) {
		int n_args;
		ybase = YMAX;	/* the 0 value vertical coordinate */
		maxval = 1.0;
		minval = 0.0;
		DBPRT(ybase);
		vscale = (maxval-minval)/(double) (YMAX-YMIN);
		DBPRT(vscale);
		reset_curve((Panel *) NULL, (Panel_item *) NULL); 
		curve_to_vals(vscale, ybase, points, list, &n_args);
		first_time = 0;
	}
	else {
		draw_scrn();
	}

	again = 1;

	while(again) {
		XNextEvent(dpy, &evt);
		if (lxt_event(&evt)) 
			continue;
		switch (evt.type) {    /* process pointer & kbd. events */
		case MotionNotify:
			cpm_select((XPointerMovedEvent *) &evt);
			break;
		case ButtonPress:
			cbp_select((XButtonPressedEvent *) &evt);
			break;
		case ButtonRelease:
			cbr_select((XButtonReleasedEvent *) &evt);
			break;
		}
	}
									
	if(ret_code == YES) {
		int n_args;
		XChangeActivePointerGrab(dpy, wait_mask, wait_curs, evt_time);

		/* create interpolated table */
		curve_to_vals(vscale, ybase, points, list, &n_args);
		vals_to_array(list, array, n_args);
	}
	XUngrabPointer(dpy, evt_time);
	XUnmapSubwindows(dpy, lwin);
	XUnmapWindow(dpy, lwin);
	return(ret_code);
}

move_window(win, xpos, ypos)
Window win;
int xpos, ypos;
{
	int max_x = DisplayWidth(dpy, DefaultScreen(dpy));
	int max_y = DisplayHeight(dpy, DefaultScreen(dpy));
	if(xpos+LFRAME_WIDTH > max_x) xpos = max_x - LFRAME_WIDTH - 5;
	if(ypos+LFRAME_HEIGHT > max_y) ypos = max_y - LFRAME_HEIGHT - 5;
	XMoveWindow(dpy, win, xpos, ypos);
}

void
confirm_curve(p, pi)
Panel *p;
Panel_item *pi;
{
	again = 0;
	evt_time = *((Time *) panelitem_get(p, pi, LXPI_TIMESTAMP));
	ret_code = YES;
	XFlush(dpy);
}

void
cancel_curve(p, pi)
Panel *p;
Panel_item *pi;
{
	again = 0;
	evt_time = *((Time *) panelitem_get(p, pi, LXPI_TIMESTAMP));
	ret_code = CANCEL;
	lineset = 0;
	XFlush(dpy);
}

void
reset_curve(p, pi)	/* restores flat curve at 0.0 */
Panel *p;
Panel_item *pi;
{
	register int i;
	DBCOM(entering reset_curve);

	/* reset data points */

	reset_list(list);
	for(i=0; i < S_WIDTH; i++) points[i].x = -1;

	dpnt[0].x = points[0].x = 0;	/* left endpoint */
	dpnt[0].y = points[0].y = ybase;
	dpnt[2].x = points[XMAX].x = XMAX;	/* right endpoint */
	dpnt[2].y = points[XMAX].y = ybase;

	draw_scrn();	/* redisplay the curve */
	DBCOM(leaving reset_curve);
}

cbp_select(evt)
XButtonPressedEvent *evt;
{
	switch (evt->button){
	case Button1:
		XChangeActivePointerGrab(dpy, e_mask, cross_curs, evt->time);
		new_point(evt->x,evt->y);
		break;
	case Button2:
		XChangeActivePointerGrab(dpy, e_mask, draw_curs, evt->time);
		new_line(evt->x,evt->y);
		break;
	case Button3:
		menu_show(curve_menu, evt);
		break;
	}
}

cpm_select(evt)
XPointerMovedEvent *evt;
{
	if(evt->state & Button1Mask) 
		move_point(evt->x,evt->y);
	else if(evt->state & Button2Mask) 
		draw_line(evt->x,evt->y);
}

cbr_select(evt)
XButtonReleasedEvent *evt;
{
	switch (evt->button){
	case Button1:
		XChangeActivePointerGrab(dpy, e_mask, arrow_curs, evt->time);
		set_point(evt->x,evt->y);
		break;
	case Button2:
		XChangeActivePointerGrab(dpy, e_mask, arrow_curs, evt->time);
		break;
	}
}

/**********************************************************************/

static int npoints;
FLAG lbflag, ubflag;

sc_newPoint(s,x,y)	/* button 1 clicked */
	Mv_screen *s;
	int x,y;
{
	XPoint pt[2];
	register int resetpt = 0;
	width = s->frameWidth(s) - 1;
	height = s->frameHeight(s) - 1;
	bw = s->vmarg;
	DBCOM(entering new_point);

	x = (x < 0) ? 0 : (x > width) ? width : x;	/* limit to screen */
	y = (y <= bw) ? bw : (y >= height) ? height : y;
	npoints = 3;
	new.x =  x;
	new.y = y;

	/* if resetting (erasing) an existing point */

	if(get_endpoints(x, &epnt[0], &epnt[2])) { 
		if(epnt[0].x == x) {		/* reset lower edge */
			dpnt[0] = new;
			dpnt[2] = dpnt[1] = epnt[2];
			ubflag = 0;
			lbflag = 1;
			npoints = 2;
		}
		else if(epnt[2].x == x) {	/* reset upper edge */
			dpnt[0] = epnt[0];
			dpnt[2] = dpnt[1] = new;
			ubflag = 1;
			lbflag = 0;
			npoints = 2;
		}
		else {
			epnt[1].x = points[x].x; /* middle point to erase */
			epnt[1].y = points[x].y;
			resetpt = 1;
			dpnt[0] = epnt[0];	  /* lower point to draw  */
			dpnt[1] = new;		  /* middle point to draw */
			dpnt[2] = epnt[2];	  /* upper point to draw  */
			lbflag = ubflag = 0;	  /* reset boundry flags  */
		}
		/* remove old inflection pt. from list */

		points[x].x = -1; 
		DBCOM(resetting existing point);
	}
	else {			/* if setting new point */
		dpnt[0] = epnt[0];
		dpnt[1] = new;
		dpnt[2] = epnt[2];
		lbflag = ubflag = 0;
	}

	/* erase old segment(s) */

	if(resetpt) {			/* if shifting (erasing) old point */
		DBCOM(erase two old segments AND infl. pt.);
		DBPRT(npoints);
		s->drawLines(s,epnt,npoints);
	}
	else {		/* if setting new inflection pt. */
		DBCOM(no resetpt - just erase the old segment);
		pt[0].x = epnt[0].x;
		pt[0].y = epnt[0].y
		pt[1].x = epnt[2].x;
		pt[1]y = epnt[2].y;
		s->drawLines(s, pt, 2);
	}

	/* draw new segments */

	DBCOM(draw new segments);
	DBPRT(npoints);
	s->drawLines(dpnt,npoints);
	print_loc(x, y);
	old = new;
	DBCOM(leaving new_point);
}

move_point(s,x,y)	/* button 1 dragged */
	Mv_screen *s;
	int x, y;
{
	int xloc, lpoint = epnt[0].x, rpoint = epnt[2].x;
	DBCOM(entering move_point);
	npoints = 3;
	width = s->frameWidth(s) - 1;
	height = s->frameHeight(s) - 1;
	bw = s->vmarg;

	y = (y <= bw) ? bw : (y >= height) ? height : y;

	/* always bounded by previous hi and lo inflection pts. */

	if(lbflag) {			/* moving lower boundry */
		xloc = 0;
		npoints = 2;
		epnt[0] = dpnt[0];
		epnt[1] = dpnt[1];
		dpnt[0].x = xloc;
		dpnt[0].y = y;
		old = dpnt[0];
	}
	else if(ubflag) {		/* moving upper boundry */
		xloc = width;
		npoints = 2;
		epnt[0] = dpnt[0];
		epnt[1] = dpnt[1];
		dpnt[1].x = xloc;
		dpnt[1].y = y;
		old = dpnt[1];
	}
	else {				/* moving intermediate points */
		xloc = (x <= lpoint) ? lpoint+1 : (x >= rpoint) ? rpoint-1 : x;
		epnt[0] = dpnt[0];
		epnt[1] = dpnt[1];
		epnt[2] = dpnt[2];
		dpnt[1].x = xloc;
		dpnt[1].y = y;
		old = dpnt[1];
	}
	s->drawLines(s, epnt, npoints);	/* erase old segments */
	s->drawLines(s, dpnt, npoints);	/* draw new segment(s) */
	print_loc(xloc, y);
	DBCOM(leaving move_point);
}

sc_setPoint(s,x,y)	/* button 1 released */
	Mv_screen *s;
	int x, y;
{
	int xloc, lpoint = epnt[0].x, rpoint = epnt[2].x;
	width = s->frameWidth(s) - 1;
	height = s->frameHeight(s) - 1;
	bw = s->vmarg;
	DBCOM(entering set_point);
	if(x >= width) xloc = width;
	else if(x <= 0) xloc = 0;
	else xloc = (x < lpoint) ? lpoint : (x > rpoint) ? rpoint : x;
	y = (y <= 0) ? 0 : (y >= height) ? height : y;
	points[xloc].x = xloc;
	points[xloc].y = y;
	DBCOM(leaving set_point);
}

sc_startCurve(s,x,y)	/* button 2 depressed */
	Mv_screen *s;
	int x,y;
{
	XPoint lower, upper;
	int nepts = 2, ndpts = 3;
	width = s->frameWidth(s) - 1;
	height = s->frameHeight(s) - 1;
	bw = s->vmarg;

	DBCOM(entering drawCurve);

	x = (x <= 0) ? 1 : (x > width) ? width : x;
	y = (y < bw) ? bw : (y > height) ? height : y;
	new.x = x;
	new.y = y;

	/* if resetting an existing point */

	if(get_endpoints(x, &lower, &upper)) {
		epnt[1] = points[x];
		epnt[2] = lower;
		nepts = 3;
		points[x].x = -1; /* remove point from list -- to be reset */
		DBCOM(resetting existing point);
	}
	else epnt[1] = lower;
	epnt[0] = upper; 	/* segment to erase */

	DBPRT(new.x); DBPRT(old.x); DBPRT(lower.x); DBPRT(upper.x);

	dpnt[0] = upper;	/* segment to draw: */
	dpnt[1] = new;
	dpnt[2] = lower;

	
	s->drawLines(s,epnt, nepts);	/* erase old segment(s) */
	s->drawLines(dpnt, ndpts);		/* draw new segments */
	print_loc(x, y);
	points[x] = new;
	old = new;
	DBCOM(leaving new_line);
}

sc_drawCurve(s,x,y)	/* button 2 dragged */
	Mv_screen *s;
	int x, y;
{
	XPoint lower, upper;
	register int i, nepts, ndpts = 3;
	width = s->frameWidth(s) - 1;
	height = s->frameHeight(s) - 1;
	bw = s->vmarg;
	DBCOM(entering draw_line);

	x = (x <= 0) ? 1 : (x > width) ? width : x;
	y = (y < bw) ? bw : (y > height) ? height : y;
	new.x = x;
	new.y = y;

	/* set the old segment endpoints for erase */

	get_endpoints(x, &lower, &upper); 
	nepts = 0;

	if(x > old.x) { /* if moving to right */
		epnt[nepts++] = upper; /* upper endpt. */
		i = upper.x;
		DBCOM(intermed pts);
		while(--i > old.x) { /* get all intermediate pts. to erase */
			if(points[i].x >= 0) {
				epnt[nepts++] = points[i];
				points[i].x = -1;
			}
		}
		epnt[nepts++] = old; /* lower endpt */

		/* set the new segment endpts */

		dpnt[0] = upper;
		dpnt[1] = new;
		dpnt[2] = old; /* = lower or old */
	}
	else if(x < old.x ) {
		nepts = 0;
		epnt[nepts++] = lower; /* lower endpt. */
		i = lower.x;
		DBCOM(intermed. pts:);
		while(++i < old.x) { /* get all intermediate pts. to erase */
			if(points[i].x >= 0) {
				epnt[nepts++] = points[i];
				points[i].x = -1;
			}
		}
		epnt[nepts++] = old; /* upper endpt */
		DBPRT(nepts);

		/* set the new segment endpts */

		dpnt[0] = lower;
		dpnt[1] = new;
		dpnt[2] = old; /* = upper or old */
	}	
	else { /* if x = x.old */
		epnt[0] = upper; 
		epnt[1] = old; 
		epnt[2] = lower;

		/* set the new segment endpts */

		dpnt[0] = upper;
		dpnt[1] = new;
		dpnt[2] = lower;
		points[old.x].x = -1;
		nepts = ndpts = 3;
	}
	points[x].x = x; /* add current point to point list */
	points[x].y = y;

	s->drawLines(s, epnt, nepts);
	s->drawLines(s, dpnt, ndpts);
	DBPRT(ndpts);
	print_loc(x, y);
	old = new;
	DBCOM(leaving draw_line);
}

/* find current upper and lower graph point bounds for new point */

get_endpoints(xstart, lpoint, upoint)
int xstart;
XPoint *lpoint, *upoint;
{
	register int i = xstart, val = 0;
	XPoint *current;
	DBCOM(entering get_endpoints);
	DBPRT(points[xstart].x);
	if((current = points->getPoint(points, xstart)) != NULL) val = 1;	/* if new pt. already set */
	DBCOM(upper);
	if(xstart == XMAX) {		/* resetting right edge value */
		upoint->x = points[XMAX].x;
		upoint->y = points[XMAX].y;
	}
	else {				/* find regular upper bound */
		while(i++ <= XMAX) {
			if(points[i].x >= 0) {
				upoint->x = points[i].x;
				upoint->y = points[i].y;
				break;
			}
		}
	}
	DBCOM(lower);
	i = xstart;
	if(xstart == 0) {		/* resetting left edge value */
		lpoint->x = points[0].x;
		lpoint->y = points[0].y;
	}
	else {				/* find regular lower bound */
		while(i-- >= 0) {
			if(points[i].x >= 0) {
				lpoint->x = points[i].x;
				lpoint->y = points[i].y;
				break;
			}
		}
	}
	DBCOM(leaving get_endpoints);
	return(val);
}

print_loc(x, y) /* prints current loc value at the top of screen */
int x, y;
{
	static int tx, ty, ax;
	int ly = 8, vx = 10, vy = -20, axmax = s->framewidth - 30;
	static char string[6], string2[16];
	double val;
	char *sprintf();
	/* draw over (erase) previous location and value labels if present */

	if (locflag) {
		s->string(s,tx, ly,string);
		s->string(s,ax+vx, ty+vy,string2);
	}
	/* print new location and value  */

/*	val = (ybase - y) * vscale; 	*/
	sprintf(string, "%s %d", s->hairlabel, x);
/*	sprintf(string2, "%3.4f", val);	*/
	ax = (x < axmax) ? x : axmax;
	tx = x;
	ty = y;
	s->string(s,tx, ly,string);
/*	s->string(s,ax+vx, ty+vy,string2);	*/
	locflag = 1;
}

/* converts linked data-location list into array of data via linear interpolation */

sc_pointsToData(s, d)
	Mv_screen *s;
	Data_buff *d;
{
	int length = d->nElements(d, chan);
	double increm;
	register int i,j,k,points;
	Graph_points *pts = s->gpoints;
	Pt_link *point, *next, *head = pts->head(pts, chan), *last = pts->last(pts, chan);
	increm = (last->xval - head->xval)/length;
	for(point=head,i=0; point != NULL, point->next != NULL; point = next) {
		points = (int)((point->next->xval - point->xval) / increm +.5);
		if(point->next->xval != point->xval) {
			if(points <= 0) points = 1;
			for(k=0; k < points; k++) {
				array[i++] = ((double)k/(double)points)
					* (point->next->yval - point->yval) + point->yval;
				if(i == length) return;
				next = point->next;
			}
		}
	}
	i--;
	while(++i < length) array[i] = array[i-1]; 
}

reset_list(vlist)
double *vlist;
{
	register int i;
	static int end = SIZE * 2;
	DBCOM(entering reset_list);
	for(i = 0; i < end; i+=2) {
		vlist[i] = -1.0;
		vlist[i+1] = 0.0;
	}

	vlist[0] = 0.0;			/* first pair in list */
	vlist[1] = 0.0;
	vlist[2] = SIZE - 1;		/* last pair in list */
	vlist[3] = 0.0;
	DBCOM(leaving reset_list);
}
