/* 
	Datawin.c 
*/

#include "main.h"
#include "dialog.h"
#include "screen.h"
#include "databuff.h"
#include "datawin.h"
#include "nobug.h"

#define MAXDATAWINS 16
extern Display *dpy;

int ndatawins = 0;
Data_win *datawinlist[MAXDATAWINS];

Data_win *
new_datawin(filename)
	char *filename;
{
	unsigned width = DATAWIN_WIDTH, height = DATAWIN_HEIGHT, bwidth = 2;
	unsigned panelwidth = DATAWIN_WIDTH, panelheight = DATAPANEL_HEIGHT;
	Data_win *d;
	static long f_mask = ExposureMask|StructureNotifyMask|PropertyChangeMask
		|EnterWindowMask;
	static int dframe_x = 200, dframe_y = 200;
	if(ndatawins == (MAXDATAWINS - 1)) {
		mv_alert("Maximum number of datawins exceeded.");
		return (Data_win *) NULL;
	}
	d = (Data_win *) mv_alloc(sizeof(Data_win));
	d->dpy = dpy;
	d->frame= new_window(DefaultRootWindow(d->dpy), width, height, 
		dframe_x, dframe_y, bwidth, f_mask);
	/* next window will be offset */
	dframe_x += 20; dframe_y += 20;
/* functions */
	d->setInsert = dw_setInsert;
	d->setRegion = dw_setRegion;
	d->shiftInsert = dw_shiftInsert;
	d->shiftRegion = dw_shiftRegion;
	d->display = dw_display;
	d->setScanChannels = dw_setScanChannels;
	d->getScanChannels = dw_getScanChannels;
	d->setHorizValues = dw_setHorizValues;
	d->getHorizValues = dw_getHorizValues;
	d->setHorizScale = dw_setHorizScale;
	d->block = dw_block;
	d->unBlock = dw_unBlock;
	d->setFrameLabel = dw_setFrameLabel;
	d->destroy = dw_destroy;
	d->datawinEvent = dw_datawinEvent;
	d->frameResize = dw_frameResize;
	d->screen_clear_proc = dw_screen_clear_proc;
	d->screen_redraw_proc = dw_screen_redraw_proc;
	d->data_fit_proc = dw_data_fit_proc;
	d->close_proc = dw_close_proc;
	d->setDataFormat = db_setDataFormat;	/* inherited */
	d->datawin_number = ndatawins;
	d->scrn = new_screen((void *) d, width, height-panelheight, 0, 
			(int) panelheight);
	d->dpanel = (Mv_panel *) new_panel(d->frame, panelwidth,
			panelheight, 0, 0, 0);
	d->data = (Data_buff *) new_databuff();
	if(strlen(filename)) 	/* if name present, read file */
		d->data->readFile(d->data, filename);
/* set defaults */
	datawinlist[ndatawins++] = d;	/* load into list */
	datawinlist[ndatawins] = NULL;	/* null-terminate list for safety */
	return d;
}

void
dw_destroy(d)
	Data_win *d;
{	
	XUnmapWindow(d->dpy, d->frame);
	XFlush(d->dpy);
	if(d->dpanel) d->dpanel->destroy(d->dpanel);
	if(d->scrn) d->scrn->destroy(d->scrn);
	if(d->data) d->data->destroy(d->data);
	XDestroyWindow(d->dpy, d->frame);
	datawinlist[d->datawin_number] = NULL;
	cfree((char *) d);
	d = (Data_win *) NULL;
}
	
void
dw_display(d)
	Data_win *d;
{
	unsigned x, y;
	Mv_screen *s = d->scrn;
	Data_buff *data = d->data;
	x = s->frameWidth(s) * .75;
	y = s->frameHeight(s) * .75;
	/* set default values for this */
	if(d->hmin == 0.0 && d->hmax == 0.0)
		d->hmax = data->nElements(data, 0);
	XMapWindow(d->dpy, d->frame);
	XMapSubwindows(d->dpy, d->frame);
	clear_events();
	s->is_mapped = True;		/* set flag here for now */
	s->display(d->scrn);
	data->loadData(data, s);
	d->setHorizScale(d);
	s->plot(s);
}

int
dw_setScanChannels(d, first, last)
	Data_win *d;
	int first, last;
{
	if(first < 0 || last < first) {
		mv_alert("dw_setScanChannels: invalid channel choices");
		return -1;
	}
	d->scrn->chanfirst = first;
	d->scrn->chanlast = last;
	return 1;
}

void
dw_getScanChannels(d, first, last)
	Data_win *d;
	int *first, *last;
{
	*first = d->scrn->chanfirst;
	*last = d->scrn->chanlast;
}

void
dw_setHorizValues(d, min, max)
	Data_win *d;
	double min, max;
{
	d->hmin = min;
	d->hmax = max;
}

void
dw_setHorizScale(d)
	Data_win *d;
{
	Mv_screen *s = d->scrn;
	int max, min;
	s->getHorizRange(s, &min, &max);
	s->setHorizScale(s, (d->hmax-d->hmin)/(double) (max-min));
}

void
dw_getHorizValues(d, min, max)
	Data_win *d;
	double *min, *max;
{
	*min = d->hmin;
	*max = d->hmax;
}

void
dw_block(d)
	Data_win *d;
{
	d->dpanel->block(d->dpanel);
	d->scrn->block(d->scrn);
}

void
dw_unBlock(d)
	Data_win *d;
{
	d->dpanel->unBlock(d->dpanel);
	d->scrn->unBlock(d->scrn);
}

int
dw_datawinEvent(d, event)
	Data_win *d;
	XEvent *event;
{
	switch(event->type) {
	case ConfigureNotify:
		if(event->xconfigure.window != d->frame) return False;
		d->frameResize(d, (XConfigureEvent *) event);
		current_datawin = d;
		break;
	case EnterNotify:
	case PropertyNotify:
	case NoExpose:
		if(event->xcrossing.window != d->frame) return False;
		/* set this datawin to be the current window */
		current_datawin = d;
		break;
	default:
		return False;
		break;
	}
	return True;
}

void
dw_frameResize(d, event)
	Data_win *d;
	XConfigureEvent *event;
{
	unsigned new_width, new_height;
	XEvent evt;
	Mv_screen *scrn = d->scrn;
	Mv_panel *pan = d->dpanel;
	/* compress events before calling routine */
	while(XEventsQueued(event->display, QueuedAfterReading) > 0)
	{
		XEvent ahead;
		XPeekEvent(event->display, &ahead);
		if(ahead.type != ConfigureNotify) break;
		XNextEvent(event->display, &evt);
		event = (XConfigureEvent *) &evt;
	}
	new_width = event->width;	/* new dimensions for screen */
	new_height = event->height - pan->frameHeight(pan);
	scrn->resize(scrn, new_width, new_height);
}

void
dw_setFrameLabel(d, label)
	Data_win *d;
	char *label;
{
	XStoreName(d->dpy, d->frame, label);
	XSetIconName(d->dpy, d->frame, label);
}

void
dw_setInsert(d)
	Data_win *d;
{
}

void
dw_setRegion(d)
	Data_win *d;
{
}

void
dw_shiftInsert(d)
	Data_win *d;
{
}

void
dw_shiftRegion(d)
	Data_win *d;
{
}

void
dw_screen_clear_proc(p, pi)	/* button on panel for clearing screen */
	Panel *p;
	Panel_item *pi;
{
	Mv_screen *s;
	if(!current_datawin) {
		mv_alert("dw_screen_clear_proc: null datawin");
		return;
	}
	s = current_datawin->scrn;
	s->clear(s);
}

void
dw_screen_redraw_proc(p, pi)	/* button on panel for redrawing screen */
	Panel *p;
	Panel_item *pi;
{
	Mv_screen *s;
	if(!current_datawin) {
		mv_alert("dw_screen_redraw_proc: null datawin");
		return;
	}
	s = current_datawin->scrn;
	s->plot(s);
}

void
dw_data_fit_proc(p, pi)	/* button on panel for fitting data in window */
	Panel *p;
	Panel_item *pi;
{
	Mv_screen *s;
	Data_buff *d;
	Data_win *win;
	int totalframes, frames, width, height, min, max;
	if(!(win = current_datawin)) {
		mv_alert("dw_data_fit_proc: null datawin");
		return;
	}
	s = win->scrn;
	d = win->data;
	s->getHorizRange(s, &min, &max);
	totalframes = d->nElements(d, 0);
	frames = ABS(max-min);	/* currently asked-for horiz display */
	width = s->frameWidth(s);
	/* width cannot be smaller than total number of frames right now */
	if(frames > width)
		mv_alert("Data will not fit within a frame of this width.");
	width = (frames > width) ? frames : width;
	height = s->frameHeight(s);
	s->newCanvas(s, width, height); /* resize canvas to just fit frame */
	s->setHorizRange(s, min, max);
	/* win->setHorizScale(win); */
	s->plot(s);
}

void
dw_close_proc(m, mi)	/* menu process for closing datawin */
	Menu *m;
	Menu_item *mi;
{
	if(!current_datawin) {
		mv_alert("dw_close_proc: null datawin");
		return;
	}
	current_datawin->destroy(current_datawin);
	current_datawin = NULL; /* temporarily force this to NULL -- DAS */
}

void
dw_vscale_set_proc(m, mi)
	Menu *m;
	Menu_item *mi;
{
	Mv_screen *s;
	Data_win *win;
	double max, min;
	int chan;
	if(!(win = current_datawin)) {
		mv_alert("dw_vscale_set_proc: null datawin");
		return;
	}
	s = win->scrn;
	if(dialog->call(dialog, D_QSVSCALE, "") < 1) return;
	win->block(win);
	chan = dialog->getChoice(dialog, 0);
	max = dialog->getValue(dialog, 0);
	min = dialog->getValue(dialog, 1);
	s->setVertScale(s, chan, min, max);
	s->display(s);
	win->unBlock(win);
}

void
dw_hscale_set_proc(m, mi)
	Menu *m;
	Menu_item *mi;
{
	Mv_screen *s;
	Data_win *win;
	int min, max;
	double scale;
	if(!(win = current_datawin)) {
		mv_alert("dw_hscale_set_proc: null datawin");
		return;
	}
	s = win->scrn;
	s->getHorizRange(s, &min, &max);
	scale = s->getHorizScale(s);
	dialog->setValue(dialog, 0, min*scale);
	dialog->setValue(dialog, 1, max*scale);
	if(dialog->call(dialog, D_QSHSCALE, "") < 1) return;
	min = ROUND(dialog->getValue(dialog, 0)/scale);
	max = ROUND(dialog->getValue(dialog, 1)/scale);
	if(ABS(max-min) < 2) {
		mv_alert("This scale is too small!  Setting to minimum.");
		max = min + 2;
	}
	win->block(win);
	s->setHorizRange(s, min, max);
	s->display(s);
	win->unBlock(win);
}

void
dw_edit_proc(m, mi)
	Menu *m;
	Menu_item *mi;
{
}

void
dw_file_save_proc(m, mi)
Menu *m;
Menu_item *mi;
{
	Data_buff *d;
	if(!current_datawin) {
		mv_alert("dw_save_file_proc: null datawin");
		return;
	}
	current_datawin->block(current_datawin);
	d = current_datawin->data;
	d->writeFile(d);
	current_datawin->unBlock(current_datawin);
}

boolean
datawin_event(event) /* called from main event loop */
	XEvent *event;
{
	register int i;
	Data_win *dwin;
	for(i=0; i<ndatawins; i++) {
		if((dwin = datawinlist[i]) != (Data_win *) NULL)
			if(dwin->datawinEvent(dwin, event)) return True;
	}
	return False;
}

void
set_channel_vals(p, pi)
	Panel *p;
	Panel_item *pi;
{
	Mv_screen *s;
	Data_buff *d;
	Dialog_win *dw;
	double max, min;
	char string[24];
	int chan = *((int *) panelitem_get(p, pi, LXPENUM_VALUE));
	if(!current_datawin) {
		mv_alert("set_channel_vals: null data window.");
		return;
	}
	s = current_datawin->scrn;
	d = current_datawin->data;
	if(chan >= d->nChannels(d)) return;
	if((dw = dialog->getDialog(dialog, 
		dialog->winIndex(dialog, D_QSVSCALE))) == NULL) {
		mv_alert("set_channel_vals: cannot get dialog window.");
		return;
	}
	s->getVertScale(s, chan, &min, &max);
	sprintf(string, "%1.3lf", max);
	dw->panel->setText(dw->panel, "Maximum:", string);
	sprintf(string, "%1.3lf", min);
	dw->panel->setText(dw->panel, "Minimum:", string);
	dw->panel->redisplay(dw->panel, "Maximum:");
	dw->panel->redisplay(dw->panel, "Minimum:");
}
