/*	FGENS.C Modified by Robin Whittle 28 August 1995 
 *
 * 	Alterations:
 *
 * 1 - 	ftfindp()
 * --------------
 *
 *	Addition of a new function for finding function tables during
 *	performance time. Based on ftfind() - which only works at init time -
 *	the new function ftfindp() produces error messages in a way 
 *	appropriate for perf time.  Any error message thus causes the 
 *	instrument to be deactivated. 
 *	
 * 	ftfindp() is used by my new ugens2.c (which has new table read code
 *	with bugs removed, and the possibility of k rate table number control) 
 *	and ugrw1.c (which has table write, with the possibility of k rate
 *	table number control).
 *
 *	Neither this ugens2.c nor ugrw1.c will compile properly without
 *	this new version of fgens.c, and without the following function 
 *	prototype being added to prototyp.h:
 *
 * FUNC *ftfindp(float *); <<<< Add this line just after:
 *
 * FUNC *ftfind(float *), *ftnp2find(float *);
 *
 *
 * 2 - Maximum table number = 1000
 * -------------------------------
 *
 * 	FMAX (below) used to be 200.  Make it 1000 to allow for more 
 *	adventurous table activities - especially with k rate control over
 *	table read and write.
 *
 */

			/*				FGENS.C		*/
#include "cs.h"			
#include <stdlib.h>
#include "soundio.h"
#include "window.h"
#include <math.h>
#include "cmath.h"

#define	FMAX	1000
#define GENMAX  21

extern OPARMS  O;

typedef void    (*GEN)(void);

static void   gen01(void), gen01raw(), gen02(void), gen03(void), gen04(void), 
              gen05(void);
static void   gen06(void), gen07(void), gen08(void), gen09(void), gen10(void);
static void   gen11(void), gen12(void), gen13(void), gen14(void), gen15(void);
static void   gn1314(void), gen17(void), gen19(void), gen20(void), gen21(void);

static	FUNC	*flist[FMAX+1], *ftp;
static	GEN	gensub[GENMAX+1] = { NULL, gen01, gen02, gen03, gen04, gen05,
					   gen06, gen07, gen08, gen09, gen10,
					   gen11, gen12, gen13, gen14, gen15,
					   NULL,  gen17, NULL,  gen19, gen20, 
					   gen21 };
static	EVTBLK	*e;

static	double	tpdlen, tpd360 = 0.017453293;
static	int	fno, guardreq, nargs, fterrcnt;
static	long	flen, flenp1, lenmask;
static  void    fterror(char *), ftresdisp(void), ftalloc(void);

#define	FTERR(s)	{fterror(s);  return;}

void fgens(EVTBLK *evtblkp)	/* create ftable using evtblk data */
{
	long	ltest, lobits, lomod, genum;

	e = evtblkp;
	fterrcnt = 0;
	if ((fno = (int)e->p[1]) < 0) {			/* fno < 0: remove */
		if ((fno = -fno) > FMAX)
			FTERR("illegal ftable number")
		if ((ftp = flist[fno]) == NULL)
			FTERR("ftable does not exist")
		flist[fno] = NULL;
		free((char *)ftp);
		printf("ftable %d now deleted\n",fno);
		return;
	}
	if (!fno)				/* fno = 0, return	*/
		return;
	if (fno > FMAX)
		FTERR("illegal ftable number")
	if ((nargs = e->pcnt - 4) <= 0)		/* chk minimum arg count */
		FTERR("insufficient gen arguments")
	if ((genum = e->p[4]) < 0)
		genum = -genum;
	if (!genum || genum > GENMAX)		/*   & legal gen number */
		FTERR("illegal gen number")
	if ((flen = e->p[3])) {			/* if user flen	given       */
	    guardreq = flen & 01;		/*   set guard request flg  */
	    flen &= -2;				/*   flen now w/o guardpt   */
	    flenp1 = flen + 1;			/*   & flenp1 with guardpt  */
	    if (flen <= 0 || flen > MAXLEN)
		FTERR("illegal table length")
	    for (ltest=flen,lobits=0; (ltest & MAXLEN) == 0; lobits++,ltest<<=1);
	    if (ltest != MAXLEN)		/*   flen must be power-of-2 */
		FTERR("illegal table length")
	    lenmask = flen-1;
	    ftalloc();				/*   alloc ftable space now */
	    ftp->flen = flen;
	    ftp->lenmask = lenmask;  		/*   init hdr w powof2 data */
	    ftp->lobits = lobits;
	    lomod = MAXLEN / flen;
	    ftp->lomask = lomod - 1;
	    ftp->lodiv = 1./((float)lomod);	/*    & other useful vals    */
	    tpdlen = twopi / flen;
	    ftp->nchnls = 1;                    /*    presume mono for now   */
	    ftp->flenfrms = flen;
	}
	else if (genum != 1)                    /* else defer alloc to gen01 */
	    FTERR("deferred size for GEN01 only")
	printf("ftable %d:\n", fno);
	(*gensub[genum])();			/* call gen subroutine	*/
	if (!fterrcnt)
	    ftresdisp();			/* rescale and display */
}

static void needsiz(long maxend)
{
        register long nxtpow;
	maxend -= 1; nxtpow = 2;
	while (maxend >>= 1)
	    nxtpow <<= 1;
	printf("non-deferred ftable %d needs size %ld\n", (int)fno, nxtpow);
}

static void gen01(void)             /* read ftable values from a sound file */
{				/* stops reading when table is full	*/
	if (nargs < 4)
		FTERR("insufficient args")
        if (O.gen01defer) {
        /* We're deferring the soundfile load until performance time,
           so allocate the function table descriptor, save the arguments,
           and get out */
	  ftalloc();
          ftp->gen01args.gen01 = e->p[4];
          ftp->gen01args.ifilno = e->p[5];
	  ftp->gen01args.iskptim = e->p[6];
	  ftp->gen01args.iformat = e->p[7];
	  ftp->gen01args.channel = e->p[8];
	  strcpy(ftp->gen01args.strarg,e->strarg);
          ftp->flen = flen;                   
          return;
        }  
        gen01raw();
}

static void gen01raw()             /* read ftable values from a sound file */
{				/* stops reading when table is full	*/
extern  int	close(int);
static	ARGOFFS	argoffs = {0};		/* OUTOCOUNT-not applicable yet */
static	OPTXT	optxt;			/* create dummy optext	*/
register SOUNDIN *p;			/*   for sndgetset	*/
        AIFFDAT *adp;
extern  int     sndgetset(SOUNDIN *);
extern  long    getsndin(int, float *, long, SOUNDIN *);
	SOUNDIN	tmpspace;		/* create temporary opds */
        int     fd, truncmsg = 0;
        long    inlocs = 0;

	optxt.t.outoffs = &argoffs;      /* point to dummy OUTOCOUNT */
	p = &tmpspace;
	p->h.optext = &optxt;
	p->ifilno = &e->p[5];
	p->iskptim = &e->p[6];
	p->iformat = &e->p[7];
	p->channel = e->p[8];
	if (p->channel < 0 || p->channel > 4) {
	    sprintf(errmsg,"channel %d illegal",(int)p->channel);
	    FTERR(errmsg);
	}
	if (p->channel == 0)                    /* snd is chan 1,2,3,4 or all */
	    p->channel = ALLCHNLS;
	p->analonly = 0;
	p->STRARG = e->strarg;
	if (!flen)
		printf("deferred alloc\n");
	if (!(fd = sndgetset(p))) 		/* sndinset to open the file */
		FTERR (errmsg)
	if (p->endfile) {
	        printf("GEN01 early end-of-file\n");
		goto gn1rtn;
	}
	if (!flen) {                            /* deferred ftalloc requestd: */
	    if ((flen = p->framesrem) <= 0)       /*   get minsize from soundin */
		FTERR ("deferred size, but filesize unknown")
	    if (p->channel == ALLCHNLS)
	        flen *= p->nchnls;
	    guardreq = 1;
	    flenp1 = flen;                      /* presum this includes guard */
	    flen -= 1;
	    ftalloc();                          /*   alloc now, and           */
	    ftp->flen = flen;
	    ftp->lenmask = 0;                   /*   mark hdr partly filled   */
	    ftp->nchnls = p->nchnls;
	    ftp->flenfrms = flen / p->nchnls;  /* ?????????? */
	}
	ftp->cvtbas = LOFACT * p->sr / esr;
        if ((adp = p->aiffdata) != NULL) {            /* if file was aiff,    */
	    /* set up some necessary header stuff if not in aiff file */
	    if (adp->natcps == 0)                      /* from Jeff Fried */
    		adp->natcps = ftp->cvtbas;
	    if(adp->gainfac == 0)
    		adp->gainfac = 1.0;
	    ftp->cpscvt = ftp->cvtbas / adp->natcps;  /*    copy data to FUNC */
	    ftp->loopmode1 = adp->loopmode1;          /* (getsndin does gain) */
	    ftp->loopmode2 = adp->loopmode2;
	    ftp->begin1 = adp->begin1;
	    ftp->begin2 = adp->begin2;
	    if (ftp->loopmode1)	/* Greg Sullivan */
	      ftp->end1 = adp->end1;
	    else
	      ftp->end1 = ftp->flenfrms; 
	    ftp->end1 = adp->end1;
	    ftp->end2 = adp->end2;
	    if (ftp->end1 > flen || ftp->end2 > flen) {
	        long maxend;
	        warning("GEN01: input file truncated by ftable size");
		if ((maxend = ftp->end1) < ftp->end2)
		    maxend = ftp->end2;
		printf("\tlooping endpoint %ld exceeds ftsize %ld\n",maxend,flen);
		needsiz(maxend);
		truncmsg = 1;
	    }
	}
	else {
	    ftp->cpscvt = 0.;                  /* else no looping possible   */
	    ftp->loopmode1 = 0;
	    ftp->loopmode2 = 0;
	    ftp->end1 = ftp->flenfrms; /* Greg Sullivan */
	}
	if ((inlocs = getsndin(fd, ftp->ftable, flenp1, p)) < 0)  /* read sound */
	    fterror("GEN01 read error");                       /* with opt gain */
gn1rtn: if (p->audrem > 0 && !truncmsg) {
	    warning("GEN01: aiff file truncated by ftable size");
	    printf("\taudio samps %ld exceeds ftsize %ld\n", p->framesrem, flen);
            needsiz(p->framesrem);     /* ????????????  */
	}
        ftp->soundend = inlocs / ftp->nchnls;   /* record end of sound samps */
        ftresdisp();
        close(fd);
}

static void gen02(void)             /* read ftable values directly from p-args */
{
register float	*fp = ftp->ftable, *pp = &e->p[5];
register int	nvals = nargs;

	if (nvals > flenp1)
		nvals = flenp1;			/* for all vals up to flen+1 */
	do  *fp++ = *pp++;			/*   copy into ftable	*/
	while (--nvals);
}

static void gen03(void)
{
	int	ncoefs;
	float	xintvl, xscale;
register int	xloc, nlocs;
register float	*fp = ftp->ftable, x, sum, *coefp, *coef0, *coeflim;

	if ((ncoefs = nargs - 2) <= 0)
		FTERR("no coefs present")
	coef0 = &e->p[7];
	coeflim = coef0 + ncoefs;
	if ((xintvl = e->p[6] - e->p[5]) <= 0)
		FTERR("illegal x interval")
	xscale = xintvl / (float)flen;
	xloc = e->p[5] / xscale;		/* initial xloc	*/
	nlocs = flenp1;
	do {					/* for each loc:	*/
		x = xloc++ * xscale;
		coefp = coeflim;
		sum = *--coefp;			/* init sum to coef(n)	*/
		while (coefp > coef0) {
			sum *= x;		/*  & accum by Horner's rule */
			sum += *--coefp;
		}
		*fp++ = sum;
	} while (--nlocs);
}

static void gen04(void)
{
register float	*valp, *rvalp, *fp = ftp->ftable;
register int	n, r;
register FUNC	*srcftp;
	float	val, max, maxinv;
	int	srcno, srcpts, ptratio;

	if (nargs < 2)
		FTERR("insufficient args")
	if ((srcno = (int)e->p[5]) <= 0 || srcno > FMAX
	  || (srcftp = flist[srcno]) == NULL)
	  	FTERR("unknown srctable number")
	if (!e->p[6]) {
		srcpts = srcftp->flen;
		valp = &srcftp->ftable[0];
		rvalp = NULL;
	}
	else {
		srcpts = srcftp->flen >>1;
		valp = &srcftp->ftable[srcpts];
		rvalp = valp - 1;
	}
	if ((ptratio = srcpts / flen) < 1)
		FTERR("table size too large")
	if (val = *valp++) {
		if (val < 0.)	val = -val;
		max = val;
		maxinv = 1. / max;
	}
	else {
		max = 0.;
		maxinv = 1.;
	}
	*fp++ = maxinv;
	for (n = flen; n--; ) {
		for (r = ptratio; r--; ) {
			if (val = *valp++) {
				if (val < 0.)	val = -val;
				if (val > max) {
					max = val;
					maxinv = 1. / max;
				}
			}
			if (rvalp != NULL && (val = *rvalp--)) {
				if (val < 0.)	val = -val;
				if (val > max) {
					max = val;
					maxinv = 1. / max;
				}
			}
			*fp++ = maxinv;
		}
	}
	guardreq = 1;			/* disable new guard point */
	e->p[4] = -4.;			/*   and rescaling	   */
}

static void gen05(void)
{
register int	nsegs, seglen;
register float	*valp, *fp, *finp;
register float	amp1, mult;

	if ((nsegs = (nargs - 1) >> 1) <= 0)	     /* nsegs = nargs-1 /2 */
		return;
	valp = &e->p[5];
	fp = ftp->ftable;
	finp = fp + flen;
	if (*valp == 0) goto gn5er2;
	do {	amp1 = *valp++;
		if (!(seglen = *valp++)) continue;
		if (seglen < 0) goto gn5er1;
		if ((mult = *valp/amp1) <= 0) goto gn5er2;
		mult = pow( (double)mult, (double)1/seglen );
		while (seglen--) {
			*fp++ = amp1;
			amp1 *= mult;
			if (fp > finp) return;
		}
	} while (--nsegs);
	if (fp == finp)			/* if 2**n pnts, add guardpt */
		*fp = amp1;
	return;

gn5er1: fterror("gen call has negative segment size:");
        return;
gn5er2:	fterror("illegal input vals for gen call, beginning:");
}

static void gen07(void)
{
register int	nsegs, seglen;
register float	*valp, *fp, *finp;
register float	amp1, incr;

	if ((nsegs = (nargs - 1) >> 1) <= 0)	     /* nsegs = nargs-1 /2 */
		return;
	valp = &e->p[5];
	fp = ftp->ftable;
	finp = fp + flen;
	do {	amp1 = *valp++;
		if (!(seglen = *valp++)) continue;
		if (seglen < 0) goto gn7err;
		incr = (*valp - amp1) / seglen;
		while (seglen--) {
			*fp++ = amp1;
			amp1 += incr;
			if (fp > finp) return;
		}
	} while (--nsegs);
	if (fp == finp)			/* if 2**n pnts, add guardpt */
		*fp = amp1;
        return;

gn7err: fterror("gen call has negative segment size:");
}

static void gen06(void)
{
register float	*segp, *extremp, *inflexp, *segptsp, *fp, *finp;
	float	y, diff2;
register int	pntno, pntinc, nsegs, npts;

	if ((nsegs = (nargs - 1) >>1) < 1)
		FTERR("insufficient args")
	fp = ftp->ftable;
	finp = fp + flen;
	pntinc = 1;
	for (segp = &e->p[3]; nsegs > 0; nsegs--) {
		segp += 2;
		segptsp = segp + 1;
		if ((npts = *segptsp) < 0)
			FTERR("negative segsiz")
		if (pntinc > 0) {
			pntno = 0;
			inflexp = segp + 2;
			extremp = segp;
		}
		else {
			pntno = npts;
			inflexp = segp;
			extremp = segp + 2;
		}
		diff2 = (*inflexp - *extremp) / 2.;
		for ( ; npts > 0 && fp < finp; pntno += pntinc, npts--) {
			y = (float)pntno / *segptsp;
			*fp++ = (3.-y) * y * y * diff2 + *extremp;
		}
		pntinc = -pntinc;
	}
	*fp = *(segp + 2);			/* write last target point */
}

static void gen08(void)
{
register float	R, x, c3, c2, c1, c0, *fp, *fplim, *valp;
	float	f2, f1, f0, df1, df0, dx01, dx02, dx12, curx;
	float	slope, resd1, resd0;
	int	nsegs, npts;

	if ((nsegs = (nargs - 1) >>1) <= 0)
		FTERR("insufficient args");
	valp = &e->p[5];
	fp = ftp->ftable;
	fplim = fp + flen;
	f0 = *valp++;			/* 1st 3 params give vals at x0, x1  */
	if ((dx01 = *valp++) <= 0.)	/*	and dist between	     */
		FTERR("illegal x interval");
	f1 = *valp++;
	curx = df0 = 0.;		/* init x to origin; slope at x0 = 0 */
	do {				/* for each spline segmnt (x0 to x1) */
	    if (nsegs > 1) {			/* if another seg to follow  */
		if ((dx12 = *valp++) <= 0.)	/*    read its distance	     */
			FTERR("illegal x interval");
		f2 = *valp++;			/*    and the value at x2    */
		dx02 = dx01 + dx12;
		df1 = ( f2*dx01*dx01 + f1*(dx12-dx01)*dx02 - f0*dx12*dx12 )
			/ (dx01*dx02*dx12);
	    }				   /* df1 is slope of parabola at x1 */
	    else df1 = 0.;
	    if ((npts = dx01 - curx) > fplim - fp)
		npts = fplim - fp;
	    if (npts > 0) {			/* for non-trivial segment: */
		slope = (f1 - f0) / dx01;	/*   get slope x0 to x1	    */
		resd0 = df0 - slope;		/*   then residual slope    */
		resd1 = df1 - slope;		/*     at x0 and x1	    */
		c3 = (resd0 + resd1) / (dx01*dx01);
		c2 = - (resd1 + 2.*resd0) / dx01;
		c1 = df0;			/*   and calc cubic coefs   */
		c0 = f0;
		for (x = curx; npts>0; --npts, x += 1.) {
		    R = c3;
		    R *= x;
		    R += c2;	     /* f(x) = ((c3 x + c2) x + c1) x + c0  */
		    R *= x;
		    R += c1;
		    R *= x;
		    R += c0;
		    *fp++ = R;			/* store n pts for this seg */
		}
		curx = x;
	    }
	    curx -= dx01;		/* back up x by length last segment */
	    dx01 = dx12;		/* relocate to the next segment	*/
	    f0 = f1;			/*   by assuming its parameters	*/
	    f1 = f2;
	    df0 = df1;
	}
	while (--nsegs && fp<fplim);	/* loop for remaining segments	*/
	while (fp <= fplim)
	    *fp++ = f0;			/* & repeat the last value	*/
}

static void gen09(void)
{
register int	hcnt;
register float	*valp, *fp, *finp;
	double	phs, inc, amp;

	if ((hcnt = nargs / 3) <= 0)		/* hcnt = nargs / 3 */
		return;
	valp = &e->p[5];
	finp = &ftp->ftable[flen];
	do	for (inc=(*valp++)*tpdlen, amp=(*valp++),
		     phs=(*valp++)*tpd360, fp=ftp->ftable; fp<=finp; fp++) {
			*fp += sin(phs) * amp;
			if ((phs += inc) >= twopi)
				phs -= twopi;
		}
	while (--hcnt);
}

static void gen10(void)
{
register long	phs, hcnt;
register float	amp, *fp, *finp;

	if ((hcnt = nargs) <= 0)			/* hcnt is nargs   */
		return;
	finp = &ftp->ftable[flen];
	do if ((amp = e->p[hcnt+4]) != 0)		/* for non-0 amps,  */
		for (phs=0, fp=ftp->ftable; fp<=finp; fp++) {
			*fp += sin(phs*tpdlen) * amp;	/* accum sin pts  */
			phs += hcnt;			/* phsinc is hno   */
			phs &= lenmask;
		}
	while (--hcnt);
}

static void gen11(void)
{
register float  *fp, *finp;
register long   phs;
	double	x;
	float	denom, r, scale;
	int	n, k;
  
	if (nargs < 1)
		FTERR ("insufficient arguments");
	if ((n = e->p[5]) < 1)
		FTERR ("nh partials < 1");
	k = 1;
	r = 1.;
	if (nargs > 1)
		k = e->p[6];
	if (nargs > 2)
		r = e->p[7];
	fp = ftp->ftable;
	finp = fp + flen;
	if (nargs == 1 || k == 1 && r == 1.) {     /* simple "buzz" case */
		int tnp1;
		float pdlen;

		tnp1 = (n << 1) + 1;
		scale = .5 / n;
		pdlen = tpdlen / 2.;
		for (phs = 0; fp <= finp; phs++) {
			x = phs * pdlen;
			if (!(denom = sin(x)))
				*fp++ = 1.;
			else *fp++ = (sin(tnp1 * x) / denom - 1.) * scale;
		}
	}
	else {                                   /* complex "gbuzz" case */
		float numer, twor, rsqp1, rtn, rtnp1, absr;
		int   km1, kpn, kpnm1;

		km1   = k - 1;
		kpn   = k + n;
		kpnm1 = kpn - 1;
		twor  = r * 2.;
		rsqp1 = r * r + 1.;
		rtn   = pow((double) r, (double) n);
		rtnp1 = rtn * r;
		if ((absr = fabs(r)) > .999 && absr < 1.001)
			scale = 1. / n;
		else scale = (1. - absr) / (1. - fabs(rtn));
		for (phs=0; fp <= finp; phs++) {
			x = phs * tpdlen;
			numer = cos(x*k) - r * cos(x*km1) - rtn * cos(x*kpn)
				+ rtnp1 * cos(x*kpnm1);
			if ((denom = rsqp1 - twor*cos(x)) > .0001
			  || denom < -.0001)
			  	*fp++ = numer / denom * scale;
			else *fp++ = 1.;
		}
	}
}

static void gen12(void)
{
static double coefs[] = { 3.5156229, 3.0899424, 1.2067492,
			  0.2659732, 0.0360768, 0.0045813 };
register double *coefp, sum, tsquare, evenpowr, *cplim = coefs + 6;
register int    n;
register float	*fp;
register double xscale;

	if (nargs < 1)
		FTERR ("insufficient arguments");
	xscale = (double) e->p[5] / flen / 3.75;
	for (n=0,fp=ftp->ftable; n<=flen; n++) {
	        tsquare = (double) n * xscale;
		tsquare *= tsquare;
		for (sum=evenpowr=1.0, coefp=coefs; coefp<cplim; coefp++) {
			evenpowr *= tsquare;
			sum += *coefp * evenpowr;
	        }
		*fp++ = (float) log(sum);
	}
}

static	float	mxval, mxscal;
static  void   gn1314(void), gen03(void);

static void gen13(void)
{
	mxval = 2.;
	mxscal = .5;
	gn1314();
}

static void gen14(void)
{
	mxval = 1.;
	mxscal = 1.;
	gn1314();
}

static void gn1314(void)
{
register long	nh, nn;
register float	*mp, *mspace, *hp, *oddhp;
	float	xamp, xintvl, scalfac, sum, prvm;

	if ((nh = nargs - 2) <= 0)
		FTERR("insufficient args")
	if ((xintvl = e->p[5]) <= 0)
		FTERR("illegal xint value")
	if ((xamp = e->p[6]) <= 0)
		FTERR("illegal xamp value")
	e->p[5] = -xintvl;
	e->p[6] = xintvl;
        nn = nh * sizeof(float) / 2;	    /* alloc spc for terms 3,5,7,... */
	mp = mspace = (float *)mcalloc(nn);     /* of 1st row of matrix, and */
	for (nn = (nh + 1) >>1; --nn; )		/* form array of non-0 terms */
		*mp++ = mxval = -mxval;		/*  -val, val, -val, val ... */
	scalfac = 2 / xamp;
	hp = &e->p[7];				/* beginning with given h0,  */
	do {
		mp = mspace;
		oddhp = hp;
		sum = *oddhp++;			/* sum = diag(=1) * this h   */
		for (nn = (nh+1) >>1; --nn; ) {
			oddhp++;		/*  + odd terms * h+2,h+4,.. */
			sum += *mp++ * *oddhp++;
		}
		*hp++ = sum * mxscal;		/* repl this h w. coef (sum) */
		mp = mspace;
		prvm = 1;
		for (nn = nh>>1; --nn > 0; mp++)/* calc nxt row matrix terms */
			*mp = prvm = *mp - prvm;
		mxscal *= scalfac;
	} while (--nh);				/* loop til all h's replaced */
	free((char *)mspace);
	gen03();				/* then call gen03 to write */
}

static void gen15(void)
{
	float	xint, xamp, hsin[PMAX/2], h, angle;
register float	*fp, *cosp, *sinp;
register int	n, nh;
register long	*lp, *lp13;

	if (nargs & 01)
		FTERR("uneven number of args");
	nh = (nargs - 2) >>1;
	fp = &e->p[5];					/* save p5, p6	*/
	xint = *fp++;
	xamp = *fp++;
	for (n = nh, cosp = fp, sinp = hsin; n > 0; n--) {
		h = *fp++;				/* rpl h,angle pairs */
		angle = *fp++ * tpd360;
		*cosp++ = h * cos((double)angle);	/*  with h cos angle */
		*sinp++ = h * sin((double)angle);	/* and save the sine */
	}
	nargs -= nh;
	gen13();					/* call gen13	*/
	if (fterrcnt) return;
	ftresdisp();					/* and display fno   */
	lp13 = (long *)ftp;
	fno++;					/* alloc eq. space for fno+1 */
	ftalloc();
	for (lp = (long *)ftp; lp < (long *)ftp->ftable; )  /* & copy header */
		*lp++ = *lp13++;
	fp = &e->p[5];
	*fp++ = xint;					/* restore p5, p6,   */
	*fp++ = xamp;
	for (n = nh-1, sinp = hsin+1; n > 0; n--)	/* then skip h0*sin  */
		*fp++ = *sinp++;			/* & copy rem hn*sin */
	nargs--;		
	gen14();					/* now draw ftable   */
}

static void gen17(void)
{
register int	nsegs, ndx, nxtndx;
register float	*valp, *fp, *finp;
register float	val;

	if ((nsegs = nargs >> 1) <= 0)	     /* nsegs = nargs /2 */
	    goto gn17err;
	valp = &e->p[5];
	fp = ftp->ftable;
	finp = fp + flen;
	if ((ndx = *valp++) != 0)
	    goto gn17err;
	while (--nsegs) {
	    val = *valp++;
	    if ((nxtndx = *valp++) <= ndx)
		goto gn17err;
	    do {
	        *fp++ = val;
		if (fp > finp) return;
	    } while (++ndx < nxtndx);
	}
	val = *valp;
	while (fp <= finp)			/* include 2**n + 1 guardpt */
		*fp++ = val;
        return;

gn17err: fterror("gen call has illegal x-ordinate values:");
}

static void gen19(void)
{
register int	hcnt;
register float	*valp, *fp, *finp;
	double	phs, inc, amp, dc;

	if ((hcnt = nargs / 4) <= 0)		/* hcnt = nargs / 4 */
		return;
	valp = &e->p[5];
	finp = &ftp->ftable[flen];
	do	for (inc=(*valp++)*tpdlen, amp=(*valp++),
		     phs=(*valp++)*tpd360, dc=(*valp++),
		     fp=ftp->ftable; fp<=finp; fp++) {
			*fp += sin(phs) * amp + dc;   /* dc after str scale */
			if ((phs += inc) >= twopi)
				phs -= twopi;
		}
	while (--hcnt);
}

/*  GEN20 and GEN21 by Paris Smaragdis 1994 B.C.M. Csound development team  */
static void gen20()
{
    register float cf[4], *ft;
    register double arg, i, xarg, beta;

    ft = ftp->ftable;
    xarg = 1;

    if (e->p[4] < 0 )       {
	xarg = e->p[6];
	if ( nargs < 2 ) xarg = 1;
    }

    if (nargs > 2)
	beta = e->p[7];

    switch( (int)e->p[5])  {
    case 1:
	cf[0] = .54;
	cf[1] = .46;
	cf[2] = cf[3] = 0;
	break;
    case 2:
	cf[0] = cf[1] = .5;
	cf[2] = cf[3] = 0;
	break;
    case 3:
	arg = 2.0/flen;
	for (i = 0 ; i < flen/2 ; i++)
	    *ft++ = i*arg*xarg;
	for (; i < flen ; i++)
	    *ft++ = (2.0 - i*arg)*xarg;
	return;
    case 4:
	cf[0] = .42;
	cf[1] = .5;
	cf[2] = .08;
	cf[3] = 0;
	break;
    case 5:
	cf[0] = .35878;
	cf[1] = .48829;
	cf[2] = .14128;
	cf[3] = .01168;
	break;
    case 6:
	arg = 12.0/flen;
	for (i = -6 ; i < 0 ; i += arg)
	    *ft++ = xarg * (pow( 2.71828, -(i*i)/2));
	for (i = arg ; i < 6 ; i += arg)
	    *ft++ = xarg * (pow( 2.71828, -(i*i)/2));
	return;
    case 7:
	for (i = -flen/2 + .1 ; i < flen/2 ; i++)
	    *ft++ = xarg * besseli((beta * sqrt(1-pow((2*i/(flen - 1)), 2)))) /
		besseli( beta);
	return;
    case 8:
	for (i = 0 ; i < flen ; i++)
	    *ft++ = 1;
	return;
    case 9:
	arg = twopi / flen;
	for (i = -twopi/2 ; i < 0 ; i += arg)
	    *ft++ = xarg * sin(i) / i;
	*ft++ = 1;
	for (i = arg ; i < twopi/2 ; i += arg)
	    *ft++ = xarg * sin(i) / i;
	return;
    default:
	fterror("No such window!");
	return;
    }

    arg = twopi/flen;
	
    for (i = 0 ; i < twopi ; i += arg)
	*ft++ = xarg * (cf[0] - cf[1]*cos(i) + cf[2]*cos(2*i) - cf[3]*cos(3*i));
}

float besseli( double x)
{
    float ax, ans;
    double y;

    if (( ax = fabs( x)) < 3.75)     {
	y = x / 3.75;
	y *= y;
	ans = 1.0 + y * ( 3.5156229 +
			  y * ( 3.0899424 +
				y * ( 1.2067492 +
				      y * ( 0.2659732 +
					    y * ( 0.360768e-1 +
						  y * 0.45813e-2)))));
    }
    else {
	y = 3.75 / ax;
	ans = (exp ( ax) / sqrt(ax))
	    * (0.39894228 +
	       y * (0.1328592e-1 +
		    y * (0.225319e-2 +
			 y * (-0.157565e-2 +
			      y * (0.916281e-2 +
				   y * (-0.2057706e-1 +
					y * (0.2635537e-1 +
					     y * (-0.1647633e-1 +
						  y * 0.392377e-2))))))));
    }
    return ans;
}

static void gen21(void)
{
    register long i;
    register float *ft;

    ft = ftp->ftable;

    switch ((int)e->p[5])  {
    case 1:
	for (i = 0 ; i < flen ; i++)
	    *ft++ = unifrand();
	break;
    case 2:
	if (nargs < 1)  {
	    fterror("Wrong number of input arguments\n");
	    return;
	}               
	for (i = 0 ; i < flen ; i++)
	    *ft++ = linrand( (float) e->p[6]);
	break;
    case 3:
	if (nargs < 1)  {
	    fterror("Wrong number of input arguments\n");
	    return;
	}               
	for (i = 0 ; i < flen ; i++)
	    *ft++ = trirand( (float) e->p[6]);
	break;
    case 4:
	if (nargs < 1)  {
	    fterror("Wrong number of input arguments\n");
	    return;
	}               
	for (i = 0 ; i < flen ; i++)
	    *ft++ = exprand( (float) e->p[6]);
	break;
    case 5:
	if (nargs < 1)  {
	    fterror("Wrong number of input arguments\n");
	    return;
	}               
	for (i = 0 ; i < flen ; i++)
	    *ft++ = biexprand( (float) e->p[6]);
	break;
    case 6:
	if (nargs < 1)  {
	    fterror("Wrong number of input arguments\n");
	    return;
	}               
	for (i = 0 ; i < flen ; i++)
	    *ft++ = gaussrand( (float) e->p[6]);
	break;
    case 7:
	if (nargs < 2)  {
	    fterror("Wrong number of input arguments\n");
	    return;
	}               
	for (i = 0 ; i < flen ; i++)
	    *ft++ = cauchrand( (float) e->p[7]);
	break;
    case 8:
	if (nargs < 2)  {
	    fterror("Wrong number of input arguments\n");
	    return;
	}               
	for (i = 0 ; i < flen ; i++)
	    *ft++ = pcauchrand( (float) e->p[7]);
	break;
    case 9:
	if (nargs < 3)  {
	    fterror("Wrong number of input arguments\n");
	    return;
	}               
	for (i = 0 ; i < flen ; i++)
	    *ft++ = betarand( (float) e->p[6],(float) e->p[7],(float)
			      e->p[8] );
	break;
    case 10:
	if (nargs < 3)  {
	    fterror("Wrong number of input arguments\n");
	    return;
	}               
	for (i = 0 ; i < flen ; i++)
	    *ft++ = weibrand( (float) e->p[7], (float) e->p[8]);
	break;
    case 11:
	if (nargs < 2)  {
	    fterror("Wrong number of input arguments\n");
	    return;
	}               
	for (i = 0 ; i < flen ; i++)
	    *ft++ = poissrand( (float) e->p[7]);
	break;
    default:
	fterror("Unknown distribution\n");
    }
}

static void fterror(s)
 char *s;
{
	printf("FTERROR, ftable %d: %s\n",fno,s);
	printf("f%3.0f%8.2f%8.2f%8.2f",e->p[1],e->p2orig,e->p3orig,e->p[4]);
	if (e->p[5] == SSTRCOD)
	    printf("  \"%s\" ...\n",e->strarg);
	else printf("%8.2f ...\n",e->p[5]);
	fterrcnt++;
}

static void ftresdisp(void)   /* set guardpt, rescale the function, and display it */
{
register float	*fp, *finp = &ftp->ftable[flen];
register float	abs, maxval;
static	WINDAT	dwindow;

	if (!guardreq)				/* if no guardpt yet, do it */
	  ftp->ftable[flen] = ftp->ftable[0];
	if (e->p[4] > 0.) {			/* if genum positve, rescale */
	  for (fp=ftp->ftable, maxval = 0.0; fp<=finp; ) {
	    if ((abs = *fp++) < 0.)
	      abs = -abs;
	    if (abs > maxval)
	      maxval = abs;
	  }
	  if (maxval != 0. && maxval != 1.)
	    for (fp=ftp->ftable; fp<=finp; fp++)
	      *fp /= maxval;
	}
	    sprintf(strmsg,"ftable %d:",fno);
	    dispset(&dwindow,ftp->ftable,(long)(flen+guardreq),strmsg,0,"ftable");
	    display(&dwindow);
}

static void ftalloc(void)   /* alloc ftable space for fno (or replace one)  */
{			/*	set ftp to point to that structure	*/
	if ((ftp = flist[fno]) != NULL) {
	    printf("replacing previous ftable %d\n",fno);
	    if (flen != ftp->flen) {    	/* if redraw & diff len, */
		extern INSDS actanchor;
		free((char *)ftp);      	/*   release old space   */
		flist[fno] = NULL;
		if (actanchor.nxtact != NULL) { /*   & chk for danger    */
		    sprintf(errmsg,"ftable %d relocating due to size change\n\
  currently active instruments may find this disturbing", fno);
		    warning(errmsg);
		}
	    }
	    else {				/* else clear it to zero */
	        register float	*fp = ftp->ftable;
		register float	*finp = &ftp->ftable[flen];
		while (fp <= finp)
		    *fp++ = 0;
	    }
	}
	if ((ftp = flist[fno]) == NULL) {	/*   alloc space as reqd */
	    ftp = (FUNC *) mcalloc((long)sizeof(FUNC) + flen*sizeof(float));
	    flist[fno] = ftp;
	}
}

/*****************************************************************************/
					
					/* The original ftfind() function.
					 */

 FUNC *
ftfind(float *argp)     /* find the ptr to an existing ftable structure */
			/*   called by oscils, etc at init time         */
{
register int	fno;
register FUNC	*ftp;

	if ((fno = *argp) <= 0 || fno > FMAX || (ftp = flist[fno]) == NULL) {
		sprintf(errmsg, "invalid ftable no. %f", *argp);
		initerror(errmsg);
		return(NULL);
	}
	else if (!ftp->lenmask) {
		sprintf(errmsg, "deferred-size ftable %f illegal here", *argp);
		initerror(errmsg);
		return(NULL);
	}
	else return(ftp);
}

					/*************************************/

					/* ftfindp()
					 *
					 * New function to find a function 
					 * table at performance time.
					 * Based on ftfind() which is intended 
					 * to run at init time only. 
					 *
					 * This function can be called from
					 * other modules - such as ugrw1.c.
					 *
					 * It returns a pointer to a FUNC
					 * data structure which contains
					 * all the details of the desired 
					 * table.  0 is returned if
					 * it cannot be found.
					 *	
					 * This does not handle deferred
					 * function table loads (gen01).
					 *	
					 * Maybe this could be achieved, but
					 * some exploration would be required
					 * to see that this is feasible at
					 * performance time.
					 * 
					 */ 
FUNC * ftfindp(float *argp)     		
{
register int	fno;
register FUNC	*ftp;
					/* Check limits, and then index 
					 * directly into the flist[] which
					 * contains pointers to FUNC data
					 * structures for each table.
					 */
	if (    (fno = *argp) <= 0 
	     || fno > FMAX 
	     || (ftp = flist[fno]) == NULL
	   ) 
	{
		sprintf(errmsg, "Invalid ftable no. %f", *argp);
		perferror(errmsg);
		return(NULL);
	}
	else if (!ftp->lenmask) {
					/* Now check that the table has
					 * a length > 0.
					 * This should only occur for 
					 * tables which have not been loaded
					 * yet.
					 */	 
		sprintf(errmsg, 
		"Deferred-size ftable %f load not available at perf time.", *argp);
		perferror(errmsg);
		return(NULL);
	}
	else return(ftp);
}

/*****************************************************************************/

 FUNC *
ftnp2find(float *argp)  /* find ptr to a deferred-size ftable structure */
			/*   called by loscil at init time              */
{
    EVTBLK evt;
    char strarg[SSTRSIZ];

    if ((fno = *argp) <= 0 || fno > FMAX || (ftp = flist[fno]) == NULL) {
	sprintf(errmsg, "invalid ftable no. %f", *argp);
	initerror(errmsg);
	return(NULL);
    }
    else {
	if (ftp->flen == 0) {
                /* The soundfile hasn't been loaded yet, so call GEN01 */
	    flen = 0;
	    e = &evt;
	    e->p[4] = ftp->gen01args.gen01;
	    e->p[5] = ftp->gen01args.ifilno;
	    e->p[6] = ftp->gen01args.iskptim;
	    e->p[7] = ftp->gen01args.iformat;
	    e->p[8] = ftp->gen01args.channel;
	    strcpy(strarg,ftp->gen01args.strarg);
	    e->strarg = strarg;
	    gen01raw();
	}
	return (ftp);
    }
}
