/*
---------------------------------------------------------------------------------------------
			MIDI Sequencer - Final Year Project, A.J. Green
---------------------------------------------------------------------------------------------

File Name:	Menu.c

Description:	Code to set up the pulldown menus using the Y library.

Author:		AJG

History:

Update	Date		Programmer	Comments
======	====		==========	========
001	16/02/94	AJG		File Created.

--------------------------------------------------------------------------------------------
*/

#include <MidiXInclude.h>
#include <MidiFile.h>
#include <MidiTrack.h>
#include <ILClient.h>

#include "Types.h"
#include "Menu.h"
#include "Main.h"
#include "Globals.h"
#include "MainWindow.h"
#include "TrackList.h"
#include "EventListWindow.h"
#include "PianoRoll.h"
#include "Message.h"
#include "Undo.h"
#include "Clipboard.h"
#include "Consts.h"
#include "Sequence.h"
#include "MidiSetup.h"
#include "Record.h"
#include "Dispatch.h"
#include "EventListMenu.h"
#include "Csound.h"

#include <string.h>
#include <unistd.h>

#include <Yawn.h>
#include <Debug.h>


MIDIHeaderChunk MIDIHeaderBuffer;
MIDIFileHandle	MIDIFile;
char	       *MIDIFileName;
EventList      *MIDITracks;
char	       *MIDItempFileName = NULL;

void Unimplemented(Widget w, XtPointer a, XtPointer b)
{
BEGIN("Unimplemented");

  (void)YQuery(topLevel, "Sorry, that function is not yet implemented.",
		1, 0, 0, "Continue", NULL);
END;
}




Widget Midi_GetWidgetFromPointerPos(void)
{
Window       Root;
Window       Child;
int          RootX, RootY, WinX, WinY;
unsigned int Mask;
Widget	     w;

BEGIN("Midi_GetWidgetFromPointerPos");

	XQueryPointer(display, RootWindowOfScreen(XtScreen(topLevel)), &Root, &Child,
		      &RootX, &RootY, &WinX, &WinY, &Mask);

	w = XtWindowToWidget(display, Child);


RETURN_WIDGET(w);
}

Widget Midi_GetShellWidget(Widget w)
{
BEGIN("Midi_GetShellWidget");

	if (w == RoseLabel) w = Midi_GetWidgetFromPointerPos();

	while(XtParent(w)) w = XtParent(w);

RETURN_WIDGET(w);
}
	

void Midi_RecordCB(Widget w, XtPointer a, XtPointer b)
{
BEGIN("Midi_RecordCB");

	w = Midi_GetShellWidget(w);

	if (w != topLevel) END;

	Midi_SeqReadTrack();

END;
}


void Midi_FileInfoCB(Widget w, XtPointer a, XtPointer b)
{
char   *FileFormat;
char 	InfoBuffer[1024];
int	i, NumEvents;

BEGIN("Midi_FileInfoCB");

	w = Midi_GetShellWidget(w);

	if (w != topLevel) END;

	switch(MIDIHeaderBuffer.Format)
	{
	case MIDI_SINGLE_TRACK_FILE:

		FileFormat = "1: Single Track File.";
		break;

	case MIDI_SIMULTANEOUS_TRACK_FILE:

		FileFormat = "2: Simultaneous Tracks.";
		break;

	case MIDI_SEQUENTIAL_TRACK_FILE:

		FileFormat = "3: Sequential Tracks.";
		break;
	}

	NumEvents = 0;

	for (i = 0; i < MIDIHeaderBuffer.NumTracks; ++i)
	{
		NumEvents += Length(MIDITracks[i]);
	}

	sprintf(InfoBuffer, 
		"%s\n\n	File Format %s\n\nNumber Of Tracks: %d\n\nTimebase: %d\n\nTotal No. of Events: %d\n",
		MIDIFileName, FileFormat, MIDIHeaderBuffer.NumTracks, 
		MIDIHeaderBuffer.Timing.Division, NumEvents);

	YQuery(topLevel, InfoBuffer, 1, 0, 0, "OK", "Sequencer File - Info");

END;
} 



void Midi_ShowClipboardCB(Widget w, XtPointer a, XtPointer b)
{
BEGIN("Midi_ShowClipboardCB");

	Midi_ClipboardShowContents();

END;
}




void Midi_NotateAckCB(IL_ReturnCode Rtn)
{
BEGIN("Midi_NotateAckCB");

	unlink(MIDItempFileName);

	XtFree(MIDItempFileName);
	MIDItempFileName = NULL;

	switch(Rtn)
	{
	case IL_NO_SUCH_SERVICE:

		YQuery(topLevel, "Notation service unavailable.", 1, 0, 0, "Continue", NULL);
		break;

	case IL_SERVICE_BUSY:

		YQuery(topLevel, "Notation service is currently busy.\nPlease try again later.", 
		       1, 0, 0, "Continue", NULL);
		break;

	case IL_SERVICE_FAILED:

		YQuery(topLevel, "Notation editor was unable to parse MIDI file.",
		       1, 0, 0, "Continue", NULL);
		break;

	case IL_SERVICE_OK:
	default:

		break;
	}
	
	Midi_SetBusy(False);
END;
}

	
void Midi_NotateCB(Widget w, XtPointer a, XtPointer b)
{
BEGIN("Midi_NotateCB");

	w = Midi_GetShellWidget(w);

	if (w != topLevel) END;

	if (MIDItempFileName) 
	{
		YQuery(topLevel, "Notation Editor has yet to\nacknowledge previous request.",
		       1, 0, 0, "Continue", NULL);
		END;
	}

	if (!(MIDItempFileName = tmpnam(NULL)))
	{
		YQuery(topLevel, "Sorry, I couldn't get a temporary file name.",
		       1, 0, 0, "Continue", NULL);
		END;
	}
	
	Midi_SaveFile(MIDItempFileName);

	IL_RequestService("Edit", Midi_NotateAckCB, MIDItempFileName, strlen(MIDItempFileName) + 1);

	Midi_SetBusy(True);
END;
}


void Midi_CloseFile(void)
{
int i;
char MsgBuffer[256];
char *SaveFileName;

BEGIN("Midi_CloseFile");
	
	if (MIDIfileModified)
	{
		if (MIDIneverSaved)
		{
			if (!YQuery(topLevel, "Save changes to current file?", 2, 0, 0, "Yes", "No", NULL))
			{
				

				SaveFileName = YFileGetWriteFilename(topLevel, "Sequencer File - Save");

				if (SaveFileName) Midi_SaveFile(SaveFileName);
				else if (MIDIinServitude)
				{
					IL_AcknowledgeRequest("Mseq", IL_SERVICE_BUSY);
					END;
				}
			}
		}
		else
		{
			sprintf(MsgBuffer, "Save changes to '%s'?", MIDIFileName);

			if (!YQuery(topLevel, MsgBuffer,  2, 0, 0, "Yes", "No", NULL))
			{
				Midi_SaveFile(MIDIFileName);
			}
		}
	}

	if (MIDIHeaderBuffer.Format != MIDI_NO_FILE_LOADED)
	{
		for(i = 0; i < MIDIHeaderBuffer.NumTracks; ++i)
		{
			Midi_TrackDelete(MIDITracks[i]);
			MIDITracks[i] = NULL;
		}

		Midi_FileClose(MIDIFile);

	}

	Midi_EventListDeleteAllWindows();
	Midi_PianoRollDeleteAllWindows();

	MIDIHeaderBuffer.Format    = MIDI_NO_FILE_LOADED;
	MIDIHeaderBuffer.NumTracks = 0;

	Midi_TrackListSetup();
	Midi_EnterMenuMode(NoFileLoadedMode);

END;
}


void Midi_CloseCB(Widget w, XtPointer a, XtPointer b)
{
int i;

BEGIN("Midi_CloseCB");


	w = Midi_GetShellWidget(w);

	if (w != topLevel) END;

	Midi_CloseFile();

END;
}

void Midi_LoadFile(char *FileName, Boolean DispMsgs)
{
MIDIHeaderChunk LoadingBuffer;
MIDIFileHandle	NewFile;
short		i;
char		TrackBuff[24];
Cursor		SandsOfTime;

BEGIN("Midi_LoadFile");

	Midi_SetBusy(True);

	if ((NewFile = Midi_FileOpen(FileName, &LoadingBuffer, MIDI_READ)) == NULL)
	{
		Midi_SetBusy(False);
		END;
	}

	MIDIHeaderBuffer = LoadingBuffer;

	MIDIFileName = FileName;

	MIDIFile = NewFile;

	MIDITracks = (EventList *)XtMalloc(MIDIHeaderBuffer.NumTracks * sizeof(EventList));

	for(i = 0; i < MIDIHeaderBuffer.NumTracks; ++i)
	{
		Midi_FileSkipToNextChunk(MIDIFile, MIDI_TRACK_HEADER);

		if (DispMsgs)
		{
			sprintf(TrackBuff, "Reading Track %d of %d", i, MIDIHeaderBuffer.NumTracks - 1);
			Midi_DisplayPermanentMessage(TrackBuff);

			while(XtAppPending(appContext))
			{
				XtAppProcessEvent(appContext, XtIMAll);
			}
		}

		SandsOfTime = HourglassAnimCur[(int)((float)i/MIDIHeaderBuffer.NumTracks * HOUR_FRAMES)];

		XDefineCursor(display, XtWindow(TrackListBox), SandsOfTime);

		MIDITracks[i] = Midi_FileReadTrack(MIDIFile);
		Midi_TrackAggregateDeltas(MIDITracks[i]);
		Midi_TrackConvertToOnePointRepresentation(MIDITracks[i]);
	}

	Midi_TrackListSetup();
	Midi_LeaveMenuMode(NoFileLoadedMode);

	if (DispMsgs)
	{
		Midi_DisplayPermanentMessage(" ");
	}

	Midi_SetBusy(False);
END;
}

void Midi_LoadCB(Widget w, XtPointer a, XtPointer b)
{
String 		FileName;
char		TitleBuffer[256];
char		machine[100];
Arg		arg[2];

BEGIN("Midi_LoadCB");


	w = Midi_GetShellWidget(w);

	if (w != topLevel) END;

	FileName = YFileGetReadFilename(topLevel, "Sequencer File - Open");

	if (!FileName) END;

	if (MIDIHeaderBuffer.Format != MIDI_NO_FILE_LOADED) Midi_CloseFile();

	Midi_LoadFile(FileName, True);
	MIDIneverSaved = False;

	Midi_SetTitleBar();
END;
}


void Midi_SaveFile(char *SaveFileName)
{
short          i;
MIDIFileHandle SaveFile;
EventList      ExpandedTrack;

BEGIN("Midi_SaveFile");

	Midi_SetBusy(True);

	if ((SaveFile = Midi_FileOpen(SaveFileName, &MIDIHeaderBuffer, MIDI_WRITE)) == NULL)
	{
		Midi_SetBusy(False);
		YQuery(topLevel, "Unable to open output file.", 1, 0, 0, "Continue", NULL);
		END;
	}

	for(i = 0; i < MIDIHeaderBuffer.NumTracks; ++i)
	{
		ExpandedTrack = Midi_TrackConvertToTwoPointRepresentation(MIDITracks[i]);
		Midi_FileWriteTrack(SaveFile, ExpandedTrack);
		Midi_TrackDelete(ExpandedTrack);
	}

	Midi_FileClose(SaveFile);
	Midi_SetBusy(False);

END;
}

void Midi_SaveAsCB(Widget w, XtPointer a, XtPointer b)
{
String          SaveFileName;
short           i;
MIDIFileHandle  SaveFile;
EventList	ExpandedTrack;

BEGIN("Midi_SaveAsCB");


	w = Midi_GetShellWidget(w);

	if (w != topLevel) END;

	SaveFileName = YFileGetWriteFilename(topLevel, "Sequencer File - Save");

	if (!SaveFileName) END;
	Midi_SaveFile(SaveFileName);
	XtFree(MIDIFileName);
	MIDIFileName = SaveFileName;
	MIDIneverSaved = False;
	Midi_SetTitleBar();
END;
}

void Midi_SaveCB(Widget w, XtPointer a, XtPointer b)
{
BEGIN("Midi_SaveCB");

	
	w = Midi_GetShellWidget(w);

	if (w != topLevel) END;

	if (MIDIFileName)
	{
		Midi_SaveFile(MIDIFileName);
		MIDIneverSaved = False;
	}
	else Midi_SaveAsCB(w, a, b);

END;
}

void Midi_SetTimebaseCB(Widget w, XtPointer a, XtPointer b)
{
char 	OldTimebase[12];
char   *NewTimebaseStr;
short	NewTimebase;

BEGIN("Midi_SetTimebaseCB");


	w = Midi_GetShellWidget(w);

	if (w != topLevel) END;

	sprintf(OldTimebase, "%hd", MIDIHeaderBuffer.Timing.Division);

	NewTimebaseStr = YGetUserInput(topLevel, "Set Timebase for file to:", 
				       OldTimebase, YOrientHorizontal, "Sequencer File - Set Timebase");

	if (NewTimebaseStr == NULL) END;

	NewTimebase = (short)atoi(NewTimebaseStr);

	MIDIHeaderBuffer.Timing.Division = NewTimebase;

END;
}

void Midi_PlayCB(Widget w, XtPointer a, XtPointer b)
{
BEGIN("Midi_PlayCB");


	w = Midi_GetShellWidget(w);

	if (w != topLevel) END;

	Midi_SeqPlayFile();

END;
}


		
void Midi_UndoCB(Widget w, XtPointer a, XtPointer b)
{
BEGIN("Midi_UndoCB");

	Midi_UndoLastOperation();

END;
}



YMenuElement FileMenu[] =
{
	"Record",	NullMode,		Midi_RecordCB,		NULL,
	"Open",		NullMode,		Midi_LoadCB,		NULL,
	YMenuDivider,
	"Close",	NoFileLoadedMode,	Midi_CloseCB,		NULL,
	"Save",		NoFileLoadedMode,	Midi_SaveCB,	  	NULL,
	"Save As",	NoFileLoadedMode,	Midi_SaveAsCB,		NULL,
	YMenuDivider,
	"File Info",	NoFileLoadedMode,	Midi_FileInfoCB,	NULL,
	"Set Timebase", NoFileLoadedMode,	Midi_SetTimebaseCB,	NULL,
	"Play",		NoFileLoadedMode,	Midi_PlayCB,		NULL,
	YMenuDivider,
	"Exit",		NullMode,		Midi_QuitCB,		NULL
};

YMenuId	FileMenuId;

YMenuElement	EditMenu[] =
{
	"Undo",			NoFileLoadedMode | NothingDoneMode,	Midi_UndoCB,		NULL,
	"Delete",		NoFileLoadedMode | NothingSelectedMode,	Midi_DispatchDeleteCB,	NULL,
	YMenuDivider,
	"Cut",			NoFileLoadedMode | NothingSelectedMode,	Midi_DispatchCutCB,	NULL,
	"Copy",			NoFileLoadedMode | NothingSelectedMode,	Midi_DispatchCopyCB,	NULL,
	"Paste",		NoFileLoadedMode | NothingCutMode,	Midi_DispatchPasteCB,	NULL,
	"Show Clipboard",	NullMode,				Midi_ShowClipboardCB,	NULL
};

YMenuId EditMenuId;



YMenuElement	TrackMenu[] =
{
	"Event List...",     NoFileLoadedMode | NothingSelectedMode, Midi_TrackEventListCB,	  NULL,
	"Piano Roll...",     NoFileLoadedMode | NothingSelectedMode, Midi_TrackPianoRollCB,	  NULL,
	YMenuDivider,
	"Rename",	     NoFileLoadedMode | NothingSelectedMode, Midi_TrackRenameCB,	  NULL,
	"Track Info",	     NoFileLoadedMode | NothingSelectedMode, Midi_TrackInfoCB,		  NULL,
	YMenuDivider,
	"Clone",	     NoFileLoadedMode | NothingSelectedMode, Midi_TrackCloneCB,		  NULL,
	"Merge",	     NoFileLoadedMode | NothingSelectedMode, Midi_TrackMergeCB,		  NULL,
	YMenuDivider,
	"Filter By Channel", NoFileLoadedMode | NothingSelectedMode, Midi_TrackFilterByChannelCB, NULL,
	"Filter By Event",   NoFileLoadedMode | NothingSelectedMode, Midi_TrackFilterByEventCB,   NULL,
	"Filter By Pitch",   NoFileLoadedMode | NothingSelectedMode, Midi_TrackFilterByPitchCB,	  NULL,
	YMenuDivider,
	"Split By Channel",  NoFileLoadedMode | NothingSelectedMode, Unimplemented,		  NULL,
	"Split By Pitch",    NoFileLoadedMode | NothingSelectedMode, Unimplemented,		  NULL,
	"Change Channel",    NoFileLoadedMode | NothingSelectedMode, Midi_TrackChangeChannelCB,	  NULL,
	YMenuDivider,
	"Quantize",	     NoFileLoadedMode,			     Midi_TrackQuantizeCB,	  NULL,
	"Transpose",	     NoFileLoadedMode,			     Midi_TrackTransposeCB,	  NULL,
};

YMenuId TrackMenuId;



YMenuElement	MidiMenu[] =
{
	"MIDI Setup...",	NullMode,	Midi_SetupCB, 	NULL,
	"Set Initial Patches",	NullMode,	Unimplemented,	NULL,
	YMenuDivider,
	"Reset",		NullMode,	Unimplemented,	NULL,
	"System Reset",		NullMode,	Unimplemented,  NULL,
	YMenuDivider,
	"System Exclusive",	NullMode,	Unimplemented,	NULL
};

YMenuId MidiMenuId;



YMenuElement	ToolsMenu[] =
{
	"Notate!",		NoFileLoadedMode,	Midi_NotateCB,	NULL,
	"Generate CSound",	NoFileLoadedMode, 	Midi_2CsoundCB,	NULL,
	"Preferences",		NullMode,		Unimplemented,	NULL
};

YMenuId ToolsMenuId;



void Midi_InstallFileMenu(Widget File)
{
BEGIN("Midi_InstallFileMenu");

	FileMenuId = YCreateMenu(File, "File Menu", XtNumber(FileMenu), FileMenu);

END;
}


void Midi_InstallEditMenu(Widget Edit)
{
BEGIN("Midi_InstallEditMenu");

	EditMenuId = YCreateMenu(Edit, "Edit Menu", XtNumber(EditMenu), EditMenu);

END;
}


	
void Midi_InstallTrackMenu(Widget Track)
{
BEGIN("Midi_InstallTrackMenu");

	TrackMenuId = YCreateMenu(Track, "Track Menu", XtNumber(TrackMenu), TrackMenu);

END;
}


void Midi_InstallMidiMenu(Widget Midi)
{
BEGIN("Midi_InstallMidiMenu");

	MidiMenuId = YCreateMenu(Midi, "Midi Menu", XtNumber(MidiMenu), MidiMenu);

END;
}


void Midi_InstallToolsMenu(Widget Tools)
{
BEGIN("Midi_InstallToolsMenu");

	ToolsMenuId = YCreateMenu(Tools, "Tools Menu", XtNumber(ToolsMenu), ToolsMenu);

END;
}


void Midi_EnterMenuMode(unsigned long MenuMode)
{
BEGIN("Midi_EnterMenuMode");

	YEnterMenuMode(FileMenuId,  MenuMode);
	YEnterMenuMode(EditMenuId,  MenuMode);
	YEnterMenuMode(TrackMenuId, MenuMode);
	YEnterMenuMode(MidiMenuId,  MenuMode);
	YEnterMenuMode(ToolsMenuId, MenuMode);

	Midi_ELAllWindowsEnterMenuMode(MenuMode);
END;
}


void Midi_LeaveMenuMode(unsigned long MenuMode)
{
BEGIN("Midi_LeaveMenuMode");

	YLeaveMenuMode(FileMenuId,  MenuMode);
	YLeaveMenuMode(EditMenuId,  MenuMode);
	YLeaveMenuMode(TrackMenuId, MenuMode);
	YLeaveMenuMode(MidiMenuId,  MenuMode);
	YLeaveMenuMode(ToolsMenuId, MenuMode);

	Midi_ELAllWindowsLeaveMenuMode(MenuMode);
END;
}
