
/* Musical Notation Editor for X, Chris Cannam 1994 */
/* Stave cursor handling code */

#include "General.h"
#include "Classes.h"
#include "GC.h"
#include "Stave.h"
#include "StavePrivate.h"
#include "StaveCursor.h"
#include "StaveEdit.h"
#include "ItemList.h"
#include "Menu.h"

#include <Yawn.h>

static int        cursorMarkedStave = -1;
static List       cursorMarkedBar   = NULL;
static List       cursorExtendedBar = NULL;
static PointerRec cursorMark   = { NULL, 0 };
static PointerRec cursorExtent = { NULL, 0 };
static XPoint     a, b;

int        StaveGetPointedStave (MajorStave, XPoint);
List       StaveGetPointedBar   (MajorStave, XPoint, int);
PointerRec StaveGetPointedItem  (MajorStave, XPoint, int, List);


/* As a matter of convention, we will agree that a pointer-rec will      */
/* contain two 0 fields iff the pointer-location code couldn't find any  */
/* candidate items; if the pointer is off the left end of the stave, and */
/* hence has no `left' item, the `left' field will be NULL (a great loss */
/* to the world's artistic subculture) and the `bound' will be non-NULL, */
/* giving a suitable bound for drawing the cursor.                       */


/* To find the itemlist to the left of the cursor, given the bar in the  */
/* cursor:  scan from the right-hand end of the bar, taking advantage of */
/* the fact that list back-pointers ignore groups.  Stop and return the  */
/* first item whose (x + width) is less than `off'.  If there is no such */
/* item, return the item before the bar item but set `bound' to the left */
/* hand edge of the leftmost (failed) item.                              */

static PointerRec GetBarPointed(StaveEltList barList,
				int staveNo, Position off)
{
  int        x;
  int        bound;
  PointerRec rtn;
  ItemList   list;
  Dimension  wd;

  Begin("GetBarPointed");

  MakeNullRec(rtn);

  list = barList->bars[staveNo]->group.end;
  if (list->item->generic.object_class == GroupClass)
    list = ((Group *)list->item)->group.end;

  if (!list) Return(rtn);
  bound = (barList->bars[staveNo]->item.x +
	   barList->bars[staveNo]->bar.width - (Prev(barList) ? 10 : 5));

  while (list && list != (ItemList)Prev(barList->bars[staveNo]->group.start)) {

    if (list->item->generic.object_class == GroupClass) {
      list = (ItemList)Prev(list);
      continue;
    }

    wd = list->item->item.methods->get_min_width((MusicObject)list->item);

    if (wd == 0) {
      list = (ItemList)Prev(list);
      continue;
    }

    x = list->item->item.x + wd/2 - 2;

    if (x < off) {

      rtn.left = list;
      rtn.bound = bound;

      /* Undo the "-2" compensation on bounds   */
      /* whose leftward neighbour is very close */

      if (list) {

	bound = list->item->item.x + wd + 2;
	if (bound > rtn.bound) rtn.bound = bound;
      }

      Return(rtn);
    }

    bound = list->item->item.x - 2;
    list = (ItemList)Prev(list);
  }

  rtn.left  = (ItemList)Prev(barList->bars[staveNo]->group.start);
  rtn.bound = bound ? bound : 4;

  Return(rtn);
}



PointerRec StaveGetPointedItem(MajorStave sp, XPoint p,
			       int staveNo, List elist)
{
  PointerRec     rtn;
  MajorStaveRec *mstave = (MajorStaveRec *)sp;

  Begin("StaveGetPointedItem");

  MakeNullRec(rtn);

  if (!mstave || !staveLabel || staveNo < 0 ||
      !elist || !(((StaveEltList)elist)->bars[staveNo])) Return(rtn);

  while (TestNullRec(rtn) && elist && ((StaveEltList)elist)->bars[staveNo]) {
    rtn = GetBarPointed((StaveEltList)elist, staveNo, p.x);
    elist = Next(elist);
  }

  Return(rtn);
}



List StaveGetPointedBar(MajorStave sp, XPoint p, int staveNo)
{
  int            i;
  int            accWidth;
  Bar           *bar;
  Dimension      wWidth;
  StaveEltList   elist;
  MajorStaveRec *mstave = (MajorStaveRec *)sp;

  Begin("StaveGetPointedBar");

  if (!mstave || !staveLabel || staveNo < 0) Return(NULL);

  YGetValue(staveLabel, XtNwidth, &wWidth);

  for (elist = mstave->bar_list, i = mstave->display_start;
       elist && Next(elist) && i > 0; elist = (StaveEltList)Next(elist), --i);

  accWidth = 0;
  if (mstave->display_start == -1) {
    for (i = 0; i < mstave->staves; ++i)
      if (mstave->name_lengths[i] + 20 > accWidth)
	accWidth = mstave->name_lengths[i] + 20;
  }

  if (p.x >= accWidth) {

    while(accWidth < wWidth && elist && elist->bars[staveNo] &&
	  Next(elist) && ((StaveEltList)Next(elist))->bars[staveNo] &&
	  (p.x < accWidth || p.x >= accWidth + elist->widths[staveNo])) {
    
      accWidth += elist->widths[staveNo];
      elist = (StaveEltList)Next(elist);
    }
  }

  if (!elist || !elist->bars[staveNo]) Return(NULL);
  else Return((List)elist);
}



int StaveGetPointedStave(MajorStave sp, XPoint p)
{
  int            staveNo;
  Dimension      iHeight;
  Dimension      wHeight;
  MajorStaveRec *mstave = (MajorStaveRec *)sp;
  
  Begin("StaveGetPointedStave");
  if (!mstave || !staveLabel) Return(-1);
  
  YGetValue(staveLabel, XtNinternalHeight, &iHeight);
  YGetValue(staveLabel, XtNheight,         &wHeight);

  p.y = p.y - iHeight - (wHeight - StaveTotalHeight(mstave))/2;

  staveNo = p.y / (StaveTopHeight(1) - StaveUpperGap);

  if (StaveTopHeight(staveNo) - p.y > StaveUpperGap - 20) Return(-1);

  if (staveNo < 0) staveNo = 0;
  if (staveNo >= mstave->staves) staveNo = mstave->staves - 1;

  Return(staveNo);
}


Pitch StaveGetPointedPitch(MajorStave sp, XPoint p, int staveNo, ClefTag clef)
{
  Pitch          rtn;
  int            y = p.y;
  Dimension      iHeight;
  Dimension      wHeight;
  MajorStaveRec *mstave = (MajorStaveRec *)sp;

  Begin("StaveGetPointedPitch");
  if (!mstave || !staveLabel) Return(0);

  YGetValue(staveLabel, XtNinternalHeight, &iHeight);
  YGetValue(staveLabel, XtNheight,         &wHeight);

  y = y - iHeight - (wHeight - StaveTotalHeight(mstave)) / 2;
  y = y - StaveTopHeight(staveNo);

  if (y > StaveHeight + NoteHeight/4) {

    if (y > StaveHeight + StaveLowerGap - NoteHeight/2)
      y   = StaveHeight + StaveLowerGap - NoteHeight/2;

    y = y - StaveHeight - NoteHeight/2;

    rtn = y / (NoteHeight + 1);
    rtn = 2*rtn + ((rtn == (y + NoteHeight/2) / (NoteHeight + 1)) ? 0 : 1);
    rtn = -1 - rtn;

  } else {

    if (y < -(StaveUpperGap-NoteHeight/2)) y = -(StaveUpperGap-NoteHeight/2);
    y = StaveHeight + NoteHeight/2 - y - NoteHeight/2 + 1;
    
    rtn = y / (NoteHeight + 1);
    rtn = 2*rtn + ((rtn == (y + NoteHeight/2) / (NoteHeight + 1)) ? 0 : 1);
  }

  rtn -= ClefPitchOffset(clef);

  Return(rtn);
}


/* This has to be a bit weird, because the xorGC will normally only */
/* work on mono displays.  In colour, we're using (essentially) the */
/* copyGC, so we want to avoid as far as possible drawing on top of */
/* other things on the screen (which would be rubbed out when we    */
/* removed the cursor).  So we have completely different cursor     */
/* designs in mono and colour.  Nasty, but it'll have to do for the */
/* moment.                                                          */

static void StavePlotRectangle(void)
{
  Begin("StavePlotRectangle");

  if (oneD) {

    if (a.x == b.x) {

      XDrawLine(display, XtWindow(staveLabel), xorGC,
		a.x - NoteWidth/2, a.y + 20, a.x + NoteWidth/2, a.y + 20);

      XDrawLine(display, XtWindow(staveLabel), xorGC,
		a.x, a.y + 20, b.x, b.y - 23);

      XDrawLine(display, XtWindow(staveLabel), xorGC,
		b.x - NoteWidth/2, b.y - 23, b.x + NoteWidth/2, b.y - 23);

    } else {
    
      /* multiple XDrawLine calls seem to interact better */
      /* when XORing than a single call to XDrawRectangle */

      XDrawLine(display, XtWindow(staveLabel), xorGC, a.x, a.y, b.x, a.y);
      XDrawLine(display, XtWindow(staveLabel), xorGC, b.x, a.y, b.x, b.y);
      XDrawLine(display, XtWindow(staveLabel), xorGC, b.x, b.y, a.x, b.y);
      XDrawLine(display, XtWindow(staveLabel), xorGC, a.x, b.y, a.x, a.y);
    }
  } else {

    if (a.x == b.x) {

      XDrawLine(display, XtWindow(staveLabel), copyGC,
		a.x - NoteWidth/2, a.y + 20 - NoteWidth/2, a.x, a.y + 20);

      XDrawLine(display, XtWindow(staveLabel), copyGC,
		a.x, a.y + 20, a.x + NoteWidth/2, a.y + 20 - NoteWidth/2);

      XDrawLine(display, XtWindow(staveLabel), copyGC,
		b.x - NoteWidth/2, b.y - 23 + NoteWidth/2, b.x, b.y - 23);

      XDrawLine(display, XtWindow(staveLabel), copyGC,
		b.x, b.y - 23, b.x + NoteWidth/2, b.y - 23 + NoteWidth/2);

    } else {
      
      XDrawLine(display, XtWindow(staveLabel), copyGC,
		a.x, a.y + NoteWidth/2, a.x, a.y);

      XDrawLine(display, XtWindow(staveLabel), copyGC,
		a.x, a.y, b.x, a.y);

      XDrawLine(display, XtWindow(staveLabel), copyGC,
		b.x, a.y + NoteWidth/2, b.x, a.y);
      
      XDrawLine(display, XtWindow(staveLabel), copyGC,
		a.x, b.y - NoteWidth/2, a.x, b.y);

      XDrawLine(display, XtWindow(staveLabel), copyGC,
		a.x, b.y, b.x, b.y);

      XDrawLine(display, XtWindow(staveLabel), copyGC,
		b.x, b.y - NoteWidth/2, b.x, b.y);
    }
  }

  End;
}


static void StaveUnplotRectangle(void)
{
  GC saveGC;
  
  Begin("StaveUnplotRectangle");

  if (oneD) StavePlotRectangle();
  else {

    saveGC = copyGC;
    copyGC = clearGC;

    StavePlotRectangle();

    copyGC = saveGC;
  }

  End;
}


static void StaveCursorDrawRectangle(MajorStave sp, XPoint p)
{
  Dimension      iHeight;
  Dimension      wHeight;
  ItemList       prev;
  MajorStaveRec *mstave = (MajorStaveRec *)sp;
  
  Begin("StaveCursorDrawRectangle");

  if (!mstave || !staveLabel) End;
  
  YGetValue(staveLabel, XtNinternalHeight, &iHeight);
  YGetValue(staveLabel, XtNheight,         &wHeight);

  if (TestNullRec(cursorMark)) {
    MakeNullRec(cursorExtent);
    End;
  }

  cursorExtent =
    StaveGetPointedItem(sp, p, cursorMarkedStave,
			StaveGetPointedBar(sp, p, cursorMarkedStave));
  if (TestNullRec(cursorExtent)) End;

  a.y = b.y =
    iHeight + (wHeight - StaveTotalHeight(mstave))/2 -
      (StaveUpperGap + StaveLowerGap)/2;

  a.y += StaveTopHeight(cursorMarkedStave)  + 15;
  b.y += StaveTopHeight(cursorMarkedStave+1) - 5;
  a.x  = cursorMark.bound - 3;

  if (cursorMark.left == cursorExtent.left) b.x = a.x;
  else b.x = cursorExtent.bound - 3;

  StavePlotRectangle();

  End;
}


void StaveRemoveRectangle(void)
{
  Begin("StaveRemoveRectangle");

  if (TestNullRec(cursorExtent)) End;
  MakeNullRec(cursorExtent);

  StaveUnplotRectangle();

  End;
}


void StaveCursorDrawX(XPoint p)
{
  Begin("StaveCursorDrawX");

  XDrawLine(display, XtWindow(staveLabel), xorGC,
	    p.x - 8, p.y, p.x - 8, p.y + NoteHeight-1);

  XDrawLine(display, XtWindow(staveLabel), xorGC,
	    p.x + 9, p.y, p.x + 9, p.y + NoteHeight-1);

  End;
}


void StaveCursorMark(MajorStave sp, XPoint p)
{
  MajorStaveRec *mstave = (MajorStaveRec *)sp;

  Begin("StaveCursorMark");

  if (!TestNullRec(cursorExtent)) StaveRemoveRectangle();
  ((MajorStaveRec *)sp)->sweep.swept = False;

  cursorMarkedStave = StaveGetPointedStave(sp, p);
  cursorMarkedBar   = StaveGetPointedBar(sp, p, cursorMarkedStave);
  cursorMark        =
    StaveGetPointedItem(sp, p, cursorMarkedStave, cursorMarkedBar);

  if (TestNullRec(cursorMark)) {

    XBell(display, 70);
    mstave->sweep.swept = False;

    LeaveMenuMode(CursorPlacedMode | AreaSweptMode | MultipleItemsSweptMode);
    EnterMenuMode(CursorNotPlacedMode | NoAreaSweptMode);
  }
  
  End;
}


void StaveCursorExtend(MajorStave sp, XPoint p)
{
  Begin("StaveCursorExtend");

  if (!TestNullRec(cursorExtent)) StaveRemoveRectangle();
  StaveCursorDrawRectangle(sp, p);

  End;
}


void StaveCursorExpose(MajorStave sp)
{
  Begin("StaveCursorNoteExposure");

  if (staveMoved) {

    MakeNullRec(cursorExtent);
    ((MajorStaveRec *)sp)->sweep.swept = False;

    LeaveMenuMode(   AreaSweptMode | CursorPlacedMode    );
    EnterMenuMode( NoAreaSweptMode | CursorNotPlacedMode );

  } else {

    if (!TestNullRec(cursorExtent)) StavePlotRectangle();
  }

  End;
}


void StaveCursorRemove(MajorStave sp)
{
  Begin("StaveCursorRemove");

  if (stave && staveLabel) {

    staveMoved = True;
    StaveForceUpdate();
  }

  End;
}


void StaveCursorFinish(MajorStave sp, XPoint p)
{
  MajorStaveRec *mstave = (MajorStaveRec *)sp;

  Begin("StaveCursorFinish");

  if (!TestNullRec(cursorMark)) {

    StaveCursorExtend(sp, p);

    mstave->sweep.swept   = True;
    mstave->sweep.stave   = cursorMarkedStave;
    cursorExtendedBar     = StaveGetPointedBar(sp, p, cursorMarkedStave);

    if (cursorMark.bound < cursorExtent.bound) {

      mstave->sweep.from = cursorMark;
      mstave->sweep.to   = cursorExtent;
      mstave->sweep.a    = a;
      mstave->sweep.b    = b;
      mstave->sweep.first_bar = ((StaveEltList)cursorMarkedBar)
	->bars[mstave->sweep.stave]->bar.number;

    } else {

      mstave->sweep.from = cursorExtent;
      mstave->sweep.to   = cursorMark;
      mstave->sweep.a    = b;
      mstave->sweep.b    = a;
      mstave->sweep.first_bar = ((StaveEltList)cursorExtendedBar)
	->bars[mstave->sweep.stave]->bar.number;
    }
  }

  if (a.x == b.x) {
    LeaveMenuMode( CursorNotPlacedMode |   AreaSweptMode );
    EnterMenuMode( CursorPlacedMode    | NoAreaSweptMode );
  } else {
    LeaveMenuMode( NoAreaSweptMode | CursorPlacedMode    );
    EnterMenuMode(   AreaSweptMode | CursorNotPlacedMode );
  }

  if (mstave->sweep.to.left &&
      ((ItemList)Prev(mstave->sweep.to.left) == mstave->sweep.from.left)) {

    LeaveMenuMode(MultipleItemsSweptMode);

  } else {

    EnterMenuMode(MultipleItemsSweptMode);
  }

  MakeNullRec(cursorMark);
  staveMoved = False;

  End;
}


