
/* Musical Notation Editor for X, Chris Cannam 1994 */
/* MajorStave constructors, methods &c.             */

#include <stdio.h>

#include "General.h"
#include "Tags.h"
#include "Notes.h"
#include "Classes.h"
#include "GC.h"
#include "Palette.h"
#include "Visuals.h"
#include "Widgets.h"
#include "Stave.h"
#include "Yawn.h"
#include "Format.h"
#include "StaveEdit.h"
#include "StavePrivate.h"
#include "ItemList.h"

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Scrollbar.h>

Boolean       staveMoved     = False;
Widget        staveLabel     = NULL;
static Widget staveScrollbar = NULL;

static void StaveFormatBars(MajorStave, int, int); /* stave, end bar number */
static void StaveScrollbarSet(Widget);
static void StaveScrollbarSetTop(Widget);

MajorStave   stave = NULL;
static Key * initialKey = NULL;





StaveEltList NewStaveEltList(Bar **bars, BarTag *starts,
			     BarTag *ends, Dimension *widths)
{
  StaveEltList list;

  Begin("NewStaveEltList");

  list = (StaveEltList)NewList(sizeof(StaveEltListElement));
  list->bars   = bars;
  list->starts = starts;
  list->ends   = ends;
  list->widths = widths;

  Return(list);
}



StaveSweep NewSweep(void)
{
  StaveSweep sweep;

  Begin("NewSweep");
 
  sweep.swept      = False;
  sweep.stave      = 0;
  sweep.from.left  = NULL;
  sweep.from.bound = 0;
  sweep.to.left    = NULL;
  sweep.to.bound   = 0;
  sweep.a.x        = 0;
  sweep.a.y        = 0;
  sweep.b.x        = 0;
  sweep.b.y        = 0;

  Return(sweep);
}



StaveEltList AddBarArray(StaveEltList list, int staves)
{
  int          i;
  StaveEltList newlist;

  Begin("AddBarArray");

  newlist = (StaveEltList)
    Last(Nconc(list,
	       NewStaveEltList
	       ((Bar **)XtMalloc(staves * sizeof(Bar *)),
		(BarTag *)XtMalloc(staves * sizeof(BarTag)),
		(BarTag *)XtMalloc(staves * sizeof(BarTag)),
		(Dimension *)XtMalloc(staves * sizeof(Dimension)))));

  for (i = 0; i < staves; ++i) {
    newlist->widths[i] = 0; newlist->bars[i] = NULL; 
    newlist->starts[i] =    newlist->ends[i] = SingleBar;
  }

  Return(newlist);
}




void StaveInitialise(Widget parent)
{
  Begin("StaveInitialise");

  staveLabel = YCreateWidget("Music", labelWidgetClass, parent);

  YSetValue(staveLabel, XtNinternalHeight, 0);
  YSetValue(staveLabel, XtNinternalWidth,  0);
  YSetValue(staveLabel, XtNborderWidth,    0);
  YSetValue(staveLabel, XtNlabel,       "  ");

  StaveEditInitialise(staveLabel);

  End;
}


/* Call before exiting, but after all staves */
/* have been unmapped and destroyed.  Can be */
/* safely called before StaveInitialise...   */

void StaveCleanUp(void)
{
  Pixmap bitmap = 0;

  Begin("StaveCleanUp");

  if (!display) End;

  if (staveLabel) {
    YGetValue(staveLabel, XtNbitmap, &bitmap);
    XtDestroyWidget(staveLabel);
  }

  if (bitmap) XFreePixmap(display, bitmap);

  End;
}


/* This function SHARES the `music' argument you give it, and   */
/* sets all list pointers in the music array back to the starts */
/* of their lists.  You'd be wise to create a new array for     */
/* each new stave, and allow the stave to get on with it.  (Not */
/* forgetting to free the array when you destroy the stave.)    */

MajorStave NewStave(int staves, ItemList *music)
{
  MajorStaveRec *mstave;
  int            i;

  Begin("NewStave");

  if (staves < 1) Error("NewStave called for empty stave list");

  mstave = (MajorStaveRec *)XtMalloc(sizeof(MajorStaveRec));

  mstave->staves  = staves;
  mstave->formats =
    (StaveFormattingRec *)XtMalloc(staves * sizeof(StaveFormattingRec));

  mstave->display_start  = 0;
  mstave->display_number = 0;
  mstave->bars           = 1;
  mstave->bar_list       = AddBarArray(NULL, staves);
  mstave->sweep          = NewSweep();
  mstave->names          = (String    *)XtCalloc(staves, sizeof(String));
  mstave->name_lengths   = (int *)XtMalloc(staves * sizeof(int));
  mstave->visible        = False;
  mstave->music          = music;

  for (i = 0; i < staves; ++i) {

    mstave->music[i]         = (ItemList)First(mstave->music[i]);
    mstave->formats[i].items = mstave->music[i];
    mstave->formats[i].tied  = False;
    mstave->formats[i].key   = NULL;
    mstave->formats[i].time  = NULL;
    mstave->formats[i].clef  = NULL;
    mstave->formats[i].next  = 0;
    mstave->name_lengths[i]  = -1;

    mstave->names[i]         = (String)XtMalloc(10);
    sprintf(mstave->names[i], "Stave %d", i);

    StaveFormatBars((MajorStave)mstave, i, -1);
  }

  Return((MajorStave)mstave);
}



/* This is a cop-out.  It ensures that the stave is reformatted   */
/* after a change by setting the markers for "next bar to format" */
/* to the very first bar.  This is pretty inefficient, & doesn't  */
/* exactly make much use of the capacity for incremental format   */
/* that's provided in Stave.c.                                    */

void StaveResetFormatting(MajorStave sp, int staveNo)
{
  MajorStaveRec *mstave = (MajorStaveRec *)sp;

  Begin("StaveResetFormatting");

  mstave->formats[staveNo].next  = 0;
  mstave->formats[staveNo].time  = NULL;
  mstave->formats[staveNo].key   = NULL;
  mstave->formats[staveNo].clef  = NULL;
  mstave->formats[staveNo].items = mstave->music[staveNo];

  End;
}



/* Go through a stave, sorting its items into bars, starting at bar       */
/* number `barNo'.  First we throw away all bars from `barNo' onwards,    */
/* and then we start at the point on the item-list where `barNo' pointed, */
/* in each staff, and use the Bar's Format method to track down from      */
/* there, creating new StaveEltList bar structures pointing to the items  */
/* down the list.  If we run out of list in one staff before the others,  */
/* we should fill that staff up with NULL bars.                           */


static void StaveFormatBars(MajorStave sp, int staveNo, int endBar)
          				    /* -1 in endBar for "all of it" */
{
  int                 i;
  int                 bars, barNo;     /* should be longs, really */
  ItemList            items;
  StaveEltList        list;
  MajorStaveRec      *mstave = (MajorStaveRec *)sp;
  StaveFormattingRec *frec = &(mstave->formats[staveNo]);
  TimeSignature      *time = NULL;
  Clef               *clef = NULL;
  Key                *key  = NULL;

  Begin("StaveFormatBars");

  if (!initialKey) initialKey = NewKey(NULL, KeyC);

  barNo = frec->next;
  if (barNo >= (bars = mstave->bars)) End;

  for (list = (StaveEltList)First(mstave->bar_list), i = 0;
       i < barNo && list && list->bars[staveNo];
       i ++, list = (StaveEltList)Next(list)) {

    time = list->bars[staveNo]->bar.time;
    clef = list->bars[staveNo]->bar.clef;
    key  = list->bars[staveNo]->bar.key;
  }

  barNo = i;

  if (!frec->clef) frec->clef = clef;
  if (!frec->time) frec->time = time;
  if (!frec->key)  frec->key  = key ? key : initialKey;

  for (items = mstave->music[staveNo];
       items && (!(frec->time) || !(frec->clef));
       items = (ItemList)Next(items)) {

    if (!frec->clef && items->item->generic.object_class==ClefClass)
      frec->clef = (Clef *)(items->item);

    if (!frec->time && items->item->generic.object_class==TimeSignatureClass)
      frec->time = (TimeSignature *)(items->item);
  }

  if (!frec->time) frec->time = &defaultTimeSignature;
  if (!frec->clef) frec->clef = &defaultClef;

  items = frec->items;
  while(items && (endBar == -1 || barNo <= endBar)) {

    if (!(list->bars[staveNo]))
      list->bars[staveNo] = NewBar(NULL, (unsigned long)barNo,
				   SingleBar, SingleBar);
/*
    fprintf(stderr,"Formatting bar %d in stave %d, address 0x%p\n",
	    list->bars[staveNo]->bar.number, staveNo, list->bars[staveNo]);

    fprintf(stderr,"List entry 0x%p, previous 0x%p and next 0x%p\n",
	    list, Prev(list), Next(list));
*/
    items = FormatBar(list->bars[staveNo],
		      frec->tied, &(frec->tied), items,
		      frec->time, &(frec->time),
		      frec->clef, &(frec->clef),
		      frec->key,  &(frec->key));
    barNo ++;

#ifdef DEBUG
    fprintf(stderr,"\n\nFormatted bar no %d, items are\n\n",
	    list->bars[staveNo]->bar.number);

    PrintItemList(list->bars[staveNo]->group.start);
#endif
    
    if (!Next(list) || barNo >= bars) {
      list = AddBarArray(list, mstave->staves);
      bars ++; mstave->bars ++;
    } else list = (StaveEltList)Next(list);
  }

  frec->next  = barNo;
  frec->items = items;

  while(list) {

    if (list->bars[staveNo]) {
      XtFree((void *)(list->bars[staveNo]));
      list->bars[staveNo] = NULL;
    }
    list = (StaveEltList)Next(list);
  }

  End;
}


void StaveForceUpdate(void)
{
  static XExposeEvent event =
    { Expose, 0L, True, NULL, NULL, 0, 0, 1, 1000, 0 };

  Begin("StaveForceUpdate");

  if (XtIsRealized(staveLabel) && XtIsManaged(staveLabel)) {

    event.display = XtDisplay(staveLabel);
    event.window  =  XtWindow(staveLabel);

    XSendEvent(display, XtWindow(staveLabel),
	       False, ExposureMask, (XEvent *)&event);
  }

  End;
}


static Drawable StaveGetLabelBitmap(MajorStave sp,
				  Dimension *wd, Dimension *ht)
{
  Pixmap           bitmap;
  Dimension        width;
  static Dimension prevHeight = 0;
  static Dimension totalWidth = 0;
  MajorStaveRec  * mstave = (MajorStaveRec *)sp;

  Begin("StaveGetLabelBitmap");

  YGetValue(staveLabel, XtNwidth,  &width);
  YGetValue(staveLabel, XtNbitmap, &bitmap);

  if (!bitmap || width != totalWidth ||
      StaveTotalHeight(mstave) != prevHeight) {

    totalWidth = width;
    prevHeight = StaveTotalHeight(mstave);
    if (bitmap) XFreePixmap(display, bitmap);

    bitmap =
      XCreatePixmap(XtDisplay(staveLabel),
		    RootWindowOfScreen(XtScreen(staveLabel)),
		    totalWidth, prevHeight,
		    DefaultDepthOfScreen(XtScreen(staveLabel)));

    YSetValue(staveLabel, XtNbitmap, bitmap);
  }

  XFillRectangle(display, bitmap, clearGC, 0, 0, totalWidth, prevHeight);

  if (wd) *wd = totalWidth;
  if (ht) *ht = prevHeight;

  Return(bitmap);
}


Dimension StaveDrawNames(MajorStave sp, Drawable drawable)
{
  int              i;
  MajorStaveRec  * mstave = (MajorStaveRec *)sp;
  Dimension        maxWidth = 0;
  int              asc, dsc, dir;
  XCharStruct      info;
  XGCValues        values;
  Boolean          haveValues = False;

  Begin("StaveDrawNames");

  for (i = 0; i < mstave->staves; ++i) {

    if (mstave->name_lengths[i] == -1) {

      if (!haveValues) {
	XGetGCValues(display, bigTextGC, GCFont, &values);
	haveValues = True;
      }

      if (mstave->names[i][0]) {

	XQueryTextExtents(display, values.font, mstave->names[i],
			  strlen(mstave->names[i]), &dir, &asc, &dsc, &info);

	mstave->name_lengths[i] = info.width;

      } else mstave->name_lengths[i] = 0;
    }

    if (mstave->name_lengths[i] > maxWidth)
      maxWidth = mstave->name_lengths[i];
  }

  for (i = 0; i < mstave->staves; ++i) {

    XDrawString(display, drawable, bigTextGC,
		10 + (maxWidth - mstave->name_lengths[i]),
		StaveUpperGap + StaveHeight/2 + 4 +
		i * (StaveUpperGap + StaveHeight + StaveLowerGap),
		mstave->names[i], strlen(mstave->names[i]));

    XDrawLine(display, drawable, drawingGC,
	      maxWidth + 19,
	      StaveUpperGap + i*(StaveUpperGap+ StaveHeight + StaveLowerGap),
	      maxWidth + 19, StaveHeight - 1 +
	      StaveUpperGap + i*(StaveUpperGap+ StaveHeight + StaveLowerGap));
  }

  Return(maxWidth + 20);
}


void StaveDrawSmallNames(MajorStave sp, Drawable drawable)
{
  int             i;
  MajorStaveRec * mstave = (MajorStaveRec *)sp;

  Begin("StaveDrawSmallNames");

  for (i = 0; i < mstave->staves; ++i) {

    XDrawString(display, drawable, tinyTextGC, 8,
		StaveUpperGap + StaveHeight + StaveLowerGap + 
		i * (StaveUpperGap + StaveHeight + StaveLowerGap),
		mstave->names[i], strlen(mstave->names[i]));
  }
  
  End;
}


void StaveRefresh(MajorStave sp, int barNo)    /* -1 is "start", with names */
{
  int              i;
  MajorStaveRec  * mstave = (MajorStaveRec *)sp;
  StaveEltList     barList = (StaveEltList)First(mstave->bar_list);
  Dimension        minWidth;
  Dimension        barWidth;
  Dimension        tempWidth;
  Dimension        incWidth = 0;
  Boolean          countedThisBar;
  Drawable         drawable;
  Bar            * bar;
  double           tempd;
  int              staveNo;
  int              y;
  int              bars = mstave->bars;
  Dimension        totalWidth = 0;
  
  Begin("StaveRefresh");

  if (bars == 0) Error("Cannot refresh zero-bar stave");

  mstave->display_start  = barNo;
  mstave->display_number = 0;
  staveMoved = True;

  drawable = (Drawable)StaveGetLabelBitmap(sp, &totalWidth, NULL);
  if (barNo == -1) { incWidth += StaveDrawNames(sp, drawable); barNo = 0; }
  else StaveDrawSmallNames(sp, drawable);

  for (i = 0; barList && Next(barList) && (i < barNo); ++ i)
    barList = (StaveEltList)Next(barList);

  while (incWidth < totalWidth) {

    barWidth = 0;
    minWidth = 10000;

    if (barList) {

      for (staveNo = 0; staveNo < mstave->staves; ++staveNo) {

	if (mstave->formats[staveNo].next <= barNo)
	  StaveFormatBars(sp, staveNo, barNo);

	if (barList->bars[staveNo]) {

	  bar = barList->bars[staveNo];
	  bar->bar.start_bar = barList->starts[staveNo];
	  bar->bar.end_bar = barList->ends[staveNo];

	  tempWidth = bar->item.methods->get_min_width((MusicObject)bar);

	  if (tempWidth < minWidth) minWidth = tempWidth;
	  if (tempWidth > barWidth) barWidth = tempWidth;
	}
      }
    }

    /* I reckon that a bar width of max * 0.6 * (1 + 1 / ((max/min)^3))
       should give reasonable results. */
/*
    if (barWidth > minWidth) {

      tempd = ((double)barWidth) / ((double)minWidth);
      tempWidth = barWidth;

      barWidth = (Dimension)
	((double)barWidth * 0.6 * (1.0 + 1.0 / (tempd * tempd * tempd)));
      
      if (barWidth > tempWidth) barWidth = tempWidth;
    }
*/
    /* No bar should be wider than the window.  In fact we force  */
    /* them to be a bit narrower, just so the user knows that the */
    /* whole bar is visible and none has disappeared off the end. */

    if (barWidth > totalWidth - 10) barWidth = totalWidth - 10;
    if (barWidth <= 0) barWidth = ClefWidth + NoteWidth;

    countedThisBar = False;

    for (staveNo = 0; staveNo < mstave->staves; ++staveNo) {

      if (barList && barList->bars[staveNo]) {

	bar = barList->bars[staveNo];

	(void)bar->item.methods->draw
	  ((MusicObject)bar, drawable, incWidth, StaveTopHeight(staveNo),
	   ClefPitchOffset(bar->bar.clef->clef.clef), barWidth, NULL);

	/* tie together section-ending bars in all staffs */

	if (staveNo < mstave->staves-1) {

	  if (barList->starts[staveNo] != SingleBar)
	    XDrawLine(display, drawable, dashedGC, incWidth,
		      StaveTopHeight(staveNo) + StaveHeight + 2, incWidth,
		      StaveTopHeight(staveNo + 1));

	  if (barList->ends[staveNo] != SingleBar)
	    XDrawLine(display, drawable, dashedGC, incWidth + barWidth - 1,
		      StaveTopHeight(staveNo) + StaveHeight,
		      incWidth + barWidth - 1, StaveTopHeight(staveNo + 1));

	  if (barNo == 0)
	    XDrawLine(display, drawable, dashedGC, incWidth - 1,
		      StaveTopHeight(staveNo) + StaveHeight, incWidth - 1,
		      StaveTopHeight(staveNo + 1));
	}

	bar->bar.width = barWidth;

	if (!countedThisBar) {
	  mstave->display_number ++;
	  countedThisBar = True;
	}
      }

      if (barList) barList->widths[staveNo] = barWidth;
    }

    if (barList) barList = (StaveEltList)Next(barList);
    incWidth += barWidth;
    ++ barNo;
  }

  mstave->visible = True;
  StaveScrollbarSet(staveScrollbar);
  StaveForceUpdate();

  End;
}
 

void StaveRefreshAsDisplayed(MajorStave sp)
{
  Begin("StaveRefreshAsDisplayed");

  StaveRefresh(sp, ((MajorStaveRec *)sp)->display_start);

  End;
}


static void StaveRefreshProportion(float distance) /* 0.0 left -> 1.0 right */
{
  MajorStaveRec *mstave = (MajorStaveRec *)stave;
  int            barNo  = (int)((float)(mstave->bars) * distance);

  Begin("StaveRefreshProportion");

  if (barNo >= mstave->bars) barNo = mstave->bars - 1;
  if (mstave->display_start != barNo - 1) StaveRefresh(stave, barNo - 1);

  End;
}



void StaveUnmap(MajorStave sp)
{
  int            i;
  Pixmap         bitmap;
  StaveEltList   list;
  MajorStaveRec *mstave = (MajorStaveRec *)sp;

  Begin("StaveUnmap");

  for (list = (StaveEltList)First(mstave->bar_list);
       list; list = (StaveEltList)Next(list)) {

    for (i = 0; i < mstave->staves; ++i)
      if (list->bars[i]) list->bars[i]->bar.still_as_drawn = False;
  }

  YGetValue(staveLabel, XtNbitmap, &bitmap);
  if (bitmap) XFreePixmap(display, bitmap);
  YSetValue(staveLabel, XtNbitmap, NULL);
  StaveForceUpdate();

  End;
}


void StaveRenameStave(MajorStave sp, int staveNo, String name)
{
  MajorStaveRec *mstave = (MajorStaveRec *)sp;
  
  Begin("StaveRenameStave");

  if (mstave->names[staveNo]) XtFree(mstave->names[staveNo]);

  mstave->names[staveNo] = XtNewString(name);
  mstave->name_lengths[staveNo] = -1;

  if (mstave->visible) StaveRefresh(sp, -1);
  
  End;
}


/* if freeP is True, the music fields will be razed and the        */
/* stave structure's own memory set free.  We have the technology. */

void StaveDestroy(MajorStave sp, Boolean freeP)
{
  MajorStaveRec *mstave = (MajorStaveRec *)sp;
  StaveEltList   list;
  int            i;

  Begin("StaveDestroy");

  for (list = (StaveEltList)First(mstave->bar_list); list;
       list = (StaveEltList)Next(list)) {

    for (i = 0; i < mstave->staves; ++i)
      if (list->bars[i]) DestroyBar((MusicObject)(list->bars[i]));

    XtFree((void *)(list->widths));
    XtFree((void *)(list->starts));
    XtFree((void *)(list->ends));
    XtFree((void *)(list->bars));
  }

  DestroyList(mstave->bar_list);

  for (i = 0; i < mstave->staves; ++i)
    if (mstave->names[i]) XtFree(mstave->names[i]);

  XtFree((void *)(mstave->formats));
  if (freeP) {
    for (i = 0; i < mstave->staves; ++i) DestroyItemList(mstave->music[i]);
    XtFree((void *)(mstave));
  }

  End;
}


void StaveSetBarStyle(MajorStave sp, int barNo, Boolean start, BarTag tag)
{
  int            i;
  StaveEltList   j;
  MajorStaveRec *mstave = (MajorStaveRec *)sp;

  Begin("StaveSetBarStyle");

  for (j = (StaveEltList)First(mstave->bar_list), i = barNo; j && i;
       j = (StaveEltList)Next(j), --i);

  if (!j) End;

  for (i = 0; i < mstave->staves; ++i)

    if (j->bars[i]) {

      if (start) j->starts[i] = tag;
      else       j->  ends[i] = tag;
    }

  End;
}



void StaveWriteToFile(MajorStave sp, FILE *file)
{
  int            n, m;
  ItemList       i;
  StaveEltList   j;
  MajorStaveRec *mstave = (MajorStaveRec *)sp;

  Begin("StaveWriteToFile");

  StaveBusyStartCount(mstave->staves*2 + 1);
  fprintf(file, "\nStaves %d\n\n", mstave->staves);

  for (n = 0; n < mstave->staves; ++n) {

    StaveBusyMakeCount(n*2);
    UnformatItemList((ItemList *)&mstave->music[n], NULL);
    StaveResetFormatting(sp, n);
    StaveBusyMakeCount(n*2 + 1);

    m = 0;
    for (ItemList_ITERATE(i, mstave->music[n])) ++m;

    fprintf(file, "\nName %s\nItems %d\n",
	    mstave->names[n] ? mstave->names[n] : "<none>", m);

    for (ItemList_ITERATE(i, mstave->music[n]))
      i->item->item.methods->write((MusicObject)(i->item), file, 0);
  }

  StaveBusyMakeCount(mstave->staves * 2);

  for (j = (StaveEltList)First(mstave->bar_list), n = 0; j;
       j = (StaveEltList)Next(j), ++n)

    if (j->bars[0]) {

      if (j->starts[0] != SingleBar)
	fprintf(file, "\nBar %d starts with %s",
		j->bars[0]->bar.number,
		j->starts[0] == DoubleBar ? "Double" : "Repeat");

      if (j->ends[0] != SingleBar)
	fprintf(file, "\nBar %d ends with %s",
		j->bars[0]->bar.number,
		j->ends[0] == DoubleBar ? "Double" : "Repeat");
    }

  StaveBusyFinishCount();

  End;
}


static void StaveScrollbarSetTop(Widget scrollbar)
{
  Arg            arg;
  float          top;
  XtArgVal      *ld;
  MajorStaveRec *mstave = (MajorStaveRec *)stave;

  Begin("StaveScrollbarSetTop");

  top = ((float)(mstave->display_start + 1))/((float)(mstave->bars));

  if (sizeof(float) > sizeof(XtArgVal)) XtSetArg(arg, XtNtopOfThumb, &top);
  else {
    ld = (XtArgVal *)&top;
    XtSetArg(arg, XtNtopOfThumb, *ld);
  }

  XtSetValues(scrollbar, &arg, 1);

  End;
}


static void StaveScrollbarSet(Widget scrollbar)
{
  Arg            arg;
  static char    label[] = "Bar 0000";
  static Arg     barg = { XtNlabel, (XtArgVal)label };
  float          shown;
  XtArgVal      *ld;
  MajorStaveRec *mstave  = (MajorStaveRec *)stave;

  Begin("StaveScrollbarSet");

  shown = ((float)(mstave->display_number))/((float)(mstave->bars));

  if (sizeof(float) > sizeof(XtArgVal)) XtSetArg(arg, XtNshown, &shown);
  else {
    ld = (XtArgVal *)&shown;
    XtSetArg(arg, XtNshown, *ld);
  }

  sprintf(label + 4, "%d", mstave->display_start + 1);

  XtSetValues(scrollbar,   &arg, 1);
  XtSetValues(pageButton, &barg, 1);

  XtSetSensitive(pageButton, True);

  End;
}


void StaveLeftCallback(Widget w, XtPointer client, XtPointer call)
{
  int            barNo;
  MajorStaveRec *mstave = (MajorStaveRec *)stave;

  Begin("StaveLeftCallback");

  if (!stave) End;

  barNo = mstave->display_start - (client ? mstave->display_number : 1);
  if (barNo == mstave->display_start) barNo = mstave->display_start - 1;
  if (barNo < -1) barNo = -1;

  StaveRefresh(stave, barNo);
  StaveScrollbarSetTop(staveScrollbar);
  StaveScrollbarSet(staveScrollbar);

  End;
}


void StaveRightCallback(Widget w, XtPointer client, XtPointer call)
{
  int            barNo;
  MajorStaveRec *mstave = (MajorStaveRec *)stave;

  Begin("StaveRightCallback");

  if (!stave) End;

  barNo = mstave->display_start + (client ? mstave->display_number : 1);
  if (barNo == mstave->display_start) barNo = mstave->display_start + 1;
  if (barNo >= mstave->bars - 1) barNo = mstave->bars - 2;

  StaveRefresh(stave, barNo);
  StaveScrollbarSetTop(staveScrollbar);
  StaveScrollbarSet(staveScrollbar);

  End;
}


void StavePageCallback(Widget w, XtPointer client, XtPointer call)
{
  int    bar;
  String label;
  String reply;

  Begin("StavePageCallback");

  if (!stave) End;

  YGetValue(w, XtNlabel, &label);

  if ((reply = YGetUserInput
       (XtParent(w), "Jump to bar number:", label + 4,
	YOrientHorizontal, NULL)) == NULL)
    End;

  bar = atoi(reply) - 1;

  if (bar > ((MajorStaveRec *)stave)->bars - 2)
      bar = ((MajorStaveRec *)stave)->bars - 2;
  if (bar <= 0) bar = -1;

  StaveRefresh(stave, bar);
  StaveScrollbarSetTop(staveScrollbar);
  StaveScrollbarSet(staveScrollbar);

  End;
}


void StaveScrollbarCallback(Widget scrollbar, XtPointer client, XtPointer call)
{
  int            barNo;
  int            move;
  Dimension      length;
  MajorStaveRec *mstave = (MajorStaveRec *)stave;

  Begin("StaveScrollbarCallback");

  if (!stave) End;

  YGetValue(scrollbar, XtNlength, &length);

  move  = mstave->display_number * ((int)call) / ((int)length);
  barNo = mstave->display_start + move;

  if (barNo == mstave->display_start)
    barNo = mstave->display_start + ((int)call > 0 ? -1 : 1);

  if (barNo >= mstave->bars - 1) barNo = mstave->bars - 2;
  if (barNo < -1) barNo = -1;

  StaveRefresh(stave, barNo);
  StaveScrollbarSetTop(staveScrollbar);
  StaveScrollbarSet(staveScrollbar);

  End;
}


void StaveJumpCallback(Widget scrollbar, XtPointer client, XtPointer call)
{
  Begin("StaveJumpCallback");

  if (stave) StaveRefreshProportion(*((float *)call));

  End;
}


void StaveInitialiseScrollbar(Widget scrollbar)
{
  Arg       arg;
  XtArgVal *ld;
  float     shown = 1.0;

  Begin("StaveInitialiseScrollbar");

  if (sizeof(float) > sizeof(XtArgVal)) XtSetArg(arg, XtNshown, &shown);
  else {
    ld = (XtArgVal *)&shown;
    XtSetArg(arg, XtNshown, *ld);
  }

  staveScrollbar = scrollbar;
  XtSetValues(scrollbar, &arg, 1);

  End;
}


