
/* Musical Notation Editor for X, Chris Cannam 1994    */
/* Class methods, excluding constructors, Draw methods */
/* and all methods within the MajorStave               */

/* Feb 95: occurrences of "this" changed to "curr" to  */
/* satisfy C++ compilers                               */

#include <Lists.h>

#include "General.h"
#include "Tags.h"
#include "Visuals.h"
#include "Classes.h"
#include "Notes.h"
#include "GC.h"
#include "ItemList.h"

Methods itemMethods = {
  GetZeroWidth,   GetLongestChord,  GetLowestNote,
  GetHighestNote, GetZeroLength,    CloneItem,
  WriteNothing,   MidiWriteNothing, DrawNothing,
  DestroyItem,  
};

Methods indicatorMethods = {
  GetZeroWidth,   GetLongestChord,  GetLowestNote,
  GetHighestNote, GetZeroLength,    CloneIndicator,
  WriteNothing,   MidiWriteNothing, DrawNothing,
  DestroyIndicator,
};

Methods metronomeMethods = {
  GetZeroWidth,   GetLongestChord,  GetLowestNote,
  GetHighestNote, GetZeroLength,    CloneMetronome,
  WriteMetronome, MidiWriteNothing, DrawMetronome,
  DestroyMetronome,
};

Methods clefMethods = {
  GetClefWidth,   GetLongestChord,  GetLowestNote,
  GetHighestNote, GetZeroLength,    CloneClef,
  WriteClef,      MidiWriteNothing, DrawClef,
  DestroyClef,  
};

Methods keyMethods = {
  GetKeyWidth,    GetLongestChord,  GetLowestNote,
  GetHighestNote, GetZeroLength,    CloneKey,
  WriteKey,       MidiWriteNothing, DrawKey,
  DestroyKey,   
};

Methods timeSigMethods = {
  GetTimeSignatureWidth, GetLongestChord,   GetLowestNote,
  GetHighestNote,        GetZeroLength,     CloneTimeSignature,
  WriteTimeSignature,    MidiWriteNothing,  DrawTimeSignature,
  DestroyTimeSignature,
};

Methods textMethods = {
  GetZeroWidth,   GetLongestChord,  GetLowestNote,
  GetHighestNote, GetZeroLength,    CloneText,
  WriteText,      MidiWriteText,    DrawText,
  DestroyText,  
};

Methods phraseMethods = {
  GetZeroWidth,   GetLongestChord,  GetLowestNote,
  GetHighestNote, GetZeroLength,    ClonePhrase,
  WriteNothing,   MidiWriteNothing, DrawNothing,
  DestroyPhrase,
};

/* rests are written to MIDI even though they're empty, so as to */
/* enable emphasis deduced from their existence, or something:   */

Methods restMethods = {
  GetRestWidth,   GetRestShortest,  GetLowestNote,
  GetHighestNote, GetRestLength,    CloneRest,
  WriteRest,      MidiWriteRest,    DrawRest,
  DestroyRest,  
};

Methods chordMethods = {
  GetChordWidth,  GetChordShortest, GetChordHighest,
  GetChordLowest, GetChordLength,   CloneChord,
  WriteChord,     MidiWriteChord,   DrawChord,
  DestroyChord,  
};

Methods groupMethods = {
  GetGroupWidth,  GetGroupShortest, GetGroupHighest,
  GetGroupLowest, GetGroupLength,   CloneGroup,
  WriteGroup,     MidiWriteNothing, DrawGroup,
  DestroyGroup,  
};

Methods barMethods = {
  GetBarWidth,    GetGroupShortest, GetGroupHighest,
  GetGroupLowest, GetGroupLength,   CloneBar,
  WriteNothing,   MidiWriteNothing, DrawBar,
  DestroyBar,
};


NoteVoice highestNoteVoice =
{
  (Pitch)20, ModSharp, ModSharp,
};

NoteVoice lowestNoteVoice =
{
  (Pitch)-10, ModFlat, ModFlat,
};

NoteVoice metronomeNoteVoice =
{
  (Pitch)3, ModNone, ModNone,
};

Chord longestChord =
{
  { ChordClass,           },
  { &chordMethods, 0,     },
  { False, False,         },
  {
    longestNoteTime,
    ModNone, &(noteVisuals[LongestNote].dotted),
    &lowestNoteVoice, (short)1, NoteModWidth,
  },
};

Chord restChord =
{
  { ChordClass,           },
  { &chordMethods, 0,     },
  { False, False,         },
  { 
    longestNoteTime,
    ModNone, &(noteVisuals[LongestNote].dotted),
    &lowestNoteVoice, (short)1, NoteModWidth,
  },
};

TimeSignature defaultTimeSignature =
{
  { TimeSignatureClass,   },
  { &timeSigMethods, 0,   },
  { ' ',                  },
  { 4, 4, 64L,            },
};

Clef defaultClef =
{
  { ClefClass,            },
  { &clefMethods, 0,      },
  { ' ',                  },
  {
    TrebleClef,
    &clefVisuals[0],
  },
};



Pitch ClefPitchOffset(ClefTag clef)
{
  Begin("ClefPitchOffset");

  switch(clef) {

  case TrebleClef: Return( 0);
  case   BassClef: Return(-2);
  case   AltoClef: Return(-1);
  case  TenorClef: Return( 1);
  }

  Return(0);
}


Dimension GetZeroWidth(MusicObject item)
{
  Begin("GetZeroWidth");
  Return(0);
}

MTime GetZeroLength(MusicObject item)
{
  Begin("GetZeroLength");
  Return(zeroTime);
}

NoteVoice *GetHighestNote(MusicObject item)
{
  Begin("GetHighestNote");
  Return(&highestNoteVoice);
}

NoteVoice *GetLowestNote(MusicObject item)
{
  Begin("GetLowestNote");
  Return (&lowestNoteVoice);
}

MusicObject GetLongestChord(MusicObject item)
{
  Begin("GetLongestChord");
  Return((MusicObject)&longestChord);
}



Dimension GetChordWidth(MusicObject chord)
{
  Begin("GetChordWidth");
  Return(chord ? (NoteWidth +
		  (((Chord *)chord)->chord.note_modifier_width) +
		  (((Chord *)chord)->chord.visual->dotted? DotWidth : 0)): 0);
}


MTime GetChordLength(MusicObject chord)
{
  Begin("GetChordLength");
  Return(chord ? ((Chord *)chord)->chord.length : zeroTime);
}


MusicObject GetChordShortest(MusicObject obj)
{
  Begin("GetChordShortest");
  Return(obj ? obj : (MusicObject)&longestChord);
}


NoteVoice *GetChordHighest(MusicObject obj)
{
  int        i;
  NoteVoice *voices = ((Chord *)obj)->chord.voices;
  NoteVoice *highest;

  Begin("GetChordHighest");

  highest = GetLowestNote((MusicObject)NULL);
  if (!obj) Return(highest);

  for (i = 0; i < ((Chord *)obj)->chord.voice_count; ++i)
    if (voices[i].pitch > highest->pitch) highest = &(voices[i]);

  Return(highest);
}


NoteVoice *GetChordLowest(MusicObject obj)
{
  int        i;
  NoteVoice *voices = ((Chord *)obj)->chord.voices;
  NoteVoice *lowest;

  Begin("GetChordLowest");

  lowest = GetHighestNote((MusicObject)NULL);
  if (!obj) Return(lowest);

  for (i = 0; i < ((Chord *)obj)->chord.voice_count; ++i)
    if (voices[i].pitch < lowest->pitch) lowest = &(voices[i]);

  Return(lowest);
}


Dimension GetClefWidth(MusicObject clef)
{
  Begin("GetClefWidth");
  Return(clef ? ClefWidth + 3 : 0);
}


Dimension GetKeyWidth(MusicObject key)
{
  Begin("GetKeyWidth");

  Return(key ? ((Key *)key)->key.visual->number * (NoteModWidth - 2) + 3 : 0);
}


Dimension GetTimeSignatureWidth(MusicObject timesig)
{
  XGCValues          values;
  static int         dir, asc, dsc = -1;
  static XCharStruct info;

  Begin("GetTimeSignatureWidth");
  
  if (dsc == -1) {
    XGetGCValues(display, timeSigGC, GCFont, &values);
    XQueryTextExtents(display, values.font, "4", 1, &dir, &asc, &dsc, &info);
    dsc = 0;
  }

  if (((TimeSignature *)timesig)->timesig.numerator   > 9 ||
      ((TimeSignature *)timesig)->timesig.denominator > 9)
    Return(2 * info.width + 8);
  else  Return(info.width + 14); /* was 8 */
}


Dimension GetRestWidth(MusicObject rest)
{
  Begin("GetRestWidth");
  Return(rest ? ((Rest *)rest)->rest.visual->width : 0);
}


MTime GetRestLength(MusicObject rest)
{
  Begin("GetRestLength");
  Return(rest ? ((Rest *)rest)->rest.length : zeroTime);
}


MusicObject GetRestShortest(MusicObject obj)
{
  Rest *rest = (Rest *)obj;

  Begin("GetRestShortest");

  if (!rest) Return((MusicObject)&longestChord);

  (void)NewChord(&restChord, &lowestNoteVoice, 0,
		 ModNone, rest->rest.visual->type, rest->rest.visual->dotted);

  Return((MusicObject)&restChord);
}


Dimension GetGroupWidth(MusicObject obj)
{
  Dimension  width = 0;
  Group     *group = (Group *)obj;
  ItemList   list;

  Begin("GetGroupWidth");

  if (!group) Return(0);

  for (ItemList_ITERATE_GROUP(list, group))
    width +=
      list->item->item.methods->get_min_width((MusicObject)(list->item));

  Return(width);
}


MTime GetGroupLength(MusicObject obj)
{
  MTime     curr;
  MTime     shortest;
  MTime     length = zeroTime;
  Group    *group  = (Group *)obj;
  Boolean   force;              		/* Triplets, &c. ? */
  ItemList  list;

  Begin("GetGroupLength");

  if (!group || (group->group.type == GroupDeGrace)) Return(zeroTime);

  shortest = longestChord.chord.length;
  force = (group->group.type == GroupTupled);
  
  for (ItemList_ITERATE_GROUP(list, group)) {

    curr   = list->item->item.methods->get_length((MusicObject)(list->item));
    length = AddMTime(length, curr);

    if (force && (MTimeLesser(curr, shortest))) shortest = curr;
  }

  if (force) {			/* This *might* handle triplet lengths */

    while (MTimeLesser(shortest, length) < 0)
      shortest = MultiplyMTime(shortest, 2);

    length = DivideMTime(shortest, 2);
  }

  Return(length);
}


MusicObject GetGroupShortest(MusicObject obj)
{
  Chord   *curr;
  Chord   *shortest;
  Group   *group = (Group *)obj;
  ItemList list;

  Begin("GetGroupShortest");

  shortest = (Chord *)GetLongestChord((MusicObject)NULL);
  if (!obj || group->group.type == GroupDeGrace) Return(shortest);

  for (ItemList_ITERATE_GROUP(list, group)) {

    curr = (Chord *)(list->item->item.methods->get_shortest
		     ((MusicObject)(list->item)));
    if (MTimeLesser(curr->chord.length, shortest->chord.length))
      shortest = curr;
  }

  Return((MusicObject)shortest);
}


NoteVoice *GetGroupHighest(MusicObject obj)
{
  NoteVoice *curr;
  NoteVoice *highest;
  Group     *group = (Group *)obj;
  ItemList   list;

  Begin("GetGroupHighest");

  highest = GetLowestNote((MusicObject)NULL);
  if (!obj || group->group.type == GroupDeGrace) Return(highest);

  for (ItemList_ITERATE_GROUP(list, group)) {

    curr = list->item->item.methods->get_highest((MusicObject)(list->item));
    if (curr->pitch > highest->pitch) highest = curr;
  }

  Return(highest);
}


NoteVoice *GetGroupLowest(MusicObject obj)
{
  NoteVoice *curr;
  NoteVoice *lowest;
  Group     *group = (Group *)obj;
  ItemList   list;

  Begin("GetGroupLowest");

  lowest = GetHighestNote((MusicObject)NULL);
  if (!obj || group->group.type == GroupDeGrace) Return(lowest);

  for (ItemList_ITERATE_GROUP(list, group)) {

    curr = list->item->item.methods->get_lowest((MusicObject)(list->item));
    if (curr->pitch < lowest->pitch) lowest = curr;
  }

  Return(lowest);
}



/* To find the minimum comfortable width of a whole bar, we need  */
/* the absolute minimum width of the bar (by "casting" it to a    */
/* Group, and getting the width of that), and then add on as much */
/* comfortable width as would be needed if the bar was made up    */
/* entirely of repetitions of its shortest note (this being the   */
/* scenario that would take most space).  This gap is the product */
/* of the number of the bar's shortest notes that will fit in the */
/* length of the bar, and the comfortable gap for each of those.  */

Dimension GetBarWidth(MusicObject obj)
{
  Bar      *bar = (Bar *)obj;
  Dimension width = 0;
  Dimension minWidth;
  ItemList  list;
  ItemList  subList;
  Chord    *curr;
  Chord    *shortest;
  MTime     length;
  MTime     totalLen;

  Begin("GetBarWidth");

  if (!bar) Return(ClefWidth + NoteWidth);              /* arbitrary */

  /* this fails to cope with the stretching that StaveRefresh now does: */
  /*  if (bar->bar.still_as_drawn) Return(bar->bar.width); */

  shortest = (Chord *)
    ((Item *)GetLongestChord((MusicObject)NULL))->item.methods->clone
      ((MusicObject)GetLongestChord((MusicObject)NULL));

  totalLen = zeroTime;

  for (ItemList_ITERATE_GROUP(list, bar)) {

    if (list->item->generic.object_class == GroupClass)
      for (ItemList_ITERATE_GROUP(subList, ((Group *)list->item))) {

	length = subList->item->item.methods->get_length
	  ((MusicObject)(subList->item));
	
	if (MTimeEqual(length, zeroTime))
	  width += subList->item->item.methods->get_min_width
	    ((MusicObject)(subList->item));
      }

    length = list->item->item.methods->get_length((MusicObject)(list->item));
    totalLen += length;

    if (MTimeEqual(length, zeroTime)) {

      width +=
	list->item->item.methods->get_min_width((MusicObject)(list->item));

    } else {

      curr = (Chord *)(list->item->item.methods->
		       get_shortest((MusicObject)(list->item)));

      if (MTimeLesser(curr->chord.length, shortest->chord.length)) {
	if (shortest) shortest->item.methods->destroy((MusicObject)shortest);
	shortest = (Chord *)curr->item.methods->clone((MusicObject)curr);
      }
    }
  }

  minWidth = shortest->item.methods->get_min_width((MusicObject)shortest);
  if (shortest->chord.note_modifier_width == 0) minWidth += NoteModWidth/2;

  width +=
  (MTimeToNumber(totalLen) *
    (shortest->chord.visual->comfortable_gap + minWidth) /
     MTimeToNumber(shortest->item.methods->get_length((MusicObject)shortest)));

  width +=
    (bar->bar.start_bar == SingleBar ? 3 :
     bar->bar.start_bar == RepeatBar ? 5 + DotWidth : 5);

  width +=
    (bar->bar.end_bar == SingleBar ? 3 :
     bar->bar.end_bar == RepeatBar ? 10 + DotWidth : 10);

  if (shortest) shortest->item.methods->destroy((MusicObject)shortest);

  Return(width + 3);
}

