
/* Musical Notation Editor for X, Chris Cannam 1994 */
/* Actions available from the Group menu            */

#include "General.h"
#include "Widgets.h"
#include "Menu.h"
#include "Stave.h"
#include "StaveEdit.h"
#include "StavePrivate.h"
#include "Yawn.h"
#include "ItemList.h"
#include "MidiOut.h"		/* for VoiceToMidiPitch */
#include "MidiIn.h"		/* for MidiPitchToVoice */

#include <X11/Xaw/Form.h>
#include <X11/Xaw/Paned.h>

void GroupMenuBeam(Widget, XtPointer, XtPointer);

static int transposeValueMaxWidth = 100;



void GroupMenuBeam(Widget w, XtPointer a, XtPointer b)
{
  MajorStaveRec *mstave  = (MajorStaveRec *)stave;
  ItemList       start   = mstave->sweep.from.left;
  ItemList       end     = mstave->sweep.to.left;
  int            staveNo = mstave->sweep.stave;

  Begin("GroupMenuBeam");

  if (!stave || !mstave->sweep.swept || !end || start == end) {

    XBell(display, 70);

  } else {

    mstave->music[staveNo] =
      ItemListExpungeGroups(mstave->music[staveNo], start, end);

    ItemListEnGroup
      (GroupBeamed,
       start ? (ItemList)Next(start) : mstave->music[staveNo], end);

    FileMenuMarkChanged(stave, True);
    StaveResetFormatting(stave, staveNo);
    StaveRefreshAsDisplayed(stave);
  }

  End;
}


void GroupMenuBreakGroup(Widget w, XtPointer a, XtPointer b)
{
  MajorStaveRec *mstave  = (MajorStaveRec *)stave;
  ItemList       start   = mstave->sweep.from.left;
  ItemList       end     = mstave->sweep.to.left;
  int            staveNo = mstave->sweep.stave;

  Begin("GroupMenuBeam");

  if (!stave || !mstave->sweep.swept || !end || start == end) {

    XBell(display, 70);

  } else {

    mstave->music[staveNo] =
      ItemListExpungeGroups(mstave->music[staveNo], start, end);

    FileMenuMarkChanged(stave, True);
    StaveResetFormatting(stave, staveNo);
    StaveRefreshAsDisplayed(stave);
  }

  End;
}


void GroupMenuChord(Widget w, XtPointer a, XtPointer b)
{
  MajorStaveRec *mstave  = (MajorStaveRec *)stave;
  ItemList       start   = mstave->sweep.from.left;
  ItemList       end     = mstave->sweep.to.left;
  int            staveNo = mstave->sweep.stave;
  int            voiceCount, i, j;
  NoteVoice     *voices;
  ItemList       list;
  Chord         *chord;
  ChordMods      mods = ModNone;
  NoteTag        tag;
  Boolean        dotted;

  Begin("GroupMenuChord");

  if (!stave || !mstave->sweep.swept || !end || start == end) {

    XBell(display, 70);

  } else {

    mstave->music[staveNo] =
      ItemListExpungeGroups(mstave->music[staveNo], start, end);

    for (voiceCount = 0, list = (ItemList)Next(start);
	 list && (ItemList)Prev(list) != end; list = (ItemList)Next(list)) {

      if (list->item->generic.object_class !=  ChordClass &&
	  list->item->generic.object_class !=   TextClass) {

	IssueMenuComplaint
	  ("You can only chord notes, not other musical objects.");

	End;
      }

      if (list->item->generic.object_class == ChordClass)
	voiceCount += ((Chord *)list->item)->chord.voice_count;
    }

    if (voiceCount == 0) {

      IssueMenuComplaint("There are no notes here to chord.");
      End;
    }

    voices = (NoteVoice *)XtMalloc(voiceCount * sizeof(NoteVoice));
    voiceCount = 0;
    list = (ItemList)Next(start);

    while (list) {

      if (list->item->generic.object_class == ChordClass) {

	chord  = (Chord *)list->item;
	mods  |= chord->chord.modifiers;
	tag    = chord->chord.visual->type;
	dotted = chord->chord.visual->dotted;

	for (i = 0; i < chord->chord.voice_count; ++i) {

	  for (j = 0; j < voiceCount; ++j)
	    if (voices[j].pitch     == chord->chord.voices[i].pitch &&
		voices[j].modifiers == chord->chord.voices[i].modifiers) break;

	  if (j >= voiceCount) voices[voiceCount++] = chord->chord.voices[i];
	}

	chord->item.methods->destroy((MusicObject)chord);

	if (list == end) { Remove(list); break; }
	else list = (ItemList)Remove(list);

      } else list = (ItemList)Next(list);
    }

    if (Next(start))
      Insert(NewItemList((Item *)NewChord(NULL, voices, voiceCount,
					  mods, tag, dotted)), Next(start));
    else
      Nconc (start, NewItemList((Item *)NewChord(NULL, voices, voiceCount,
						 mods, tag, dotted)));

    FileMenuMarkChanged(stave, True);
    StaveResetFormatting(stave, staveNo);
    StaveRefreshAsDisplayed(stave);
  }

  End;
}


void GroupMenuUnchord(Widget w, XtPointer a, XtPointer b)
{
  MajorStaveRec *mstave  = (MajorStaveRec *)stave;
  ItemList       start   = mstave->sweep.from.left;
  ItemList       end     = mstave->sweep.to.left;
  int            staveNo = mstave->sweep.stave;
  int            voiceCount, i;
  NoteVoice     *voices;
  NoteVoice     *newVoice;
  Chord         *chord;
  ChordMods      mods = ModNone;
  NoteTag        tag;
  Boolean        dotted;

  Begin("GroupMenuUnchord");

  if (!stave || !mstave->sweep.swept || !end || start == end) {

    XBell(display, 70);

  } else {

    if (start != (ItemList)Prev(end)) {
      IssueMenuComplaint("You can only unchord a single chord.");
      End;
    }

    if (end->item->generic.object_class != ChordClass) {
      IssueMenuComplaint("You can't unchord that; it's not a chord.");
      End;
    }

    chord      = (Chord *)end->item;
    voices     = chord->chord.voices;
    voiceCount = chord->chord.voice_count;
    mods       = chord->chord.modifiers;
    tag        = chord->chord.visual->type;
    dotted     = chord->chord.visual->dotted;

    for (i = 0; i < voiceCount; ++i) {

      newVoice = (NoteVoice *)XtMalloc(sizeof(NoteVoice));
     *newVoice = voices[i];

      Insert
	(NewItemList((Item *)NewChord(NULL, newVoice, 1, mods, tag, dotted)),
	 end);
    }

    chord->item.methods->destroy((MusicObject)chord);
    Remove(end);

    FileMenuMarkChanged(stave, True);
    StaveResetFormatting(stave, staveNo);
    StaveRefreshAsDisplayed(stave);
  }
}



static void Transpose(ItemList from, ItemList to, int semitones)
{
  int         mp;		/* MIDI pitch, used as intermediate repn */
  ItemList    i, j;
  short       v;
  Chord      *c;
  Key        *k  = NULL;
  static Key *k0 = NULL;
  NoteVoice   voice;

  Begin("Transpose");

  for (j = from; j; j = (ItemList)Prev(j)) /* bug fixed cc mid 94 */
    if (j->item->generic.object_class == KeyClass) k = (Key *)(j->item);

  if (!k)
    if (!k0) k = k0 = NewKey(NULL, KeyC);
    else k = k0;

  for (i = from; i; i = i ? (ItemList)Next(i) : i) {

    if (i->item->generic.object_class == ChordClass)
      for (v = 0, c = (Chord *)(i->item); v < c->chord.voice_count; ++v) {

	mp = VoiceToMidiPitch(c->chord.voices[v], TrebleClef);
	voice = MidiPitchToVoice(mp + semitones, k->key.visual->sharps);
	c->chord.voices[v] = voice;
      }

    if (i == to) i = NULL;
  }

  End;
}



static void TransposeUp(Widget w, XtPointer p, XtPointer b)
{
  int          n;
  String       temp;
  static char  buffer[24];
  Widget       value = (Widget)p;

  Begin("TransposeUp");

  YGetValue(value, XtNlabel, &temp);

  n = atoi(temp) + 1;
  if (n > 99) n = 99;

  sprintf(buffer, " %d Semitone%c ", n, (n == 1 || n == -1) ? '\0' : 's');

  XtUnmanageChild(value);

  YSetValue(value, XtNlabel, buffer);
  YSetValue(value, XtNwidth, transposeValueMaxWidth);

  XtManageChild(value);

  End;
}


static void TransposeDown(Widget w, XtPointer p, XtPointer b)
{
  int          n;
  String       temp;
  static char  buffer[24];
  Widget       value = (Widget)p;

  Begin("TransposeUp");

  YGetValue(value, XtNlabel, &temp);

  n = atoi(temp) - 1;
  if (n < -99) n = -99;

  sprintf(buffer, " %d Semitone%c ", n, (n == 1 || n == -1) ? '\0' : 's');

  XtUnmanageChild(value);

  YSetValue(value, XtNlabel, buffer);
  YSetValue(value, XtNwidth, transposeValueMaxWidth);

  XtManageChild(value);

  End;
}


void TransposeOK(Widget w, XtPointer a, XtPointer b)
{
  Begin("TransposeOK");

  *((int *)a) = 1;

  End;
}


void TransposeCancel(Widget w, XtPointer a, XtPointer b)
{
  Begin("TransposeOK");

  *((int *)a) = 0;

  End;
}


void GroupMenuTranspose(Widget w, XtPointer a, XtPointer b)
{
  Widget         tShell;
  Widget         tPane;
  Widget         tTopBox;
  Widget         tLabel;
  Widget         tValue;
  Widget         tForm;
  Widget         tUp;
  Widget         tDown;
  Widget         tBottomBox;
  Widget         tOK;
  Widget         tCancel;
  Widget         tHelp = NULL;
  XtAppContext   context;

  Dimension      w1;
  XPoint         op;
  String         temp;
  Position       px, py;
  char           title[32];
  int            i, result = -1;

  MajorStaveRec *mstave  = (MajorStaveRec *)stave;
  ItemList       start   = mstave->sweep.from.left;
  ItemList       end     = mstave->sweep.to.left;
  int            staveNo = mstave->sweep.stave;
  int            scope;

  Begin("GroupMenuTranspose");
 
  if (!mstave->sweep.swept || !start) scope = 3;  /* whole piece   */
  else if (start == end)              scope = 2;  /* rest of staff */
  else                                scope = 1;  /* swept area    */

  tShell  = XtCreatePopupShell("Transpose", transientShellWidgetClass, 
			      topLevel, NULL, 0);

  tPane   = YCreateWidget("Transpose Pane", panedWidgetClass, tShell);

  tTopBox = YCreateShadedWidget("Transpose Title Box", boxWidgetClass,
				tPane, MediumShade);

  switch(scope) {
  case 1:  sprintf(title, "Transpose Area");          break;
  case 2:  sprintf(title, "Transpose Rest of Staff"); break;
  default: sprintf(title, "Transpose Whole Piece");   break;
  }

  tLabel     = YCreateLabel(title, tTopBox);

  tForm      = YCreateShadedWidget
    ("Transpose Form", formWidgetClass, tPane, LightShade);

  tUp        = YCreateArrowButton("Up",       tForm, YArrowUp);
  tDown      = YCreateArrowButton("Down",     tForm, YArrowDown);
  tValue     = YCreateLabel(" XXX Semitones ", tForm);

  tBottomBox = YCreateShadedWidget
    ("Transpose Button Box", formWidgetClass, tPane, MediumShade);

  tOK	     = YCreateCommand("OK",     tBottomBox);
  tCancel    = YCreateCommand("Cancel", tBottomBox);

  YSetValue(XtParent(tDown) ,  XtNfromVert,     XtParent(tUp));
  YSetValue(XtParent(tValue),  XtNfromHoriz,    XtParent(tDown));
  YSetValue(XtParent(tCancel), XtNfromHoriz,    XtParent(tOK));
  YSetValue(XtParent(tValue),  XtNvertDistance, 16);

  if (appData.interlockWindow) {

    tHelp = YCreateCommand("Help", tBottomBox);
    YSetValue(XtParent(tHelp), XtNfromHoriz, XtParent(tCancel));

    XtAddCallback
      (tHelp, XtNcallback, yHelpCallbackCallback,
       (XtPointer)"Editor Edit - Transpose");
  }

  XtAddCallback(tOK,     XtNcallback, TransposeOK,     (XtPointer)&result);
  XtAddCallback(tCancel, XtNcallback, TransposeCancel, (XtPointer)&result);
  XtAddCallback(tUp,     XtNcallback, TransposeUp,     (XtPointer)tValue);
  XtAddCallback(tDown,   XtNcallback, TransposeDown,   (XtPointer)tValue);

  XtSetMappedWhenManaged(tShell, False);
  XtRealizeWidget(tShell);

  YGetValue(tTopBox, XtNwidth, &w1);
  YGetValue(tValue,  XtNwidth, &transposeValueMaxWidth);

  XtUnrealizeWidget(tShell);

  YSetValue(tValue, XtNlabel,  "0 Semitones");
  YSetValue(tValue, XtNwidth,   transposeValueMaxWidth);

  XtSetMappedWhenManaged(tShell, True);
  op = YPlacePopupAndWarp(tShell, XtGrabExclusive, tOK, tCancel);
  YAssertDialogueActions(tShell, tOK, tCancel, tHelp);

  context = XtWidgetToApplicationContext(tShell);
  while (result < 0 || XtAppPending(context))
    XtAppProcessEvent(context, XtIMAll);

  if (op.x || op.y) (void) YPopPointerPosition();

  if (result) {
    YGetValue(tValue, XtNlabel, &temp);
    result = atoi(temp);
  }

  YPopdown(tShell);
  YRetractDialogueActions(tShell);
  XtDestroyWidget(tShell);

  if (!result) End;

  switch(scope) {

  case 1:

    Transpose((ItemList)Next(start), end, result);

    mstave->formats[staveNo].next  = 0;
    mstave->formats[staveNo].time  = NULL;
    mstave->formats[staveNo].key   = NULL;
    mstave->formats[staveNo].clef  = NULL;
    mstave->formats[staveNo].items = mstave->music[staveNo];
    break;

  case 2:

    Transpose((ItemList)Next(start), NULL, result);

    mstave->formats[staveNo].next  = 0;
    mstave->formats[staveNo].time  = NULL;
    mstave->formats[staveNo].key   = NULL;
    mstave->formats[staveNo].clef  = NULL;
    mstave->formats[staveNo].items = mstave->music[staveNo];

    break;

  default:

    for (i = 0; i < mstave->staves; ++i) {

      Transpose(mstave->music[i], NULL, result);

      mstave->formats[i].next  = 0;
      mstave->formats[i].time  = NULL;
      mstave->formats[i].key   = NULL;
      mstave->formats[i].clef  = NULL;
      mstave->formats[i].items = mstave->music[i];
    }

    break;
  }

  FileMenuMarkChanged(stave, True);
  StaveRefreshAsDisplayed(stave);
  End;
}



void GroupMenuAutoBeam(Widget w, XtPointer a, XtPointer b)
{
  MajorStaveRec *mstave  = (MajorStaveRec *)stave;
  ItemList       start   = mstave->sweep.from.left;
  ItemList       end     = mstave->sweep.to.left;
  ItemList       beamS;
  int            staveNo = mstave->sweep.stave;
  int            scope;
  ItemList       timeL, last;
  TimeSignature *time;

  Begin("GroupMenuAutoBeam");

  if (!mstave->sweep.swept || !start) scope = 3;  /* whole piece   */
  else if (start == end)              scope = 2;  /* rest of staff */
  else                                scope = 1;  /* swept area    */

  if (start) beamS = (ItemList)Next(start);
     
  switch(scope) {

  case 1:

    for (timeL = start;
	 timeL && timeL->item->generic.object_class != TimeSignatureClass;
	 timeL = (ItemList)Prev(timeL));

    if (!timeL) time = &defaultTimeSignature;
    else time = (TimeSignature *)(timeL->item);

    if (beamS && beamS->item->generic.object_class == GroupClass)
      beamS = ((Group *)(beamS->item))->group.start;
    if (!beamS) break;

    mstave->music[staveNo] =
      ItemListExpungeGroups
	((ItemList)First(mstave->music[staveNo]), start, end);

    ItemListAutoBeam(time, beamS, end);

    mstave->formats[staveNo].next  = 0;
    mstave->formats[staveNo].time  = NULL;
    mstave->formats[staveNo].key   = NULL;
    mstave->formats[staveNo].clef  = NULL;
    mstave->formats[staveNo].items = mstave->music[staveNo];

    break;

  case 2:

    if (YQuery(topLevel, "Auto-beam the rest of the staff?", 2, 1, 1,
	       "Continue", "Cancel", "Group - Auto Beam")) break;

    for (timeL = start;
	 timeL && timeL->item->generic.object_class != TimeSignatureClass;
	 timeL = (ItemList)Prev(timeL));

    if (!timeL) time = &defaultTimeSignature;
    else time = (TimeSignature *)(timeL->item);

    if (beamS && beamS->item->generic.object_class == GroupClass)
      beamS = ((Group *)(beamS->item))->group.start;
    if (!beamS) break;

    mstave->music[staveNo] =
      ItemListExpungeGroups
	((ItemList)First(mstave->music[staveNo]), start,
	 last = (ItemList)Last(mstave->music[staveNo]));

    ItemListAutoBeam(time, beamS, last);

    mstave->formats[staveNo].next  = 0;
    mstave->formats[staveNo].time  = NULL;
    mstave->formats[staveNo].key   = NULL;
    mstave->formats[staveNo].clef  = NULL;
    mstave->formats[staveNo].items = mstave->music[staveNo];

    break;

  case 3:

    if (YQuery(topLevel, "Auto-beam the whole piece?", 2, 1, 1,
	       "Continue", "Cancel", "Group - Auto Beam")) break;

    for (staveNo = 0; staveNo < mstave->staves; ++staveNo) {

      mstave->music[staveNo] = (ItemList)First(mstave->music[staveNo]);

      for (timeL = mstave->music[staveNo];
	   timeL && timeL->item->generic.object_class != TimeSignatureClass;
	   timeL = (ItemList)Next(timeL));
      
      if (!timeL) time = &defaultTimeSignature;
      else time = (TimeSignature *)(timeL->item);

      mstave->music[staveNo] =
	ItemListExpungeGroups(mstave->music[staveNo], NULL,
			      last = (ItemList)Last(mstave->music[staveNo]));

      ItemListAutoBeam(time, mstave->music[staveNo], last);

      mstave->formats[staveNo].next  = 0;
      mstave->formats[staveNo].time  = NULL;
      mstave->formats[staveNo].key   = NULL;
      mstave->formats[staveNo].clef  = NULL;
      mstave->formats[staveNo].items = mstave->music[staveNo];
    }

    break;
  }

  FileMenuMarkChanged(stave, True);
  StaveRefreshAsDisplayed(stave);
  End;
}


/* Invert and Retrograde functions added JPff 94 */
      
static void Invert(ItemList from, ItemList to, int centre)
{
  int         mp;		/* MIDI pitch, used as intermediate repn */
  ItemList    i, j;
  short       v;
  Chord      *c;
  Key        *k  = NULL;
  static Key *k0 = NULL;
  NoteVoice   voice;
  Boolean     first = True;

  Begin("Invert");

  for (j = from; j; j = (ItemList)Prev(j))
    if (j->item->generic.object_class == KeyClass) k = (Key *)(j->item);

  if (!k)
    if (!k0) k = k0 = NewKey(NULL, KeyC);
    else k = k0;

  for (i = from; i; i = i ? (ItemList)Next(i) : i) {

    if (i->item->generic.object_class == ChordClass)
      for (v = 0, c = (Chord *)(i->item); v < c->chord.voice_count; ++v) {

	mp = VoiceToMidiPitch(c->chord.voices[v], TrebleClef);
	if (first) centre = centre + mp, first = False;
	voice = MidiPitchToVoice(centre + centre - mp, k->key.visual->sharps);
	c->chord.voices[v] = voice;
      }

    if (i == to) i = NULL;
  }

  End;
}

static void InvertUp(Widget w, XtPointer p, XtPointer b)
{
  int          n;
  String       temp;
  static char  buffer[24];
  Widget       value = (Widget)p;

  Begin("InvertUp");

  YGetValue(value, XtNlabel, &temp);

  n = atoi(temp) + 1;
  if (n > 99) n = 99;

  sprintf(buffer, " %d Semitone%c ", n, (n == 1 || n == -1) ? '\0' : 's');

  XtUnmanageChild(value);

  YSetValue(value, XtNlabel, buffer);
  YSetValue(value, XtNwidth, transposeValueMaxWidth);

  XtManageChild(value);

  End;
}


static void InvertDown(Widget w, XtPointer p, XtPointer b)
{
  int          n;
  String       temp;
  static char  buffer[24];
  Widget       value = (Widget)p;

  Begin("InvertUp");

  YGetValue(value, XtNlabel, &temp);

  n = atoi(temp) - 1;
  if (n < -99) n = -99;

  sprintf(buffer, " %d Semitone%c ", n, (n == 1 || n == -1) ? '\0' : 's');

  XtUnmanageChild(value);

  YSetValue(value, XtNlabel, buffer);
  YSetValue(value, XtNwidth, transposeValueMaxWidth);

  XtManageChild(value);

  End;
}


void InvertOK(Widget w, XtPointer a, XtPointer b)
{
  Begin("InvertOK");

  *((int *)a) = 1;

  End;
}


void InvertCancel(Widget w, XtPointer a, XtPointer b)
{
  Begin("InvertOK");

  *((int *)a) = 0;

  End;
}


void GroupMenuInvert(Widget w, XtPointer a, XtPointer b)
{
  Widget         tShell;
  Widget         tPane;
  Widget         tTopBox;
  Widget         tLabel;
  Widget         tValue;
  Widget         tForm;
  Widget         tUp;
  Widget         tDown;
  Widget         tBottomBox;
  Widget         tOK;
  Widget         tCancel;
  Widget         tHelp = NULL;
  XtAppContext   context;

  Dimension      w1;
  XPoint         op;
  String         temp;
  Position       px, py;
  char           title[32];
  int            i, result = -1;

  MajorStaveRec *mstave  = (MajorStaveRec *)stave;
  ItemList       start   = mstave->sweep.from.left;
  ItemList       end     = mstave->sweep.to.left;
  int            staveNo = mstave->sweep.stave;
  int            scope;

  Begin("GroupMenuInvert");
 
  if (!mstave->sweep.swept || !start) scope = 3;  /* whole piece   */
  else if (start == end)              scope = 2;  /* rest of staff */
  else                                scope = 1;  /* swept area    */

  tShell  = XtCreatePopupShell("Inversion", transientShellWidgetClass, 
			      topLevel, NULL, 0);

  tPane   = YCreateWidget("Inversion Pane", panedWidgetClass, tShell);

  tTopBox = YCreateShadedWidget("Inversion Title Box", boxWidgetClass,
				tPane, MediumShade);

  switch(scope) {
  case 1:  sprintf(title, "Invert Area around pitch");          break;
  case 2:  sprintf(title, "Invert Rest of Staff around pitch"); break;
  default: sprintf(title, "Invert Whole Piece around pitch");   break;
  }

  tLabel     = YCreateLabel(title, tTopBox);

  tForm      = YCreateShadedWidget
    ("Invert Form", formWidgetClass, tPane, LightShade);

  tUp        = YCreateArrowButton("Up",       tForm, YArrowUp);
  tDown      = YCreateArrowButton("Down",     tForm, YArrowDown);
  tValue     = YCreateLabel(" XXX Semitones ", tForm);

  tBottomBox = YCreateShadedWidget
    ("Invert Button Box", formWidgetClass, tPane, MediumShade);

  tOK	     = YCreateCommand("OK",     tBottomBox);
  tCancel    = YCreateCommand("Cancel", tBottomBox);

  YSetValue(XtParent(tDown) ,  XtNfromVert,     XtParent(tUp));
  YSetValue(XtParent(tValue),  XtNfromHoriz,    XtParent(tDown));
  YSetValue(XtParent(tCancel), XtNfromHoriz,    XtParent(tOK));
  YSetValue(XtParent(tValue),  XtNvertDistance, 16);

  if (appData.interlockWindow) {

    tHelp = YCreateCommand("Help", tBottomBox);
    YSetValue(XtParent(tHelp), XtNfromHoriz, XtParent(tCancel));

    XtAddCallback
      (tHelp, XtNcallback, yHelpCallbackCallback,
       (XtPointer)"Editor Edit - Invert");
  }

  XtAddCallback(tOK,     XtNcallback, InvertOK,     (XtPointer)&result);
  XtAddCallback(tCancel, XtNcallback, InvertCancel, (XtPointer)&result);
  XtAddCallback(tUp,     XtNcallback, InvertUp,     (XtPointer)tValue);
  XtAddCallback(tDown,   XtNcallback, InvertDown,   (XtPointer)tValue);

  XtSetMappedWhenManaged(tShell, False);
  XtRealizeWidget(tShell);

  YGetValue(tTopBox, XtNwidth, &w1);
  YGetValue(tValue,  XtNwidth, &transposeValueMaxWidth);

  XtUnrealizeWidget(tShell);

  YSetValue(tValue, XtNlabel,  "0 Semitones");
  YSetValue(tValue, XtNwidth,   transposeValueMaxWidth);

  XtSetMappedWhenManaged(tShell, True);
  op = YPlacePopupAndWarp(tShell, XtGrabExclusive, tOK, tCancel);
  YAssertDialogueActions(tShell, tOK, tCancel, tHelp);

  context = XtWidgetToApplicationContext(tShell);
  while (result < 0 || XtAppPending(context))
    XtAppProcessEvent(context, XtIMAll);

  if (op.x || op.y) (void) YPopPointerPosition();

  if (result) {
    YGetValue(tValue, XtNlabel, &temp);
    i = atoi(temp);
  }

  YPopdown(tShell);
  YRetractDialogueActions(tShell);
  XtDestroyWidget(tShell);

  if (!result) End;

  result = i;

  switch(scope) {

  case 1:

    Invert((ItemList)Next(start), end, result);

    mstave->formats[staveNo].next  = 0;
    mstave->formats[staveNo].time  = NULL;
    mstave->formats[staveNo].key   = NULL;
    mstave->formats[staveNo].clef  = NULL;
    mstave->formats[staveNo].items = mstave->music[staveNo];
    break;

  case 2:

    Invert((ItemList)Next(start), NULL, result);

    mstave->formats[staveNo].next  = 0;
    mstave->formats[staveNo].time  = NULL;
    mstave->formats[staveNo].key   = NULL;
    mstave->formats[staveNo].clef  = NULL;
    mstave->formats[staveNo].items = mstave->music[staveNo];

    break;

  default:

    for (i = 0; i < mstave->staves; ++i) {

      Invert(mstave->music[i], NULL, result);

      mstave->formats[i].next  = 0;
      mstave->formats[i].time  = NULL;
      mstave->formats[i].key   = NULL;
      mstave->formats[i].clef  = NULL;
      mstave->formats[i].items = mstave->music[i];
    }

    break;
  }

  FileMenuMarkChanged(stave, True);
  StaveRefreshAsDisplayed(stave);
  End;
}


static ItemList tempboard = NULL;

void GroupMenuRetrograde(Widget w, XtPointer a, XtPointer b)
{
  ItemList       list;
  Item          *newItem;
  MajorStaveRec *mstave  = (MajorStaveRec *)stave;
  ItemList       start   = mstave->sweep.from.left;
  ItemList       end     = mstave->sweep.to.left;
  int            staveNo = mstave->sweep.stave;

  Begin("GroupMenuRetrograde");

  if (!stave || !mstave->sweep.swept || !start || start==end) {
    XBell(display, 70);
    End;
  }

  tempboard = NULL;

  if (start) list = (ItemList)Next(start);
  else       list = mstave->music[mstave->sweep.stave];

  while (list) {

    if (list->item->generic.object_class == GroupClass) {

      if (list == end) break;
      list = (ItemList)Next(list);
      continue;
    }

    newItem = (Item *)list->item->item.methods->clone((MusicObject)list->item);
    tempboard = (ItemList)Nconc(NewItemList(newItem), tempboard);

    if (list == end) break;
    list = (ItemList)Next(list);
  }

  EditMenuDelete(w, a, b);

  if (start) {
    if (Next(start)) Insert(tempboard, Next(start));
    else Nconc(start, tempboard);
  }
  else start = (ItemList)Nconc(tempboard, start);

  FileMenuMarkChanged(stave, True);
  StaveResetFormatting(stave, staveNo);
  StaveRefreshAsDisplayed(stave);
  End;
}

