
/* Musical Notation Editor for X, Chris Cannam 1994       */
/* Functions for dealing with manipulations of ItemLists. */
/* This is not as easy as just dealing with Lists; the    */
/* presence of Groups within the ItemList, with their     */
/* accompanying pointer inconsistency, makes things much  */
/* more complicated.                                      */

#include "Classes.h"
#include "Lists.h"
#include "ItemList.h"



/* ItemListEnsureIntegrity: call this on a newly-read itemlist in  */
/* format A.  These are lists in which groups' itemlists are not   */
/* sublists of the main lists, but instead are complete lists with */
/* pointers to NULL at both ends. EnsureItemListIntegrity converts */
/* these to format B, with the groups' itemlists as sublists.      */

ItemList ItemListEnsureIntegrity(ItemList i)
{
  ItemList rtn;

  Begin("ItemListEnsureIntegrity");

  i = rtn = (ItemList)First(i);

  while (i) {

    if (i->item->generic.object_class == GroupClass) {

      if (Next(i)) {

	i = (ItemList)Insert(((Group *)i->item)->group.start, Next(i));

      } else {

	Nconc(i, ((Group *)i->item)->group.start);
	i = (ItemList)Next(i);
      }

    } else i = (ItemList)Next(i);
  }

  Return(rtn);
}
  
  
/* Pass grouping type and first and last items in prospective */
/* group's item list.  Returns new (group) item in list.      */

ItemList ItemListEnGroup(GroupTag type, ItemList begin, ItemList end)
{
  Begin("ItemListEnGroup");

  if (!begin || !end)
    Error("ItemListEnGroup called with invalid argument");

  Return((ItemList)Insert
	 (NewItemList((Item *)NewGroup(NULL, type, begin, end)), begin));
}


/* Pass group item in list.  Returns list that was first in */
/* group's itemlist.                                        */

ItemList ItemListUnGroup(ItemList groupList)
{
  ItemList last;
  Group   *group;

  Begin("ItemListUnGroup");

  if (groupList->item->generic.object_class != GroupClass)
    Error("ItemListUnGroup called with invalid argument");

  groupList->item->item.methods->destroy((MusicObject)groupList->item);
  Return((ItemList)Remove(groupList));
}



/* This deals with group references in a sublist.  It ensures    */
/* that no group references any object within the sublist, and   */
/* that the sublist contains no group objects (any previously    */
/* beginning within the sublist being moved to just after).      */

/* start and end are the `left' items; start is the item to the  */
/* left of the first item, end the item to the left of the last. */
/* Returns first item in the whole list (which may have changed).*/

ItemList ItemListExpungeGroups(ItemList first, ItemList start, ItemList end)
{
  ItemList list;
  ItemList temp;
  ItemList groupl = NULL;
  ItemList groupa = NULL;
  ItemList groupz = NULL;
  Group   *tempGroup;

  Begin("ItemListExpungeGroups");
  
  /* How do we deal with groups in cutting, &c.?  If any groups start */
  /* and end in the cut section, they are lost.  If any start before  */
  /* and end during, they are truncated.  If any start during and end */
  /* after, they are truncated.  If any start before and end after,   */
  /* their middle sections are (trivially) swiped.                    */

  /* We only need look for the nearest group in either direction,    */
  /* because groups can't contain groups.                            */

  for (list = start; list; list = (ItemList)Prev(list)) {
    if (list->item->generic.object_class == GroupClass) {
      groupl = list;
      groupa = ((Group *)list->item)->group.start;
      groupz = ((Group *)list->item)->group.end;
      break;
    }
  }

  if (start) list = (ItemList)Next(start);
  else list = first;

  while (list) {

    if (list == groupa)	{	   /* group item before, starts within? Zap */
      ItemListUnGroup(groupl);
      groupl = groupa = groupz = NULL;
      list = (ItemList)Next(list);
      continue;
    }

    if (list == groupz) {	   /* starts before, ends within?  Truncate */
      ((Group *)groupl->item)->group.end = start;
      groupl = groupa = groupz = NULL;
      list = (ItemList)Next(list);
      continue;
    }

    if (list->item->generic.object_class == GroupClass) { /* starts within? */

      if (list == first) first = (ItemList)Next(list);

      for (temp = ((Group *)list->item)->group.start;
	   temp; temp = (ItemList)Next(temp))

	if (temp == ((Group *)list->item)->group.end) {

	  list = ItemListUnGroup(list);         /* ... and ends within? Zap */
	  break;

	} else if (temp == end && Next(temp)) {       /* ... ends without?  */

	  if (Next(end)) ItemListEnGroup
	    (((Group *)list->item)->group.type, (ItemList)Next(end),
	     ((Group *)list->item)->group.end);

	  list = ItemListUnGroup(list);
	  break;

	} else if (temp == end) {
	
	  list = ItemListUnGroup(list);
	  break;
	}

    } else list = (ItemList)Next(list);

    if (list && Prev(list) && end == (ItemList)Prev(list)) break;
  }

  Return(first);
}



/* Auto-Beaming Heuristic:                                   */
/*                                                           */
/* First, we need the number of units to try to beam in.  If */
/* we're in 4/4, we probably beam quavers in twos (and semis */
/* in fours and so on); if 6/8, we beam quavers in threes;   */
/* if 2/4, in twos again; if something weird (5/8? 7/8?) we  */
/* probably just beam in fives or sevens.  Seems we just     */
/* want the lowest prime divisor for the number of quavers,  */
/* and to take semis and so on from there.                   */
/*                                                           */
/* Then we skip through the section, accumulating times.     */
/* On finding that the accumulated time is a multiple of the */
/* length of a beamed group (two quavers in 4/4, three in    */
/* 6/8, &c.), we check to see if there's a note ending       */
/* cleanly at the end of another beamed group's length, and  */
/* if so, and if there are at least two beamable items in    */
/* between here and there, we beam them up.  (...Scotty)     */
/*                                                           */
/* This returns the item list; if `start' was the first item */
/* and has been engrouped, the return value will be the new  */
/* group item; otherwise it'll just be `start'.              */

ItemList ItemListAutoBeamSub(MTime avg, MTime min, MTime max,
			     ItemList start, ItemList end, MTime barLength)
{
  MTime    accumulator;		/* ah, for the days of *real* CPUs */
  MTime    prospective;
  MTime    temp;
  MTime    tmin;
  MTime    beamLength;
  ClassTag oclass;
  MTime    current;
  int      beamable;
  int      longerThanDemi;
  ItemList i, j, k;
  ItemList rtn;
  
  Begin("ItemList");

  rtn = start;

  for (i = start, accumulator = zeroTime;
       i && ((ItemList)Prev(i) != end); i = (ItemList)Next(i)) {

    oclass = i->item->generic.object_class;

    if (oclass == GroupClass) continue;

    if (MTimeToNumber(accumulator) % MTimeToNumber(barLength) == 0)
      accumulator = zeroTime;

    if ((MTimeToNumber(accumulator) % MTimeToNumber(avg) == 0) &&
	(oclass == ChordClass || oclass ==  RestClass) &&
	(i->item->item.methods->get_length((MusicObject)i->item) <
	 TagToMTime(Crotchet, False))) {

      k              = NULL;
      tmin           = min;
      temp           = zeroTime;
      beamable       = 0;
      longerThanDemi = 0;

      for (j = i; j && ((ItemList)Prev(j) != end); j = (ItemList)Next(j)) {

	oclass  = j->item->generic.object_class;
	current = j->item->item.methods->get_length((MusicObject)j->item);

	if (oclass == ChordClass) {

	  if (((Chord *)j->item)->chord.visual->type < Crotchet) ++beamable;
	  if (((Chord *)j->item)->chord.visual->type >= Semiquaver)
	    ++longerThanDemi;
	}

	temp = AddMTime(temp, current);

	if (MTimeGreater(temp, zeroTime) &&
	    MTimeToNumber(temp) % MTimeToNumber(tmin) == 0) {
	  k = j;
	  beamLength = temp;

	  /* Will this group have crossed a bar line?  If so, wrap */
	  /* around the prospective accumulator from the bar mark  */

	  prospective =
	    NumberToMTime(MTimeToNumber(AddMTime(accumulator, temp)) %
			  MTimeToNumber(barLength));

	  tmin = NumberToMTime(2 * MTimeToNumber(tmin));
	}

	/* Decide to stop scanning for items to join this beamed group. */
	/* We stop if we've got the maximum length of beamed group, if  */
	/* we've got more than 4 semis or quavers, if we're at the end  */
	/* of the piece or of a bar, if there's a rest ahead or if the  */
	/* next chord is longer than the current one.                   */

	/* All this, of course, results is an absolutely magnificent    */
	/* conditional.  Ha!  I spurn your puny temporary variables.    */

	if (!MTimeLesser(temp, max) || longerThanDemi >= 4 || !Next(j) ||
	    (k && !MTimeToNumber(prospective)) ||
	    ((ItemList)Next(j))->item->generic.object_class == RestClass ||
	    (oclass == ChordClass &&
	     ((ItemList)Next(j))->item->generic.object_class == ChordClass &&
	     MTimeGreater
	     (((ItemList)Next(j))->item->item.methods->get_length
	      ((MusicObject)((ItemList)Next(j))->item), current))) {

	  if (k && beamable >= 2) {

	    if (i == start && !Prev(i))
	      rtn = ItemListEnGroup(GroupBeamed, i, k);
	    else
	      (void)ItemListEnGroup(GroupBeamed, i, k);
	  }

	  /* If this group is longer than the check threshold (`avg'), */
	  /* its length must be a multiple of the threshold and hence  */
	  /* we can keep scanning from the end of the group without    */
	  /* losing the modulo properties of the accumulator.  Other-  */
	  /* wise, we continue from where we were.  (The latter action */
	  /* must be safe -- we can't get another group starting half- */
	  /* way through the last one, because we know the last one    */
	  /* is shorter than the group start check threshold.)         */

	  if (k && !MTimeLesser(beamLength, avg)) {

	    i = k;
	    accumulator = prospective;

	  } else {

	    accumulator = AddMTime
	      (accumulator,
	       i->item->item.methods->get_length((MusicObject)i->item));
	  }

	  break;
	}
      }
    } else {

      accumulator = AddMTime
	(accumulator,
	 i->item->item.methods->get_length((MusicObject)i->item));
    }
  }

  Return(rtn);
}


ItemList ItemListAutoBeam(TimeSignature *time, ItemList start, ItemList end)
{
  int   num;
  MTime avg;
  MTime min;
  MTime barLength;

  Begin("ItemList");

  if (time) {

    barLength = time->timesig.bar_length;

    /* Minimal number for grouping should ideally be smallest prime */
    /* divisor of bar's numerator.  We'll relax the "prime" bit,    */
    /* but ensure that it's at least 2.                             */

    /* Later comment: no, this isn't true.  If the denominator is   */
    /* 2 or 4, we should always beam in twos (in 3/4, 6/2 or stng.) */

    /* (Actually this isn't right either.  I think if the numerator */
    /* is prime, perhaps, we should beam up to the length of one    */
    /* beat at a time?  Something like that, anyway.  Leave it for  */
    /* a rainy day sometime.)                                       */

    if (time->timesig.denominator == 2 ||
	time->timesig.denominator == 4) {

      if (time->timesig.numerator % 3) {

	avg = NumberToMTime(TagToNumber(Quaver,     False));
	min = NumberToMTime(TagToNumber(Semiquaver, False));

      } else avg = min = NumberToMTime(TagToNumber(Semiquaver, False));

    } else {

      /* Special hack for 6/8.  This is getting dodgier by the minute */

      if (time->timesig.numerator   == 6 &&
	  time->timesig.denominator == 8) {

	avg = NumberToMTime(3 * TagToNumber(Quaver, False));
	min = NumberToMTime(MTimeToNumber(avg) / 2);

      } else {

	for (num = 2; time->timesig.numerator % num != 0; ++num);
	
	avg = NumberToMTime(num * TagToNumber(Semiquaver, False));
	min = NumberToMTime(MTimeToNumber(avg) / 2);
      }

      /* older code, works okay for most time signatures:

	 avg =
	 ((time->timesig.numerator== 4) ? (2 * TagToMTime(Semiquaver, False)) :
	 (time->timesig.numerator == 6) ? (3 * TagToMTime(Semiquaver, False)) :
	 (time->timesig.numerator * TagToMTime(Semiquaver, False)));

	 min =
	 ((time->timesig.numerator== 4) ? TagToMTime(Semiquaver, False) :
	 (time->timesig.numerator == 6) ? TagToMTime(Semiquaver,  True) :
	 TagToMTime(Semiquaver, False));
      */
    }
  } else {

    barLength = 4 * TagToMTime(Crotchet, False);

    avg = TagToMTime(Quaver, False);
    min = TagToMTime(Semiquaver, False);
  }

  if (time->timesig.denominator > 4) avg /= 2;

  Return(ItemListAutoBeamSub(avg, min,
			     NumberToMTime(4 * MTimeToNumber(avg)),
			     start, end, barLength));
}

