/* PVTool - Copyright 1996 Richard W.E. Furse */

#include <stdio.h>
#include <stdlib.h>

#include "pvoc.h"
#define EXCLUDE_EXTERNALS 1
#include "top.x"

int bins;
float *input_buffer[2];
int input_buffer_frames[2];
float *output_buffer;
int output_buffer_frames = 0;
int resize (int frames);
float fparameter[MAX_NUMBER_OF_PARAMETERS];
char *parameter[MAX_NUMBER_OF_PARAMETERS];
int parameter_count;
char *input_filename_ptr[2];

PVSTRUCT *phdr1, *phdr2, *phdr_out = NULL;

char * static_version_string = "VERSION: " VERSION_STRING;

extern char *programName;
char error_space[MAX_ERROR_LENGTH];

void
print_pv (PVSTRUCT * phdr)
{
  printf ("magic = %ld\n", phdr->magic);
  printf ("headBsize = %ld\n", phdr->headBsize);
  printf ("dataBsize = %ld\n", phdr->dataBsize);
  printf ("dataFormat = %ld\n", phdr->dataFormat);
  printf ("samplingRate = %f\n", phdr->samplingRate);
  printf ("channels = %ld\n", phdr->channels);
  printf ("frameSize = %ld\n", phdr->frameSize);
  printf ("frameIncr = %ld\n", phdr->frameIncr);
  printf ("frameBsize = %ld\n", phdr->frameBsize);
  printf ("frameFormat = %ld\n", phdr->frameFormat);
  printf ("minFreq = %f\n", phdr->minFreq);
  printf ("maxFreq = %f\n", phdr->maxFreq);
  printf ("freqFormat = %ld\n", phdr->freqFormat);
}

float 
bin_to_freq (int bin)
{
  /* This always reads the first file. If used with pvanal etc this
     aint a problem. */
  return (phdr1->minFreq
	  + (phdr1->maxFreq - phdr1->minFreq)
	  * (0.5 + (float) bin)
	  / (float) bins);
}

float 
sampling_rate (void)
{
  return (phdr1->samplingRate);
}

char *
input_filename (int input_number)
{
  return (input_filename_ptr[input_number - 1]);
}

float 
sound_length (int input_number)
{
  PVSTRUCT *phdr;
  float frames, new_samples_per_frame, frame_samples;

  if (input_number == 2)
    phdr = phdr2;
  else
    phdr = phdr1;

  frames = (float) input_buffer_frames[input_number - 1];
  new_samples_per_frame = (float) phdr->frameIncr;
  frame_samples = (float) phdr->frameSize;

  return ((frame_samples + (frames - 1) * new_samples_per_frame)
	  / phdr->samplingRate);
}

int
pvtool_error (char *message)
{
  fprintf (stderr, "%s: %s\n", programName, message);
  return (-1);
}

int
pvtool_warning (char *message)
{
  fprintf (stderr, "%s: %s\n", programName, message);
  return (0);
}

int
resize (int frames)
{
  int err = 0;
  if (phdr_out != NULL)
    PVFree (phdr_out);
  err = PVAlloc (&phdr_out,
		 sizeof (float) * 2 * (bins + 1) * frames,
		 phdr1->dataFormat,
		 phdr1->samplingRate,
		 phdr1->channels,
		 phdr1->frameSize,
		 phdr1->frameIncr,
		 phdr1->frameBsize,
		 phdr1->frameFormat,
		 phdr1->minFreq,
		 phdr1->maxFreq,
		 phdr1->freqFormat,
		 phdr1->headBsize - sizeof (PVSTRUCT) + PVDFLTBYTS);
  if (!err)
    {
      output_buffer = (float *) (phdr_out->headBsize + (char *) phdr_out);
      output_buffer_frames = frames;
    }
  return (err);
}

int
check_ok_format (PVSTRUCT * phdr)
{
  int err = 0;
  if (phdr->magic != PVMAGIC)
    err = pvtool_error
      ("Input_Buffer file does not carry correct magic number.");

  if (phdr->frameFormat != PVPVOC)
    err = pvtool_error
      ("Input_Buffer phase vocoder analysis file does not use the PVPVOC format.\n");

  if (phdr->dataFormat != PVFLOAT)
    err = pvtool_error
      ("Input_Buffer phase vocoder analysis file does not use floats.\n");

  if (phdr->channels != 1)
    err = pvtool_error
      ("Input_Buffer phase vocoder analysis file must have exactly one channel.\n");

  if (phdr->freqFormat != PVLIN)
    err = pvtool_error
      ("Input_Buffer frequency format is not linear.\n");

  return (err);
}

int
pvmain (int argc,
	char *argv[],
	int double_input,
	int extra_parameters,
	int usage_message_provided,
	int no_output)
     /* Top level interface to library code */
{
  int err = 0, i;
  int target_output_buffer_size;

  input_filename_ptr[0] = argv[1];
  input_filename_ptr[1] = argv[2];
  /* Check parameter count makes sense: */
  if (argc < 3 + double_input - no_output
      || (extra_parameters >= 0
	  && argc != (3 + double_input - no_output
		      + extra_parameters)))
    {
      /* Display a usage message of some kind and stop */
      err = 1;
      printf ("%s is a pvtool " VERSION_STRING " utility.\n",
	      programName);
      if (usage_message_provided)
	usage_message ();
      else
	{
	  /* Basic usage message if code doesn't have one: */
	  printf ("Basic usage:\n");
	  printf ("\n\t%s <pv input file>", programName);
	  if (double_input)
	    printf (" <pv input file>");
	  if (!no_output)
	    printf (" <pv output file>");
	  if (extra_parameters == -1)
	    printf (" [<further parameters>]");
	  else
	    for (i = 0; i < extra_parameters; i++)
	      printf (" <number>");
	  printf ("\n\n");
	}
    }
  else
    {
      /* Store parameters to feed to pvaction(): */
      parameter_count = argc - (3 + double_input - no_output);
      for (i = 0; i < parameter_count; i++)
	{
	  sscanf (argv[i + (3 + double_input - no_output)],
		  "%f",
		  &(fparameter[i]));
	  parameter[i] = argv[i + (3 + double_input - no_output)];
	}
    }

  if (!err)
    {
      /* Read in first input_buffer file: */
#ifdef DEBUG
      printf ("Loading pv input file %s.\n", argv[1]);
#endif
      if (PVReadFile (argv[1], &phdr1) != 0)
	{
	  sprintf (error_space, "Failed to open input file %s.", argv[1]);
	  err = pvtool_error (error_space);
	}
      else
	err = check_ok_format (phdr1);
    }

  if (!err && double_input)
    {
      /* Read in second input_buffer file: */
#ifdef DEBUG
      printf ("Loading pv input file %s.\n", argv[2]);
#endif
      if (PVReadFile (argv[2], &phdr2) != 0)
	{
	  sprintf (error_space, "Failed to open input file %s.", argv[2]);
	  err = pvtool_error
	    (error_space);
	}
      else
	err = check_ok_format (phdr2);
    }

  if (!err)
    {
      /* Work out how many frequency bins are in use: */
      bins = (phdr1->frameSize) / 2;
      if (double_input)
	{
	  if ((phdr2->frameSize / 2) != bins)
	    err = pvtool_error
	      ("Input files have conflicting numbers of frequency bins.");
	  if (phdr1->samplingRate != phdr2->samplingRate)
	    err = pvtool_warning
	      ("Input files have different sampling rates. Taking first.");
	}
    }

  if (!err)
    {
      /* Set up input_buffer/output_buffer buffers and relevant variables for
         pvaction(): */
      /* Set up input buffer 1: */
      input_buffer_frames[0] = (phdr1->dataBsize /
				((bins + 1) * 2 * sizeof (float)));
      input_buffer[0] = (float *) (phdr1->headBsize + (char *) phdr1);
#ifdef DEBUG
      if (double_input)
	printf ("Input_Buffer file 1 has frame count %ld.\n",
		input_buffer_frames[0]);
      else
	printf ("Input_Buffer file has frame count %ld.\n",
		input_buffer_frames[0]);
#endif
      target_output_buffer_size = input_buffer_frames[0];

      if (double_input)
	{
	  /* Set up input_buffer buffer 2: */
	  input_buffer_frames[1] = (phdr2->dataBsize /
				    ((bins + 1) * 2 * sizeof (float)));
	  input_buffer[1] = (float *) (phdr2->headBsize + (char *) phdr2);
#ifdef DEBUG
	  printf ("Input_Buffer file 2 has frame count %ld.\n",
		  input_buffer_frames[1]);
#endif
	  if (input_buffer_frames[1] < target_output_buffer_size)
	    target_output_buffer_size = input_buffer_frames[1];
	}

      if (!no_output)
	{
	  /* Set up output_buffer buffer: */
	  resize (target_output_buffer_size);
	}
    }

  if (!err)
    {
      /* Call pvaction(): */
#ifdef DEBUG
      printf ("Processing pv data\n");
#endif
      err = pvaction ();
    }

  if (!err && !no_output)
    {
      /* Write output file: */
#ifdef DEBUG
      printf ("Output file has frame count %ld.\n", output_buffer_frames);
      printf ("Writing output_buffer pv file %s.\n",
	      argv[2 + double_input - no_output]);
#endif
      if (PVWriteFile (argv[2 + double_input - no_output], phdr_out) != 0)
	{
	  sprintf (error_space,
		   "Failed to open value file %s.",
		   argv[2 + double_input]);
	  err = pvtool_error (error_space);
	}
    }

  /* Return error level: */
  return (err);
}


float
input_mag (int input_buffer_number, int frame, int channel)
{
  float value;
  int err = 0;

  if (input_buffer_number < 1 || input_buffer_number > 2 ||
      frame < 0 || frame >= input_buffer_frames[input_buffer_number - 1] ||
      channel < 0 || channel >= bins)
    {
      sprintf (error_space,
	       "input_buffer_mag(%d,%d,%d) has parameters beyond range.",
	       input_buffer_number, frame, channel);
      err = pvtool_error (error_space);
    }

  if (err)
    value = 0;
  else
    value = *(input_buffer[input_buffer_number - 1]
	      + ((bins + 1) * (frame) + channel) * 2);

  return (value);
}


float
input_phidot (int input_buffer_number, int frame, int channel)
{
  float value;
  int err = 0;

  if (input_buffer_number < 1 || input_buffer_number > 2 ||
      frame < 0 || frame >= input_buffer_frames[input_buffer_number - 1] ||
      channel < 0 || channel >= bins)
    {
      sprintf (error_space,
	       "input_buffer_phidot(%d,%d,%d) has parameters beyond range.",
	       input_buffer_number, frame, channel);
      err = pvtool_error (error_space);
    }

  if (err)
    value = 0;
  else
    value = *(input_buffer[input_buffer_number - 1]
	      + ((bins + 1) * (frame) + channel) * 2 + 1);

  return (value);
}


float *
output_mag (int frame, int channel)
{
  float *value;
  int err = 0;

  if (frame < 0 || frame >= output_buffer_frames ||
      channel < 0 || channel >= bins)
    {
      sprintf (error_space,
	       "output_mag(%d,%d) has parameters beyond range.",
	       frame, channel);
      err = pvtool_error (error_space);
    }

  if (err)
    value = NULL;
  else
    value = output_buffer + ((bins + 1) * (frame) + channel) * 2;

  return (value);
}


float *
output_phidot (int frame, int channel)
{
  float *value;
  int err = 0;

  if (frame < 0 || frame >= output_buffer_frames ||
      channel < 0 || channel >= bins)
    {
      sprintf (error_space,
	       "output_phidot(%d,%d) has parameters beyond range.",
	       frame, channel);
      err = pvtool_error (error_space);
    }

  if (err)
    value = NULL;
  else
    value = output_buffer + ((bins + 1) * (frame) + channel) * 2 + 1;

  return (value);
}
